/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.spi;

import java.util.Collection;

import org.jboss.deployers.spi.management.ManagementView;
import org.jboss.deployers.spi.management.deploy.DeploymentManager;

/**
 * The entry point service for accessing/administering server profiles
 *  
 * @todo this should be broken up into different feature plugin services
 * 
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public interface ProfileService
{
   
   /**
    * Get the unique domains for which profiles exist.
    * 
    * @return array of profile domains
    */
   public String[] getDomains();

   /**
    * Get the keys for all registered profiles.
    * 
    * @return keys for all registered profiles.
    */
   public Collection<ProfileKey> getProfileKeys();

   /**
    * Obtain the registered profile for the key.
    * 
    * @param key - the key for the profile
    * @return the matching profile.
    * @throws NoSuchProfileException
    */
   public Profile getProfile(ProfileKey key)
      throws NoSuchProfileException;

   /**
    * Get the keys for all active profiles.
    * 
    * @return keys for all active profiles.
    */
   public Collection<ProfileKey> getActiveProfileKeys();
   
   /**
    * Obtain a activated profile for the key.
    * 
    * @param key - the key for the profile
    * @return the active profile
    * @throws NoSuchProfileException thrown if there is no
    *    active profile 
    */
   public Profile getActiveProfile(ProfileKey key)
      throws NoSuchProfileException;
   
   /**
    * Obtain the ManagementView plugin
    * @return the ManagementView plugin if supported
    */
   public ManagementView getViewManager();

   /**
    * Obtain the DeploymentManager plugin
    * @return the DeploymentManager plugin if supported
    */
   public DeploymentManager getDeploymentManager();
   
   /**
    * Register a Profile.
    * 
    * @param profile the profile.
    * @throws Exception
    */
   public void registerProfile(Profile profile) throws Exception;
   
   /**
    * Activate a registered profile.
    * 
    * @param key the profile key.
    * @throws NoSuchProfileException if the profile is not registered. 
    * @throws Exception
    */
   public void activateProfile(ProfileKey key) throws NoSuchProfileException, Exception;
   
   /**
    * Release the profile.
    * 
    * @param key the profile key.
    * @throws NoSuchProfileException if the profile is not active.
    */
   public void releaseProfile(ProfileKey key) throws NoSuchProfileException;
   

   /**
    * Unregister a profile.
    * 
    * @param profile
    * @throws NoSuchProfileException if the profile is not registered.
    */
   public void unregisterProfile(Profile profile) throws NoSuchProfileException;

}
