/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.profileservice.spi;

import java.util.Collection;
import java.util.Set;

/**
 * A profile represents a named collection of deployments on a server
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public interface Profile
{
   /**
    * Get the key used to create the Profile
    * @return key used to create the Profile
    */
   ProfileKey getKey();

   /**
    * Get the system time in milliseconds the profile was last modified. 
    * @return System.currentTimeMillis of last modification
    */
   long getLastModified();
   
   /**
    * Internally create the profile.
    * 
    * @throws Exception
    */
   void create() throws Exception;

   /**
    * Internally destroy the profile.
    */
   void destroy();
   
   /**
    * Get the profile dependencies.
    * 
    * @return the collection of profile keys.
    */
   Collection<ProfileKey> getSubProfiles();
   
   /**
    * Get the names of the deployments in the profile
    * @return names of deployments
    */
   Set<String> getDeploymentNames();

   /**
    * Add a deployment
    * 
    * @param d the deployment
    * deployment is loaded
    * @throws Exception for any error
    */
   void addDeployment(ProfileDeployment d) throws Exception;

   /**
    * Remove a deployment
    * 
    * @param name the deployment name
    * @return the DeploymentContext for the name if found
    * @throws Exception for any error
    */
   ProfileDeployment removeDeployment(String name) throws Exception;

   /**
    * Get a named deployment.
    * 
    * @param name - the deployment name
    * @return the named bootstrap
    * @throws Exception for any error
    * @throws NoSuchDeploymentException - if there is no such bootstrap
    */
   ProfileDeployment getDeployment(String name) throws Exception, NoSuchDeploymentException;

   /**
    * Get all deployments defined in this profile
    * 
    * @return the deployment instances in this profile.
    * @throws Exception for any error
    */
   Collection<ProfileDeployment> getDeployments() throws Exception;

   /**
    * Get the deployments that have been modified since the last
    * check. 
    * 
    * @return the modified deployments, empty if none exist or
    * the profile does not support hot deployments.
    * 
    * @throws Exception - thrown on error scanning for deployments
    */
   Collection<ModificationInfo> getModifiedDeployments() throws Exception;

   /**
    * Enable/disable the getModifiedDeployments results. This can be
    * used to disable the getModifiedDeployments results during
    * periods such as bootstrap where changes should be delayed.
    * @see #getModifiedDeployments
    * @param flag - the enable/disable flag
    */
   void enableModifiedDeploymentChecks(boolean flag);

   /**
    * Checks whether a deployment context is available in the basic profile service.
    * 
    * @param    name    the name of the deployment
    * @return   true if the deployment is found or false otherwise
    */
   boolean hasDeployment(String name);
   
}
