/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.spi;

import java.io.Serializable;

/**
 * The key for a Profile. It consists of the domain, server and name.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class ProfileKey
   implements Comparable, Serializable
{
   private static final long serialVersionUID = 1;
   /** The DEFAULT value for domain, server, name */
   public static final String DEFAULT = "default";

   /** The profile domain/cluster */
   private String domain;
   /** The server/node */
   private String server;
   /** The profile name */
   private String name;

   /**
    * Calls this this(DEFAULT, DEFAULT, name)
    * @param name - the profile name
    */
   public ProfileKey(String name)
   {
      this(DEFAULT, DEFAULT, name);
   }

   /**
    * Build a profile key from the domain, server and name. If any parameter
    * is null it defaulted to "default".
    * 
    * @param domain - the admin domain
    * @param server - the server instance name
    * @param name - the profile name
    */
   public ProfileKey(String domain, String server, String name)
   {
      if( domain == null )
         domain = DEFAULT;
      this.domain = domain;
      if( server == null )
         server = DEFAULT;
      this.server = server;
      if( name == null )
         name = DEFAULT;
      this.name = name;
   }

   public String getName()
   {
      return name;
   }
   public void setName(String name)
   {
      this.name = name;
   }

   public String getDomain()
   {
      return domain;
   }
   public void setDomain(String domain)
   {
      this.domain = domain;
   }

   public String getServer()
   {
      return server;
   }
   public void setServer(String server)
   {
      this.server = server;
   }

   /**
    * Is this a default key. A default key is one where all
    * fields are {@link #DEFAULT}
    * @return true if default, false otherwise.
    */
   public boolean isDefaultKey()
   {
      return DEFAULT.equalsIgnoreCase(domain)
         && DEFAULT.equalsIgnoreCase(server)
         && DEFAULT.equalsIgnoreCase(name);
   }

   /**
    * Compare based on domain, then server and finally name.
    * 
    * @param o - the ProfileKey instance to compare to
    * @return < 0, 0, > 0 based on whether this is less than, equal to
    *    or greater than o.
    */
   public int compareTo(Object o)
   {
      ProfileKey pk = (ProfileKey) o;
      int compareTo = domain.compareTo(pk.domain);
      if( compareTo == 0 )
      {
         compareTo = server.compareTo(pk.server);
         if( compareTo == 0 )
         {
            compareTo = name.compareTo(pk.name);
         }
      }
      return compareTo;
   }
   public boolean equals(Object o)
   {
      if(o instanceof ProfileKey == false)
         return false;
      
      return compareTo(o) == 0;
   }
   public int hashCode()
   {
      int hash = domain.hashCode() + server.hashCode() + name.hashCode();
      return hash;
   }

   public String toString()
   {
      StringBuilder tmp = new StringBuilder();
      tmp.append(getClass().getSimpleName());
      tmp.append('@').append(Integer.toHexString(System.identityHashCode(this)));
      tmp.append('[');
      tmp.append(" domain=");
      tmp.append(domain);
      tmp.append(", server=");
      tmp.append(server);
      tmp.append(", name=");
      tmp.append(name);
      tmp.append(']');
      return tmp.toString();
   }
}
