/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployers.spi.management;

import org.jboss.managed.api.ComponentType;

/**
 * A collection of enums for the well known component types.
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 90185 $
 */
public interface KnownComponentTypes
{
   // constants for well known subtypes
   public static final String ANY = "*";
   public static final String XA = "XA";
   public static final String LOCAL_TX = "LocalTx";
   public static final String NO_TX = "NoTx";
   public static final String TX = "Tx";
   public static final String QUEUE = "Queue";
   public static final String TOPIC = "Topic";
   public static final String DURABLE = "DurableTopic";


   /**
    * A component type that should match any other
    */
   public static ComponentType ANY_TYPE = new ComponentType(ANY, ANY);

   /**
    * Enums for the DataSource types
    */
   public enum DataSourceTypes
   {
      XA("DataSource", KnownComponentTypes.XA), 
      LocalTx("DataSource", LOCAL_TX), 
      NoTx("DataSource", NO_TX);

      private final String type;
      private final String subtype;
      private DataSourceTypes(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };

   /**
    * Enums for generic ConnectionFactory types
    */
   public enum ConnectionFactoryTypes
   {
      /**
       * @deprecated use {@link #Tx}
       */
      XA("ConnectionFactory", KnownComponentTypes.XA),
      /**
       * A transacted connection factory
       */
      Tx("ConnectionFactory", TX),
      /**
       * A connection factory that does not support transactions
       */
      NoTx("ConnectionFactory", NO_TX);

      private final String type;
      private final String subtype;
      private ConnectionFactoryTypes(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };

   /**
    * Enums for the JMSDestination/{Queue,Topic,DurableTopic} types
    */
   public enum JMSDestination
   {
      Queue("JMSDestination", "Queue"), 
      Topic("JMSDestination", "Topic"),
      DurableTopic("JMSDestination", "DurableTopic");

      private final String type;
      private final String subtype;
      private JMSDestination(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };

   /**
    * Enums for the EJB/{StatelessSession,StatefulSession,Entity,MDB} types
    */
   public enum EJB
   {
      StatelessSession("EJB", "StatelessSession"),
      StatefulSession("EJB", "StatefulSession"),
      Entity("EJB", "Entity"),
      MDB("EJB", "MDB");

      private final String type;
      private final String subtype;
      private EJB(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };
   
   /**
    * Enums for the EJB3/{StatelessSession,StatefulSession,MessageDriven} types
    */   
   public enum EJB3
   {
      StatelessSession("EJB3", "SLSB"),
      StatefulSession("EJB3", "SFSB"),
      MessageDriven("EJB3", "MDB");

      private final String type;
      private final String subtype;
      private EJB3(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };
   
   /**
    * Enums for JPA entity bean.
    */
   public enum JPA
   {
      Entity("JPA", "Entity");
      
      private final String type;
      private final String subtype;
      private JPA(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }      
   }

   /**
    * Enums for the MBean/{Dynamic,Standard,Model,Open,Persistent,XMBean} types
    */
   public enum MBean
   {
      Dynamic("MBean", "Dynamic"),
      Standard("MBean", "Standard"),
      Model("MBean", "Model"),
      Open("MBean", "Open"),
      Persistent("MBean", "Persistent"),
      XMBean("MBean", "XMBean"),
      Platform("MBean", "Platform");

      private final String type;
      private final String subtype;
      private MBean(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   };

   /**
    * Enums for the MCBean types
    */
   public enum MCBean
   {
      Any("MCBean", "*");

      private final String type;
      private final String subtype;
      private MCBean(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   }
   
   /**
    * Enums for the web component types
    */
   public enum Web
   {
      Application("Web", "Application"),
      ApplicationManager("Web", "ApplicationManager"),
      Connector("Web", "Connector"),
      Filter("Web", "Filter"),
      Host("Web", "Host"),
      Listener("Web", "Listener"),
      RequestProcessor("Web", "RequestProcessor"),
      ThreadPool("Web", "ThreadPool"),
      Servlet("Web", "Servlet");

      private final String type;
      private final String subtype;
      private Web(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   }
   /**
    * Enums for Server component types
    */
   public enum Server
   {
      ServerInfo("Server", "ServerInfo"),
      ServerConfig("Server", "ServerConfig"),
      MCServer("Server", "MCServer");

      private final String type;
      private final String subtype;
      private Server(String type, String subtype)
      {
         this.type = type;
         this.subtype = subtype;
      }

      public String type()
      {
         return type;
      }
      public String subtype()
      {
         return subtype;
      }
      public ComponentType getType()
      {
         return new ComponentType(type, subtype);
      }
   }
}
