/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2016, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.dmr.stream;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigDecimal;

import org.jboss.dmr.ModelType;
import org.junit.Test;

/**
 * @author <a href="mailto:ropalka@redhat.com">Richard Opalka</a>
 */
public final class InvalidJsonReaderTestCase extends AbstractModelStreamsTestCase {

    @Override
    public ModelReader getModelReader( final String data ) throws IOException {
        final ByteArrayInputStream bais = new ByteArrayInputStream( data.getBytes() );
        return ModelStreamFactory.getInstance( true ).newModelReader( bais );
    }

    @Test
    public void noData() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    @Test
    public void emptyState() throws IOException, ModelException {
        read_unexpectedChar();
        read_colon();
        read_comma();
        read_objectEnd();
        read_listEnd();

        read_whitespace_unexpectedChar();
        read_whitespace_colon();
        read_whitespace_comma();
        read_whitespace_objectEnd();
        read_whitespace_listEnd();
    }

    @Test
    public void unexpectedContentFollowingDmrData() throws IOException, ModelException {
        read_list_unexpectedChar();
        read_object_unexpectedChar();
        read_property_unexpectedChar();
        read_string_unexpectedChar();
        read_int_unexpectedChar();
        read_long_unexpectedChar();
        read_double_unexpectedChar();
        read_bigInteger_unexpectedChar();
        read_bigDecimal_unexpectedChar();
        read_bytes_unexpectedChar();
        read_expression_unexpectedChar();
        read_listType_unexpectedChar();
        read_objectType_unexpectedChar();
        read_propertyType_unexpectedChar();
        read_stringType_unexpectedChar();
        read_intType_unexpectedChar();
        read_longType_unexpectedChar();
        read_doubleType_unexpectedChar();
        read_bigIntegerType_unexpectedChar();
        read_bigDecimalType_unexpectedChar();
        read_bytesType_unexpectedChar();
        read_expressionType_unexpectedChar();
        read_typeType_unexpectedChar();
        read_booleanType_unexpectedChar();
        read_undefinedType_unexpectedChar();
        read_false_unexpectedChar();
        read_true_unexpectedChar();
        read_undefined_unexpectedChar();
    }

    @Test
    public void brokenPrimitives() throws IOException, ModelException {
        // string
        read_brokenString_unexpectedEOF();
        // bytes
        read_brokenBytes_unexpectedChar();
        read_brokenBytes_unexpectedEOF();
        read_brokenBytes_missingColon();
        read_brokenBytes_missingString();
        read_brokenBytes_invalidBase64Value_unexpectedEOF();
        read_brokenBytes_invalidBase64Value_unexpectedChar();
        read_brokenBytes_invalidBase64Value_wrongLength();
        // expression
        read_brokenExpressionString_unexpectedEOF();
        // types
        read_brokenType_unexpectedChar();
        read_brokenType_unexpectedEOF();
        // double specials
        read_brokenInfinity2_unexpectedChar();
        read_brokenInfinity2_unexpectedEOF();
        read_brokenInfinity3_unexpectedChar();
        read_brokenInfinity3_unexpectedEOF();
        read_brokenNegativeInfinity1_unexpectedChar();
        read_brokenNegativeInfinity1_unexpectedEOF();
        read_brokenNegativeInfinity2_unexpectedChar();
        read_brokenNegativeInfinity2_unexpectedEOF();
        read_brokenNegativeInfinity3_unexpectedChar();
        read_brokenNegativeInfinity3_unexpectedEOF();
        read_brokenPositiveInfinity1_unexpectedChar();
        read_brokenPositiveInfinity1_unexpectedEOF();
        read_brokenPositiveInfinity2_unexpectedChar();
        read_brokenPositiveInfinity2_unexpectedEOF();
        read_brokenPositiveInfinity3_unexpectedChar();
        read_brokenPositiveInfinity3_unexpectedEOF();
        read_brokenNaN2_unexpectedChar();
        read_brokenNaN2_unexpectedEOF();
        read_brokenNaN3_unexpectedChar();
        read_brokenNaN3_unexpectedEOF();
        read_brokenNegativeNaN1_unexpectedChar();
        read_brokenNegativeNaN1_unexpectedEOF();
        read_brokenNegativeNaN2_unexpectedChar();
        read_brokenNegativeNaN2_unexpectedEOF();
        read_brokenNegativeNaN3_unexpectedChar();
        read_brokenNegativeNaN3_unexpectedEOF();
        read_brokenPositiveNaN1_unexpectedChar();
        read_brokenPositiveNaN1_unexpectedEOF();
        read_brokenPositiveNaN2_unexpectedChar();
        read_brokenPositiveNaN2_unexpectedEOF();
        read_brokenPositiveNaN3_unexpectedChar();
        read_brokenPositiveNaN3_unexpectedEOF();
        // boolean
        read_brokenTrue_unexpectedChar();
        read_brokenTrue_unexpectedEOF();
        read_brokenFalse_unexpectedChar();
        read_brokenFalse_unexpectedEOF();
        // null
        read_brokenUndefined_unexpectedChar();
        read_brokenUndefined_unexpectedEOF();
    }

    @Test
    public void emptyObjectStartState() throws IOException, ModelException {
        read_objectStart_unexpectedChar();
        read_objectStart_colon();
        read_objectStart_comma();
        read_objectStart_listEnd();
        read_objectStart_listStart();
        read_objectStart_objectStart();
        read_objectStart_number();
        read_objectStart_false();
        read_objectStart_true();
        read_objectStart_null();

        read_objectStart_whitespace_unexpectedChar();
        read_objectStart_whitespace_colon();
        read_objectStart_whitespace_comma();
        read_objectStart_whitespace_listEnd();
        read_objectStart_whitespace_listStart();
        read_objectStart_whitespace_objectStart();
        read_objectStart_whitespace_number();
        read_objectStart_whitespace_false();
        read_objectStart_whitespace_true();
        read_objectStart_whitespace_null();
    }

    @Test
    public void emptyListStartState() throws IOException, ModelException {
        read_listStart_unexpectedChar();
        read_listStart_colon();
        read_listStart_comma();
        read_listStart_objectEnd();

        read_listStart_whitespace_unexpectedChar();
        read_listStart_whitespace_colon();
        read_listStart_whitespace_comma();
        read_listStart_whitespace_objectEnd();
    }

    @Test
    public void notEmptyListMissingCommaAfterFirstItem() throws IOException, ModelException {
        read_listStart_listStart_listEnd_unexpectedChar();
        read_listStart_listStart_listEnd_unexpectedEOF();
        read_listStart_listStart_listEnd_colon();
        read_listStart_listStart_listEnd_listStart();
        read_listStart_listStart_listEnd_objectEnd();
        read_listStart_listStart_listEnd_objectStart();
        read_listStart_listStart_listEnd_string();
        read_listStart_listStart_listEnd_numberBigInteger();
        read_listStart_listStart_listEnd_numberBigDecimal();
        read_listStart_listStart_listEnd_bytes();
        read_listStart_listStart_listEnd_expression();
        read_listStart_listStart_listEnd_typeObject();
        read_listStart_listStart_listEnd_typeList();
        read_listStart_listStart_listEnd_typeProperty();
        read_listStart_listStart_listEnd_typeString();
        read_listStart_listStart_listEnd_typeInt();
        read_listStart_listStart_listEnd_typeLong();
        read_listStart_listStart_listEnd_typeDouble();
        read_listStart_listStart_listEnd_typeBigDecimal();
        read_listStart_listStart_listEnd_typeBigInteger();
        read_listStart_listStart_listEnd_typeBoolean();
        read_listStart_listStart_listEnd_typeBytes();
        read_listStart_listStart_listEnd_typeExpression();
        read_listStart_listStart_listEnd_typeType();
        read_listStart_listStart_listEnd_typeUndefined();
        read_listStart_listStart_listEnd_false();
        read_listStart_listStart_listEnd_true();
        read_listStart_listStart_listEnd_null();

        read_listStart_objectStart_objectEnd_unexpectedChar();
        read_listStart_objectStart_objectEnd_unexpectedEOF();
        read_listStart_objectStart_objectEnd_colon();
        read_listStart_objectStart_objectEnd_listStart();
        read_listStart_objectStart_objectEnd_objectEnd();
        read_listStart_objectStart_objectEnd_objectStart();
        read_listStart_objectStart_objectEnd_string();
        read_listStart_objectStart_objectEnd_numberBigInteger();
        read_listStart_objectStart_objectEnd_numberBigDecimal();
        read_listStart_objectStart_objectEnd_bytes();
        read_listStart_objectStart_objectEnd_expression();
        read_listStart_objectStart_objectEnd_typeObject();
        read_listStart_objectStart_objectEnd_typeList();
        read_listStart_objectStart_objectEnd_typeProperty();
        read_listStart_objectStart_objectEnd_typeString();
        read_listStart_objectStart_objectEnd_typeInt();
        read_listStart_objectStart_objectEnd_typeLong();
        read_listStart_objectStart_objectEnd_typeDouble();
        read_listStart_objectStart_objectEnd_typeBigDecimal();
        read_listStart_objectStart_objectEnd_typeBigInteger();
        read_listStart_objectStart_objectEnd_typeBoolean();
        read_listStart_objectStart_objectEnd_typeBytes();
        read_listStart_objectStart_objectEnd_typeExpression();
        read_listStart_objectStart_objectEnd_typeType();
        read_listStart_objectStart_objectEnd_typeUndefined();
        read_listStart_objectStart_objectEnd_false();
        read_listStart_objectStart_objectEnd_true();
        read_listStart_objectStart_objectEnd_null();

        read_listStart_string_unexpectedChar();
        read_listStart_string_unexpectedEOF();
        read_listStart_string_colon();
        read_listStart_string_listStart();
        read_listStart_string_objectEnd();
        read_listStart_string_objectStart();
        read_listStart_string_string();
        read_listStart_string_numberBigInteger();
        read_listStart_string_numberBigDecimal();
        read_listStart_string_bytes();
        read_listStart_string_expression();
        read_listStart_string_typeObject();
        read_listStart_string_typeList();
        read_listStart_string_typeProperty();
        read_listStart_string_typeString();
        read_listStart_string_typeInt();
        read_listStart_string_typeLong();
        read_listStart_string_typeDouble();
        read_listStart_string_typeBigInteger();
        read_listStart_string_typeBigDecimal();
        read_listStart_string_typeBoolean();
        read_listStart_string_typeBytes();
        read_listStart_string_typeExpression();
        read_listStart_string_typeType();
        read_listStart_string_typeUndefined();
        read_listStart_string_false();
        read_listStart_string_true();
        read_listStart_string_null();

        read_listStart_numberBigInteger_unexpectedChar();
        read_listStart_numberBigInteger_unexpectedEOF();
        read_listStart_numberBigInteger_colon();
        read_listStart_numberBigInteger_listStart();
        read_listStart_numberBigInteger_objectEnd();
        read_listStart_numberBigInteger_objectStart();
        read_listStart_numberBigInteger_string();
        read_listStart_numberBigInteger_numberBigInteger();
        read_listStart_numberBigInteger_numberBigDecimal();
        read_listStart_numberBigInteger_bytes();
        read_listStart_numberBigInteger_expression();
        read_listStart_numberBigInteger_typeObject();
        read_listStart_numberBigInteger_typeList();
        read_listStart_numberBigInteger_typeProperty();
        read_listStart_numberBigInteger_typeString();
        read_listStart_numberBigInteger_typeInt();
        read_listStart_numberBigInteger_typeLong();
        read_listStart_numberBigInteger_typeDouble();
        read_listStart_numberBigInteger_typeBigInteger();
        read_listStart_numberBigInteger_typeBigDecimal();
        read_listStart_numberBigInteger_typeBoolean();
        read_listStart_numberBigInteger_typeBytes();
        read_listStart_numberBigInteger_typeExpression();
        read_listStart_numberBigInteger_typeType();
        read_listStart_numberBigInteger_typeUndefined();
        read_listStart_numberBigInteger_false();
        read_listStart_numberBigInteger_true();
        read_listStart_numberBigInteger_null();

        read_listStart_numberBigDecimal_unexpectedChar();
        read_listStart_numberBigDecimal_unexpectedEOF();
        read_listStart_numberBigDecimal_colon();
        read_listStart_numberBigDecimal_listStart();
        read_listStart_numberBigDecimal_objectEnd();
        read_listStart_numberBigDecimal_objectStart();
        read_listStart_numberBigDecimal_string();
        read_listStart_numberBigDecimal_numberBigInteger();
        read_listStart_numberBigDecimal_numberBigDecimal();
        read_listStart_numberBigDecimal_bytes();
        read_listStart_numberBigDecimal_expression();
        read_listStart_numberBigDecimal_typeObject();
        read_listStart_numberBigDecimal_typeList();
        read_listStart_numberBigDecimal_typeProperty();
        read_listStart_numberBigDecimal_typeString();
        read_listStart_numberBigDecimal_typeInt();
        read_listStart_numberBigDecimal_typeLong();
        read_listStart_numberBigDecimal_typeDouble();
        read_listStart_numberBigDecimal_typeBigInteger();
        read_listStart_numberBigDecimal_typeBigDecimal();
        read_listStart_numberBigDecimal_typeBoolean();
        read_listStart_numberBigDecimal_typeBytes();
        read_listStart_numberBigDecimal_typeExpression();
        read_listStart_numberBigDecimal_typeType();
        read_listStart_numberBigDecimal_typeUndefined();
        read_listStart_numberBigDecimal_false();
        read_listStart_numberBigDecimal_true();
        read_listStart_numberBigDecimal_null();

        read_listStart_bytes_unexpectedChar();
        read_listStart_bytes_unexpectedEOF();
        read_listStart_bytes_colon();
        read_listStart_bytes_listStart();
        read_listStart_bytes_objectEnd();
        read_listStart_bytes_objectStart();
        read_listStart_bytes_string();
        read_listStart_bytes_numberBigInteger();
        read_listStart_bytes_numberBigDecimal();
        read_listStart_bytes_bytes();
        read_listStart_bytes_expression();
        read_listStart_bytes_typeObject();
        read_listStart_bytes_typeList();
        read_listStart_bytes_typeProperty();
        read_listStart_bytes_typeString();
        read_listStart_bytes_typeInt();
        read_listStart_bytes_typeLong();
        read_listStart_bytes_typeDouble();
        read_listStart_bytes_typeBigInteger();
        read_listStart_bytes_typeBigDecimal();
        read_listStart_bytes_typeBoolean();
        read_listStart_bytes_typeBytes();
        read_listStart_bytes_typeExpression();
        read_listStart_bytes_typeType();
        read_listStart_bytes_typeUndefined();
        read_listStart_bytes_false();
        read_listStart_bytes_true();
        read_listStart_bytes_null();

        read_listStart_expression_unexpectedChar();
        read_listStart_expression_unexpectedEOF();
        read_listStart_expression_colon();
        read_listStart_expression_listStart();
        read_listStart_expression_objectEnd();
        read_listStart_expression_objectStart();
        read_listStart_expression_string();
        read_listStart_expression_numberBigInteger();
        read_listStart_expression_numberBigDecimal();
        read_listStart_expression_bytes();
        read_listStart_expression_expression();
        read_listStart_expression_typeObject();
        read_listStart_expression_typeList();
        read_listStart_expression_typeProperty();
        read_listStart_expression_typeString();
        read_listStart_expression_typeInt();
        read_listStart_expression_typeLong();
        read_listStart_expression_typeDouble();
        read_listStart_expression_typeBigInteger();
        read_listStart_expression_typeBigDecimal();
        read_listStart_expression_typeBoolean();
        read_listStart_expression_typeBytes();
        read_listStart_expression_typeExpression();
        read_listStart_expression_typeType();
        read_listStart_expression_typeUndefined();
        read_listStart_expression_false();
        read_listStart_expression_true();
        read_listStart_expression_null();

        read_listStart_typeObject_unexpectedChar();
        read_listStart_typeObject_unexpectedEOF();
        read_listStart_typeObject_colon();
        read_listStart_typeObject_listStart();
        read_listStart_typeObject_objectEnd();
        read_listStart_typeObject_objectStart();
        read_listStart_typeObject_string();
        read_listStart_typeObject_numberBigInteger();
        read_listStart_typeObject_numberBigDecimal();
        read_listStart_typeObject_bytes();
        read_listStart_typeObject_expression();
        read_listStart_typeObject_typeObject();
        read_listStart_typeObject_typeList();
        read_listStart_typeObject_typeProperty();
        read_listStart_typeObject_typeString();
        read_listStart_typeObject_typeInt();
        read_listStart_typeObject_typeLong();
        read_listStart_typeObject_typeDouble();
        read_listStart_typeObject_typeBigInteger();
        read_listStart_typeObject_typeBigDecimal();
        read_listStart_typeObject_typeBoolean();
        read_listStart_typeObject_typeBytes();
        read_listStart_typeObject_typeExpression();
        read_listStart_typeObject_typeType();
        read_listStart_typeObject_typeUndefined();
        read_listStart_typeObject_false();
        read_listStart_typeObject_true();
        read_listStart_typeObject_null();

        read_listStart_typeList_unexpectedChar();
        read_listStart_typeList_unexpectedEOF();
        read_listStart_typeList_colon();
        read_listStart_typeList_listStart();
        read_listStart_typeList_objectEnd();
        read_listStart_typeList_objectStart();
        read_listStart_typeList_string();
        read_listStart_typeList_numberBigInteger();
        read_listStart_typeList_numberBigDecimal();
        read_listStart_typeList_bytes();
        read_listStart_typeList_expression();
        read_listStart_typeList_typeObject();
        read_listStart_typeList_typeList();
        read_listStart_typeList_typeProperty();
        read_listStart_typeList_typeString();
        read_listStart_typeList_typeInt();
        read_listStart_typeList_typeLong();
        read_listStart_typeList_typeDouble();
        read_listStart_typeList_typeBigInteger();
        read_listStart_typeList_typeBigDecimal();
        read_listStart_typeList_typeBoolean();
        read_listStart_typeList_typeBytes();
        read_listStart_typeList_typeExpression();
        read_listStart_typeList_typeType();
        read_listStart_typeList_typeUndefined();
        read_listStart_typeList_false();
        read_listStart_typeList_true();
        read_listStart_typeList_null();

        read_listStart_typeProperty_unexpectedChar();
        read_listStart_typeProperty_unexpectedEOF();
        read_listStart_typeProperty_colon();
        read_listStart_typeProperty_listStart();
        read_listStart_typeProperty_objectEnd();
        read_listStart_typeProperty_objectStart();
        read_listStart_typeProperty_string();
        read_listStart_typeProperty_numberBigInteger();
        read_listStart_typeProperty_numberBigDecimal();
        read_listStart_typeProperty_bytes();
        read_listStart_typeProperty_expression();
        read_listStart_typeProperty_typeObject();
        read_listStart_typeProperty_typeList();
        read_listStart_typeProperty_typeProperty();
        read_listStart_typeProperty_typeString();
        read_listStart_typeProperty_typeInt();
        read_listStart_typeProperty_typeLong();
        read_listStart_typeProperty_typeDouble();
        read_listStart_typeProperty_typeBigInteger();
        read_listStart_typeProperty_typeBigDecimal();
        read_listStart_typeProperty_typeBoolean();
        read_listStart_typeProperty_typeBytes();
        read_listStart_typeProperty_typeExpression();
        read_listStart_typeProperty_typeType();
        read_listStart_typeProperty_typeUndefined();
        read_listStart_typeProperty_false();
        read_listStart_typeProperty_true();
        read_listStart_typeProperty_null();

        read_listStart_typeString_unexpectedChar();
        read_listStart_typeString_unexpectedEOF();
        read_listStart_typeString_colon();
        read_listStart_typeString_listStart();
        read_listStart_typeString_objectEnd();
        read_listStart_typeString_objectStart();
        read_listStart_typeString_string();
        read_listStart_typeString_numberBigInteger();
        read_listStart_typeString_numberBigDecimal();
        read_listStart_typeString_bytes();
        read_listStart_typeString_expression();
        read_listStart_typeString_typeObject();
        read_listStart_typeString_typeList();
        read_listStart_typeString_typeProperty();
        read_listStart_typeString_typeString();
        read_listStart_typeString_typeInt();
        read_listStart_typeString_typeLong();
        read_listStart_typeString_typeDouble();
        read_listStart_typeString_typeBigInteger();
        read_listStart_typeString_typeBigDecimal();
        read_listStart_typeString_typeBoolean();
        read_listStart_typeString_typeBytes();
        read_listStart_typeString_typeExpression();
        read_listStart_typeString_typeType();
        read_listStart_typeString_typeUndefined();
        read_listStart_typeString_false();
        read_listStart_typeString_true();
        read_listStart_typeString_null();

        read_listStart_typeInt_unexpectedChar();
        read_listStart_typeInt_unexpectedEOF();
        read_listStart_typeInt_colon();
        read_listStart_typeInt_listStart();
        read_listStart_typeInt_objectEnd();
        read_listStart_typeInt_objectStart();
        read_listStart_typeInt_string();
        read_listStart_typeInt_numberBigInteger();
        read_listStart_typeInt_numberBigDecimal();
        read_listStart_typeInt_bytes();
        read_listStart_typeInt_expression();
        read_listStart_typeInt_typeObject();
        read_listStart_typeInt_typeList();
        read_listStart_typeInt_typeProperty();
        read_listStart_typeInt_typeString();
        read_listStart_typeInt_typeInt();
        read_listStart_typeInt_typeLong();
        read_listStart_typeInt_typeDouble();
        read_listStart_typeInt_typeBigInteger();
        read_listStart_typeInt_typeBigDecimal();
        read_listStart_typeInt_typeBoolean();
        read_listStart_typeInt_typeBytes();
        read_listStart_typeInt_typeExpression();
        read_listStart_typeInt_typeType();
        read_listStart_typeInt_typeUndefined();
        read_listStart_typeInt_false();
        read_listStart_typeInt_true();
        read_listStart_typeInt_null();

        read_listStart_typeLong_unexpectedChar();
        read_listStart_typeLong_unexpectedEOF();
        read_listStart_typeLong_colon();
        read_listStart_typeLong_listStart();
        read_listStart_typeLong_objectEnd();
        read_listStart_typeLong_objectStart();
        read_listStart_typeLong_string();
        read_listStart_typeLong_numberBigInteger();
        read_listStart_typeLong_numberBigDecimal();
        read_listStart_typeLong_bytes();
        read_listStart_typeLong_expression();
        read_listStart_typeLong_typeObject();
        read_listStart_typeLong_typeList();
        read_listStart_typeLong_typeProperty();
        read_listStart_typeLong_typeString();
        read_listStart_typeLong_typeInt();
        read_listStart_typeLong_typeLong();
        read_listStart_typeLong_typeDouble();
        read_listStart_typeLong_typeBigInteger();
        read_listStart_typeLong_typeBigDecimal();
        read_listStart_typeLong_typeBoolean();
        read_listStart_typeLong_typeBytes();
        read_listStart_typeLong_typeExpression();
        read_listStart_typeLong_typeType();
        read_listStart_typeLong_typeUndefined();
        read_listStart_typeLong_false();
        read_listStart_typeLong_true();
        read_listStart_typeLong_null();

        read_listStart_typeDouble_unexpectedChar();
        read_listStart_typeDouble_unexpectedEOF();
        read_listStart_typeDouble_colon();
        read_listStart_typeDouble_listStart();
        read_listStart_typeDouble_objectEnd();
        read_listStart_typeDouble_objectStart();
        read_listStart_typeDouble_string();
        read_listStart_typeDouble_numberBigInteger();
        read_listStart_typeDouble_numberBigDecimal();
        read_listStart_typeDouble_bytes();
        read_listStart_typeDouble_expression();
        read_listStart_typeDouble_typeObject();
        read_listStart_typeDouble_typeList();
        read_listStart_typeDouble_typeProperty();
        read_listStart_typeDouble_typeString();
        read_listStart_typeDouble_typeInt();
        read_listStart_typeDouble_typeLong();
        read_listStart_typeDouble_typeDouble();
        read_listStart_typeDouble_typeBigInteger();
        read_listStart_typeDouble_typeBigDecimal();
        read_listStart_typeDouble_typeBoolean();
        read_listStart_typeDouble_typeBytes();
        read_listStart_typeDouble_typeExpression();
        read_listStart_typeDouble_typeType();
        read_listStart_typeDouble_typeUndefined();
        read_listStart_typeDouble_false();
        read_listStart_typeDouble_true();
        read_listStart_typeDouble_null();

        read_listStart_typeBigInteger_unexpectedChar();
        read_listStart_typeBigInteger_unexpectedEOF();
        read_listStart_typeBigInteger_colon();
        read_listStart_typeBigInteger_listStart();
        read_listStart_typeBigInteger_objectEnd();
        read_listStart_typeBigInteger_objectStart();
        read_listStart_typeBigInteger_string();
        read_listStart_typeBigInteger_numberBigInteger();
        read_listStart_typeBigInteger_numberBigDecimal();
        read_listStart_typeBigInteger_bytes();
        read_listStart_typeBigInteger_expression();
        read_listStart_typeBigInteger_typeObject();
        read_listStart_typeBigInteger_typeList();
        read_listStart_typeBigInteger_typeProperty();
        read_listStart_typeBigInteger_typeString();
        read_listStart_typeBigInteger_typeInt();
        read_listStart_typeBigInteger_typeLong();
        read_listStart_typeBigInteger_typeDouble();
        read_listStart_typeBigInteger_typeBigInteger();
        read_listStart_typeBigInteger_typeBigDecimal();
        read_listStart_typeBigInteger_typeBoolean();
        read_listStart_typeBigInteger_typeBytes();
        read_listStart_typeBigInteger_typeExpression();
        read_listStart_typeBigInteger_typeType();
        read_listStart_typeBigInteger_typeUndefined();
        read_listStart_typeBigInteger_false();
        read_listStart_typeBigInteger_true();
        read_listStart_typeBigInteger_null();

        read_listStart_typeBigDecimal_unexpectedChar();
        read_listStart_typeBigDecimal_unexpectedEOF();
        read_listStart_typeBigDecimal_colon();
        read_listStart_typeBigDecimal_listStart();
        read_listStart_typeBigDecimal_objectEnd();
        read_listStart_typeBigDecimal_objectStart();
        read_listStart_typeBigDecimal_string();
        read_listStart_typeBigDecimal_numberBigInteger();
        read_listStart_typeBigDecimal_numberBigDecimal();
        read_listStart_typeBigDecimal_bytes();
        read_listStart_typeBigDecimal_expression();
        read_listStart_typeBigDecimal_typeObject();
        read_listStart_typeBigDecimal_typeList();
        read_listStart_typeBigDecimal_typeProperty();
        read_listStart_typeBigDecimal_typeString();
        read_listStart_typeBigDecimal_typeInt();
        read_listStart_typeBigDecimal_typeLong();
        read_listStart_typeBigDecimal_typeDouble();
        read_listStart_typeBigDecimal_typeBigInteger();
        read_listStart_typeBigDecimal_typeBigDecimal();
        read_listStart_typeBigDecimal_typeBoolean();
        read_listStart_typeBigDecimal_typeBytes();
        read_listStart_typeBigDecimal_typeExpression();
        read_listStart_typeBigDecimal_typeType();
        read_listStart_typeBigDecimal_typeUndefined();
        read_listStart_typeBigDecimal_false();
        read_listStart_typeBigDecimal_true();
        read_listStart_typeBigDecimal_null();

        read_listStart_typeBytes_unexpectedChar();
        read_listStart_typeBytes_unexpectedEOF();
        read_listStart_typeBytes_colon();
        read_listStart_typeBytes_listStart();
        read_listStart_typeBytes_objectEnd();
        read_listStart_typeBytes_objectStart();
        read_listStart_typeBytes_string();
        read_listStart_typeBytes_numberBigInteger();
        read_listStart_typeBytes_numberBigDecimal();
        read_listStart_typeBytes_bytes();
        read_listStart_typeBytes_expression();
        read_listStart_typeBytes_typeObject();
        read_listStart_typeBytes_typeList();
        read_listStart_typeBytes_typeProperty();
        read_listStart_typeBytes_typeString();
        read_listStart_typeBytes_typeInt();
        read_listStart_typeBytes_typeLong();
        read_listStart_typeBytes_typeDouble();
        read_listStart_typeBytes_typeBigInteger();
        read_listStart_typeBytes_typeBigDecimal();
        read_listStart_typeBytes_typeBoolean();
        read_listStart_typeBytes_typeBytes();
        read_listStart_typeBytes_typeExpression();
        read_listStart_typeBytes_typeType();
        read_listStart_typeBytes_typeUndefined();
        read_listStart_typeBytes_false();
        read_listStart_typeBytes_true();
        read_listStart_typeBytes_null();

        read_listStart_typeExpression_unexpectedChar();
        read_listStart_typeExpression_unexpectedEOF();
        read_listStart_typeExpression_colon();
        read_listStart_typeExpression_listStart();
        read_listStart_typeExpression_objectEnd();
        read_listStart_typeExpression_objectStart();
        read_listStart_typeExpression_string();
        read_listStart_typeExpression_numberBigInteger();
        read_listStart_typeExpression_numberBigDecimal();
        read_listStart_typeExpression_bytes();
        read_listStart_typeExpression_expression();
        read_listStart_typeExpression_typeObject();
        read_listStart_typeExpression_typeList();
        read_listStart_typeExpression_typeProperty();
        read_listStart_typeExpression_typeString();
        read_listStart_typeExpression_typeInt();
        read_listStart_typeExpression_typeLong();
        read_listStart_typeExpression_typeDouble();
        read_listStart_typeExpression_typeBigInteger();
        read_listStart_typeExpression_typeBigDecimal();
        read_listStart_typeExpression_typeBoolean();
        read_listStart_typeExpression_typeBytes();
        read_listStart_typeExpression_typeExpression();
        read_listStart_typeExpression_typeType();
        read_listStart_typeExpression_typeUndefined();
        read_listStart_typeExpression_false();
        read_listStart_typeExpression_true();
        read_listStart_typeExpression_null();

        read_listStart_typeType_unexpectedChar();
        read_listStart_typeType_unexpectedEOF();
        read_listStart_typeType_colon();
        read_listStart_typeType_listStart();
        read_listStart_typeType_objectEnd();
        read_listStart_typeType_objectStart();
        read_listStart_typeType_string();
        read_listStart_typeType_numberBigInteger();
        read_listStart_typeType_numberBigDecimal();
        read_listStart_typeType_bytes();
        read_listStart_typeType_expression();
        read_listStart_typeType_typeObject();
        read_listStart_typeType_typeList();
        read_listStart_typeType_typeProperty();
        read_listStart_typeType_typeString();
        read_listStart_typeType_typeInt();
        read_listStart_typeType_typeLong();
        read_listStart_typeType_typeDouble();
        read_listStart_typeType_typeBigInteger();
        read_listStart_typeType_typeBigDecimal();
        read_listStart_typeType_typeBoolean();
        read_listStart_typeType_typeBytes();
        read_listStart_typeType_typeExpression();
        read_listStart_typeType_typeType();
        read_listStart_typeType_typeUndefined();
        read_listStart_typeType_false();
        read_listStart_typeType_true();
        read_listStart_typeType_null();

        read_listStart_typeUndefined_unexpectedChar();
        read_listStart_typeUndefined_unexpectedEOF();
        read_listStart_typeUndefined_colon();
        read_listStart_typeUndefined_listStart();
        read_listStart_typeUndefined_objectEnd();
        read_listStart_typeUndefined_objectStart();
        read_listStart_typeUndefined_string();
        read_listStart_typeUndefined_numberBigInteger();
        read_listStart_typeUndefined_numberBigDecimal();
        read_listStart_typeUndefined_bytes();
        read_listStart_typeUndefined_expression();
        read_listStart_typeUndefined_typeObject();
        read_listStart_typeUndefined_typeList();
        read_listStart_typeUndefined_typeProperty();
        read_listStart_typeUndefined_typeString();
        read_listStart_typeUndefined_typeInt();
        read_listStart_typeUndefined_typeLong();
        read_listStart_typeUndefined_typeDouble();
        read_listStart_typeUndefined_typeBigInteger();
        read_listStart_typeUndefined_typeBigDecimal();
        read_listStart_typeUndefined_typeBoolean();
        read_listStart_typeUndefined_typeBytes();
        read_listStart_typeUndefined_typeExpression();
        read_listStart_typeUndefined_typeType();
        read_listStart_typeUndefined_typeUndefined();
        read_listStart_typeUndefined_false();
        read_listStart_typeUndefined_true();
        read_listStart_typeUndefined_null();

        read_listStart_true_unexpectedChar();
        read_listStart_true_unexpectedEOF();
        read_listStart_true_colon();
        read_listStart_true_listStart();
        read_listStart_true_objectEnd();
        read_listStart_true_objectStart();
        read_listStart_true_string();
        read_listStart_true_numberBigInteger();
        read_listStart_true_numberBigDecimal();
        read_listStart_true_bytes();
        read_listStart_true_expression();
        read_listStart_true_typeObject();
        read_listStart_true_typeList();
        read_listStart_true_typeProperty();
        read_listStart_true_typeString();
        read_listStart_true_typeInt();
        read_listStart_true_typeLong();
        read_listStart_true_typeDouble();
        read_listStart_true_typeBigInteger();
        read_listStart_true_typeBigDecimal();
        read_listStart_true_typeBoolean();
        read_listStart_true_typeBytes();
        read_listStart_true_typeExpression();
        read_listStart_true_typeType();
        read_listStart_true_typeUndefined();
        read_listStart_true_false();
        read_listStart_true_true();
        read_listStart_true_null();

        read_listStart_false_unexpectedChar();
        read_listStart_false_unexpectedEOF();
        read_listStart_false_colon();
        read_listStart_false_listStart();
        read_listStart_false_objectEnd();
        read_listStart_false_objectStart();
        read_listStart_false_string();
        read_listStart_false_numberBigInteger();
        read_listStart_false_numberBigDecimal();
        read_listStart_false_bytes();
        read_listStart_false_expression();
        read_listStart_false_typeObject();
        read_listStart_false_typeList();
        read_listStart_false_typeProperty();
        read_listStart_false_typeString();
        read_listStart_false_typeInt();
        read_listStart_false_typeLong();
        read_listStart_false_typeDouble();
        read_listStart_false_typeBigInteger();
        read_listStart_false_typeBigDecimal();
        read_listStart_false_typeBoolean();
        read_listStart_false_typeBytes();
        read_listStart_false_typeExpression();
        read_listStart_false_typeType();
        read_listStart_false_typeUndefined();
        read_listStart_false_false();
        read_listStart_false_true();
        read_listStart_false_null();

        read_listStart_null_unexpectedChar();
        read_listStart_null_unexpectedEOF();
        read_listStart_null_colon();
        read_listStart_null_listStart();
        read_listStart_null_objectEnd();
        read_listStart_null_objectStart();
        read_listStart_null_string();
        read_listStart_null_numberBigInteger();
        read_listStart_null_numberBigDecimal();
        read_listStart_null_bytes();
        read_listStart_null_expression();
        read_listStart_null_typeObject();
        read_listStart_null_typeList();
        read_listStart_null_typeProperty();
        read_listStart_null_typeString();
        read_listStart_null_typeInt();
        read_listStart_null_typeLong();
        read_listStart_null_typeDouble();
        read_listStart_null_typeBigInteger();
        read_listStart_null_typeBigDecimal();
        read_listStart_null_typeBoolean();
        read_listStart_null_typeBytes();
        read_listStart_null_typeExpression();
        read_listStart_null_typeType();
        read_listStart_null_typeUndefined();
        read_listStart_null_false();
        read_listStart_null_true();
        read_listStart_null_null();
    }

    @Test
    public void notEmptyObjectCommaBeforeRightBracket() throws IOException, ModelException {
        read_objectStart_string_colon_listStart_listEnd_comma_objectEnd();
        read_objectStart_string_colon_objectStart_objectEnd_comma_objectEnd();
        read_objectStart_string_colon_string_comma_objectEnd();
        read_objectStart_string_colon_numberBigInteger_comma_objectEnd();
        read_objectStart_string_colon_numberBigDecimal_comma_objectEnd();
        read_objectStart_string_colon_bytes_comma_objectEnd();
        read_objectStart_string_colon_expression_comma_objectEnd();
        read_objectStart_string_colon_typeObject_comma_objectEnd();
        read_objectStart_string_colon_typeList_comma_objectEnd();
        read_objectStart_string_colon_typeProperty_comma_objectEnd();
        read_objectStart_string_colon_typeString_comma_objectEnd();
        read_objectStart_string_colon_typeInt_comma_objectEnd();
        read_objectStart_string_colon_typeLong_comma_objectEnd();
        read_objectStart_string_colon_typeDouble_comma_objectEnd();
        read_objectStart_string_colon_typeBigInteger_comma_objectEnd();
        read_objectStart_string_colon_typeBigDecimal_comma_objectEnd();
        read_objectStart_string_colon_typeBytes_comma_objectEnd();
        read_objectStart_string_colon_typeExpression_comma_objectEnd();
        read_objectStart_string_colon_typeType_comma_objectEnd();
        read_objectStart_string_colon_typeBoolean_comma_objectEnd();
        read_objectStart_string_colon_typeUndefined_comma_objectEnd();
        read_objectStart_string_colon_false_comma_objectEnd();
        read_objectStart_string_colon_true_comma_objectEnd();
        read_objectStart_string_colon_null_comma_objectEnd();
    }

    @Test
    public void notEmptyListCommaBeforeRightBracket() throws IOException, ModelException {
        read_listStart_listStart_listEnd_comma_listEnd();
        read_listStart_objectStart_objectEnd_comma_listEnd();
        read_listStart_string_comma_listEnd();
        read_listStart_numberBigInteger_comma_listEnd();
        read_listStart_numberBigDecimal_comma_listEnd();
        read_listStart_bytes_comma_listEnd();
        read_listStart_expression_comma_listEnd();
        read_listStart_typeObject_comma_listEnd();
        read_listStart_typeList_comma_listEnd();
        read_listStart_typeProperty_comma_listEnd();
        read_listStart_typeString_comma_listEnd();
        read_listStart_typeInt_comma_listEnd();
        read_listStart_typeLong_comma_listEnd();
        read_listStart_typeDouble_comma_listEnd();
        read_listStart_typeBigInteger_comma_listEnd();
        read_listStart_typeBigDecimal_comma_listEnd();
        read_listStart_typeBytes_comma_listEnd();
        read_listStart_typeExpression_comma_listEnd();
        read_listStart_typeType_comma_listEnd();
        read_listStart_typeBoolean_comma_listEnd();
        read_listStart_typeUndefined_comma_listEnd();
        read_listStart_false_comma_listEnd();
        read_listStart_true_comma_listEnd();
        read_listStart_null_comma_listEnd();
    }

    @Test
    public void notEmptyListBrokenAfterComma() throws IOException, ModelException {
        read_listStart_listStart_listEnd_comma_objectEnd();
        read_listStart_objectStart_objectEnd_comma_objectEnd();
        read_listStart_string_comma_objectEnd();
        read_listStart_numberBigInteger_comma_objectEnd();
        read_listStart_numberBigDecimal_comma_objectEnd();
        read_listStart_bytes_comma_objectEnd();
        read_listStart_expression_comma_objectEnd();
        read_listStart_typeObject_comma_objectEnd();
        read_listStart_typeList_comma_objectEnd();
        read_listStart_typeProperty_comma_objectEnd();
        read_listStart_typeString_comma_objectEnd();
        read_listStart_typeInt_comma_objectEnd();
        read_listStart_typeLong_comma_objectEnd();
        read_listStart_typeDouble_comma_objectEnd();
        read_listStart_typeBigInteger_comma_objectEnd();
        read_listStart_typeBigDecimal_comma_objectEnd();
        read_listStart_typeBytes_comma_objectEnd();
        read_listStart_typeExpression_comma_objectEnd();
        read_listStart_typeType_comma_objectEnd();
        read_listStart_typeBoolean_comma_objectEnd();
        read_listStart_typeUndefined_comma_objectEnd();
        read_listStart_false_comma_objectEnd();
        read_listStart_true_comma_objectEnd();
        read_listStart_null_comma_objectEnd();

        read_listStart_listStart_listEnd_comma_colon();
        read_listStart_objectStart_objectEnd_comma_colon();
        read_listStart_string_comma_colon();
        read_listStart_numberBigInteger_comma_colon();
        read_listStart_numberBigDecimal_comma_colon();
        read_listStart_bytes_comma_colon();
        read_listStart_expression_comma_colon();
        read_listStart_typeObject_comma_colon();
        read_listStart_typeList_comma_colon();
        read_listStart_typeProperty_comma_colon();
        read_listStart_typeString_comma_colon();
        read_listStart_typeInt_comma_colon();
        read_listStart_typeLong_comma_colon();
        read_listStart_typeDouble_comma_colon();
        read_listStart_typeBigInteger_comma_colon();
        read_listStart_typeBigDecimal_comma_colon();
        read_listStart_typeBytes_comma_colon();
        read_listStart_typeExpression_comma_colon();
        read_listStart_typeType_comma_colon();
        read_listStart_typeBoolean_comma_colon();
        read_listStart_typeUndefined_comma_colon();
        read_listStart_false_comma_colon();
        read_listStart_true_comma_colon();
        read_listStart_null_comma_colon();

        read_listStart_listStart_listEnd_comma_comma();
        read_listStart_objectStart_objectEnd_comma_comma();
        read_listStart_string_comma_comma();
        read_listStart_numberBigInteger_comma_comma();
        read_listStart_numberBigDecimal_comma_comma();
        read_listStart_bytes_comma_comma();
        read_listStart_expression_comma_comma();
        read_listStart_typeObject_comma_comma();
        read_listStart_typeList_comma_comma();
        read_listStart_typeProperty_comma_comma();
        read_listStart_typeString_comma_comma();
        read_listStart_typeInt_comma_comma();
        read_listStart_typeLong_comma_comma();
        read_listStart_typeDouble_comma_comma();
        read_listStart_typeBigInteger_comma_comma();
        read_listStart_typeBigDecimal_comma_comma();
        read_listStart_typeBytes_comma_comma();
        read_listStart_typeExpression_comma_comma();
        read_listStart_typeType_comma_comma();
        read_listStart_typeBoolean_comma_comma();
        read_listStart_typeUndefined_comma_comma();
        read_listStart_false_comma_comma();
        read_listStart_true_comma_comma();
        read_listStart_null_comma_comma();

        read_listStart_listStart_listEnd_comma_unexpectedChar();
        read_listStart_objectStart_objectEnd_comma_unexpectedChar();
        read_listStart_string_comma_unexpectedChar();
        read_listStart_numberBigInteger_comma_unexpectedChar();
        read_listStart_numberBigDecimal_comma_unexpectedChar();
        read_listStart_bytes_comma_unexpectedChar();
        read_listStart_expression_comma_unexpectedChar();
        read_listStart_typeObject_comma_unexpectedChar();
        read_listStart_typeList_comma_unexpectedChar();
        read_listStart_typeProperty_comma_unexpectedChar();
        read_listStart_typeString_comma_unexpectedChar();
        read_listStart_typeInt_comma_unexpectedChar();
        read_listStart_typeLong_comma_unexpectedChar();
        read_listStart_typeDouble_comma_unexpectedChar();
        read_listStart_typeBigInteger_comma_unexpectedChar();
        read_listStart_typeBigDecimal_comma_unexpectedChar();
        read_listStart_typeBytes_comma_unexpectedChar();
        read_listStart_typeExpression_comma_unexpectedChar();
        read_listStart_typeType_comma_unexpectedChar();
        read_listStart_typeBoolean_comma_unexpectedChar();
        read_listStart_typeUndefined_comma_unexpectedChar();
        read_listStart_false_comma_unexpectedChar();
        read_listStart_true_comma_unexpectedChar();
        read_listStart_null_comma_unexpectedChar();

        read_listStart_listStart_listEnd_comma_unexpectedEOF();
        read_listStart_objectStart_objectEnd_comma_unexpectedEOF();
        read_listStart_string_comma_unexpectedEOF();
        read_listStart_numberBigInteger_comma_unexpectedEOF();
        read_listStart_numberBigDecimal_comma_unexpectedEOF();
        read_listStart_bytes_comma_unexpectedEOF();
        read_listStart_expression_comma_unexpectedEOF();
        read_listStart_typeObject_comma_unexpectedEOF();
        read_listStart_typeList_comma_unexpectedEOF();
        read_listStart_typeProperty_comma_unexpectedEOF();
        read_listStart_typeString_comma_unexpectedEOF();
        read_listStart_typeInt_comma_unexpectedEOF();
        read_listStart_typeLong_comma_unexpectedEOF();
        read_listStart_typeDouble_comma_unexpectedEOF();
        read_listStart_typeBigInteger_comma_unexpectedEOF();
        read_listStart_typeBigDecimal_comma_unexpectedEOF();
        read_listStart_typeBytes_comma_unexpectedEOF();
        read_listStart_typeExpression_comma_unexpectedEOF();
        read_listStart_typeType_comma_unexpectedEOF();
        read_listStart_typeBoolean_comma_unexpectedEOF();
        read_listStart_typeUndefined_comma_unexpectedEOF();
        read_listStart_false_comma_unexpectedEOF();
        read_listStart_true_comma_unexpectedEOF();
        read_listStart_null_comma_unexpectedEOF();
    }

    @Test
    public void notEmptyObjectBrokenAfterColon() throws IOException, ModelException {
        read_objectStart_string_colon_objectEnd();
        read_objectStart_string_colon_listEnd();
        read_objectStart_string_colon_colon();
        read_objectStart_string_colon_comma();
        read_objectStart_string_colon_unexpectedChar();
        read_objectStart_string_colon_unexpectedEOF();
    }

    @Test
    public void notEmptyObjectMissingColon() throws IOException, ModelException {
        read_objectStart_string_unexpectedChar();
        read_objectStart_string_unexpectedEOF();
        read_objectStart_string_comma();
        read_objectStart_string_listEnd();
        read_objectStart_string_listStart();
        read_objectStart_string_objectEnd();
        read_objectStart_string_objectStart();
        read_objectStart_string_string();
        read_objectStart_string_numberBigInteger();
        read_objectStart_string_numberBigDecimal();
        read_objectStart_string_bytes();
        read_objectStart_string_expression();
        read_objectStart_string_typeObject();
        read_objectStart_string_typeList();
        read_objectStart_string_typeProperty();
        read_objectStart_string_typeString();
        read_objectStart_string_typeInt();
        read_objectStart_string_typeLong();
        read_objectStart_string_typeDouble();
        read_objectStart_string_typeBigInteger();
        read_objectStart_string_typeBigDecimal();
        read_objectStart_string_typeBoolean();
        read_objectStart_string_typeBytes();
        read_objectStart_string_typeExpression();
        read_objectStart_string_typeType();
        read_objectStart_string_typeUndefined();
        read_objectStart_string_true();
        read_objectStart_string_false();
        read_objectStart_string_null();
    }

    @Test
    public void notEmptyObjectMissingComma() throws IOException, ModelException {
        read_objectStart_string_colon_listStart_listEnd_unexpectedChar();
        read_objectStart_string_colon_objectStart_objectEnd_unexpectedChar();
        read_objectStart_string_colon_string_unexpectedChar();
        read_objectStart_string_colon_numberBigInteger_unexpectedChar();
        read_objectStart_string_colon_numberBigDecimal_unexpectedChar();
        read_objectStart_string_colon_bytes_unexpectedChar();
        read_objectStart_string_colon_expression_unexpectedChar();
        read_objectStart_string_colon_typeObject_unexpectedChar();
        read_objectStart_string_colon_typeList_unexpectedChar();
        read_objectStart_string_colon_typeProperty_unexpectedChar();
        read_objectStart_string_colon_typeString_unexpectedChar();
        read_objectStart_string_colon_typeInt_unexpectedChar();
        read_objectStart_string_colon_typeLong_unexpectedChar();
        read_objectStart_string_colon_typeDouble_unexpectedChar();
        read_objectStart_string_colon_typeBigInteger_unexpectedChar();
        read_objectStart_string_colon_typeBigDecimal_unexpectedChar();
        read_objectStart_string_colon_typeBytes_unexpectedChar();
        read_objectStart_string_colon_typeExpression_unexpectedChar();
        read_objectStart_string_colon_typeType_unexpectedChar();
        read_objectStart_string_colon_typeBoolean_unexpectedChar();
        read_objectStart_string_colon_typeUndefined_unexpectedChar();
        read_objectStart_string_colon_false_unexpectedChar();
        read_objectStart_string_colon_true_unexpectedChar();
        read_objectStart_string_colon_null_unexpectedChar();

        read_objectStart_string_colon_listStart_listEnd_unexpectedEOF();
        read_objectStart_string_colon_objectStart_objectEnd_unexpectedEOF();
        read_objectStart_string_colon_string_unexpectedEOF();
        read_objectStart_string_colon_numberBigInteger_unexpectedEOF();
        read_objectStart_string_colon_numberBigDecimal_unexpectedEOF();
        read_objectStart_string_colon_bytes_unexpectedEOF();
        read_objectStart_string_colon_expression_unexpectedEOF();
        read_objectStart_string_colon_typeObject_unexpectedEOF();
        read_objectStart_string_colon_typeList_unexpectedEOF();
        read_objectStart_string_colon_typeProperty_unexpectedEOF();
        read_objectStart_string_colon_typeString_unexpectedEOF();
        read_objectStart_string_colon_typeInt_unexpectedEOF();
        read_objectStart_string_colon_typeLong_unexpectedEOF();
        read_objectStart_string_colon_typeDouble_unexpectedEOF();
        read_objectStart_string_colon_typeBigInteger_unexpectedEOF();
        read_objectStart_string_colon_typeBigDecimal_unexpectedEOF();
        read_objectStart_string_colon_typeBytes_unexpectedEOF();
        read_objectStart_string_colon_typeExpression_unexpectedEOF();
        read_objectStart_string_colon_typeType_unexpectedEOF();
        read_objectStart_string_colon_typeBoolean_unexpectedEOF();
        read_objectStart_string_colon_typeUndefined_unexpectedEOF();
        read_objectStart_string_colon_false_unexpectedEOF();
        read_objectStart_string_colon_true_unexpectedEOF();
        read_objectStart_string_colon_null_unexpectedEOF();

        read_objectStart_string_colon_listStart_listEnd_colon();
        read_objectStart_string_colon_objectStart_objectEnd_colon();
        read_objectStart_string_colon_string_colon();
        read_objectStart_string_colon_numberBigInteger_colon();
        read_objectStart_string_colon_numberBigDecimal_colon();
        read_objectStart_string_colon_bytes_colon();
        read_objectStart_string_colon_expression_colon();
        read_objectStart_string_colon_typeObject_colon();
        read_objectStart_string_colon_typeList_colon();
        read_objectStart_string_colon_typeProperty_colon();
        read_objectStart_string_colon_typeString_colon();
        read_objectStart_string_colon_typeInt_colon();
        read_objectStart_string_colon_typeLong_colon();
        read_objectStart_string_colon_typeDouble_colon();
        read_objectStart_string_colon_typeBigInteger_colon();
        read_objectStart_string_colon_typeBigDecimal_colon();
        read_objectStart_string_colon_typeBytes_colon();
        read_objectStart_string_colon_typeExpression_colon();
        read_objectStart_string_colon_typeType_colon();
        read_objectStart_string_colon_typeBoolean_colon();
        read_objectStart_string_colon_typeUndefined_colon();
        read_objectStart_string_colon_true_colon();
        read_objectStart_string_colon_false_colon();
        read_objectStart_string_colon_null_colon();

        read_objectStart_string_colon_listStart_listEnd_objectStart();
        read_objectStart_string_colon_objectStart_objectEnd_objectStart();
        read_objectStart_string_colon_string_objectStart();
        read_objectStart_string_colon_numberBigInteger_objectStart();
        read_objectStart_string_colon_numberBigDecimal_objectStart();
        read_objectStart_string_colon_bytes_objectStart();
        read_objectStart_string_colon_expression_objectStart();
        read_objectStart_string_colon_typeObject_objectStart();
        read_objectStart_string_colon_typeList_objectStart();
        read_objectStart_string_colon_typeProperty_objectStart();
        read_objectStart_string_colon_typeString_objectStart();
        read_objectStart_string_colon_typeInt_objectStart();
        read_objectStart_string_colon_typeLong_objectStart();
        read_objectStart_string_colon_typeDouble_objectStart();
        read_objectStart_string_colon_typeBigInteger_objectStart();
        read_objectStart_string_colon_typeBigDecimal_objectStart();
        read_objectStart_string_colon_typeBytes_objectStart();
        read_objectStart_string_colon_typeExpression_objectStart();
        read_objectStart_string_colon_typeType_objectStart();
        read_objectStart_string_colon_typeBoolean_objectStart();
        read_objectStart_string_colon_typeUndefined_objectStart();
        read_objectStart_string_colon_false_objectStart();
        read_objectStart_string_colon_true_objectStart();
        read_objectStart_string_colon_null_objectStart();

        read_objectStart_string_colon_listStart_listEnd_listStart();
        read_objectStart_string_colon_objectStart_objectEnd_listStart();
        read_objectStart_string_colon_string_listStart();
        read_objectStart_string_colon_numberBigInteger_listStart();
        read_objectStart_string_colon_numberBigDecimal_listStart();
        read_objectStart_string_colon_bytes_listStart();
        read_objectStart_string_colon_expression_listStart();
        read_objectStart_string_colon_typeObject_listStart();
        read_objectStart_string_colon_typeList_listStart();
        read_objectStart_string_colon_typeProperty_listStart();
        read_objectStart_string_colon_typeString_listStart();
        read_objectStart_string_colon_typeInt_listStart();
        read_objectStart_string_colon_typeLong_listStart();
        read_objectStart_string_colon_typeDouble_listStart();
        read_objectStart_string_colon_typeBigInteger_listStart();
        read_objectStart_string_colon_typeBigDecimal_listStart();
        read_objectStart_string_colon_typeBytes_listStart();
        read_objectStart_string_colon_typeExpression_listStart();
        read_objectStart_string_colon_typeType_listStart();
        read_objectStart_string_colon_typeBoolean_listStart();
        read_objectStart_string_colon_typeUndefined_listStart();
        read_objectStart_string_colon_false_listStart();
        read_objectStart_string_colon_true_listStart();
        read_objectStart_string_colon_null_listStart();

        read_objectStart_string_colon_listStart_listEnd_listEnd();
        read_objectStart_string_colon_objectStart_objectEnd_listEnd();
        read_objectStart_string_colon_string_listEnd();
        read_objectStart_string_colon_numberBigInteger_listEnd();
        read_objectStart_string_colon_numberBigDecimal_listEnd();
        read_objectStart_string_colon_bytes_listEnd();
        read_objectStart_string_colon_expression_listEnd();
        read_objectStart_string_colon_typeObject_listEnd();
        read_objectStart_string_colon_typeList_listEnd();
        read_objectStart_string_colon_typeProperty_listEnd();
        read_objectStart_string_colon_typeString_listEnd();
        read_objectStart_string_colon_typeInt_listEnd();
        read_objectStart_string_colon_typeLong_listEnd();
        read_objectStart_string_colon_typeDouble_listEnd();
        read_objectStart_string_colon_typeBigInteger_listEnd();
        read_objectStart_string_colon_typeBigDecimal_listEnd();
        read_objectStart_string_colon_typeBytes_listEnd();
        read_objectStart_string_colon_typeExpression_listEnd();
        read_objectStart_string_colon_typeType_listEnd();
        read_objectStart_string_colon_typeBoolean_listEnd();
        read_objectStart_string_colon_typeUndefined_listEnd();
        read_objectStart_string_colon_false_listEnd();
        read_objectStart_string_colon_true_listEnd();
        read_objectStart_string_colon_null_listEnd();

        read_objectStart_string_colon_listStart_listEnd_string();
        read_objectStart_string_colon_objectStart_objectEnd_string();
        read_objectStart_string_colon_string_string();
        read_objectStart_string_colon_numberBigInteger_string();
        read_objectStart_string_colon_numberBigDecimal_string();
        read_objectStart_string_colon_bytes_string();
        read_objectStart_string_colon_expression_string();
        read_objectStart_string_colon_typeObject_string();
        read_objectStart_string_colon_typeList_string();
        read_objectStart_string_colon_typeProperty_string();
        read_objectStart_string_colon_typeString_string();
        read_objectStart_string_colon_typeInt_string();
        read_objectStart_string_colon_typeLong_string();
        read_objectStart_string_colon_typeDouble_string();
        read_objectStart_string_colon_typeBigInteger_string();
        read_objectStart_string_colon_typeBigDecimal_string();
        read_objectStart_string_colon_typeBytes_string();
        read_objectStart_string_colon_typeExpression_string();
        read_objectStart_string_colon_typeType_string();
        read_objectStart_string_colon_typeBoolean_string();
        read_objectStart_string_colon_typeUndefined_string();
        read_objectStart_string_colon_false_string();
        read_objectStart_string_colon_true_string();
        read_objectStart_string_colon_null_string();

        read_objectStart_string_colon_listStart_listEnd_numberBigInteger();
        read_objectStart_string_colon_objectStart_objectEnd_numberBigInteger();
        read_objectStart_string_colon_string_numberBigInteger();
        read_objectStart_string_colon_numberBigInteger_numberBigInteger();
        read_objectStart_string_colon_numberBigDecimal_numberBigInteger();
        read_objectStart_string_colon_bytes_numberBigInteger();
        read_objectStart_string_colon_expression_numberBigInteger();
        read_objectStart_string_colon_typeObject_numberBigInteger();
        read_objectStart_string_colon_typeList_numberBigInteger();
        read_objectStart_string_colon_typeProperty_numberBigInteger();
        read_objectStart_string_colon_typeString_numberBigInteger();
        read_objectStart_string_colon_typeInt_numberBigInteger();
        read_objectStart_string_colon_typeLong_numberBigInteger();
        read_objectStart_string_colon_typeDouble_numberBigInteger();
        read_objectStart_string_colon_typeBigInteger_numberBigInteger();
        read_objectStart_string_colon_typeBigDecimal_numberBigInteger();
        read_objectStart_string_colon_typeBytes_numberBigInteger();
        read_objectStart_string_colon_typeExpression_numberBigInteger();
        read_objectStart_string_colon_typeType_numberBigInteger();
        read_objectStart_string_colon_typeBoolean_numberBigInteger();
        read_objectStart_string_colon_typeUndefined_numberBigInteger();
        read_objectStart_string_colon_false_numberBigInteger();
        read_objectStart_string_colon_true_numberBigInteger();
        read_objectStart_string_colon_null_numberBigInteger();

        read_objectStart_string_colon_listStart_listEnd_numberBigDecimal();
        read_objectStart_string_colon_objectStart_objectEnd_numberBigDecimal();
        read_objectStart_string_colon_string_numberBigDecimal();
        read_objectStart_string_colon_numberBigInteger_numberBigDecimal();
        read_objectStart_string_colon_numberBigDecimal_numberBigDecimal();
        read_objectStart_string_colon_bytes_numberBigDecimal();
        read_objectStart_string_colon_expression_numberBigDecimal();
        read_objectStart_string_colon_typeObject_numberBigDecimal();
        read_objectStart_string_colon_typeList_numberBigDecimal();
        read_objectStart_string_colon_typeProperty_numberBigDecimal();
        read_objectStart_string_colon_typeString_numberBigDecimal();
        read_objectStart_string_colon_typeInt_numberBigDecimal();
        read_objectStart_string_colon_typeLong_numberBigDecimal();
        read_objectStart_string_colon_typeDouble_numberBigDecimal();
        read_objectStart_string_colon_typeBigInteger_numberBigDecimal();
        read_objectStart_string_colon_typeBigDecimal_numberBigDecimal();
        read_objectStart_string_colon_typeBytes_numberBigDecimal();
        read_objectStart_string_colon_typeExpression_numberBigDecimal();
        read_objectStart_string_colon_typeType_numberBigDecimal();
        read_objectStart_string_colon_typeBoolean_numberBigDecimal();
        read_objectStart_string_colon_typeUndefined_numberBigDecimal();
        read_objectStart_string_colon_false_numberBigDecimal();
        read_objectStart_string_colon_true_numberBigDecimal();
        read_objectStart_string_colon_null_numberBigDecimal();

        read_objectStart_string_colon_listStart_listEnd_bytes();
        read_objectStart_string_colon_objectStart_objectEnd_bytes();
        read_objectStart_string_colon_string_bytes();
        read_objectStart_string_colon_numberBigInteger_bytes();
        read_objectStart_string_colon_numberBigDecimal_bytes();
        read_objectStart_string_colon_bytes_bytes();
        read_objectStart_string_colon_expression_bytes();
        read_objectStart_string_colon_typeObject_bytes();
        read_objectStart_string_colon_typeList_bytes();
        read_objectStart_string_colon_typeProperty_bytes();
        read_objectStart_string_colon_typeString_bytes();
        read_objectStart_string_colon_typeInt_bytes();
        read_objectStart_string_colon_typeLong_bytes();
        read_objectStart_string_colon_typeDouble_bytes();
        read_objectStart_string_colon_typeBigInteger_bytes();
        read_objectStart_string_colon_typeBigDecimal_bytes();
        read_objectStart_string_colon_typeBytes_bytes();
        read_objectStart_string_colon_typeExpression_bytes();
        read_objectStart_string_colon_typeType_bytes();
        read_objectStart_string_colon_typeBoolean_bytes();
        read_objectStart_string_colon_typeUndefined_bytes();
        read_objectStart_string_colon_false_bytes();
        read_objectStart_string_colon_true_bytes();
        read_objectStart_string_colon_null_bytes();

        read_objectStart_string_colon_listStart_listEnd_expression();
        read_objectStart_string_colon_objectStart_objectEnd_expression();
        read_objectStart_string_colon_string_expression();
        read_objectStart_string_colon_numberBigInteger_expression();
        read_objectStart_string_colon_numberBigDecimal_expression();
        read_objectStart_string_colon_bytes_expression();
        read_objectStart_string_colon_expression_expression();
        read_objectStart_string_colon_typeObject_expression();
        read_objectStart_string_colon_typeList_expression();
        read_objectStart_string_colon_typeProperty_expression();
        read_objectStart_string_colon_typeString_expression();
        read_objectStart_string_colon_typeInt_expression();
        read_objectStart_string_colon_typeLong_expression();
        read_objectStart_string_colon_typeDouble_expression();
        read_objectStart_string_colon_typeBigInteger_expression();
        read_objectStart_string_colon_typeBigDecimal_expression();
        read_objectStart_string_colon_typeBytes_expression();
        read_objectStart_string_colon_typeExpression_expression();
        read_objectStart_string_colon_typeType_expression();
        read_objectStart_string_colon_typeBoolean_expression();
        read_objectStart_string_colon_typeUndefined_expression();
        read_objectStart_string_colon_false_expression();
        read_objectStart_string_colon_true_expression();
        read_objectStart_string_colon_null_expression();

        read_objectStart_string_colon_listStart_listEnd_typeObject();
        read_objectStart_string_colon_objectStart_objectEnd_typeObject();
        read_objectStart_string_colon_string_typeObject();
        read_objectStart_string_colon_numberBigInteger_typeObject();
        read_objectStart_string_colon_numberBigDecimal_typeObject();
        read_objectStart_string_colon_bytes_typeObject();
        read_objectStart_string_colon_expression_typeObject();
        read_objectStart_string_colon_typeObject_typeObject();
        read_objectStart_string_colon_typeList_typeObject();
        read_objectStart_string_colon_typeProperty_typeObject();
        read_objectStart_string_colon_typeString_typeObject();
        read_objectStart_string_colon_typeInt_typeObject();
        read_objectStart_string_colon_typeLong_typeObject();
        read_objectStart_string_colon_typeDouble_typeObject();
        read_objectStart_string_colon_typeBigInteger_typeObject();
        read_objectStart_string_colon_typeBigDecimal_typeObject();
        read_objectStart_string_colon_typeBytes_typeObject();
        read_objectStart_string_colon_typeExpression_typeObject();
        read_objectStart_string_colon_typeType_typeObject();
        read_objectStart_string_colon_typeBoolean_typeObject();
        read_objectStart_string_colon_typeUndefined_typeObject();
        read_objectStart_string_colon_false_typeObject();
        read_objectStart_string_colon_true_typeObject();
        read_objectStart_string_colon_null_typeObject();

        read_objectStart_string_colon_listStart_listEnd_typeList();
        read_objectStart_string_colon_objectStart_objectEnd_typeList();
        read_objectStart_string_colon_string_typeList();
        read_objectStart_string_colon_numberBigInteger_typeList();
        read_objectStart_string_colon_numberBigDecimal_typeList();
        read_objectStart_string_colon_bytes_typeList();
        read_objectStart_string_colon_expression_typeList();
        read_objectStart_string_colon_typeObject_typeList();
        read_objectStart_string_colon_typeList_typeList();
        read_objectStart_string_colon_typeProperty_typeList();
        read_objectStart_string_colon_typeString_typeList();
        read_objectStart_string_colon_typeInt_typeList();
        read_objectStart_string_colon_typeLong_typeList();
        read_objectStart_string_colon_typeDouble_typeList();
        read_objectStart_string_colon_typeBigInteger_typeList();
        read_objectStart_string_colon_typeBigDecimal_typeList();
        read_objectStart_string_colon_typeBytes_typeList();
        read_objectStart_string_colon_typeExpression_typeList();
        read_objectStart_string_colon_typeType_typeList();
        read_objectStart_string_colon_typeBoolean_typeList();
        read_objectStart_string_colon_typeUndefined_typeList();
        read_objectStart_string_colon_false_typeList();
        read_objectStart_string_colon_true_typeList();
        read_objectStart_string_colon_null_typeList();

        read_objectStart_string_colon_listStart_listEnd_typeProperty();
        read_objectStart_string_colon_objectStart_objectEnd_typeProperty();
        read_objectStart_string_colon_string_typeProperty();
        read_objectStart_string_colon_numberBigInteger_typeProperty();
        read_objectStart_string_colon_numberBigDecimal_typeProperty();
        read_objectStart_string_colon_bytes_typeProperty();
        read_objectStart_string_colon_expression_typeProperty();
        read_objectStart_string_colon_typeObject_typeProperty();
        read_objectStart_string_colon_typeList_typeProperty();
        read_objectStart_string_colon_typeProperty_typeProperty();
        read_objectStart_string_colon_typeString_typeProperty();
        read_objectStart_string_colon_typeInt_typeProperty();
        read_objectStart_string_colon_typeLong_typeProperty();
        read_objectStart_string_colon_typeDouble_typeProperty();
        read_objectStart_string_colon_typeBigInteger_typeProperty();
        read_objectStart_string_colon_typeBigDecimal_typeProperty();
        read_objectStart_string_colon_typeBytes_typeProperty();
        read_objectStart_string_colon_typeExpression_typeProperty();
        read_objectStart_string_colon_typeType_typeProperty();
        read_objectStart_string_colon_typeBoolean_typeProperty();
        read_objectStart_string_colon_typeUndefined_typeProperty();
        read_objectStart_string_colon_false_typeProperty();
        read_objectStart_string_colon_true_typeProperty();
        read_objectStart_string_colon_null_typeProperty();

        read_objectStart_string_colon_listStart_listEnd_typeString();
        read_objectStart_string_colon_objectStart_objectEnd_typeString();
        read_objectStart_string_colon_string_typeString();
        read_objectStart_string_colon_numberBigInteger_typeString();
        read_objectStart_string_colon_numberBigDecimal_typeString();
        read_objectStart_string_colon_bytes_typeString();
        read_objectStart_string_colon_expression_typeString();
        read_objectStart_string_colon_typeObject_typeString();
        read_objectStart_string_colon_typeList_typeString();
        read_objectStart_string_colon_typeProperty_typeString();
        read_objectStart_string_colon_typeString_typeString();
        read_objectStart_string_colon_typeInt_typeString();
        read_objectStart_string_colon_typeLong_typeString();
        read_objectStart_string_colon_typeDouble_typeString();
        read_objectStart_string_colon_typeBigInteger_typeString();
        read_objectStart_string_colon_typeBigDecimal_typeString();
        read_objectStart_string_colon_typeBytes_typeString();
        read_objectStart_string_colon_typeExpression_typeString();
        read_objectStart_string_colon_typeType_typeString();
        read_objectStart_string_colon_typeBoolean_typeString();
        read_objectStart_string_colon_typeUndefined_typeString();
        read_objectStart_string_colon_false_typeString();
        read_objectStart_string_colon_true_typeString();
        read_objectStart_string_colon_null_typeString();

        read_objectStart_string_colon_listStart_listEnd_typeInt();
        read_objectStart_string_colon_objectStart_objectEnd_typeInt();
        read_objectStart_string_colon_string_typeInt();
        read_objectStart_string_colon_numberBigInteger_typeInt();
        read_objectStart_string_colon_numberBigDecimal_typeInt();
        read_objectStart_string_colon_bytes_typeInt();
        read_objectStart_string_colon_expression_typeInt();
        read_objectStart_string_colon_typeObject_typeInt();
        read_objectStart_string_colon_typeList_typeInt();
        read_objectStart_string_colon_typeProperty_typeInt();
        read_objectStart_string_colon_typeString_typeInt();
        read_objectStart_string_colon_typeInt_typeInt();
        read_objectStart_string_colon_typeLong_typeInt();
        read_objectStart_string_colon_typeDouble_typeInt();
        read_objectStart_string_colon_typeBigInteger_typeInt();
        read_objectStart_string_colon_typeBigDecimal_typeInt();
        read_objectStart_string_colon_typeBytes_typeInt();
        read_objectStart_string_colon_typeExpression_typeInt();
        read_objectStart_string_colon_typeType_typeInt();
        read_objectStart_string_colon_typeBoolean_typeInt();
        read_objectStart_string_colon_typeUndefined_typeInt();
        read_objectStart_string_colon_false_typeInt();
        read_objectStart_string_colon_true_typeInt();
        read_objectStart_string_colon_null_typeInt();

        read_objectStart_string_colon_listStart_listEnd_typeLong();
        read_objectStart_string_colon_objectStart_objectEnd_typeLong();
        read_objectStart_string_colon_string_typeLong();
        read_objectStart_string_colon_numberBigInteger_typeLong();
        read_objectStart_string_colon_numberBigDecimal_typeLong();
        read_objectStart_string_colon_bytes_typeLong();
        read_objectStart_string_colon_expression_typeLong();
        read_objectStart_string_colon_typeObject_typeLong();
        read_objectStart_string_colon_typeList_typeLong();
        read_objectStart_string_colon_typeProperty_typeLong();
        read_objectStart_string_colon_typeString_typeLong();
        read_objectStart_string_colon_typeInt_typeLong();
        read_objectStart_string_colon_typeLong_typeLong();
        read_objectStart_string_colon_typeDouble_typeLong();
        read_objectStart_string_colon_typeBigInteger_typeLong();
        read_objectStart_string_colon_typeBigDecimal_typeLong();
        read_objectStart_string_colon_typeBytes_typeLong();
        read_objectStart_string_colon_typeExpression_typeLong();
        read_objectStart_string_colon_typeType_typeLong();
        read_objectStart_string_colon_typeBoolean_typeLong();
        read_objectStart_string_colon_typeUndefined_typeLong();
        read_objectStart_string_colon_false_typeLong();
        read_objectStart_string_colon_true_typeLong();
        read_objectStart_string_colon_null_typeLong();

        read_objectStart_string_colon_listStart_listEnd_typeDouble();
        read_objectStart_string_colon_objectStart_objectEnd_typeDouble();
        read_objectStart_string_colon_string_typeDouble();
        read_objectStart_string_colon_numberBigInteger_typeDouble();
        read_objectStart_string_colon_numberBigDecimal_typeDouble();
        read_objectStart_string_colon_bytes_typeDouble();
        read_objectStart_string_colon_expression_typeDouble();
        read_objectStart_string_colon_typeObject_typeDouble();
        read_objectStart_string_colon_typeList_typeDouble();
        read_objectStart_string_colon_typeProperty_typeDouble();
        read_objectStart_string_colon_typeString_typeDouble();
        read_objectStart_string_colon_typeInt_typeDouble();
        read_objectStart_string_colon_typeLong_typeDouble();
        read_objectStart_string_colon_typeDouble_typeDouble();
        read_objectStart_string_colon_typeBigInteger_typeDouble();
        read_objectStart_string_colon_typeBigDecimal_typeDouble();
        read_objectStart_string_colon_typeBytes_typeDouble();
        read_objectStart_string_colon_typeExpression_typeDouble();
        read_objectStart_string_colon_typeType_typeDouble();
        read_objectStart_string_colon_typeBoolean_typeDouble();
        read_objectStart_string_colon_typeUndefined_typeDouble();
        read_objectStart_string_colon_false_typeDouble();
        read_objectStart_string_colon_true_typeDouble();
        read_objectStart_string_colon_null_typeDouble();

        read_objectStart_string_colon_listStart_listEnd_typeBigInteger();
        read_objectStart_string_colon_objectStart_objectEnd_typeBigInteger();
        read_objectStart_string_colon_string_typeBigInteger();
        read_objectStart_string_colon_numberBigInteger_typeBigInteger();
        read_objectStart_string_colon_numberBigDecimal_typeBigInteger();
        read_objectStart_string_colon_bytes_typeBigInteger();
        read_objectStart_string_colon_expression_typeBigInteger();
        read_objectStart_string_colon_typeObject_typeBigInteger();
        read_objectStart_string_colon_typeList_typeBigInteger();
        read_objectStart_string_colon_typeProperty_typeBigInteger();
        read_objectStart_string_colon_typeString_typeBigInteger();
        read_objectStart_string_colon_typeInt_typeBigInteger();
        read_objectStart_string_colon_typeLong_typeBigInteger();
        read_objectStart_string_colon_typeDouble_typeBigInteger();
        read_objectStart_string_colon_typeBigInteger_typeBigInteger();
        read_objectStart_string_colon_typeBigDecimal_typeBigInteger();
        read_objectStart_string_colon_typeBytes_typeBigInteger();
        read_objectStart_string_colon_typeExpression_typeBigInteger();
        read_objectStart_string_colon_typeType_typeBigInteger();
        read_objectStart_string_colon_typeBoolean_typeBigInteger();
        read_objectStart_string_colon_typeUndefined_typeBigInteger();
        read_objectStart_string_colon_false_typeBigInteger();
        read_objectStart_string_colon_true_typeBigInteger();
        read_objectStart_string_colon_null_typeBigInteger();

        read_objectStart_string_colon_listStart_listEnd_typeBigDecimal();
        read_objectStart_string_colon_objectStart_objectEnd_typeBigDecimal();
        read_objectStart_string_colon_string_typeBigDecimal();
        read_objectStart_string_colon_numberBigInteger_typeBigDecimal();
        read_objectStart_string_colon_numberBigDecimal_typeBigDecimal();
        read_objectStart_string_colon_bytes_typeBigDecimal();
        read_objectStart_string_colon_expression_typeBigDecimal();
        read_objectStart_string_colon_typeObject_typeBigDecimal();
        read_objectStart_string_colon_typeList_typeBigDecimal();
        read_objectStart_string_colon_typeProperty_typeBigDecimal();
        read_objectStart_string_colon_typeString_typeBigDecimal();
        read_objectStart_string_colon_typeInt_typeBigDecimal();
        read_objectStart_string_colon_typeLong_typeBigDecimal();
        read_objectStart_string_colon_typeDouble_typeBigDecimal();
        read_objectStart_string_colon_typeBigInteger_typeBigDecimal();
        read_objectStart_string_colon_typeBigDecimal_typeBigDecimal();
        read_objectStart_string_colon_typeBytes_typeBigDecimal();
        read_objectStart_string_colon_typeExpression_typeBigDecimal();
        read_objectStart_string_colon_typeType_typeBigDecimal();
        read_objectStart_string_colon_typeBoolean_typeBigDecimal();
        read_objectStart_string_colon_typeUndefined_typeBigDecimal();
        read_objectStart_string_colon_false_typeBigDecimal();
        read_objectStart_string_colon_true_typeBigDecimal();
        read_objectStart_string_colon_null_typeBigDecimal();

        read_objectStart_string_colon_listStart_listEnd_typeBytes();
        read_objectStart_string_colon_objectStart_objectEnd_typeBytes();
        read_objectStart_string_colon_string_typeBytes();
        read_objectStart_string_colon_numberBigInteger_typeBytes();
        read_objectStart_string_colon_numberBigDecimal_typeBytes();
        read_objectStart_string_colon_bytes_typeBytes();
        read_objectStart_string_colon_expression_typeBytes();
        read_objectStart_string_colon_typeObject_typeBytes();
        read_objectStart_string_colon_typeList_typeBytes();
        read_objectStart_string_colon_typeProperty_typeBytes();
        read_objectStart_string_colon_typeString_typeBytes();
        read_objectStart_string_colon_typeInt_typeBytes();
        read_objectStart_string_colon_typeLong_typeBytes();
        read_objectStart_string_colon_typeDouble_typeBytes();
        read_objectStart_string_colon_typeBigInteger_typeBytes();
        read_objectStart_string_colon_typeBigDecimal_typeBytes();
        read_objectStart_string_colon_typeBytes_typeBytes();
        read_objectStart_string_colon_typeExpression_typeBytes();
        read_objectStart_string_colon_typeType_typeBytes();
        read_objectStart_string_colon_typeBoolean_typeBytes();
        read_objectStart_string_colon_typeUndefined_typeBytes();
        read_objectStart_string_colon_false_typeBytes();
        read_objectStart_string_colon_true_typeBytes();
        read_objectStart_string_colon_null_typeBytes();

        read_objectStart_string_colon_listStart_listEnd_typeExpression();
        read_objectStart_string_colon_objectStart_objectEnd_typeExpression();
        read_objectStart_string_colon_string_typeExpression();
        read_objectStart_string_colon_numberBigInteger_typeExpression();
        read_objectStart_string_colon_numberBigDecimal_typeExpression();
        read_objectStart_string_colon_bytes_typeExpression();
        read_objectStart_string_colon_expression_typeExpression();
        read_objectStart_string_colon_typeObject_typeExpression();
        read_objectStart_string_colon_typeList_typeExpression();
        read_objectStart_string_colon_typeProperty_typeExpression();
        read_objectStart_string_colon_typeString_typeExpression();
        read_objectStart_string_colon_typeInt_typeExpression();
        read_objectStart_string_colon_typeLong_typeExpression();
        read_objectStart_string_colon_typeDouble_typeExpression();
        read_objectStart_string_colon_typeBigInteger_typeExpression();
        read_objectStart_string_colon_typeBigDecimal_typeExpression();
        read_objectStart_string_colon_typeBytes_typeExpression();
        read_objectStart_string_colon_typeExpression_typeExpression();
        read_objectStart_string_colon_typeType_typeExpression();
        read_objectStart_string_colon_typeBoolean_typeExpression();
        read_objectStart_string_colon_typeUndefined_typeExpression();
        read_objectStart_string_colon_false_typeExpression();
        read_objectStart_string_colon_true_typeExpression();
        read_objectStart_string_colon_null_typeExpression();

        read_objectStart_string_colon_listStart_listEnd_typeType();
        read_objectStart_string_colon_objectStart_objectEnd_typeType();
        read_objectStart_string_colon_string_typeType();
        read_objectStart_string_colon_numberBigInteger_typeType();
        read_objectStart_string_colon_numberBigDecimal_typeType();
        read_objectStart_string_colon_bytes_typeType();
        read_objectStart_string_colon_expression_typeType();
        read_objectStart_string_colon_typeObject_typeType();
        read_objectStart_string_colon_typeList_typeType();
        read_objectStart_string_colon_typeProperty_typeType();
        read_objectStart_string_colon_typeString_typeType();
        read_objectStart_string_colon_typeInt_typeType();
        read_objectStart_string_colon_typeLong_typeType();
        read_objectStart_string_colon_typeDouble_typeType();
        read_objectStart_string_colon_typeBigInteger_typeType();
        read_objectStart_string_colon_typeBigDecimal_typeType();
        read_objectStart_string_colon_typeBytes_typeType();
        read_objectStart_string_colon_typeExpression_typeType();
        read_objectStart_string_colon_typeType_typeType();
        read_objectStart_string_colon_typeBoolean_typeType();
        read_objectStart_string_colon_typeUndefined_typeType();
        read_objectStart_string_colon_false_typeType();
        read_objectStart_string_colon_true_typeType();
        read_objectStart_string_colon_null_typeType();

        read_objectStart_string_colon_listStart_listEnd_typeBoolean();
        read_objectStart_string_colon_objectStart_objectEnd_typeBoolean();
        read_objectStart_string_colon_string_typeBoolean();
        read_objectStart_string_colon_numberBigInteger_typeBoolean();
        read_objectStart_string_colon_numberBigDecimal_typeBoolean();
        read_objectStart_string_colon_bytes_typeBoolean();
        read_objectStart_string_colon_expression_typeBoolean();
        read_objectStart_string_colon_typeObject_typeBoolean();
        read_objectStart_string_colon_typeList_typeBoolean();
        read_objectStart_string_colon_typeProperty_typeBoolean();
        read_objectStart_string_colon_typeString_typeBoolean();
        read_objectStart_string_colon_typeInt_typeBoolean();
        read_objectStart_string_colon_typeLong_typeBoolean();
        read_objectStart_string_colon_typeDouble_typeBoolean();
        read_objectStart_string_colon_typeBigInteger_typeBoolean();
        read_objectStart_string_colon_typeBigDecimal_typeBoolean();
        read_objectStart_string_colon_typeBytes_typeBoolean();
        read_objectStart_string_colon_typeExpression_typeBoolean();
        read_objectStart_string_colon_typeType_typeBoolean();
        read_objectStart_string_colon_typeBoolean_typeBoolean();
        read_objectStart_string_colon_typeUndefined_typeBoolean();
        read_objectStart_string_colon_false_typeBoolean();
        read_objectStart_string_colon_true_typeBoolean();
        read_objectStart_string_colon_null_typeBoolean();

        read_objectStart_string_colon_listStart_listEnd_typeUndefined();
        read_objectStart_string_colon_objectStart_objectEnd_typeUndefined();
        read_objectStart_string_colon_string_typeUndefined();
        read_objectStart_string_colon_numberBigInteger_typeUndefined();
        read_objectStart_string_colon_numberBigDecimal_typeUndefined();
        read_objectStart_string_colon_bytes_typeUndefined();
        read_objectStart_string_colon_expression_typeUndefined();
        read_objectStart_string_colon_typeObject_typeUndefined();
        read_objectStart_string_colon_typeList_typeUndefined();
        read_objectStart_string_colon_typeProperty_typeUndefined();
        read_objectStart_string_colon_typeString_typeUndefined();
        read_objectStart_string_colon_typeInt_typeUndefined();
        read_objectStart_string_colon_typeLong_typeUndefined();
        read_objectStart_string_colon_typeDouble_typeUndefined();
        read_objectStart_string_colon_typeBigInteger_typeUndefined();
        read_objectStart_string_colon_typeBigDecimal_typeUndefined();
        read_objectStart_string_colon_typeBytes_typeUndefined();
        read_objectStart_string_colon_typeExpression_typeUndefined();
        read_objectStart_string_colon_typeType_typeUndefined();
        read_objectStart_string_colon_typeBoolean_typeUndefined();
        read_objectStart_string_colon_typeUndefined_typeUndefined();
        read_objectStart_string_colon_false_typeUndefined();
        read_objectStart_string_colon_true_typeUndefined();
        read_objectStart_string_colon_null_typeUndefined();

        read_objectStart_string_colon_listStart_listEnd_true();
        read_objectStart_string_colon_objectStart_objectEnd_true();
        read_objectStart_string_colon_string_true();
        read_objectStart_string_colon_numberBigInteger_true();
        read_objectStart_string_colon_numberBigDecimal_true();
        read_objectStart_string_colon_bytes_true();
        read_objectStart_string_colon_expression_true();
        read_objectStart_string_colon_typeObject_true();
        read_objectStart_string_colon_typeList_true();
        read_objectStart_string_colon_typeProperty_true();
        read_objectStart_string_colon_typeString_true();
        read_objectStart_string_colon_typeInt_true();
        read_objectStart_string_colon_typeLong_true();
        read_objectStart_string_colon_typeDouble_true();
        read_objectStart_string_colon_typeBigInteger_true();
        read_objectStart_string_colon_typeBigDecimal_true();
        read_objectStart_string_colon_typeBytes_true();
        read_objectStart_string_colon_typeExpression_true();
        read_objectStart_string_colon_typeType_true();
        read_objectStart_string_colon_typeBoolean_true();
        read_objectStart_string_colon_typeUndefined_true();
        read_objectStart_string_colon_false_true();
        read_objectStart_string_colon_true_true();
        read_objectStart_string_colon_null_true();

        read_objectStart_string_colon_listStart_listEnd_false();
        read_objectStart_string_colon_objectStart_objectEnd_false();
        read_objectStart_string_colon_string_false();
        read_objectStart_string_colon_numberBigInteger_false();
        read_objectStart_string_colon_numberBigDecimal_false();
        read_objectStart_string_colon_bytes_false();
        read_objectStart_string_colon_expression_false();
        read_objectStart_string_colon_typeObject_false();
        read_objectStart_string_colon_typeList_false();
        read_objectStart_string_colon_typeProperty_false();
        read_objectStart_string_colon_typeString_false();
        read_objectStart_string_colon_typeInt_false();
        read_objectStart_string_colon_typeLong_false();
        read_objectStart_string_colon_typeDouble_false();
        read_objectStart_string_colon_typeBigInteger_false();
        read_objectStart_string_colon_typeBigDecimal_false();
        read_objectStart_string_colon_typeBytes_false();
        read_objectStart_string_colon_typeExpression_false();
        read_objectStart_string_colon_typeType_false();
        read_objectStart_string_colon_typeBoolean_false();
        read_objectStart_string_colon_typeUndefined_false();
        read_objectStart_string_colon_false_false();
        read_objectStart_string_colon_true_false();
        read_objectStart_string_colon_null_false();

        read_objectStart_string_colon_listStart_listEnd_null();
        read_objectStart_string_colon_objectStart_objectEnd_null();
        read_objectStart_string_colon_string_null();
        read_objectStart_string_colon_numberBigInteger_null();
        read_objectStart_string_colon_numberBigDecimal_null();
        read_objectStart_string_colon_bytes_null();
        read_objectStart_string_colon_expression_null();
        read_objectStart_string_colon_typeObject_null();
        read_objectStart_string_colon_typeList_null();
        read_objectStart_string_colon_typeProperty_null();
        read_objectStart_string_colon_typeString_null();
        read_objectStart_string_colon_typeInt_null();
        read_objectStart_string_colon_typeLong_null();
        read_objectStart_string_colon_typeDouble_null();
        read_objectStart_string_colon_typeBigInteger_null();
        read_objectStart_string_colon_typeBigDecimal_null();
        read_objectStart_string_colon_typeBytes_null();
        read_objectStart_string_colon_typeExpression_null();
        read_objectStart_string_colon_typeType_null();
        read_objectStart_string_colon_typeBoolean_null();
        read_objectStart_string_colon_typeUndefined_null();
        read_objectStart_string_colon_false_null();
        read_objectStart_string_colon_true_null();
        read_objectStart_string_colon_null_null();
    }

    @Test
    public void notEmptyObjectBrokenAfterComma() throws IOException, ModelException {
        read_objectStart_string_colon_listStart_listEnd_comma_unexpectedChar();
        read_objectStart_string_colon_objectStart_objectEnd_comma_unexpectedChar();
        read_objectStart_string_colon_string_comma_unexpectedChar();
        read_objectStart_string_colon_numberBigInteger_comma_unexpectedChar();
        read_objectStart_string_colon_numberBigDecimal_comma_unexpectedChar();
        read_objectStart_string_colon_bytes_comma_unexpectedChar();
        read_objectStart_string_colon_expression_comma_unexpectedChar();
        read_objectStart_string_colon_typeObject_comma_unexpectedChar();
        read_objectStart_string_colon_typeList_comma_unexpectedChar();
        read_objectStart_string_colon_typeProperty_comma_unexpectedChar();
        read_objectStart_string_colon_typeString_comma_unexpectedChar();
        read_objectStart_string_colon_typeInt_comma_unexpectedChar();
        read_objectStart_string_colon_typeLong_comma_unexpectedChar();
        read_objectStart_string_colon_typeDouble_comma_unexpectedChar();
        read_objectStart_string_colon_typeBigInteger_comma_unexpectedChar();
        read_objectStart_string_colon_typeBigDecimal_comma_unexpectedChar();
        read_objectStart_string_colon_typeBytes_comma_unexpectedChar();
        read_objectStart_string_colon_typeExpression_comma_unexpectedChar();
        read_objectStart_string_colon_typeType_comma_unexpectedChar();
        read_objectStart_string_colon_typeBoolean_comma_unexpectedChar();
        read_objectStart_string_colon_typeUndefined_comma_unexpectedChar();
        read_objectStart_string_colon_false_comma_unexpectedChar();
        read_objectStart_string_colon_true_comma_unexpectedChar();
        read_objectStart_string_colon_null_comma_unexpectedChar();

        read_objectStart_string_colon_listStart_listEnd_comma_unexpectedEOF();
        read_objectStart_string_colon_objectStart_objectEnd_comma_unexpectedEOF();
        read_objectStart_string_colon_string_comma_unexpectedEOF();
        read_objectStart_string_colon_numberBigInteger_comma_unexpectedEOF();
        read_objectStart_string_colon_numberBigDecimal_comma_unexpectedEOF();
        read_objectStart_string_colon_bytes_comma_unexpectedEOF();
        read_objectStart_string_colon_expression_comma_unexpectedEOF();
        read_objectStart_string_colon_typeObject_comma_unexpectedEOF();
        read_objectStart_string_colon_typeList_comma_unexpectedEOF();
        read_objectStart_string_colon_typeProperty_comma_unexpectedEOF();
        read_objectStart_string_colon_typeString_comma_unexpectedEOF();
        read_objectStart_string_colon_typeInt_comma_unexpectedEOF();
        read_objectStart_string_colon_typeLong_comma_unexpectedEOF();
        read_objectStart_string_colon_typeDouble_comma_unexpectedEOF();
        read_objectStart_string_colon_typeBigInteger_comma_unexpectedEOF();
        read_objectStart_string_colon_typeBigDecimal_comma_unexpectedEOF();
        read_objectStart_string_colon_typeBytes_comma_unexpectedEOF();
        read_objectStart_string_colon_typeExpression_comma_unexpectedEOF();
        read_objectStart_string_colon_typeType_comma_unexpectedEOF();
        read_objectStart_string_colon_typeBoolean_comma_unexpectedEOF();
        read_objectStart_string_colon_typeUndefined_comma_unexpectedEOF();
        read_objectStart_string_colon_false_comma_unexpectedEOF();
        read_objectStart_string_colon_true_comma_unexpectedEOF();
        read_objectStart_string_colon_null_comma_unexpectedEOF();

        read_objectStart_string_colon_listStart_listEnd_comma_comma();
        read_objectStart_string_colon_objectStart_objectEnd_comma_comma();
        read_objectStart_string_colon_string_comma_comma();
        read_objectStart_string_colon_numberBigInteger_comma_comma();
        read_objectStart_string_colon_numberBigDecimal_comma_comma();
        read_objectStart_string_colon_bytes_comma_comma();
        read_objectStart_string_colon_expression_comma_comma();
        read_objectStart_string_colon_typeObject_comma_comma();
        read_objectStart_string_colon_typeList_comma_comma();
        read_objectStart_string_colon_typeProperty_comma_comma();
        read_objectStart_string_colon_typeString_comma_comma();
        read_objectStart_string_colon_typeInt_comma_comma();
        read_objectStart_string_colon_typeLong_comma_comma();
        read_objectStart_string_colon_typeDouble_comma_comma();
        read_objectStart_string_colon_typeBigInteger_comma_comma();
        read_objectStart_string_colon_typeBigDecimal_comma_comma();
        read_objectStart_string_colon_typeBytes_comma_comma();
        read_objectStart_string_colon_typeExpression_comma_comma();
        read_objectStart_string_colon_typeType_comma_comma();
        read_objectStart_string_colon_typeBoolean_comma_comma();
        read_objectStart_string_colon_typeUndefined_comma_comma();
        read_objectStart_string_colon_false_comma_comma();
        read_objectStart_string_colon_true_comma_comma();
        read_objectStart_string_colon_null_comma_comma();

        read_objectStart_string_colon_listStart_listEnd_comma_colon();
        read_objectStart_string_colon_objectStart_objectEnd_comma_colon();
        read_objectStart_string_colon_string_comma_colon();
        read_objectStart_string_colon_numberBigInteger_comma_colon();
        read_objectStart_string_colon_numberBigDecimal_comma_colon();
        read_objectStart_string_colon_bytes_comma_colon();
        read_objectStart_string_colon_expression_comma_colon();
        read_objectStart_string_colon_typeObject_comma_colon();
        read_objectStart_string_colon_typeList_comma_colon();
        read_objectStart_string_colon_typeProperty_comma_colon();
        read_objectStart_string_colon_typeString_comma_colon();
        read_objectStart_string_colon_typeInt_comma_colon();
        read_objectStart_string_colon_typeLong_comma_colon();
        read_objectStart_string_colon_typeDouble_comma_colon();
        read_objectStart_string_colon_typeBigInteger_comma_colon();
        read_objectStart_string_colon_typeBigDecimal_comma_colon();
        read_objectStart_string_colon_typeBytes_comma_colon();
        read_objectStart_string_colon_typeExpression_comma_colon();
        read_objectStart_string_colon_typeType_comma_colon();
        read_objectStart_string_colon_typeBoolean_comma_colon();
        read_objectStart_string_colon_typeUndefined_comma_colon();
        read_objectStart_string_colon_false_comma_colon();
        read_objectStart_string_colon_true_comma_colon();
        read_objectStart_string_colon_null_comma_colon();

        read_objectStart_string_colon_listStart_listEnd_comma_objectStart();
        read_objectStart_string_colon_objectStart_objectEnd_comma_objectStart();
        read_objectStart_string_colon_string_comma_objectStart();
        read_objectStart_string_colon_numberBigInteger_comma_objectStart();
        read_objectStart_string_colon_numberBigDecimal_comma_objectStart();
        read_objectStart_string_colon_bytes_comma_objectStart();
        read_objectStart_string_colon_expression_comma_objectStart();
        read_objectStart_string_colon_typeObject_comma_objectStart();
        read_objectStart_string_colon_typeList_comma_objectStart();
        read_objectStart_string_colon_typeProperty_comma_objectStart();
        read_objectStart_string_colon_typeString_comma_objectStart();
        read_objectStart_string_colon_typeInt_comma_objectStart();
        read_objectStart_string_colon_typeLong_comma_objectStart();
        read_objectStart_string_colon_typeDouble_comma_objectStart();
        read_objectStart_string_colon_typeBigInteger_comma_objectStart();
        read_objectStart_string_colon_typeBigDecimal_comma_objectStart();
        read_objectStart_string_colon_typeBytes_comma_objectStart();
        read_objectStart_string_colon_typeExpression_comma_objectStart();
        read_objectStart_string_colon_typeType_comma_objectStart();
        read_objectStart_string_colon_typeBoolean_comma_objectStart();
        read_objectStart_string_colon_typeUndefined_comma_objectStart();
        read_objectStart_string_colon_false_comma_objectStart();
        read_objectStart_string_colon_true_comma_objectStart();
        read_objectStart_string_colon_null_comma_objectStart();

        read_objectStart_string_colon_listStart_listEnd_comma_listStart();
        read_objectStart_string_colon_objectStart_objectEnd_comma_listStart();
        read_objectStart_string_colon_string_comma_listStart();
        read_objectStart_string_colon_numberBigInteger_comma_listStart();
        read_objectStart_string_colon_numberBigDecimal_comma_listStart();
        read_objectStart_string_colon_bytes_comma_listStart();
        read_objectStart_string_colon_expression_comma_listStart();
        read_objectStart_string_colon_typeObject_comma_listStart();
        read_objectStart_string_colon_typeList_comma_listStart();
        read_objectStart_string_colon_typeProperty_comma_listStart();
        read_objectStart_string_colon_typeString_comma_listStart();
        read_objectStart_string_colon_typeInt_comma_listStart();
        read_objectStart_string_colon_typeLong_comma_listStart();
        read_objectStart_string_colon_typeDouble_comma_listStart();
        read_objectStart_string_colon_typeBigInteger_comma_listStart();
        read_objectStart_string_colon_typeBigDecimal_comma_listStart();
        read_objectStart_string_colon_typeBytes_comma_listStart();
        read_objectStart_string_colon_typeExpression_comma_listStart();
        read_objectStart_string_colon_typeType_comma_listStart();
        read_objectStart_string_colon_typeBoolean_comma_listStart();
        read_objectStart_string_colon_typeUndefined_comma_listStart();
        read_objectStart_string_colon_false_comma_listStart();
        read_objectStart_string_colon_true_comma_listStart();
        read_objectStart_string_colon_null_comma_listStart();

        read_objectStart_string_colon_listStart_listEnd_comma_listEnd();
        read_objectStart_string_colon_objectStart_objectEnd_comma_listEnd();
        read_objectStart_string_colon_string_comma_listEnd();
        read_objectStart_string_colon_numberBigInteger_comma_listEnd();
        read_objectStart_string_colon_numberBigDecimal_comma_listEnd();
        read_objectStart_string_colon_bytes_comma_listEnd();
        read_objectStart_string_colon_expression_comma_listEnd();
        read_objectStart_string_colon_typeObject_comma_listEnd();
        read_objectStart_string_colon_typeList_comma_listEnd();
        read_objectStart_string_colon_typeProperty_comma_listEnd();
        read_objectStart_string_colon_typeString_comma_listEnd();
        read_objectStart_string_colon_typeInt_comma_listEnd();
        read_objectStart_string_colon_typeLong_comma_listEnd();
        read_objectStart_string_colon_typeDouble_comma_listEnd();
        read_objectStart_string_colon_typeBigInteger_comma_listEnd();
        read_objectStart_string_colon_typeBigDecimal_comma_listEnd();
        read_objectStart_string_colon_typeBytes_comma_listEnd();
        read_objectStart_string_colon_typeExpression_comma_listEnd();
        read_objectStart_string_colon_typeType_comma_listEnd();
        read_objectStart_string_colon_typeBoolean_comma_listEnd();
        read_objectStart_string_colon_typeUndefined_comma_listEnd();
        read_objectStart_string_colon_false_comma_listEnd();
        read_objectStart_string_colon_true_comma_listEnd();
        read_objectStart_string_colon_null_comma_listEnd();

        read_objectStart_string_colon_listStart_listEnd_comma_listStart();
        read_objectStart_string_colon_objectStart_objectEnd_comma_listStart();
        read_objectStart_string_colon_string_comma_listStart();
        read_objectStart_string_colon_numberBigInteger_comma_listStart();
        read_objectStart_string_colon_numberBigDecimal_comma_listStart();
        read_objectStart_string_colon_bytes_comma_listStart();
        read_objectStart_string_colon_expression_comma_listStart();
        read_objectStart_string_colon_typeObject_comma_listStart();
        read_objectStart_string_colon_typeList_comma_listStart();
        read_objectStart_string_colon_typeProperty_comma_listStart();
        read_objectStart_string_colon_typeString_comma_listStart();
        read_objectStart_string_colon_typeInt_comma_listStart();
        read_objectStart_string_colon_typeLong_comma_listStart();
        read_objectStart_string_colon_typeDouble_comma_listStart();
        read_objectStart_string_colon_typeBigInteger_comma_listStart();
        read_objectStart_string_colon_typeBigDecimal_comma_listStart();
        read_objectStart_string_colon_typeBytes_comma_listStart();
        read_objectStart_string_colon_typeExpression_comma_listStart();
        read_objectStart_string_colon_typeType_comma_listStart();
        read_objectStart_string_colon_typeBoolean_comma_listStart();
        read_objectStart_string_colon_typeUndefined_comma_listStart();
        read_objectStart_string_colon_false_comma_listStart();
        read_objectStart_string_colon_true_comma_listStart();
        read_objectStart_string_colon_null_comma_listStart();

        read_objectStart_string_colon_listStart_listEnd_comma_numberBigInteger();
        read_objectStart_string_colon_objectStart_objectEnd_comma_numberBigInteger();
        read_objectStart_string_colon_string_comma_numberBigInteger();
        read_objectStart_string_colon_numberBigInteger_comma_numberBigInteger();
        read_objectStart_string_colon_numberBigDecimal_comma_numberBigInteger();
        read_objectStart_string_colon_bytes_comma_numberBigInteger();
        read_objectStart_string_colon_expression_comma_numberBigInteger();
        read_objectStart_string_colon_typeObject_comma_numberBigInteger();
        read_objectStart_string_colon_typeList_comma_numberBigInteger();
        read_objectStart_string_colon_typeProperty_comma_numberBigInteger();
        read_objectStart_string_colon_typeString_comma_numberBigInteger();
        read_objectStart_string_colon_typeInt_comma_numberBigInteger();
        read_objectStart_string_colon_typeLong_comma_numberBigInteger();
        read_objectStart_string_colon_typeDouble_comma_numberBigInteger();
        read_objectStart_string_colon_typeBigInteger_comma_numberBigInteger();
        read_objectStart_string_colon_typeBigDecimal_comma_numberBigInteger();
        read_objectStart_string_colon_typeBytes_comma_numberBigInteger();
        read_objectStart_string_colon_typeExpression_comma_numberBigInteger();
        read_objectStart_string_colon_typeType_comma_numberBigInteger();
        read_objectStart_string_colon_typeBoolean_comma_numberBigInteger();
        read_objectStart_string_colon_typeUndefined_comma_numberBigInteger();
        read_objectStart_string_colon_false_comma_numberBigInteger();
        read_objectStart_string_colon_true_comma_numberBigInteger();
        read_objectStart_string_colon_null_comma_numberBigInteger();

        read_objectStart_string_colon_listStart_listEnd_comma_numberBigDecimal();
        read_objectStart_string_colon_objectStart_objectEnd_comma_numberBigDecimal();
        read_objectStart_string_colon_string_comma_numberBigDecimal();
        read_objectStart_string_colon_numberBigInteger_comma_numberBigDecimal();
        read_objectStart_string_colon_numberBigDecimal_comma_numberBigDecimal();
        read_objectStart_string_colon_bytes_comma_numberBigDecimal();
        read_objectStart_string_colon_expression_comma_numberBigDecimal();
        read_objectStart_string_colon_typeObject_comma_numberBigDecimal();
        read_objectStart_string_colon_typeList_comma_numberBigDecimal();
        read_objectStart_string_colon_typeProperty_comma_numberBigDecimal();
        read_objectStart_string_colon_typeString_comma_numberBigDecimal();
        read_objectStart_string_colon_typeInt_comma_numberBigDecimal();
        read_objectStart_string_colon_typeLong_comma_numberBigDecimal();
        read_objectStart_string_colon_typeDouble_comma_numberBigDecimal();
        read_objectStart_string_colon_typeBigInteger_comma_numberBigDecimal();
        read_objectStart_string_colon_typeBigDecimal_comma_numberBigDecimal();
        read_objectStart_string_colon_typeBytes_comma_numberBigDecimal();
        read_objectStart_string_colon_typeExpression_comma_numberBigDecimal();
        read_objectStart_string_colon_typeType_comma_numberBigDecimal();
        read_objectStart_string_colon_typeBoolean_comma_numberBigDecimal();
        read_objectStart_string_colon_typeUndefined_comma_numberBigDecimal();
        read_objectStart_string_colon_false_comma_numberBigDecimal();
        read_objectStart_string_colon_true_comma_numberBigDecimal();
        read_objectStart_string_colon_null_comma_numberBigDecimal();

        read_objectStart_string_colon_listStart_listEnd_comma_bytes();
        read_objectStart_string_colon_objectStart_objectEnd_comma_bytes();
        read_objectStart_string_colon_string_comma_bytes();
        read_objectStart_string_colon_numberBigInteger_comma_bytes();
        read_objectStart_string_colon_numberBigDecimal_comma_bytes();
        read_objectStart_string_colon_bytes_comma_bytes();
        read_objectStart_string_colon_expression_comma_bytes();
        read_objectStart_string_colon_typeObject_comma_bytes();
        read_objectStart_string_colon_typeList_comma_bytes();
        read_objectStart_string_colon_typeProperty_comma_bytes();
        read_objectStart_string_colon_typeString_comma_bytes();
        read_objectStart_string_colon_typeInt_comma_bytes();
        read_objectStart_string_colon_typeLong_comma_bytes();
        read_objectStart_string_colon_typeDouble_comma_bytes();
        read_objectStart_string_colon_typeBigInteger_comma_bytes();
        read_objectStart_string_colon_typeBigDecimal_comma_bytes();
        read_objectStart_string_colon_typeBytes_comma_bytes();
        read_objectStart_string_colon_typeExpression_comma_bytes();
        read_objectStart_string_colon_typeType_comma_bytes();
        read_objectStart_string_colon_typeBoolean_comma_bytes();
        read_objectStart_string_colon_typeUndefined_comma_bytes();
        read_objectStart_string_colon_false_comma_bytes();
        read_objectStart_string_colon_true_comma_bytes();
        read_objectStart_string_colon_null_comma_bytes();

        read_objectStart_string_colon_listStart_listEnd_comma_expression();
        read_objectStart_string_colon_objectStart_objectEnd_comma_expression();
        read_objectStart_string_colon_string_comma_expression();
        read_objectStart_string_colon_numberBigInteger_comma_expression();
        read_objectStart_string_colon_numberBigDecimal_comma_expression();
        read_objectStart_string_colon_bytes_comma_expression();
        read_objectStart_string_colon_expression_comma_expression();
        read_objectStart_string_colon_typeObject_comma_expression();
        read_objectStart_string_colon_typeList_comma_expression();
        read_objectStart_string_colon_typeProperty_comma_expression();
        read_objectStart_string_colon_typeString_comma_expression();
        read_objectStart_string_colon_typeInt_comma_expression();
        read_objectStart_string_colon_typeLong_comma_expression();
        read_objectStart_string_colon_typeDouble_comma_expression();
        read_objectStart_string_colon_typeBigInteger_comma_expression();
        read_objectStart_string_colon_typeBigDecimal_comma_expression();
        read_objectStart_string_colon_typeBytes_comma_expression();
        read_objectStart_string_colon_typeExpression_comma_expression();
        read_objectStart_string_colon_typeType_comma_expression();
        read_objectStart_string_colon_typeBoolean_comma_expression();
        read_objectStart_string_colon_typeUndefined_comma_expression();
        read_objectStart_string_colon_false_comma_expression();
        read_objectStart_string_colon_true_comma_expression();
        read_objectStart_string_colon_null_comma_expression();

        read_objectStart_string_colon_listStart_listEnd_comma_typeObject();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeObject();
        read_objectStart_string_colon_string_comma_typeObject();
        read_objectStart_string_colon_numberBigInteger_comma_typeObject();
        read_objectStart_string_colon_numberBigDecimal_comma_typeObject();
        read_objectStart_string_colon_bytes_comma_typeObject();
        read_objectStart_string_colon_expression_comma_typeObject();
        read_objectStart_string_colon_typeObject_comma_typeObject();
        read_objectStart_string_colon_typeList_comma_typeObject();
        read_objectStart_string_colon_typeProperty_comma_typeObject();
        read_objectStart_string_colon_typeString_comma_typeObject();
        read_objectStart_string_colon_typeInt_comma_typeObject();
        read_objectStart_string_colon_typeLong_comma_typeObject();
        read_objectStart_string_colon_typeDouble_comma_typeObject();
        read_objectStart_string_colon_typeBigInteger_comma_typeObject();
        read_objectStart_string_colon_typeBigDecimal_comma_typeObject();
        read_objectStart_string_colon_typeBytes_comma_typeObject();
        read_objectStart_string_colon_typeExpression_comma_typeObject();
        read_objectStart_string_colon_typeType_comma_typeObject();
        read_objectStart_string_colon_typeBoolean_comma_typeObject();
        read_objectStart_string_colon_typeUndefined_comma_typeObject();
        read_objectStart_string_colon_false_comma_typeObject();
        read_objectStart_string_colon_true_comma_typeObject();
        read_objectStart_string_colon_null_comma_typeObject();

        read_objectStart_string_colon_listStart_listEnd_comma_typeList();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeList();
        read_objectStart_string_colon_string_comma_typeList();
        read_objectStart_string_colon_numberBigInteger_comma_typeList();
        read_objectStart_string_colon_numberBigDecimal_comma_typeList();
        read_objectStart_string_colon_bytes_comma_typeList();
        read_objectStart_string_colon_expression_comma_typeList();
        read_objectStart_string_colon_typeObject_comma_typeList();
        read_objectStart_string_colon_typeList_comma_typeList();
        read_objectStart_string_colon_typeProperty_comma_typeList();
        read_objectStart_string_colon_typeString_comma_typeList();
        read_objectStart_string_colon_typeInt_comma_typeList();
        read_objectStart_string_colon_typeLong_comma_typeList();
        read_objectStart_string_colon_typeDouble_comma_typeList();
        read_objectStart_string_colon_typeBigInteger_comma_typeList();
        read_objectStart_string_colon_typeBigDecimal_comma_typeList();
        read_objectStart_string_colon_typeBytes_comma_typeList();
        read_objectStart_string_colon_typeExpression_comma_typeList();
        read_objectStart_string_colon_typeType_comma_typeList();
        read_objectStart_string_colon_typeBoolean_comma_typeList();
        read_objectStart_string_colon_typeUndefined_comma_typeList();
        read_objectStart_string_colon_false_comma_typeList();
        read_objectStart_string_colon_true_comma_typeList();
        read_objectStart_string_colon_null_comma_typeList();

        read_objectStart_string_colon_listStart_listEnd_comma_typeProperty();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeProperty();
        read_objectStart_string_colon_string_comma_typeProperty();
        read_objectStart_string_colon_numberBigInteger_comma_typeProperty();
        read_objectStart_string_colon_numberBigDecimal_comma_typeProperty();
        read_objectStart_string_colon_bytes_comma_typeProperty();
        read_objectStart_string_colon_expression_comma_typeProperty();
        read_objectStart_string_colon_typeObject_comma_typeProperty();
        read_objectStart_string_colon_typeList_comma_typeProperty();
        read_objectStart_string_colon_typeProperty_comma_typeProperty();
        read_objectStart_string_colon_typeString_comma_typeProperty();
        read_objectStart_string_colon_typeInt_comma_typeProperty();
        read_objectStart_string_colon_typeLong_comma_typeProperty();
        read_objectStart_string_colon_typeDouble_comma_typeProperty();
        read_objectStart_string_colon_typeBigInteger_comma_typeProperty();
        read_objectStart_string_colon_typeBigDecimal_comma_typeProperty();
        read_objectStart_string_colon_typeBytes_comma_typeProperty();
        read_objectStart_string_colon_typeExpression_comma_typeProperty();
        read_objectStart_string_colon_typeType_comma_typeProperty();
        read_objectStart_string_colon_typeBoolean_comma_typeProperty();
        read_objectStart_string_colon_typeUndefined_comma_typeProperty();
        read_objectStart_string_colon_false_comma_typeProperty();
        read_objectStart_string_colon_true_comma_typeProperty();
        read_objectStart_string_colon_null_comma_typeProperty();

        read_objectStart_string_colon_listStart_listEnd_comma_typeString();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeString();
        read_objectStart_string_colon_string_comma_typeString();
        read_objectStart_string_colon_numberBigInteger_comma_typeString();
        read_objectStart_string_colon_numberBigDecimal_comma_typeString();
        read_objectStart_string_colon_bytes_comma_typeString();
        read_objectStart_string_colon_expression_comma_typeString();
        read_objectStart_string_colon_typeObject_comma_typeString();
        read_objectStart_string_colon_typeList_comma_typeString();
        read_objectStart_string_colon_typeProperty_comma_typeString();
        read_objectStart_string_colon_typeString_comma_typeString();
        read_objectStart_string_colon_typeInt_comma_typeString();
        read_objectStart_string_colon_typeLong_comma_typeString();
        read_objectStart_string_colon_typeDouble_comma_typeString();
        read_objectStart_string_colon_typeBigInteger_comma_typeString();
        read_objectStart_string_colon_typeBigDecimal_comma_typeString();
        read_objectStart_string_colon_typeBytes_comma_typeString();
        read_objectStart_string_colon_typeExpression_comma_typeString();
        read_objectStart_string_colon_typeType_comma_typeString();
        read_objectStart_string_colon_typeBoolean_comma_typeString();
        read_objectStart_string_colon_typeUndefined_comma_typeString();
        read_objectStart_string_colon_false_comma_typeString();
        read_objectStart_string_colon_true_comma_typeString();
        read_objectStart_string_colon_null_comma_typeString();

        read_objectStart_string_colon_listStart_listEnd_comma_typeInt();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeInt();
        read_objectStart_string_colon_string_comma_typeInt();
        read_objectStart_string_colon_numberBigInteger_comma_typeInt();
        read_objectStart_string_colon_numberBigDecimal_comma_typeInt();
        read_objectStart_string_colon_bytes_comma_typeInt();
        read_objectStart_string_colon_expression_comma_typeInt();
        read_objectStart_string_colon_typeObject_comma_typeInt();
        read_objectStart_string_colon_typeList_comma_typeInt();
        read_objectStart_string_colon_typeProperty_comma_typeInt();
        read_objectStart_string_colon_typeString_comma_typeInt();
        read_objectStart_string_colon_typeInt_comma_typeInt();
        read_objectStart_string_colon_typeLong_comma_typeInt();
        read_objectStart_string_colon_typeDouble_comma_typeInt();
        read_objectStart_string_colon_typeBigInteger_comma_typeInt();
        read_objectStart_string_colon_typeBigDecimal_comma_typeInt();
        read_objectStart_string_colon_typeBytes_comma_typeInt();
        read_objectStart_string_colon_typeExpression_comma_typeInt();
        read_objectStart_string_colon_typeType_comma_typeInt();
        read_objectStart_string_colon_typeBoolean_comma_typeInt();
        read_objectStart_string_colon_typeUndefined_comma_typeInt();
        read_objectStart_string_colon_false_comma_typeInt();
        read_objectStart_string_colon_true_comma_typeInt();
        read_objectStart_string_colon_null_comma_typeInt();

        read_objectStart_string_colon_listStart_listEnd_comma_typeLong();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeLong();
        read_objectStart_string_colon_string_comma_typeLong();
        read_objectStart_string_colon_numberBigInteger_comma_typeLong();
        read_objectStart_string_colon_numberBigDecimal_comma_typeLong();
        read_objectStart_string_colon_bytes_comma_typeLong();
        read_objectStart_string_colon_expression_comma_typeLong();
        read_objectStart_string_colon_typeObject_comma_typeLong();
        read_objectStart_string_colon_typeList_comma_typeLong();
        read_objectStart_string_colon_typeProperty_comma_typeLong();
        read_objectStart_string_colon_typeString_comma_typeLong();
        read_objectStart_string_colon_typeInt_comma_typeLong();
        read_objectStart_string_colon_typeLong_comma_typeLong();
        read_objectStart_string_colon_typeDouble_comma_typeLong();
        read_objectStart_string_colon_typeBigInteger_comma_typeLong();
        read_objectStart_string_colon_typeBigDecimal_comma_typeLong();
        read_objectStart_string_colon_typeBytes_comma_typeLong();
        read_objectStart_string_colon_typeExpression_comma_typeLong();
        read_objectStart_string_colon_typeType_comma_typeLong();
        read_objectStart_string_colon_typeBoolean_comma_typeLong();
        read_objectStart_string_colon_typeUndefined_comma_typeLong();
        read_objectStart_string_colon_false_comma_typeLong();
        read_objectStart_string_colon_true_comma_typeLong();
        read_objectStart_string_colon_null_comma_typeLong();

        read_objectStart_string_colon_listStart_listEnd_comma_typeDouble();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeDouble();
        read_objectStart_string_colon_string_comma_typeDouble();
        read_objectStart_string_colon_numberBigInteger_comma_typeDouble();
        read_objectStart_string_colon_numberBigDecimal_comma_typeDouble();
        read_objectStart_string_colon_bytes_comma_typeDouble();
        read_objectStart_string_colon_expression_comma_typeDouble();
        read_objectStart_string_colon_typeObject_comma_typeDouble();
        read_objectStart_string_colon_typeList_comma_typeDouble();
        read_objectStart_string_colon_typeProperty_comma_typeDouble();
        read_objectStart_string_colon_typeString_comma_typeDouble();
        read_objectStart_string_colon_typeInt_comma_typeDouble();
        read_objectStart_string_colon_typeLong_comma_typeDouble();
        read_objectStart_string_colon_typeDouble_comma_typeDouble();
        read_objectStart_string_colon_typeBigInteger_comma_typeDouble();
        read_objectStart_string_colon_typeBigDecimal_comma_typeDouble();
        read_objectStart_string_colon_typeBytes_comma_typeDouble();
        read_objectStart_string_colon_typeExpression_comma_typeDouble();
        read_objectStart_string_colon_typeType_comma_typeDouble();
        read_objectStart_string_colon_typeBoolean_comma_typeDouble();
        read_objectStart_string_colon_typeUndefined_comma_typeDouble();
        read_objectStart_string_colon_false_comma_typeDouble();
        read_objectStart_string_colon_true_comma_typeDouble();
        read_objectStart_string_colon_null_comma_typeDouble();

        read_objectStart_string_colon_listStart_listEnd_comma_typeBigInteger();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeBigInteger();
        read_objectStart_string_colon_string_comma_typeBigInteger();
        read_objectStart_string_colon_numberBigInteger_comma_typeBigInteger();
        read_objectStart_string_colon_numberBigDecimal_comma_typeBigInteger();
        read_objectStart_string_colon_bytes_comma_typeBigInteger();
        read_objectStart_string_colon_expression_comma_typeBigInteger();
        read_objectStart_string_colon_typeObject_comma_typeBigInteger();
        read_objectStart_string_colon_typeList_comma_typeBigInteger();
        read_objectStart_string_colon_typeProperty_comma_typeBigInteger();
        read_objectStart_string_colon_typeString_comma_typeBigInteger();
        read_objectStart_string_colon_typeInt_comma_typeBigInteger();
        read_objectStart_string_colon_typeLong_comma_typeBigInteger();
        read_objectStart_string_colon_typeDouble_comma_typeBigInteger();
        read_objectStart_string_colon_typeBigInteger_comma_typeBigInteger();
        read_objectStart_string_colon_typeBigDecimal_comma_typeBigInteger();
        read_objectStart_string_colon_typeBytes_comma_typeBigInteger();
        read_objectStart_string_colon_typeExpression_comma_typeBigInteger();
        read_objectStart_string_colon_typeType_comma_typeBigInteger();
        read_objectStart_string_colon_typeBoolean_comma_typeBigInteger();
        read_objectStart_string_colon_typeUndefined_comma_typeBigInteger();
        read_objectStart_string_colon_false_comma_typeBigInteger();
        read_objectStart_string_colon_true_comma_typeBigInteger();
        read_objectStart_string_colon_null_comma_typeBigInteger();

        read_objectStart_string_colon_listStart_listEnd_comma_typeBigDecimal();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeBigDecimal();
        read_objectStart_string_colon_string_comma_typeBigDecimal();
        read_objectStart_string_colon_numberBigInteger_comma_typeBigDecimal();
        read_objectStart_string_colon_numberBigDecimal_comma_typeBigDecimal();
        read_objectStart_string_colon_bytes_comma_typeBigDecimal();
        read_objectStart_string_colon_expression_comma_typeBigDecimal();
        read_objectStart_string_colon_typeObject_comma_typeBigDecimal();
        read_objectStart_string_colon_typeList_comma_typeBigDecimal();
        read_objectStart_string_colon_typeProperty_comma_typeBigDecimal();
        read_objectStart_string_colon_typeString_comma_typeBigDecimal();
        read_objectStart_string_colon_typeInt_comma_typeBigDecimal();
        read_objectStart_string_colon_typeLong_comma_typeBigDecimal();
        read_objectStart_string_colon_typeDouble_comma_typeBigDecimal();
        read_objectStart_string_colon_typeBigInteger_comma_typeBigDecimal();
        read_objectStart_string_colon_typeBigDecimal_comma_typeBigDecimal();
        read_objectStart_string_colon_typeBytes_comma_typeBigDecimal();
        read_objectStart_string_colon_typeExpression_comma_typeBigDecimal();
        read_objectStart_string_colon_typeType_comma_typeBigDecimal();
        read_objectStart_string_colon_typeBoolean_comma_typeBigDecimal();
        read_objectStart_string_colon_typeUndefined_comma_typeBigDecimal();
        read_objectStart_string_colon_false_comma_typeBigDecimal();
        read_objectStart_string_colon_true_comma_typeBigDecimal();
        read_objectStart_string_colon_null_comma_typeBigDecimal();

        read_objectStart_string_colon_listStart_listEnd_comma_typeBytes();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeBytes();
        read_objectStart_string_colon_string_comma_typeBytes();
        read_objectStart_string_colon_numberBigInteger_comma_typeBytes();
        read_objectStart_string_colon_numberBigDecimal_comma_typeBytes();
        read_objectStart_string_colon_bytes_comma_typeBytes();
        read_objectStart_string_colon_expression_comma_typeBytes();
        read_objectStart_string_colon_typeObject_comma_typeBytes();
        read_objectStart_string_colon_typeList_comma_typeBytes();
        read_objectStart_string_colon_typeProperty_comma_typeBytes();
        read_objectStart_string_colon_typeString_comma_typeBytes();
        read_objectStart_string_colon_typeInt_comma_typeBytes();
        read_objectStart_string_colon_typeLong_comma_typeBytes();
        read_objectStart_string_colon_typeDouble_comma_typeBytes();
        read_objectStart_string_colon_typeBigInteger_comma_typeBytes();
        read_objectStart_string_colon_typeBigDecimal_comma_typeBytes();
        read_objectStart_string_colon_typeBytes_comma_typeBytes();
        read_objectStart_string_colon_typeExpression_comma_typeBytes();
        read_objectStart_string_colon_typeType_comma_typeBytes();
        read_objectStart_string_colon_typeBoolean_comma_typeBytes();
        read_objectStart_string_colon_typeUndefined_comma_typeBytes();
        read_objectStart_string_colon_false_comma_typeBytes();
        read_objectStart_string_colon_true_comma_typeBytes();
        read_objectStart_string_colon_null_comma_typeBytes();

        read_objectStart_string_colon_listStart_listEnd_comma_typeExpression();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeExpression();
        read_objectStart_string_colon_string_comma_typeExpression();
        read_objectStart_string_colon_numberBigInteger_comma_typeExpression();
        read_objectStart_string_colon_numberBigDecimal_comma_typeExpression();
        read_objectStart_string_colon_bytes_comma_typeExpression();
        read_objectStart_string_colon_expression_comma_typeExpression();
        read_objectStart_string_colon_typeObject_comma_typeExpression();
        read_objectStart_string_colon_typeList_comma_typeExpression();
        read_objectStart_string_colon_typeProperty_comma_typeExpression();
        read_objectStart_string_colon_typeString_comma_typeExpression();
        read_objectStart_string_colon_typeInt_comma_typeExpression();
        read_objectStart_string_colon_typeLong_comma_typeExpression();
        read_objectStart_string_colon_typeDouble_comma_typeExpression();
        read_objectStart_string_colon_typeBigInteger_comma_typeExpression();
        read_objectStart_string_colon_typeBigDecimal_comma_typeExpression();
        read_objectStart_string_colon_typeBytes_comma_typeExpression();
        read_objectStart_string_colon_typeExpression_comma_typeExpression();
        read_objectStart_string_colon_typeType_comma_typeExpression();
        read_objectStart_string_colon_typeBoolean_comma_typeExpression();
        read_objectStart_string_colon_typeUndefined_comma_typeExpression();
        read_objectStart_string_colon_false_comma_typeExpression();
        read_objectStart_string_colon_true_comma_typeExpression();
        read_objectStart_string_colon_null_comma_typeExpression();

        read_objectStart_string_colon_listStart_listEnd_comma_typeType();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeType();
        read_objectStart_string_colon_string_comma_typeType();
        read_objectStart_string_colon_numberBigInteger_comma_typeType();
        read_objectStart_string_colon_numberBigDecimal_comma_typeType();
        read_objectStart_string_colon_bytes_comma_typeType();
        read_objectStart_string_colon_expression_comma_typeType();
        read_objectStart_string_colon_typeObject_comma_typeType();
        read_objectStart_string_colon_typeList_comma_typeType();
        read_objectStart_string_colon_typeProperty_comma_typeType();
        read_objectStart_string_colon_typeString_comma_typeType();
        read_objectStart_string_colon_typeInt_comma_typeType();
        read_objectStart_string_colon_typeLong_comma_typeType();
        read_objectStart_string_colon_typeDouble_comma_typeType();
        read_objectStart_string_colon_typeBigInteger_comma_typeType();
        read_objectStart_string_colon_typeBigDecimal_comma_typeType();
        read_objectStart_string_colon_typeBytes_comma_typeType();
        read_objectStart_string_colon_typeExpression_comma_typeType();
        read_objectStart_string_colon_typeType_comma_typeType();
        read_objectStart_string_colon_typeBoolean_comma_typeType();
        read_objectStart_string_colon_typeUndefined_comma_typeType();
        read_objectStart_string_colon_false_comma_typeType();
        read_objectStart_string_colon_true_comma_typeType();
        read_objectStart_string_colon_null_comma_typeType();

        read_objectStart_string_colon_listStart_listEnd_comma_typeBoolean();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeBoolean();
        read_objectStart_string_colon_string_comma_typeBoolean();
        read_objectStart_string_colon_numberBigInteger_comma_typeBoolean();
        read_objectStart_string_colon_numberBigDecimal_comma_typeBoolean();
        read_objectStart_string_colon_bytes_comma_typeBoolean();
        read_objectStart_string_colon_expression_comma_typeBoolean();
        read_objectStart_string_colon_typeObject_comma_typeBoolean();
        read_objectStart_string_colon_typeList_comma_typeBoolean();
        read_objectStart_string_colon_typeProperty_comma_typeBoolean();
        read_objectStart_string_colon_typeString_comma_typeBoolean();
        read_objectStart_string_colon_typeInt_comma_typeBoolean();
        read_objectStart_string_colon_typeLong_comma_typeBoolean();
        read_objectStart_string_colon_typeDouble_comma_typeBoolean();
        read_objectStart_string_colon_typeBigInteger_comma_typeBoolean();
        read_objectStart_string_colon_typeBigDecimal_comma_typeBoolean();
        read_objectStart_string_colon_typeBytes_comma_typeBoolean();
        read_objectStart_string_colon_typeExpression_comma_typeBoolean();
        read_objectStart_string_colon_typeType_comma_typeBoolean();
        read_objectStart_string_colon_typeBoolean_comma_typeBoolean();
        read_objectStart_string_colon_typeUndefined_comma_typeBoolean();
        read_objectStart_string_colon_false_comma_typeBoolean();
        read_objectStart_string_colon_true_comma_typeBoolean();
        read_objectStart_string_colon_null_comma_typeBoolean();

        read_objectStart_string_colon_listStart_listEnd_comma_typeUndefined();
        read_objectStart_string_colon_objectStart_objectEnd_comma_typeUndefined();
        read_objectStart_string_colon_string_comma_typeUndefined();
        read_objectStart_string_colon_numberBigInteger_comma_typeUndefined();
        read_objectStart_string_colon_numberBigDecimal_comma_typeUndefined();
        read_objectStart_string_colon_bytes_comma_typeUndefined();
        read_objectStart_string_colon_expression_comma_typeUndefined();
        read_objectStart_string_colon_typeObject_comma_typeUndefined();
        read_objectStart_string_colon_typeList_comma_typeUndefined();
        read_objectStart_string_colon_typeProperty_comma_typeUndefined();
        read_objectStart_string_colon_typeString_comma_typeUndefined();
        read_objectStart_string_colon_typeInt_comma_typeUndefined();
        read_objectStart_string_colon_typeLong_comma_typeUndefined();
        read_objectStart_string_colon_typeDouble_comma_typeUndefined();
        read_objectStart_string_colon_typeBigInteger_comma_typeUndefined();
        read_objectStart_string_colon_typeBigDecimal_comma_typeUndefined();
        read_objectStart_string_colon_typeBytes_comma_typeUndefined();
        read_objectStart_string_colon_typeExpression_comma_typeUndefined();
        read_objectStart_string_colon_typeType_comma_typeUndefined();
        read_objectStart_string_colon_typeBoolean_comma_typeUndefined();
        read_objectStart_string_colon_typeUndefined_comma_typeUndefined();
        read_objectStart_string_colon_false_comma_typeUndefined();
        read_objectStart_string_colon_true_comma_typeUndefined();
        read_objectStart_string_colon_null_comma_typeUndefined();

        read_objectStart_string_colon_listStart_listEnd_comma_true();
        read_objectStart_string_colon_objectStart_objectEnd_comma_true();
        read_objectStart_string_colon_string_comma_true();
        read_objectStart_string_colon_numberBigInteger_comma_true();
        read_objectStart_string_colon_numberBigDecimal_comma_true();
        read_objectStart_string_colon_bytes_comma_true();
        read_objectStart_string_colon_expression_comma_true();
        read_objectStart_string_colon_typeObject_comma_true();
        read_objectStart_string_colon_typeList_comma_true();
        read_objectStart_string_colon_typeProperty_comma_true();
        read_objectStart_string_colon_typeString_comma_true();
        read_objectStart_string_colon_typeInt_comma_true();
        read_objectStart_string_colon_typeLong_comma_true();
        read_objectStart_string_colon_typeDouble_comma_true();
        read_objectStart_string_colon_typeBigInteger_comma_true();
        read_objectStart_string_colon_typeBigDecimal_comma_true();
        read_objectStart_string_colon_typeBytes_comma_true();
        read_objectStart_string_colon_typeExpression_comma_true();
        read_objectStart_string_colon_typeType_comma_true();
        read_objectStart_string_colon_typeBoolean_comma_true();
        read_objectStart_string_colon_typeUndefined_comma_true();
        read_objectStart_string_colon_false_comma_true();
        read_objectStart_string_colon_true_comma_true();
        read_objectStart_string_colon_null_comma_true();

        read_objectStart_string_colon_listStart_listEnd_comma_false();
        read_objectStart_string_colon_objectStart_objectEnd_comma_false();
        read_objectStart_string_colon_string_comma_false();
        read_objectStart_string_colon_numberBigInteger_comma_false();
        read_objectStart_string_colon_numberBigDecimal_comma_false();
        read_objectStart_string_colon_bytes_comma_false();
        read_objectStart_string_colon_expression_comma_false();
        read_objectStart_string_colon_typeObject_comma_false();
        read_objectStart_string_colon_typeList_comma_false();
        read_objectStart_string_colon_typeProperty_comma_false();
        read_objectStart_string_colon_typeString_comma_false();
        read_objectStart_string_colon_typeInt_comma_false();
        read_objectStart_string_colon_typeLong_comma_false();
        read_objectStart_string_colon_typeDouble_comma_false();
        read_objectStart_string_colon_typeBigInteger_comma_false();
        read_objectStart_string_colon_typeBigDecimal_comma_false();
        read_objectStart_string_colon_typeBytes_comma_false();
        read_objectStart_string_colon_typeExpression_comma_false();
        read_objectStart_string_colon_typeType_comma_false();
        read_objectStart_string_colon_typeBoolean_comma_false();
        read_objectStart_string_colon_typeUndefined_comma_false();
        read_objectStart_string_colon_false_comma_false();
        read_objectStart_string_colon_true_comma_false();
        read_objectStart_string_colon_null_comma_false();

        read_objectStart_string_colon_listStart_listEnd_comma_null();
        read_objectStart_string_colon_objectStart_objectEnd_comma_null();
        read_objectStart_string_colon_string_comma_null();
        read_objectStart_string_colon_numberBigInteger_comma_null();
        read_objectStart_string_colon_numberBigDecimal_comma_null();
        read_objectStart_string_colon_bytes_comma_null();
        read_objectStart_string_colon_expression_comma_null();
        read_objectStart_string_colon_typeObject_comma_null();
        read_objectStart_string_colon_typeList_comma_null();
        read_objectStart_string_colon_typeProperty_comma_null();
        read_objectStart_string_colon_typeString_comma_null();
        read_objectStart_string_colon_typeInt_comma_null();
        read_objectStart_string_colon_typeLong_comma_null();
        read_objectStart_string_colon_typeDouble_comma_null();
        read_objectStart_string_colon_typeBigInteger_comma_null();
        read_objectStart_string_colon_typeBigDecimal_comma_null();
        read_objectStart_string_colon_typeBytes_comma_null();
        read_objectStart_string_colon_typeExpression_comma_null();
        read_objectStart_string_colon_typeType_comma_null();
        read_objectStart_string_colon_typeBoolean_comma_null();
        read_objectStart_string_colon_typeUndefined_comma_null();
        read_objectStart_string_colon_false_comma_null();
        read_objectStart_string_colon_true_comma_null();
        read_objectStart_string_colon_null_comma_null();
    }

    private void read_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "x" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( ":" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "," );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "}" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "]" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\nx" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n:" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n," );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n}" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n]" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_list_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[]\"" );
        assertInitialState( reader );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_object_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{}\"" );
        assertInitialState( reader );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_property_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":\"\"}_" );
        assertInitialState( reader );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "\"\"_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalStringState( reader );
        reader.close();
    }

    private void read_int_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_long_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1L_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_double_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1.0_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "60000000000000000000000000000_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1.0_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\":\"\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBytesState( reader );
        reader.close();
    }

    private void read_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"EXPRESSION_VALUE\":\"\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalExpressionState( reader );
        reader.close();
    }

    private void read_listType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"LIST\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_objectType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"OBJECT\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_propertyType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_stringType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"STRING\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_intType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"INT\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_longType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"LONG\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_doubleType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bigIntegerType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bigDecimalType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bytesType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"BYTES\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_expressionType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"TYPE\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_booleanType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_undefinedType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "false_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBooleanState( reader );
        reader.close();
    }

    private void read_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "true_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBooleanState( reader );
        reader.close();
    }

    private void read_undefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "null_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "\"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\"a" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'a' while reading DMR bytes value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_missingColon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\"  \"\"}" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character '\"' while reading DMR bytes value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_missingString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\":null}" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR bytes value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_invalidBase64Value_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\":\"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_invalidBase64Value_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\":\"abcde-" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character '-' while reading DMR base64 string" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_invalidBase64Value_wrongLength() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"BYTES_VALUE\":\"abcde==\"" );
        assertInitialState( reader );
        assertModelException( reader, "Encoded base64 value is not dividable by 4" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"EXPRESSION_VALUE\":\"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"ObJECT\"}" );
        assertInitialState( reader );
        assertModelException( reader, "No enum constant org.jboss.dmr.ModelType.ObJECT" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenType_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"TYPE_MODEL_VALUE\":\"O" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'z' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'z' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'z' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTrue_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "trua" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'a' while reading DMR true token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTrue_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "tru" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenFalse_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "falsa" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'a' while reading DMR false token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenFalse_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "fals" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "nulL" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'L' while reading DMR null token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "nul" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{x" );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{:" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{," );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{]" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{[" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{{" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_number() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{false" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{true" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{null" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nx" );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n:" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n," );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n]" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n[" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n{" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_number() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nfalse" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\ntrue" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nnull" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[x" );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }
    private void read_listStart_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[:" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[," );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[}" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nx" );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n:" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n," );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n}" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[][" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}[" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"[" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+23[" );
        assertListStartState( reader );
        assertNumberState( reader, 23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n23}" );
        assertListStartState( reader );
        assertNumberState( reader, 23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23{" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 1" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 1.0" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 false" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 true" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-23 null" );
        assertListStartState( reader );
        assertNumberState( reader, -23 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+29.0[" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n[" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     [" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     [" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     [" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     [" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     }" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     }" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     }" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     }" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     \"\"" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     1" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     1" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     1" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     1" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     1.0" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     1.0" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     1.0" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     false" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     false" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     false" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     false" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     true" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"OBJECT\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"LIST\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"STRING\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"INT\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"LONG\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BYTES\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"TYPE\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}true" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[truetrue" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[falsetrue" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[nulltrue" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     null" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"OBJECT\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"LIST\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"STRING\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"INT\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"LONG\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"BYTES\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"TYPE\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}null" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[truenull" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[falsenull" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[nullnull" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n\"\"" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n1" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n1.0" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\nfalse" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\ntrue" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\nnull" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n29.0}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0{" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 1.0" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 false" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 true" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-29.0 null" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]\"\"" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]1.0" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]10" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}\"\"" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}1.0" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}10" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"\"\"" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"1" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"1.0" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"BYTES_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"EXPRESSION_VALUE\":\"\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]true" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}true" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"true" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]false" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}false" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"false" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]null" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}null" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"null" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]x" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]:" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}x" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}:" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"x" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\r\t\n    1x" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\r\t\n    10.0     x" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "10.0" ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n     x" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     x" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     x" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     x" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     x" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n    1" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n   10.0   \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "10.0" ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     " );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     " );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     " );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     " );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\":" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1:" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1.0:" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"BYTES_VALUE\" \t\r\n: \t\r\n\"fwo=\"} \r\t\n:" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"EXPRESSION_VALUE\" \t\r\n: \t\r\n\"env.JAVA_HOME\"}\t\r\n     :" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LIST\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"STRING\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"INT\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"LONG\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\t\r\n     :" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     :" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     :" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nnull\t\r\n     :" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",}" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],]" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},]" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",]" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\n]" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],:" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},:" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",:" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\n:" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],," );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},," );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",," );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],x" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},x" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",x" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]," );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}," );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\"," );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, -17 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+11.0 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "11.0" ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{ \t\r\n\"BYTES_VALUE\" \t\r\n: \t\r\n\"EAo=\" \t\r\n} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"EXPRESSION_VALUE\":\"expression\\\\Value\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_null_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nnull \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"true" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"false" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"null" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ':'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"::" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":truex" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":falsex" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":nullx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]:" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\":" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[][" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"BYTES_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"EXPRESSION_VALUE\":\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"OBJECT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"LIST\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"STRING\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"INT\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"LONG\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"BYTES\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"TYPE\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 {\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]true" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":truetrue" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":falsetrue" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":nulltrue" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]false" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":truefalse" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":falsefalse" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":nullfalse" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]null" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}null" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":truenull" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":falsenull" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_null() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":nullnull" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[]," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{}," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\"," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"}," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,," );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],:" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_colon() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,:" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1,]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0,]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 ,1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 ,1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBoolean_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BOOLEAN\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeUndefined_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"UNDEFINED\"},1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_true_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":true,1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_false_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":false,1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_null_comma_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":null,1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_listStart_listEnd_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":[],1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_objectStart_objectEnd_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\":{},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_string_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":\"value\",1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigInteger_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1 ,1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_numberBigDecimal_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":1.0 ,1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( "1.0" ) );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_bytes_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"BYTES_VALUE\":\"\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_expression_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"EXPRESSION_VALUE\":\"env.JAVA_HOME\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeList_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LIST\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeObject_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"OBJECT\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeProperty_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"PROPERTY\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeString_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"STRING\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeInt_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"INT\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeLong_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"LONG\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeDouble_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"DOUBLE\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigInteger_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_INTEGER\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBigDecimal_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BIG_DECIMAL\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeBytes_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"BYTES\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeExpression_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"EXPRESSION\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_typeType_comma_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\":{\"TYPE_MODEL_VALUE\":\"TYPE\"},1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_colon_ty