/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2016, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.dmr.stream;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.jboss.dmr.ModelType;
import org.junit.Test;

/**
 * @author <a href="mailto:ropalka@redhat.com">Richard Opalka</a>
 */
public final class InvalidModelReaderTestCase extends AbstractModelStreamsTestCase {

    @Override
    public ModelReader getModelReader( final String data ) throws IOException {
        final ByteArrayInputStream bais = new ByteArrayInputStream( data.getBytes() );
        return ModelStreamFactory.getInstance( false ).newModelReader( bais );
    }

    @Test
    public void noData() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    @Test
    public void emptyState() throws IOException, ModelException {
        read_unexpectedChar();
        read_arrow();
        read_comma();
        read_objectEnd();
        read_listEnd();
        read_propertyEnd();

        read_whitespace_unexpectedChar();
        read_whitespace_arrow();
        read_whitespace_comma();
        read_whitespace_objectEnd();
        read_whitespace_listEnd();
        read_whitespace_propertyEnd();
    }

    @Test
    public void unexpectedContentFollowingDmrData() throws IOException, ModelException {
        read_list_unexpectedChar();
        read_object_unexpectedChar();
        read_property_unexpectedChar();
        read_string_unexpectedChar();
        read_int_unexpectedChar();
        read_long_unexpectedChar();
        read_double_unexpectedChar();
        read_bigInteger_unexpectedChar();
        read_bigDecimal_unexpectedChar();
        read_bytes_unexpectedChar();
        read_expression_unexpectedChar();
        read_listType_unexpectedChar();
        read_objectType_unexpectedChar();
        read_propertyType_unexpectedChar();
        read_stringType_unexpectedChar();
        read_intType_unexpectedChar();
        read_longType_unexpectedChar();
        read_doubleType_unexpectedChar();
        read_bigIntegerType_unexpectedChar();
        read_bigDecimalType_unexpectedChar();
        read_bytesType_unexpectedChar();
        read_expressionType_unexpectedChar();
        read_typeType_unexpectedChar();
        read_booleanType_unexpectedChar();
        read_undefinedType_unexpectedChar();
        read_false_unexpectedChar();
        read_true_unexpectedChar();
        read_undefined_unexpectedChar();
    }

    @Test
    public void brokenPrimitives() throws IOException, ModelException {
        // number
        read_brokenInt_tooBigValue();
        read_brokenInt_tooSmallValue();
        read_brokenLong_tooBigValue();
        read_brokenLong_tooSmallValue();
        read_brokenDouble_tooBigValue();
        read_brokenDouble_tooSmallValue();
        read_brokenBigInteger2_unexpectedChar();
        read_brokenBigInteger2_unexpectedEOF();
        read_brokenBigInteger3_unexpectedChar();
        read_brokenBigInteger3_unexpectedEOF();
        read_brokenBigInteger5_unexpectedChar();
        read_brokenBigInteger5_unexpectedEOF();
        read_brokenBigInteger6_unexpectedChar();
        read_brokenBigInteger6_unexpectedEOF();
        read_brokenBigInteger_brokenValue_unexpectedChar();
        read_brokenBigInteger_brokenValue_unexpectedEOF();
        read_brokenBigDecimal2_unexpectedChar();
        read_brokenBigDecimal2_unexpectedEOF();
        read_brokenBigDecimal3_unexpectedChar();
        read_brokenBigDecimal3_unexpectedEOF();
        read_brokenBigDecimal5_unexpectedChar();
        read_brokenBigDecimal5_unexpectedEOF();
        read_brokenBigDecimal6_unexpectedChar();
        read_brokenBigDecimal6_unexpectedEOF();
        read_brokenBigDecimal_brokenValue_unexpectedChar();
        read_brokenBigDecimal_brokenValue_unexpectedEOF();
        // string
        read_brokenString_unexpectedEOF();
        // bytes
        read_brokenBytes2_unexpectedChar();
        read_brokenBytes2_unexpectedEOF();
        read_brokenBytes3_unexpectedChar();
        read_brokenBytes3_unexpectedEOF();
        read_brokenBytes_missingLeftBracket();
        read_brokenBytes_missingRightBracket();
        read_brokenBytes_wrongNumberValue1();
        read_brokenBytes_wrongHexedNumberValue1();
        read_brokenBytes_wrongNumberValue2();
        read_brokenBytes_wrongHexedNumberValue2();
        read_brokenBytes_missingComma();
        read_brokenBytes_duplicatedComma();
        // expression
        read_brokenExpression_unexpectedChar();
        read_brokenExpression_unexpectedEOF();
        read_brokenExpressionString_unexpectedEOF();
        // types
        read_brokenObjectType2_unexpectedChar();
        read_brokenObjectType2_unexpectedEOF();
        read_brokenObjectType3_unexpectedChar();
        read_brokenObjectType3_unexpectedEOF();
        read_brokenListType2_unexpectedChar();
        read_brokenListType2_unexpectedEOF();
        read_brokenListType3_unexpectedChar();
        read_brokenListType3_unexpectedEOF();
        read_brokenPropertyType2_unexpectedChar();
        read_brokenPropertyType2_unexpectedEOF();
        read_brokenPropertyType3_unexpectedChar();
        read_brokenPropertyType3_unexpectedEOF();
        read_brokenStringType2_unexpectedChar();
        read_brokenStringType2_unexpectedEOF();
        read_brokenStringType3_unexpectedChar();
        read_brokenStringType3_unexpectedEOF();
        read_brokenIntType2_unexpectedChar();
        read_brokenIntType2_unexpectedEOF();
        read_brokenIntType3_unexpectedChar();
        read_brokenIntType3_unexpectedEOF();
        read_brokenLongType2_unexpectedChar();
        read_brokenLongType2_unexpectedEOF();
        read_brokenLongType3_unexpectedChar();
        read_brokenLongType3_unexpectedEOF();
        read_brokenDoubleType2_unexpectedChar();
        read_brokenDoubleType2_unexpectedEOF();
        read_brokenDoubleType3_unexpectedChar();
        read_brokenDoubleType3_unexpectedEOF();
        read_brokenBigIntegerType2_unexpectedChar();
        read_brokenBigIntegerType2_unexpectedEOF();
        read_brokenBigIntegerType3_unexpectedChar();
        read_brokenBigIntegerType3_unexpectedEOF();
        read_brokenBigIntegerType5_unexpectedChar();
        read_brokenBigIntegerType5_unexpectedEOF();
        read_brokenBigIntegerType6_unexpectedChar();
        read_brokenBigIntegerType6_unexpectedEOF();
        read_brokenBigDecimalType2_unexpectedChar();
        read_brokenBigDecimalType2_unexpectedEOF();
        read_brokenBigDecimalType3_unexpectedChar();
        read_brokenBigDecimalType3_unexpectedEOF();
        read_brokenBigDecimalType5_unexpectedChar();
        read_brokenBigDecimalType5_unexpectedEOF();
        read_brokenBigDecimalType6_unexpectedChar();
        read_brokenBigDecimalType6_unexpectedEOF();
        read_brokenBytesType2_unexpectedChar();
        read_brokenBytesType2_unexpectedEOF();
        read_brokenBytesType3_unexpectedChar();
        read_brokenBytesType3_unexpectedEOF();
        read_brokenBooleanType2_unexpectedChar();
        read_brokenBooleanType2_unexpectedEOF();
        read_brokenBooleanType3_unexpectedChar();
        read_brokenBooleanType3_unexpectedEOF();
        read_brokenExpressionType2_unexpectedChar();
        read_brokenExpressionType2_unexpectedEOF();
        read_brokenExpressionType3_unexpectedChar();
        read_brokenExpressionType3_unexpectedEOF();
        read_brokenTypeType2_unexpectedChar();
        read_brokenTypeType2_unexpectedEOF();
        read_brokenTypeType3_unexpectedChar();
        read_brokenTypeType3_unexpectedEOF();
        read_brokenUndefinedType2_unexpectedChar();
        read_brokenUndefinedType2_unexpectedEOF();
        read_brokenUndefinedType3_unexpectedChar();
        read_brokenUndefinedType3_unexpectedEOF();
        // double specials
        read_brokenInfinity2_unexpectedChar();
        read_brokenInfinity2_unexpectedEOF();
        read_brokenInfinity3_unexpectedChar();
        read_brokenInfinity3_unexpectedEOF();
        read_brokenNegativeInfinity1_unexpectedChar();
        read_brokenNegativeInfinity1_unexpectedEOF();
        read_brokenNegativeInfinity2_unexpectedChar();
        read_brokenNegativeInfinity2_unexpectedEOF();
        read_brokenNegativeInfinity3_unexpectedChar();
        read_brokenNegativeInfinity3_unexpectedEOF();
        read_brokenPositiveInfinity1_unexpectedChar();
        read_brokenPositiveInfinity1_unexpectedEOF();
        read_brokenPositiveInfinity2_unexpectedChar();
        read_brokenPositiveInfinity2_unexpectedEOF();
        read_brokenPositiveInfinity3_unexpectedChar();
        read_brokenPositiveInfinity3_unexpectedEOF();
        read_brokenNaN2_unexpectedChar();
        read_brokenNaN2_unexpectedEOF();
        read_brokenNaN3_unexpectedChar();
        read_brokenNaN3_unexpectedEOF();
        read_brokenNegativeNaN1_unexpectedChar();
        read_brokenNegativeNaN1_unexpectedEOF();
        read_brokenNegativeNaN2_unexpectedChar();
        read_brokenNegativeNaN2_unexpectedEOF();
        read_brokenNegativeNaN3_unexpectedChar();
        read_brokenNegativeNaN3_unexpectedEOF();
        read_brokenPositiveNaN1_unexpectedChar();
        read_brokenPositiveNaN1_unexpectedEOF();
        read_brokenPositiveNaN2_unexpectedChar();
        read_brokenPositiveNaN2_unexpectedEOF();
        read_brokenPositiveNaN3_unexpectedChar();
        read_brokenPositiveNaN3_unexpectedEOF();
        // boolean
        read_brokenTrue_unexpectedChar();
        read_brokenTrue_unexpectedEOF();
        read_brokenFalse_unexpectedChar();
        read_brokenFalse_unexpectedEOF();
        // undefined
        read_brokenUndefined_unexpectedChar();
        read_brokenUndefined_unexpectedEOF();
        // arrow
        read_brokenArrow_unexpectedChar();
        read_brokenArrow_unexpectedEOF();
    }

    @Test
    public void emptyObjectStartState() throws IOException, ModelException {
        read_objectStart_unexpectedChar();
        read_objectStart_arrow();
        read_objectStart_comma();
        read_objectStart_listEnd();
        read_objectStart_listStart();
        read_objectStart_objectStart();
        read_objectStart_propertyEnd();
        read_objectStart_propertyStart();
        read_objectStart_numberInt();
        read_objectStart_numberLong();
        read_objectStart_numberDouble();
        read_objectStart_numberBigInteger();
        read_objectStart_numberBigDecimal();
        read_objectStart_bytes();
        read_objectStart_expression();
        read_objectStart_typeObject();
        read_objectStart_typeList();
        read_objectStart_typeProperty();
        read_objectStart_typeString();
        read_objectStart_typeInt();
        read_objectStart_typeLong();
        read_objectStart_typeDouble();
        read_objectStart_typeBigInteger();
        read_objectStart_typeBigDecimal();
        read_objectStart_typeBoolean();
        read_objectStart_typeBytes();
        read_objectStart_typeExpression();
        read_objectStart_typeType();
        read_objectStart_typeUndefined();
        read_objectStart_false();
        read_objectStart_true();
        read_objectStart_undefined();

        read_objectStart_whitespace_unexpectedChar();
        read_objectStart_whitespace_arrow();
        read_objectStart_whitespace_comma();
        read_objectStart_whitespace_listEnd();
        read_objectStart_whitespace_listStart();
        read_objectStart_whitespace_objectStart();
        read_objectStart_whitespace_propertyEnd();
        read_objectStart_whitespace_propertyStart();
        read_objectStart_whitespace_numberInt();
        read_objectStart_whitespace_numberLong();
        read_objectStart_whitespace_numberDouble();
        read_objectStart_whitespace_numberBigInteger();
        read_objectStart_whitespace_numberBigDecimal();
        read_objectStart_whitespace_bytes();
        read_objectStart_whitespace_expression();
        read_objectStart_whitespace_typeObject();
        read_objectStart_whitespace_typeList();
        read_objectStart_whitespace_typeProperty();
        read_objectStart_whitespace_typeString();
        read_objectStart_whitespace_typeInt();
        read_objectStart_whitespace_typeLong();
        read_objectStart_whitespace_typeDouble();
        read_objectStart_whitespace_typeBigInteger();
        read_objectStart_whitespace_typeBigDecimal();
        read_objectStart_whitespace_typeBoolean();
        read_objectStart_whitespace_typeBytes();
        read_objectStart_whitespace_typeExpression();
        read_objectStart_whitespace_typeType();
        read_objectStart_whitespace_typeUndefined();
        read_objectStart_whitespace_false();
        read_objectStart_whitespace_true();
        read_objectStart_whitespace_undefined();
    }

    @Test
    public void emptyListStartState() throws IOException, ModelException {
        read_listStart_unexpectedChar();
        read_listStart_arrow();
        read_listStart_comma();
        read_listStart_objectEnd();
        read_listStart_propertyEnd();

        read_listStart_whitespace_unexpectedChar();
        read_listStart_whitespace_arrow();
        read_listStart_whitespace_comma();
        read_listStart_whitespace_objectEnd();
        read_listStart_whitespace_propertyEnd();
    }

    @Test
    public void emptyPropertyStartState() throws IOException, ModelException {
        read_propertyStart_unexpectedChar();
        read_propertyStart_arrow();
        read_propertyStart_comma();
        read_propertyStart_listEnd();
        read_propertyStart_listStart();
        read_propertyStart_objectEnd();
        read_propertyStart_objectStart();
        read_propertyStart_propertyEnd();
        read_propertyStart_propertyStart();
        read_propertyStart_numberInt();
        read_propertyStart_numberLong();
        read_propertyStart_numberDouble();
        read_propertyStart_numberBigInteger();
        read_propertyStart_numberBigDecimal();
        read_propertyStart_bytes();
        read_propertyStart_expression();
        read_propertyStart_typeObject();
        read_propertyStart_typeList();
        read_propertyStart_typeProperty();
        read_propertyStart_typeString();
        read_propertyStart_typeInt();
        read_propertyStart_typeLong();
        read_propertyStart_typeDouble();
        read_propertyStart_typeBigInteger();
        read_propertyStart_typeBigDecimal();
        read_propertyStart_typeBoolean();
        read_propertyStart_typeBytes();
        read_propertyStart_typeExpression();
        read_propertyStart_typeType();
        read_propertyStart_typeUndefined();
        read_propertyStart_false();
        read_propertyStart_true();
        read_propertyStart_undefined();

        read_propertyStart_whitespace_unexpectedChar();
        read_propertyStart_whitespace_arrow();
        read_propertyStart_whitespace_comma();
        read_propertyStart_whitespace_listEnd();
        read_propertyStart_whitespace_listStart();
        read_propertyStart_whitespace_objectEnd();
        read_propertyStart_whitespace_objectStart();
        read_propertyStart_whitespace_propertyEnd();
        read_propertyStart_whitespace_propertyStart();
        read_propertyStart_whitespace_numberInt();
        read_propertyStart_whitespace_numberLong();
        read_propertyStart_whitespace_numberDouble();
        read_propertyStart_whitespace_numberBigInteger();
        read_propertyStart_whitespace_numberBigDecimal();
        read_propertyStart_whitespace_bytes();
        read_propertyStart_whitespace_expression();
        read_propertyStart_whitespace_typeObject();
        read_propertyStart_whitespace_typeList();
        read_propertyStart_whitespace_typeProperty();
        read_propertyStart_whitespace_typeString();
        read_propertyStart_whitespace_typeInt();
        read_propertyStart_whitespace_typeLong();
        read_propertyStart_whitespace_typeDouble();
        read_propertyStart_whitespace_typeBigInteger();
        read_propertyStart_whitespace_typeBigDecimal();
        read_propertyStart_whitespace_typeBoolean();
        read_propertyStart_whitespace_typeBytes();
        read_propertyStart_whitespace_typeExpression();
        read_propertyStart_whitespace_typeType();
        read_propertyStart_whitespace_typeUndefined();
        read_propertyStart_whitespace_false();
        read_propertyStart_whitespace_true();
        read_propertyStart_whitespace_undefined();
    }

    @Test
    public void notEmptyListMissingCommaAfterFirstItem() throws IOException, ModelException {
        read_listStart_listStart_listEnd_unexpectedChar();
        read_listStart_listStart_listEnd_unexpectedEOF();
        read_listStart_listStart_listEnd_arrow();
        read_listStart_listStart_listEnd_listStart();
        read_listStart_listStart_listEnd_objectEnd();
        read_listStart_listStart_listEnd_objectStart();
        read_listStart_listStart_listEnd_propertyEnd();
        read_listStart_listStart_listEnd_propertyStart();
        read_listStart_listStart_listEnd_string();
        read_listStart_listStart_listEnd_numberInt();
        read_listStart_listStart_listEnd_numberLong();
        read_listStart_listStart_listEnd_numberDouble();
        read_listStart_listStart_listEnd_numberBigInteger();
        read_listStart_listStart_listEnd_numberBigDecimal();
        read_listStart_listStart_listEnd_bytes();
        read_listStart_listStart_listEnd_expression();
        read_listStart_listStart_listEnd_typeObject();
        read_listStart_listStart_listEnd_typeList();
        read_listStart_listStart_listEnd_typeProperty();
        read_listStart_listStart_listEnd_typeString();
        read_listStart_listStart_listEnd_typeInt();
        read_listStart_listStart_listEnd_typeLong();
        read_listStart_listStart_listEnd_typeDouble();
        read_listStart_listStart_listEnd_typeBigDecimal();
        read_listStart_listStart_listEnd_typeBigInteger();
        read_listStart_listStart_listEnd_typeBoolean();
        read_listStart_listStart_listEnd_typeBytes();
        read_listStart_listStart_listEnd_typeExpression();
        read_listStart_listStart_listEnd_typeType();
        read_listStart_listStart_listEnd_typeUndefined();
        read_listStart_listStart_listEnd_false();
        read_listStart_listStart_listEnd_true();
        read_listStart_listStart_listEnd_undefined();

        read_listStart_objectStart_objectEnd_unexpectedChar();
        read_listStart_objectStart_objectEnd_unexpectedEOF();
        read_listStart_objectStart_objectEnd_arrow();
        read_listStart_objectStart_objectEnd_listStart();
        read_listStart_objectStart_objectEnd_objectEnd();
        read_listStart_objectStart_objectEnd_objectStart();
        read_listStart_objectStart_objectEnd_propertyEnd();
        read_listStart_objectStart_objectEnd_propertyStart();
        read_listStart_objectStart_objectEnd_string();
        read_listStart_objectStart_objectEnd_numberInt();
        read_listStart_objectStart_objectEnd_numberLong();
        read_listStart_objectStart_objectEnd_numberDouble();
        read_listStart_objectStart_objectEnd_numberBigInteger();
        read_listStart_objectStart_objectEnd_numberBigDecimal();
        read_listStart_objectStart_objectEnd_bytes();
        read_listStart_objectStart_objectEnd_expression();
        read_listStart_objectStart_objectEnd_typeObject();
        read_listStart_objectStart_objectEnd_typeList();
        read_listStart_objectStart_objectEnd_typeProperty();
        read_listStart_objectStart_objectEnd_typeString();
        read_listStart_objectStart_objectEnd_typeInt();
        read_listStart_objectStart_objectEnd_typeLong();
        read_listStart_objectStart_objectEnd_typeDouble();
        read_listStart_objectStart_objectEnd_typeBigDecimal();
        read_listStart_objectStart_objectEnd_typeBigInteger();
        read_listStart_objectStart_objectEnd_typeBoolean();
        read_listStart_objectStart_objectEnd_typeBytes();
        read_listStart_objectStart_objectEnd_typeExpression();
        read_listStart_objectStart_objectEnd_typeType();
        read_listStart_objectStart_objectEnd_typeUndefined();
        read_listStart_objectStart_objectEnd_false();
        read_listStart_objectStart_objectEnd_true();
        read_listStart_objectStart_objectEnd_undefined();

        read_listStart_propertyStart_propertyEnd_unexpectedChar();
        read_listStart_propertyStart_propertyEnd_unexpectedEOF();
        read_listStart_propertyStart_propertyEnd_arrow();
        read_listStart_propertyStart_propertyEnd_listStart();
        read_listStart_propertyStart_propertyEnd_objectEnd();
        read_listStart_propertyStart_propertyEnd_objectStart();
        read_listStart_propertyStart_propertyEnd_propertyEnd();
        read_listStart_propertyStart_propertyEnd_propertyStart();
        read_listStart_propertyStart_propertyEnd_string();
        read_listStart_propertyStart_propertyEnd_numberInt();
        read_listStart_propertyStart_propertyEnd_numberLong();
        read_listStart_propertyStart_propertyEnd_numberDouble();
        read_listStart_propertyStart_propertyEnd_numberBigInteger();
        read_listStart_propertyStart_propertyEnd_numberBigDecimal();
        read_listStart_propertyStart_propertyEnd_bytes();
        read_listStart_propertyStart_propertyEnd_expression();
        read_listStart_propertyStart_propertyEnd_typeObject();
        read_listStart_propertyStart_propertyEnd_typeList();
        read_listStart_propertyStart_propertyEnd_typeProperty();
        read_listStart_propertyStart_propertyEnd_typeString();
        read_listStart_propertyStart_propertyEnd_typeInt();
        read_listStart_propertyStart_propertyEnd_typeLong();
        read_listStart_propertyStart_propertyEnd_typeDouble();
        read_listStart_propertyStart_propertyEnd_typeBigDecimal();
        read_listStart_propertyStart_propertyEnd_typeBigInteger();
        read_listStart_propertyStart_propertyEnd_typeBoolean();
        read_listStart_propertyStart_propertyEnd_typeBytes();
        read_listStart_propertyStart_propertyEnd_typeExpression();
        read_listStart_propertyStart_propertyEnd_typeType();
        read_listStart_propertyStart_propertyEnd_typeUndefined();
        read_listStart_propertyStart_propertyEnd_false();
        read_listStart_propertyStart_propertyEnd_true();
        read_listStart_propertyStart_propertyEnd_undefined();

        read_listStart_string_unexpectedChar();
        read_listStart_string_unexpectedEOF();
        read_listStart_string_arrow();
        read_listStart_string_listStart();
        read_listStart_string_objectEnd();
        read_listStart_string_objectStart();
        read_listStart_string_propertyEnd();
        read_listStart_string_propertyStart();
        read_listStart_string_string();
        read_listStart_string_numberInt();
        read_listStart_string_numberLong();
        read_listStart_string_numberDouble();
        read_listStart_string_numberBigInteger();
        read_listStart_string_numberBigDecimal();
        read_listStart_string_bytes();
        read_listStart_string_expression();
        read_listStart_string_typeObject();
        read_listStart_string_typeList();
        read_listStart_string_typeProperty();
        read_listStart_string_typeString();
        read_listStart_string_typeInt();
        read_listStart_string_typeLong();
        read_listStart_string_typeDouble();
        read_listStart_string_typeBigInteger();
        read_listStart_string_typeBigDecimal();
        read_listStart_string_typeBoolean();
        read_listStart_string_typeBytes();
        read_listStart_string_typeExpression();
        read_listStart_string_typeType();
        read_listStart_string_typeUndefined();
        read_listStart_string_false();
        read_listStart_string_true();
        read_listStart_string_undefined();

        read_listStart_numberInt_unexpectedChar();
        read_listStart_numberInt_unexpectedEOF();
        read_listStart_numberInt_arrow();
        read_listStart_numberInt_listStart();
        read_listStart_numberInt_objectEnd();
        read_listStart_numberInt_objectStart();
        read_listStart_numberInt_propertyEnd();
        read_listStart_numberInt_propertyStart();
        read_listStart_numberInt_string();
        read_listStart_numberInt_numberInt();
        read_listStart_numberInt_numberLong();
        read_listStart_numberInt_numberDouble();
        read_listStart_numberInt_numberBigInteger();
        read_listStart_numberInt_numberBigDecimal();
        read_listStart_numberInt_bytes();
        read_listStart_numberInt_expression();
        read_listStart_numberInt_typeObject();
        read_listStart_numberInt_typeList();
        read_listStart_numberInt_typeProperty();
        read_listStart_numberInt_typeString();
        read_listStart_numberInt_typeInt();
        read_listStart_numberInt_typeLong();
        read_listStart_numberInt_typeDouble();
        read_listStart_numberInt_typeBigInteger();
        read_listStart_numberInt_typeBigDecimal();
        read_listStart_numberInt_typeBoolean();
        read_listStart_numberInt_typeBytes();
        read_listStart_numberInt_typeExpression();
        read_listStart_numberInt_typeType();
        read_listStart_numberInt_typeUndefined();
        read_listStart_numberInt_false();
        read_listStart_numberInt_true();
        read_listStart_numberInt_undefined();

        read_listStart_numberLong_unexpectedChar();
        read_listStart_numberLong_unexpectedEOF();
        read_listStart_numberLong_arrow();
        read_listStart_numberLong_listStart();
        read_listStart_numberLong_objectEnd();
        read_listStart_numberLong_objectStart();
        read_listStart_numberLong_propertyEnd();
        read_listStart_numberLong_propertyStart();
        read_listStart_numberLong_string();
        read_listStart_numberLong_numberInt();
        read_listStart_numberLong_numberLong();
        read_listStart_numberLong_numberDouble();
        read_listStart_numberLong_numberBigInteger();
        read_listStart_numberLong_numberBigDecimal();
        read_listStart_numberLong_bytes();
        read_listStart_numberLong_expression();
        read_listStart_numberLong_typeObject();
        read_listStart_numberLong_typeList();
        read_listStart_numberLong_typeProperty();
        read_listStart_numberLong_typeString();
        read_listStart_numberLong_typeInt();
        read_listStart_numberLong_typeLong();
        read_listStart_numberLong_typeDouble();
        read_listStart_numberLong_typeBigInteger();
        read_listStart_numberLong_typeBigDecimal();
        read_listStart_numberLong_typeBoolean();
        read_listStart_numberLong_typeBytes();
        read_listStart_numberLong_typeExpression();
        read_listStart_numberLong_typeType();
        read_listStart_numberLong_typeUndefined();
        read_listStart_numberLong_false();
        read_listStart_numberLong_true();
        read_listStart_numberLong_undefined();

        read_listStart_numberDouble_unexpectedChar();
        read_listStart_numberDouble_unexpectedEOF();
        read_listStart_numberDouble_arrow();
        read_listStart_numberDouble_listStart();
        read_listStart_numberDouble_objectEnd();
        read_listStart_numberDouble_objectStart();
        read_listStart_numberDouble_propertyEnd();
        read_listStart_numberDouble_propertyStart();
        read_listStart_numberDouble_string();
        read_listStart_numberDouble_numberInt();
        read_listStart_numberDouble_numberLong();
        read_listStart_numberDouble_numberDouble();
        read_listStart_numberDouble_numberBigInteger();
        read_listStart_numberDouble_numberBigDecimal();
        read_listStart_numberDouble_bytes();
        read_listStart_numberDouble_expression();
        read_listStart_numberDouble_typeObject();
        read_listStart_numberDouble_typeList();
        read_listStart_numberDouble_typeProperty();
        read_listStart_numberDouble_typeString();
        read_listStart_numberDouble_typeInt();
        read_listStart_numberDouble_typeLong();
        read_listStart_numberDouble_typeDouble();
        read_listStart_numberDouble_typeBigInteger();
        read_listStart_numberDouble_typeBigDecimal();
        read_listStart_numberDouble_typeBoolean();
        read_listStart_numberDouble_typeBytes();
        read_listStart_numberDouble_typeExpression();
        read_listStart_numberDouble_typeType();
        read_listStart_numberDouble_typeUndefined();
        read_listStart_numberDouble_false();
        read_listStart_numberDouble_true();
        read_listStart_numberDouble_undefined();

        read_listStart_numberBigInteger_unexpectedChar();
        read_listStart_numberBigInteger_unexpectedEOF();
        read_listStart_numberBigInteger_arrow();
        read_listStart_numberBigInteger_listStart();
        read_listStart_numberBigInteger_objectEnd();
        read_listStart_numberBigInteger_objectStart();
        read_listStart_numberBigInteger_propertyEnd();
        read_listStart_numberBigInteger_propertyStart();
        read_listStart_numberBigInteger_string();
        read_listStart_numberBigInteger_numberInt();
        read_listStart_numberBigInteger_numberLong();
        read_listStart_numberBigInteger_numberDouble();
        read_listStart_numberBigInteger_numberBigInteger();
        read_listStart_numberBigInteger_numberBigDecimal();
        read_listStart_numberBigInteger_bytes();
        read_listStart_numberBigInteger_expression();
        read_listStart_numberBigInteger_typeObject();
        read_listStart_numberBigInteger_typeList();
        read_listStart_numberBigInteger_typeProperty();
        read_listStart_numberBigInteger_typeString();
        read_listStart_numberBigInteger_typeInt();
        read_listStart_numberBigInteger_typeLong();
        read_listStart_numberBigInteger_typeDouble();
        read_listStart_numberBigInteger_typeBigInteger();
        read_listStart_numberBigInteger_typeBigDecimal();
        read_listStart_numberBigInteger_typeBoolean();
        read_listStart_numberBigInteger_typeBytes();
        read_listStart_numberBigInteger_typeExpression();
        read_listStart_numberBigInteger_typeType();
        read_listStart_numberBigInteger_typeUndefined();
        read_listStart_numberBigInteger_false();
        read_listStart_numberBigInteger_true();
        read_listStart_numberBigInteger_undefined();

        read_listStart_numberBigDecimal_unexpectedChar();
        read_listStart_numberBigDecimal_unexpectedEOF();
        read_listStart_numberBigDecimal_arrow();
        read_listStart_numberBigDecimal_listStart();
        read_listStart_numberBigDecimal_objectEnd();
        read_listStart_numberBigDecimal_objectStart();
        read_listStart_numberBigDecimal_propertyEnd();
        read_listStart_numberBigDecimal_propertyStart();
        read_listStart_numberBigDecimal_string();
        read_listStart_numberBigDecimal_numberInt();
        read_listStart_numberBigDecimal_numberLong();
        read_listStart_numberBigDecimal_numberDouble();
        read_listStart_numberBigDecimal_numberBigInteger();
        read_listStart_numberBigDecimal_numberBigDecimal();
        read_listStart_numberBigDecimal_bytes();
        read_listStart_numberBigDecimal_expression();
        read_listStart_numberBigDecimal_typeObject();
        read_listStart_numberBigDecimal_typeList();
        read_listStart_numberBigDecimal_typeProperty();
        read_listStart_numberBigDecimal_typeString();
        read_listStart_numberBigDecimal_typeInt();
        read_listStart_numberBigDecimal_typeLong();
        read_listStart_numberBigDecimal_typeDouble();
        read_listStart_numberBigDecimal_typeBigInteger();
        read_listStart_numberBigDecimal_typeBigDecimal();
        read_listStart_numberBigDecimal_typeBoolean();
        read_listStart_numberBigDecimal_typeBytes();
        read_listStart_numberBigDecimal_typeExpression();
        read_listStart_numberBigDecimal_typeType();
        read_listStart_numberBigDecimal_typeUndefined();
        read_listStart_numberBigDecimal_false();
        read_listStart_numberBigDecimal_true();
        read_listStart_numberBigDecimal_undefined();

        read_listStart_bytes_unexpectedChar();
        read_listStart_bytes_unexpectedEOF();
        read_listStart_bytes_arrow();
        read_listStart_bytes_listStart();
        read_listStart_bytes_objectEnd();
        read_listStart_bytes_objectStart();
        read_listStart_bytes_propertyEnd();
        read_listStart_bytes_propertyStart();
        read_listStart_bytes_string();
        read_listStart_bytes_numberInt();
        read_listStart_bytes_numberLong();
        read_listStart_bytes_numberDouble();
        read_listStart_bytes_numberBigInteger();
        read_listStart_bytes_numberBigDecimal();
        read_listStart_bytes_bytes();
        read_listStart_bytes_expression();
        read_listStart_bytes_typeObject();
        read_listStart_bytes_typeList();
        read_listStart_bytes_typeProperty();
        read_listStart_bytes_typeString();
        read_listStart_bytes_typeInt();
        read_listStart_bytes_typeLong();
        read_listStart_bytes_typeDouble();
        read_listStart_bytes_typeBigInteger();
        read_listStart_bytes_typeBigDecimal();
        read_listStart_bytes_typeBoolean();
        read_listStart_bytes_typeBytes();
        read_listStart_bytes_typeExpression();
        read_listStart_bytes_typeType();
        read_listStart_bytes_typeUndefined();
        read_listStart_bytes_false();
        read_listStart_bytes_true();
        read_listStart_bytes_undefined();

        read_listStart_expression_unexpectedChar();
        read_listStart_expression_unexpectedEOF();
        read_listStart_expression_arrow();
        read_listStart_expression_listStart();
        read_listStart_expression_objectEnd();
        read_listStart_expression_objectStart();
        read_listStart_expression_propertyEnd();
        read_listStart_expression_propertyStart();
        read_listStart_expression_string();
        read_listStart_expression_numberInt();
        read_listStart_expression_numberLong();
        read_listStart_expression_numberDouble();
        read_listStart_expression_numberBigInteger();
        read_listStart_expression_numberBigDecimal();
        read_listStart_expression_bytes();
        read_listStart_expression_expression();
        read_listStart_expression_typeObject();
        read_listStart_expression_typeList();
        read_listStart_expression_typeProperty();
        read_listStart_expression_typeString();
        read_listStart_expression_typeInt();
        read_listStart_expression_typeLong();
        read_listStart_expression_typeDouble();
        read_listStart_expression_typeBigInteger();
        read_listStart_expression_typeBigDecimal();
        read_listStart_expression_typeBoolean();
        read_listStart_expression_typeBytes();
        read_listStart_expression_typeExpression();
        read_listStart_expression_typeType();
        read_listStart_expression_typeUndefined();
        read_listStart_expression_false();
        read_listStart_expression_true();
        read_listStart_expression_undefined();

        read_listStart_typeObject_unexpectedChar();
        read_listStart_typeObject_unexpectedEOF();
        read_listStart_typeObject_arrow();
        read_listStart_typeObject_listStart();
        read_listStart_typeObject_objectEnd();
        read_listStart_typeObject_objectStart();
        read_listStart_typeObject_propertyEnd();
        read_listStart_typeObject_propertyStart();
        read_listStart_typeObject_string();
        read_listStart_typeObject_numberInt();
        read_listStart_typeObject_numberLong();
        read_listStart_typeObject_numberDouble();
        read_listStart_typeObject_numberBigInteger();
        read_listStart_typeObject_numberBigDecimal();
        read_listStart_typeObject_bytes();
        read_listStart_typeObject_expression();
        read_listStart_typeObject_typeObject();
        read_listStart_typeObject_typeList();
        read_listStart_typeObject_typeProperty();
        read_listStart_typeObject_typeString();
        read_listStart_typeObject_typeInt();
        read_listStart_typeObject_typeLong();
        read_listStart_typeObject_typeDouble();
        read_listStart_typeObject_typeBigInteger();
        read_listStart_typeObject_typeBigDecimal();
        read_listStart_typeObject_typeBoolean();
        read_listStart_typeObject_typeBytes();
        read_listStart_typeObject_typeExpression();
        read_listStart_typeObject_typeType();
        read_listStart_typeObject_typeUndefined();
        read_listStart_typeObject_false();
        read_listStart_typeObject_true();
        read_listStart_typeObject_undefined();

        read_listStart_typeList_unexpectedChar();
        read_listStart_typeList_unexpectedEOF();
        read_listStart_typeList_arrow();
        read_listStart_typeList_listStart();
        read_listStart_typeList_objectEnd();
        read_listStart_typeList_objectStart();
        read_listStart_typeList_propertyEnd();
        read_listStart_typeList_propertyStart();
        read_listStart_typeList_string();
        read_listStart_typeList_numberInt();
        read_listStart_typeList_numberLong();
        read_listStart_typeList_numberDouble();
        read_listStart_typeList_numberBigInteger();
        read_listStart_typeList_numberBigDecimal();
        read_listStart_typeList_bytes();
        read_listStart_typeList_expression();
        read_listStart_typeList_typeObject();
        read_listStart_typeList_typeList();
        read_listStart_typeList_typeProperty();
        read_listStart_typeList_typeString();
        read_listStart_typeList_typeInt();
        read_listStart_typeList_typeLong();
        read_listStart_typeList_typeDouble();
        read_listStart_typeList_typeBigInteger();
        read_listStart_typeList_typeBigDecimal();
        read_listStart_typeList_typeBoolean();
        read_listStart_typeList_typeBytes();
        read_listStart_typeList_typeExpression();
        read_listStart_typeList_typeType();
        read_listStart_typeList_typeUndefined();
        read_listStart_typeList_false();
        read_listStart_typeList_true();
        read_listStart_typeList_undefined();

        read_listStart_typeProperty_unexpectedChar();
        read_listStart_typeProperty_unexpectedEOF();
        read_listStart_typeProperty_arrow();
        read_listStart_typeProperty_listStart();
        read_listStart_typeProperty_objectEnd();
        read_listStart_typeProperty_objectStart();
        read_listStart_typeProperty_propertyEnd();
        read_listStart_typeProperty_propertyStart();
        read_listStart_typeProperty_string();
        read_listStart_typeProperty_numberInt();
        read_listStart_typeProperty_numberLong();
        read_listStart_typeProperty_numberDouble();
        read_listStart_typeProperty_numberBigInteger();
        read_listStart_typeProperty_numberBigDecimal();
        read_listStart_typeProperty_bytes();
        read_listStart_typeProperty_expression();
        read_listStart_typeProperty_typeObject();
        read_listStart_typeProperty_typeList();
        read_listStart_typeProperty_typeProperty();
        read_listStart_typeProperty_typeString();
        read_listStart_typeProperty_typeInt();
        read_listStart_typeProperty_typeLong();
        read_listStart_typeProperty_typeDouble();
        read_listStart_typeProperty_typeBigInteger();
        read_listStart_typeProperty_typeBigDecimal();
        read_listStart_typeProperty_typeBoolean();
        read_listStart_typeProperty_typeBytes();
        read_listStart_typeProperty_typeExpression();
        read_listStart_typeProperty_typeType();
        read_listStart_typeProperty_typeUndefined();
        read_listStart_typeProperty_false();
        read_listStart_typeProperty_true();
        read_listStart_typeProperty_undefined();

        read_listStart_typeString_unexpectedChar();
        read_listStart_typeString_unexpectedEOF();
        read_listStart_typeString_arrow();
        read_listStart_typeString_listStart();
        read_listStart_typeString_objectEnd();
        read_listStart_typeString_objectStart();
        read_listStart_typeString_propertyEnd();
        read_listStart_typeString_propertyStart();
        read_listStart_typeString_string();
        read_listStart_typeString_numberInt();
        read_listStart_typeString_numberLong();
        read_listStart_typeString_numberDouble();
        read_listStart_typeString_numberBigInteger();
        read_listStart_typeString_numberBigDecimal();
        read_listStart_typeString_bytes();
        read_listStart_typeString_expression();
        read_listStart_typeString_typeObject();
        read_listStart_typeString_typeList();
        read_listStart_typeString_typeProperty();
        read_listStart_typeString_typeString();
        read_listStart_typeString_typeInt();
        read_listStart_typeString_typeLong();
        read_listStart_typeString_typeDouble();
        read_listStart_typeString_typeBigInteger();
        read_listStart_typeString_typeBigDecimal();
        read_listStart_typeString_typeBoolean();
        read_listStart_typeString_typeBytes();
        read_listStart_typeString_typeExpression();
        read_listStart_typeString_typeType();
        read_listStart_typeString_typeUndefined();
        read_listStart_typeString_false();
        read_listStart_typeString_true();
        read_listStart_typeString_undefined();

        read_listStart_typeInt_unexpectedChar();
        read_listStart_typeInt_unexpectedEOF();
        read_listStart_typeInt_arrow();
        read_listStart_typeInt_listStart();
        read_listStart_typeInt_objectEnd();
        read_listStart_typeInt_objectStart();
        read_listStart_typeInt_propertyEnd();
        read_listStart_typeInt_propertyStart();
        read_listStart_typeInt_string();
        read_listStart_typeInt_numberInt();
        read_listStart_typeInt_numberLong();
        read_listStart_typeInt_numberDouble();
        read_listStart_typeInt_numberBigInteger();
        read_listStart_typeInt_numberBigDecimal();
        read_listStart_typeInt_bytes();
        read_listStart_typeInt_expression();
        read_listStart_typeInt_typeObject();
        read_listStart_typeInt_typeList();
        read_listStart_typeInt_typeProperty();
        read_listStart_typeInt_typeString();
        read_listStart_typeInt_typeInt();
        read_listStart_typeInt_typeLong();
        read_listStart_typeInt_typeDouble();
        read_listStart_typeInt_typeBigInteger();
        read_listStart_typeInt_typeBigDecimal();
        read_listStart_typeInt_typeBoolean();
        read_listStart_typeInt_typeBytes();
        read_listStart_typeInt_typeExpression();
        read_listStart_typeInt_typeType();
        read_listStart_typeInt_typeUndefined();
        read_listStart_typeInt_false();
        read_listStart_typeInt_true();
        read_listStart_typeInt_undefined();

        read_listStart_typeLong_unexpectedChar();
        read_listStart_typeLong_unexpectedEOF();
        read_listStart_typeLong_arrow();
        read_listStart_typeLong_listStart();
        read_listStart_typeLong_objectEnd();
        read_listStart_typeLong_objectStart();
        read_listStart_typeLong_propertyEnd();
        read_listStart_typeLong_propertyStart();
        read_listStart_typeLong_string();
        read_listStart_typeLong_numberInt();
        read_listStart_typeLong_numberLong();
        read_listStart_typeLong_numberDouble();
        read_listStart_typeLong_numberBigInteger();
        read_listStart_typeLong_numberBigDecimal();
        read_listStart_typeLong_bytes();
        read_listStart_typeLong_expression();
        read_listStart_typeLong_typeObject();
        read_listStart_typeLong_typeList();
        read_listStart_typeLong_typeProperty();
        read_listStart_typeLong_typeString();
        read_listStart_typeLong_typeInt();
        read_listStart_typeLong_typeLong();
        read_listStart_typeLong_typeDouble();
        read_listStart_typeLong_typeBigInteger();
        read_listStart_typeLong_typeBigDecimal();
        read_listStart_typeLong_typeBoolean();
        read_listStart_typeLong_typeBytes();
        read_listStart_typeLong_typeExpression();
        read_listStart_typeLong_typeType();
        read_listStart_typeLong_typeUndefined();
        read_listStart_typeLong_false();
        read_listStart_typeLong_true();
        read_listStart_typeLong_undefined();

        read_listStart_typeDouble_unexpectedChar();
        read_listStart_typeDouble_unexpectedEOF();
        read_listStart_typeDouble_arrow();
        read_listStart_typeDouble_listStart();
        read_listStart_typeDouble_objectEnd();
        read_listStart_typeDouble_objectStart();
        read_listStart_typeDouble_propertyEnd();
        read_listStart_typeDouble_propertyStart();
        read_listStart_typeDouble_string();
        read_listStart_typeDouble_numberInt();
        read_listStart_typeDouble_numberLong();
        read_listStart_typeDouble_numberDouble();
        read_listStart_typeDouble_numberBigInteger();
        read_listStart_typeDouble_numberBigDecimal();
        read_listStart_typeDouble_bytes();
        read_listStart_typeDouble_expression();
        read_listStart_typeDouble_typeObject();
        read_listStart_typeDouble_typeList();
        read_listStart_typeDouble_typeProperty();
        read_listStart_typeDouble_typeString();
        read_listStart_typeDouble_typeInt();
        read_listStart_typeDouble_typeLong();
        read_listStart_typeDouble_typeDouble();
        read_listStart_typeDouble_typeBigInteger();
        read_listStart_typeDouble_typeBigDecimal();
        read_listStart_typeDouble_typeBoolean();
        read_listStart_typeDouble_typeBytes();
        read_listStart_typeDouble_typeExpression();
        read_listStart_typeDouble_typeType();
        read_listStart_typeDouble_typeUndefined();
        read_listStart_typeDouble_false();
        read_listStart_typeDouble_true();
        read_listStart_typeDouble_undefined();

        read_listStart_typeBigInteger_unexpectedChar();
        read_listStart_typeBigInteger_unexpectedEOF();
        read_listStart_typeBigInteger_arrow();
        read_listStart_typeBigInteger_listStart();
        read_listStart_typeBigInteger_objectEnd();
        read_listStart_typeBigInteger_objectStart();
        read_listStart_typeBigInteger_propertyEnd();
        read_listStart_typeBigInteger_propertyStart();
        read_listStart_typeBigInteger_string();
        read_listStart_typeBigInteger_numberInt();
        read_listStart_typeBigInteger_numberLong();
        read_listStart_typeBigInteger_numberDouble();
        read_listStart_typeBigInteger_numberBigInteger();
        read_listStart_typeBigInteger_numberBigDecimal();
        read_listStart_typeBigInteger_bytes();
        read_listStart_typeBigInteger_expression();
        read_listStart_typeBigInteger_typeObject();
        read_listStart_typeBigInteger_typeList();
        read_listStart_typeBigInteger_typeProperty();
        read_listStart_typeBigInteger_typeString();
        read_listStart_typeBigInteger_typeInt();
        read_listStart_typeBigInteger_typeLong();
        read_listStart_typeBigInteger_typeDouble();
        read_listStart_typeBigInteger_typeBigInteger();
        read_listStart_typeBigInteger_typeBigDecimal();
        read_listStart_typeBigInteger_typeBoolean();
        read_listStart_typeBigInteger_typeBytes();
        read_listStart_typeBigInteger_typeExpression();
        read_listStart_typeBigInteger_typeType();
        read_listStart_typeBigInteger_typeUndefined();
        read_listStart_typeBigInteger_false();
        read_listStart_typeBigInteger_true();
        read_listStart_typeBigInteger_undefined();

        read_listStart_typeBigDecimal_unexpectedChar();
        read_listStart_typeBigDecimal_unexpectedEOF();
        read_listStart_typeBigDecimal_arrow();
        read_listStart_typeBigDecimal_listStart();
        read_listStart_typeBigDecimal_objectEnd();
        read_listStart_typeBigDecimal_objectStart();
        read_listStart_typeBigDecimal_propertyEnd();
        read_listStart_typeBigDecimal_propertyStart();
        read_listStart_typeBigDecimal_string();
        read_listStart_typeBigDecimal_numberInt();
        read_listStart_typeBigDecimal_numberLong();
        read_listStart_typeBigDecimal_numberDouble();
        read_listStart_typeBigDecimal_numberBigInteger();
        read_listStart_typeBigDecimal_numberBigDecimal();
        read_listStart_typeBigDecimal_bytes();
        read_listStart_typeBigDecimal_expression();
        read_listStart_typeBigDecimal_typeObject();
        read_listStart_typeBigDecimal_typeList();
        read_listStart_typeBigDecimal_typeProperty();
        read_listStart_typeBigDecimal_typeString();
        read_listStart_typeBigDecimal_typeInt();
        read_listStart_typeBigDecimal_typeLong();
        read_listStart_typeBigDecimal_typeDouble();
        read_listStart_typeBigDecimal_typeBigInteger();
        read_listStart_typeBigDecimal_typeBigDecimal();
        read_listStart_typeBigDecimal_typeBoolean();
        read_listStart_typeBigDecimal_typeBytes();
        read_listStart_typeBigDecimal_typeExpression();
        read_listStart_typeBigDecimal_typeType();
        read_listStart_typeBigDecimal_typeUndefined();
        read_listStart_typeBigDecimal_false();
        read_listStart_typeBigDecimal_true();
        read_listStart_typeBigDecimal_undefined();

        read_listStart_typeBytes_unexpectedChar();
        read_listStart_typeBytes_unexpectedEOF();
        read_listStart_typeBytes_arrow();
        read_listStart_typeBytes_listStart();
        read_listStart_typeBytes_objectEnd();
        read_listStart_typeBytes_objectStart();
        read_listStart_typeBytes_propertyEnd();
        read_listStart_typeBytes_propertyStart();
        read_listStart_typeBytes_string();
        read_listStart_typeBytes_numberInt();
        read_listStart_typeBytes_numberLong();
        read_listStart_typeBytes_numberDouble();
        read_listStart_typeBytes_numberBigInteger();
        read_listStart_typeBytes_numberBigDecimal();
        read_listStart_typeBytes_bytes();
        read_listStart_typeBytes_expression();
        read_listStart_typeBytes_typeObject();
        read_listStart_typeBytes_typeList();
        read_listStart_typeBytes_typeProperty();
        read_listStart_typeBytes_typeString();
        read_listStart_typeBytes_typeInt();
        read_listStart_typeBytes_typeLong();
        read_listStart_typeBytes_typeDouble();
        read_listStart_typeBytes_typeBigInteger();
        read_listStart_typeBytes_typeBigDecimal();
        read_listStart_typeBytes_typeBoolean();
        read_listStart_typeBytes_typeBytes();
        read_listStart_typeBytes_typeExpression();
        read_listStart_typeBytes_typeType();
        read_listStart_typeBytes_typeUndefined();
        read_listStart_typeBytes_false();
        read_listStart_typeBytes_true();
        read_listStart_typeBytes_undefined();

        read_listStart_typeExpression_unexpectedChar();
        read_listStart_typeExpression_unexpectedEOF();
        read_listStart_typeExpression_arrow();
        read_listStart_typeExpression_listStart();
        read_listStart_typeExpression_objectEnd();
        read_listStart_typeExpression_objectStart();
        read_listStart_typeExpression_propertyEnd();
        read_listStart_typeExpression_propertyStart();
        read_listStart_typeExpression_string();
        read_listStart_typeExpression_numberInt();
        read_listStart_typeExpression_numberLong();
        read_listStart_typeExpression_numberDouble();
        read_listStart_typeExpression_numberBigInteger();
        read_listStart_typeExpression_numberBigDecimal();
        read_listStart_typeExpression_bytes();
        read_listStart_typeExpression_expression();
        read_listStart_typeExpression_typeObject();
        read_listStart_typeExpression_typeList();
        read_listStart_typeExpression_typeProperty();
        read_listStart_typeExpression_typeString();
        read_listStart_typeExpression_typeInt();
        read_listStart_typeExpression_typeLong();
        read_listStart_typeExpression_typeDouble();
        read_listStart_typeExpression_typeBigInteger();
        read_listStart_typeExpression_typeBigDecimal();
        read_listStart_typeExpression_typeBoolean();
        read_listStart_typeExpression_typeBytes();
        read_listStart_typeExpression_typeExpression();
        read_listStart_typeExpression_typeType();
        read_listStart_typeExpression_typeUndefined();
        read_listStart_typeExpression_false();
        read_listStart_typeExpression_true();
        read_listStart_typeExpression_undefined();

        read_listStart_typeType_unexpectedChar();
        read_listStart_typeType_unexpectedEOF();
        read_listStart_typeType_arrow();
        read_listStart_typeType_listStart();
        read_listStart_typeType_objectEnd();
        read_listStart_typeType_objectStart();
        read_listStart_typeType_propertyEnd();
        read_listStart_typeType_propertyStart();
        read_listStart_typeType_string();
        read_listStart_typeType_numberInt();
        read_listStart_typeType_numberLong();
        read_listStart_typeType_numberDouble();
        read_listStart_typeType_numberBigInteger();
        read_listStart_typeType_numberBigDecimal();
        read_listStart_typeType_bytes();
        read_listStart_typeType_expression();
        read_listStart_typeType_typeObject();
        read_listStart_typeType_typeList();
        read_listStart_typeType_typeProperty();
        read_listStart_typeType_typeString();
        read_listStart_typeType_typeInt();
        read_listStart_typeType_typeLong();
        read_listStart_typeType_typeDouble();
        read_listStart_typeType_typeBigInteger();
        read_listStart_typeType_typeBigDecimal();
        read_listStart_typeType_typeBoolean();
        read_listStart_typeType_typeBytes();
        read_listStart_typeType_typeExpression();
        read_listStart_typeType_typeType();
        read_listStart_typeType_typeUndefined();
        read_listStart_typeType_false();
        read_listStart_typeType_true();
        read_listStart_typeType_undefined();

        read_listStart_typeUndefined_unexpectedChar();
        read_listStart_typeUndefined_unexpectedEOF();
        read_listStart_typeUndefined_arrow();
        read_listStart_typeUndefined_listStart();
        read_listStart_typeUndefined_objectEnd();
        read_listStart_typeUndefined_objectStart();
        read_listStart_typeUndefined_propertyEnd();
        read_listStart_typeUndefined_propertyStart();
        read_listStart_typeUndefined_string();
        read_listStart_typeUndefined_numberInt();
        read_listStart_typeUndefined_numberLong();
        read_listStart_typeUndefined_numberDouble();
        read_listStart_typeUndefined_numberBigInteger();
        read_listStart_typeUndefined_numberBigDecimal();
        read_listStart_typeUndefined_bytes();
        read_listStart_typeUndefined_expression();
        read_listStart_typeUndefined_typeObject();
        read_listStart_typeUndefined_typeList();
        read_listStart_typeUndefined_typeProperty();
        read_listStart_typeUndefined_typeString();
        read_listStart_typeUndefined_typeInt();
        read_listStart_typeUndefined_typeLong();
        read_listStart_typeUndefined_typeDouble();
        read_listStart_typeUndefined_typeBigInteger();
        read_listStart_typeUndefined_typeBigDecimal();
        read_listStart_typeUndefined_typeBoolean();
        read_listStart_typeUndefined_typeBytes();
        read_listStart_typeUndefined_typeExpression();
        read_listStart_typeUndefined_typeType();
        read_listStart_typeUndefined_typeUndefined();
        read_listStart_typeUndefined_false();
        read_listStart_typeUndefined_true();
        read_listStart_typeUndefined_undefined();

        read_listStart_true_unexpectedChar();
        read_listStart_true_unexpectedEOF();
        read_listStart_true_arrow();
        read_listStart_true_listStart();
        read_listStart_true_objectEnd();
        read_listStart_true_objectStart();
        read_listStart_true_propertyEnd();
        read_listStart_true_propertyStart();
        read_listStart_true_string();
        read_listStart_true_numberInt();
        read_listStart_true_numberLong();
        read_listStart_true_numberDouble();
        read_listStart_true_numberBigInteger();
        read_listStart_true_numberBigDecimal();
        read_listStart_true_bytes();
        read_listStart_true_expression();
        read_listStart_true_typeObject();
        read_listStart_true_typeList();
        read_listStart_true_typeProperty();
        read_listStart_true_typeString();
        read_listStart_true_typeInt();
        read_listStart_true_typeLong();
        read_listStart_true_typeDouble();
        read_listStart_true_typeBigInteger();
        read_listStart_true_typeBigDecimal();
        read_listStart_true_typeBoolean();
        read_listStart_true_typeBytes();
        read_listStart_true_typeExpression();
        read_listStart_true_typeType();
        read_listStart_true_typeUndefined();
        read_listStart_true_false();
        read_listStart_true_true();
        read_listStart_true_undefined();

        read_listStart_false_unexpectedChar();
        read_listStart_false_unexpectedEOF();
        read_listStart_false_arrow();
        read_listStart_false_listStart();
        read_listStart_false_objectEnd();
        read_listStart_false_objectStart();
        read_listStart_false_propertyEnd();
        read_listStart_false_propertyStart();
        read_listStart_false_string();
        read_listStart_false_numberInt();
        read_listStart_false_numberLong();
        read_listStart_false_numberDouble();
        read_listStart_false_numberBigInteger();
        read_listStart_false_numberBigDecimal();
        read_listStart_false_bytes();
        read_listStart_false_expression();
        read_listStart_false_typeObject();
        read_listStart_false_typeList();
        read_listStart_false_typeProperty();
        read_listStart_false_typeString();
        read_listStart_false_typeInt();
        read_listStart_false_typeLong();
        read_listStart_false_typeDouble();
        read_listStart_false_typeBigInteger();
        read_listStart_false_typeBigDecimal();
        read_listStart_false_typeBoolean();
        read_listStart_false_typeBytes();
        read_listStart_false_typeExpression();
        read_listStart_false_typeType();
        read_listStart_false_typeUndefined();
        read_listStart_false_false();
        read_listStart_false_true();
        read_listStart_false_undefined();

        read_listStart_undefined_unexpectedChar();
        read_listStart_undefined_unexpectedEOF();
        read_listStart_undefined_arrow();
        read_listStart_undefined_listStart();
        read_listStart_undefined_objectEnd();
        read_listStart_undefined_objectStart();
        read_listStart_undefined_propertyEnd();
        read_listStart_undefined_propertyStart();
        read_listStart_undefined_string();
        read_listStart_undefined_numberInt();
        read_listStart_undefined_numberLong();
        read_listStart_undefined_numberDouble();
        read_listStart_undefined_numberBigInteger();
        read_listStart_undefined_numberBigDecimal();
        read_listStart_undefined_bytes();
        read_listStart_undefined_expression();
        read_listStart_undefined_typeObject();
        read_listStart_undefined_typeList();
        read_listStart_undefined_typeProperty();
        read_listStart_undefined_typeString();
        read_listStart_undefined_typeInt();
        read_listStart_undefined_typeLong();
        read_listStart_undefined_typeDouble();
        read_listStart_undefined_typeBigInteger();
        read_listStart_undefined_typeBigDecimal();
        read_listStart_undefined_typeBoolean();
        read_listStart_undefined_typeBytes();
        read_listStart_undefined_typeExpression();
        read_listStart_undefined_typeType();
        read_listStart_undefined_typeUndefined();
        read_listStart_undefined_false();
        read_listStart_undefined_true();
        read_listStart_undefined_undefined();
    }

    @Test
    public void notEmptyListBrokenAfterComma() throws IOException, ModelException {
        read_listStart_listStart_listEnd_comma_propertyEnd();
        read_listStart_objectStart_objectEnd_comma_propertyEnd();
        read_listStart_propertyStart_propertyEnd_comma_propertyEnd();
        read_listStart_string_comma_propertyEnd();
        read_listStart_numberInt_comma_propertyEnd();
        read_listStart_numberLong_comma_propertyEnd();
        read_listStart_numberDouble_comma_propertyEnd();
        read_listStart_numberBigInteger_comma_propertyEnd();
        read_listStart_numberBigDecimal_comma_propertyEnd();
        read_listStart_bytes_comma_propertyEnd();
        read_listStart_expression_comma_propertyEnd();
        read_listStart_typeObject_comma_propertyEnd();
        read_listStart_typeList_comma_propertyEnd();
        read_listStart_typeProperty_comma_propertyEnd();
        read_listStart_typeString_comma_propertyEnd();
        read_listStart_typeInt_comma_propertyEnd();
        read_listStart_typeLong_comma_propertyEnd();
        read_listStart_typeDouble_comma_propertyEnd();
        read_listStart_typeBigInteger_comma_propertyEnd();
        read_listStart_typeBigDecimal_comma_propertyEnd();
        read_listStart_typeBytes_comma_propertyEnd();
        read_listStart_typeExpression_comma_propertyEnd();
        read_listStart_typeType_comma_propertyEnd();
        read_listStart_typeBoolean_comma_propertyEnd();
        read_listStart_typeUndefined_comma_propertyEnd();
        read_listStart_false_comma_propertyEnd();
        read_listStart_true_comma_propertyEnd();
        read_listStart_undefined_comma_propertyEnd();

        read_listStart_listStart_listEnd_comma_objectEnd();
        read_listStart_objectStart_objectEnd_comma_objectEnd();
        read_listStart_propertyStart_propertyEnd_comma_objectEnd();
        read_listStart_string_comma_objectEnd();
        read_listStart_numberInt_comma_objectEnd();
        read_listStart_numberLong_comma_objectEnd();
        read_listStart_numberDouble_comma_objectEnd();
        read_listStart_numberBigInteger_comma_objectEnd();
        read_listStart_numberBigDecimal_comma_objectEnd();
        read_listStart_bytes_comma_objectEnd();
        read_listStart_expression_comma_objectEnd();
        read_listStart_typeObject_comma_objectEnd();
        read_listStart_typeList_comma_objectEnd();
        read_listStart_typeProperty_comma_objectEnd();
        read_listStart_typeString_comma_objectEnd();
        read_listStart_typeInt_comma_objectEnd();
        read_listStart_typeLong_comma_objectEnd();
        read_listStart_typeDouble_comma_objectEnd();
        read_listStart_typeBigInteger_comma_objectEnd();
        read_listStart_typeBigDecimal_comma_objectEnd();
        read_listStart_typeBytes_comma_objectEnd();
        read_listStart_typeExpression_comma_objectEnd();
        read_listStart_typeType_comma_objectEnd();
        read_listStart_typeBoolean_comma_objectEnd();
        read_listStart_typeUndefined_comma_objectEnd();
        read_listStart_false_comma_objectEnd();
        read_listStart_true_comma_objectEnd();
        read_listStart_undefined_comma_objectEnd();

        read_listStart_listStart_listEnd_comma_arrow();
        read_listStart_objectStart_objectEnd_comma_arrow();
        read_listStart_propertyStart_propertyEnd_comma_arrow();
        read_listStart_string_comma_arrow();
        read_listStart_numberInt_comma_arrow();
        read_listStart_numberLong_comma_arrow();
        read_listStart_numberDouble_comma_arrow();
        read_listStart_numberBigInteger_comma_arrow();
        read_listStart_numberBigDecimal_comma_arrow();
        read_listStart_bytes_comma_arrow();
        read_listStart_expression_comma_arrow();
        read_listStart_typeObject_comma_arrow();
        read_listStart_typeList_comma_arrow();
        read_listStart_typeProperty_comma_arrow();
        read_listStart_typeString_comma_arrow();
        read_listStart_typeInt_comma_arrow();
        read_listStart_typeLong_comma_arrow();
        read_listStart_typeDouble_comma_arrow();
        read_listStart_typeBigInteger_comma_arrow();
        read_listStart_typeBigDecimal_comma_arrow();
        read_listStart_typeBytes_comma_arrow();
        read_listStart_typeExpression_comma_arrow();
        read_listStart_typeType_comma_arrow();
        read_listStart_typeBoolean_comma_arrow();
        read_listStart_typeUndefined_comma_arrow();
        read_listStart_false_comma_arrow();
        read_listStart_true_comma_arrow();
        read_listStart_undefined_comma_arrow();

        read_listStart_listStart_listEnd_comma_comma();
        read_listStart_objectStart_objectEnd_comma_comma();
        read_listStart_propertyStart_propertyEnd_comma_comma();
        read_listStart_string_comma_comma();
        read_listStart_numberInt_comma_comma();
        read_listStart_numberLong_comma_comma();
        read_listStart_numberDouble_comma_comma();
        read_listStart_numberBigInteger_comma_comma();
        read_listStart_numberBigDecimal_comma_comma();
        read_listStart_bytes_comma_comma();
        read_listStart_expression_comma_comma();
        read_listStart_typeObject_comma_comma();
        read_listStart_typeList_comma_comma();
        read_listStart_typeProperty_comma_comma();
        read_listStart_typeString_comma_comma();
        read_listStart_typeInt_comma_comma();
        read_listStart_typeLong_comma_comma();
        read_listStart_typeDouble_comma_comma();
        read_listStart_typeBigInteger_comma_comma();
        read_listStart_typeBigDecimal_comma_comma();
        read_listStart_typeBytes_comma_comma();
        read_listStart_typeExpression_comma_comma();
        read_listStart_typeType_comma_comma();
        read_listStart_typeBoolean_comma_comma();
        read_listStart_typeUndefined_comma_comma();
        read_listStart_false_comma_comma();
        read_listStart_true_comma_comma();
        read_listStart_undefined_comma_comma();

        read_listStart_listStart_listEnd_comma_unexpectedChar();
        read_listStart_objectStart_objectEnd_comma_unexpectedChar();
        read_listStart_propertyStart_propertyEnd_comma_unexpectedChar();
        read_listStart_string_comma_unexpectedChar();
        read_listStart_numberInt_comma_unexpectedChar();
        read_listStart_numberLong_comma_unexpectedChar();
        read_listStart_numberDouble_comma_unexpectedChar();
        read_listStart_numberBigInteger_comma_unexpectedChar();
        read_listStart_numberBigDecimal_comma_unexpectedChar();
        read_listStart_bytes_comma_unexpectedChar();
        read_listStart_expression_comma_unexpectedChar();
        read_listStart_typeObject_comma_unexpectedChar();
        read_listStart_typeList_comma_unexpectedChar();
        read_listStart_typeProperty_comma_unexpectedChar();
        read_listStart_typeString_comma_unexpectedChar();
        read_listStart_typeInt_comma_unexpectedChar();
        read_listStart_typeLong_comma_unexpectedChar();
        read_listStart_typeDouble_comma_unexpectedChar();
        read_listStart_typeBigInteger_comma_unexpectedChar();
        read_listStart_typeBigDecimal_comma_unexpectedChar();
        read_listStart_typeBytes_comma_unexpectedChar();
        read_listStart_typeExpression_comma_unexpectedChar();
        read_listStart_typeType_comma_unexpectedChar();
        read_listStart_typeBoolean_comma_unexpectedChar();
        read_listStart_typeUndefined_comma_unexpectedChar();
        read_listStart_false_comma_unexpectedChar();
        read_listStart_true_comma_unexpectedChar();
        read_listStart_undefined_comma_unexpectedChar();

        read_listStart_listStart_listEnd_comma_unexpectedEOF();
        read_listStart_objectStart_objectEnd_comma_unexpectedEOF();
        read_listStart_propertyStart_propertyEnd_comma_unexpectedEOF();
        read_listStart_string_comma_unexpectedEOF();
        read_listStart_numberInt_comma_unexpectedEOF();
        read_listStart_numberLong_comma_unexpectedEOF();
        read_listStart_numberDouble_comma_unexpectedEOF();
        read_listStart_numberBigInteger_comma_unexpectedEOF();
        read_listStart_numberBigDecimal_comma_unexpectedEOF();
        read_listStart_bytes_comma_unexpectedEOF();
        read_listStart_expression_comma_unexpectedEOF();
        read_listStart_typeObject_comma_unexpectedEOF();
        read_listStart_typeList_comma_unexpectedEOF();
        read_listStart_typeProperty_comma_unexpectedEOF();
        read_listStart_typeString_comma_unexpectedEOF();
        read_listStart_typeInt_comma_unexpectedEOF();
        read_listStart_typeLong_comma_unexpectedEOF();
        read_listStart_typeDouble_comma_unexpectedEOF();
        read_listStart_typeBigInteger_comma_unexpectedEOF();
        read_listStart_typeBigDecimal_comma_unexpectedEOF();
        read_listStart_typeBytes_comma_unexpectedEOF();
        read_listStart_typeExpression_comma_unexpectedEOF();
        read_listStart_typeType_comma_unexpectedEOF();
        read_listStart_typeBoolean_comma_unexpectedEOF();
        read_listStart_typeUndefined_comma_unexpectedEOF();
        read_listStart_false_comma_unexpectedEOF();
        read_listStart_true_comma_unexpectedEOF();
        read_listStart_undefined_comma_unexpectedEOF();
    }

    @Test
    public void brokenArrow() throws IOException, ModelException {
        read_objectStart_string_brokenArrow_unexpectedChar();
        read_objectStart_string_brokenArrow_unexpectedEOF();
        read_propertyStart_string_brokenArrow_unexpectedChar();
        read_propertyStart_string_brokenArrow_unexpectedEOF();
    }

    @Test
    public void notEmptyObjectBrokenAfterArrow() throws IOException, ModelException {
        read_objectStart_string_arrow_objectEnd();
        read_objectStart_string_arrow_listEnd();
        read_objectStart_string_arrow_propertyEnd();
        read_objectStart_string_arrow_arrow();
        read_objectStart_string_arrow_comma();
        read_objectStart_string_arrow_unexpectedChar();
        read_objectStart_string_arrow_unexpectedEOF();
    }

    @Test
    public void notEmptyPropertyBrokenAfterArrow() throws IOException, ModelException {
        read_propertyStart_string_arrow_objectEnd();
        read_propertyStart_string_arrow_listEnd();
        read_propertyStart_string_arrow_propertyEnd();
        read_propertyStart_string_arrow_arrow();
        read_propertyStart_string_arrow_comma();
        read_propertyStart_string_arrow_unexpectedChar();
        read_propertyStart_string_arrow_unexpectedEOF();
    }

    @Test
    public void notEmptyObjectMissingArrow() throws IOException, ModelException {
        read_objectStart_string_unexpectedChar();
        read_objectStart_string_unexpectedEOF();
        read_objectStart_string_comma();
        read_objectStart_string_listEnd();
        read_objectStart_string_listStart();
        read_objectStart_string_objectEnd();
        read_objectStart_string_objectStart();
        read_objectStart_string_propertyEnd();
        read_objectStart_string_propertyStart();
        read_objectStart_string_string();
        read_objectStart_string_numberInt();
        read_objectStart_string_numberLong();
        read_objectStart_string_numberDouble();
        read_objectStart_string_numberBigInteger();
        read_objectStart_string_numberBigDecimal();
        read_objectStart_string_bytes();
        read_objectStart_string_expression();
        read_objectStart_string_typeObject();
        read_objectStart_string_typeList();
        read_objectStart_string_typeProperty();
        read_objectStart_string_typeString();
        read_objectStart_string_typeInt();
        read_objectStart_string_typeLong();
        read_objectStart_string_typeDouble();
        read_objectStart_string_typeBigInteger();
        read_objectStart_string_typeBigDecimal();
        read_objectStart_string_typeBoolean();
        read_objectStart_string_typeBytes();
        read_objectStart_string_typeExpression();
        read_objectStart_string_typeType();
        read_objectStart_string_typeUndefined();
        read_objectStart_string_true();
        read_objectStart_string_false();
        read_objectStart_string_undefined();
    }

    @Test
    public void notEmptyPropertyMissingArrow() throws IOException, ModelException {
        read_propertyStart_string_unexpectedChar();
        read_propertyStart_string_unexpectedEOF();
        read_propertyStart_string_comma();
        read_propertyStart_string_listEnd();
        read_propertyStart_string_listStart();
        read_propertyStart_string_objectEnd();
        read_propertyStart_string_objectStart();
        read_propertyStart_string_propertyEnd();
        read_propertyStart_string_propertyStart();
        read_propertyStart_string_string();
        read_propertyStart_string_numberInt();
        read_propertyStart_string_numberLong();
        read_propertyStart_string_numberDouble();
        read_propertyStart_string_numberBigInteger();
        read_propertyStart_string_numberBigDecimal();
        read_propertyStart_string_bytes();
        read_propertyStart_string_expression();
        read_propertyStart_string_typeObject();
        read_propertyStart_string_typeList();
        read_propertyStart_string_typeProperty();
        read_propertyStart_string_typeString();
        read_propertyStart_string_typeInt();
        read_propertyStart_string_typeLong();
        read_propertyStart_string_typeDouble();
        read_propertyStart_string_typeBigInteger();
        read_propertyStart_string_typeBigDecimal();
        read_propertyStart_string_typeBoolean();
        read_propertyStart_string_typeBytes();
        read_propertyStart_string_typeExpression();
        read_propertyStart_string_typeType();
        read_propertyStart_string_typeUndefined();
        read_propertyStart_string_true();
        read_propertyStart_string_false();
        read_propertyStart_string_undefined();
    }

    @Test
    public void notEmptyPropertyMissingRightBracket() throws IOException, ModelException {
        read_propertyStart_string_arrow_listStart_listEnd_unexpectedChar();
        read_propertyStart_string_arrow_objectStart_objectEnd_unexpectedChar();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_unexpectedChar();
        read_propertyStart_string_arrow_string_unexpectedChar();
        read_propertyStart_string_arrow_numberInt_unexpectedChar();
        read_propertyStart_string_arrow_numberLong_unexpectedChar();
        read_propertyStart_string_arrow_numberDouble_unexpectedChar();
        read_propertyStart_string_arrow_numberBigInteger_unexpectedChar();
        read_propertyStart_string_arrow_numberBigDecimal_unexpectedChar();
        read_propertyStart_string_arrow_bytes_unexpectedChar();
        read_propertyStart_string_arrow_expression_unexpectedChar();
        read_propertyStart_string_arrow_typeObject_unexpectedChar();
        read_propertyStart_string_arrow_typeList_unexpectedChar();
        read_propertyStart_string_arrow_typeProperty_unexpectedChar();
        read_propertyStart_string_arrow_typeString_unexpectedChar();
        read_propertyStart_string_arrow_typeInt_unexpectedChar();
        read_propertyStart_string_arrow_typeLong_unexpectedChar();
        read_propertyStart_string_arrow_typeDouble_unexpectedChar();
        read_propertyStart_string_arrow_typeBigInteger_unexpectedChar();
        read_propertyStart_string_arrow_typeBigDecimal_unexpectedChar();
        read_propertyStart_string_arrow_typeBytes_unexpectedChar();
        read_propertyStart_string_arrow_typeExpression_unexpectedChar();
        read_propertyStart_string_arrow_typeType_unexpectedChar();
        read_propertyStart_string_arrow_typeBoolean_unexpectedChar();
        read_propertyStart_string_arrow_typeUndefined_unexpectedChar();
        read_propertyStart_string_arrow_false_unexpectedChar();
        read_propertyStart_string_arrow_true_unexpectedChar();
        read_propertyStart_string_arrow_undefined_unexpectedChar();

        read_propertyStart_string_arrow_listStart_listEnd_unexpectedEOF();
        read_propertyStart_string_arrow_objectStart_objectEnd_unexpectedEOF();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_unexpectedEOF();
        read_propertyStart_string_arrow_string_unexpectedEOF();
        read_propertyStart_string_arrow_numberInt_unexpectedEOF();
        read_propertyStart_string_arrow_numberLong_unexpectedEOF();
        read_propertyStart_string_arrow_numberDouble_unexpectedEOF();
        read_propertyStart_string_arrow_numberBigInteger_unexpectedEOF();
        read_propertyStart_string_arrow_numberBigDecimal_unexpectedEOF();
        read_propertyStart_string_arrow_bytes_unexpectedEOF();
        read_propertyStart_string_arrow_expression_unexpectedEOF();
        read_propertyStart_string_arrow_typeObject_unexpectedEOF();
        read_propertyStart_string_arrow_typeList_unexpectedEOF();
        read_propertyStart_string_arrow_typeProperty_unexpectedEOF();
        read_propertyStart_string_arrow_typeString_unexpectedEOF();
        read_propertyStart_string_arrow_typeInt_unexpectedEOF();
        read_propertyStart_string_arrow_typeLong_unexpectedEOF();
        read_propertyStart_string_arrow_typeDouble_unexpectedEOF();
        read_propertyStart_string_arrow_typeBigInteger_unexpectedEOF();
        read_propertyStart_string_arrow_typeBigDecimal_unexpectedEOF();
        read_propertyStart_string_arrow_typeBytes_unexpectedEOF();
        read_propertyStart_string_arrow_typeExpression_unexpectedEOF();
        read_propertyStart_string_arrow_typeType_unexpectedEOF();
        read_propertyStart_string_arrow_typeBoolean_unexpectedEOF();
        read_propertyStart_string_arrow_typeUndefined_unexpectedEOF();
        read_propertyStart_string_arrow_false_unexpectedEOF();
        read_propertyStart_string_arrow_true_unexpectedEOF();
        read_propertyStart_string_arrow_undefined_unexpectedEOF();

        read_propertyStart_string_arrow_listStart_listEnd_comma();
        read_propertyStart_string_arrow_objectStart_objectEnd_comma();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_comma();
        read_propertyStart_string_arrow_string_comma();
        read_propertyStart_string_arrow_numberInt_comma();
        read_propertyStart_string_arrow_numberLong_comma();
        read_propertyStart_string_arrow_numberDouble_comma();
        read_propertyStart_string_arrow_numberBigInteger_comma();
        read_propertyStart_string_arrow_numberBigDecimal_comma();
        read_propertyStart_string_arrow_bytes_comma();
        read_propertyStart_string_arrow_expression_comma();
        read_propertyStart_string_arrow_typeObject_comma();
        read_propertyStart_string_arrow_typeList_comma();
        read_propertyStart_string_arrow_typeProperty_comma();
        read_propertyStart_string_arrow_typeString_comma();
        read_propertyStart_string_arrow_typeInt_comma();
        read_propertyStart_string_arrow_typeLong_comma();
        read_propertyStart_string_arrow_typeDouble_comma();
        read_propertyStart_string_arrow_typeBigInteger_comma();
        read_propertyStart_string_arrow_typeBigDecimal_comma();
        read_propertyStart_string_arrow_typeBytes_comma();
        read_propertyStart_string_arrow_typeExpression_comma();
        read_propertyStart_string_arrow_typeType_comma();
        read_propertyStart_string_arrow_typeBoolean_comma();
        read_propertyStart_string_arrow_typeUndefined_comma();
        read_propertyStart_string_arrow_false_comma();
        read_propertyStart_string_arrow_true_comma();
        read_propertyStart_string_arrow_undefined_comma();

        read_propertyStart_string_arrow_listStart_listEnd_arrow();
        read_propertyStart_string_arrow_objectStart_objectEnd_arrow();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_arrow();
        read_propertyStart_string_arrow_string_arrow();
        read_propertyStart_string_arrow_numberInt_arrow();
        read_propertyStart_string_arrow_numberLong_arrow();
        read_propertyStart_string_arrow_numberDouble_arrow();
        read_propertyStart_string_arrow_numberBigInteger_arrow();
        read_propertyStart_string_arrow_numberBigDecimal_arrow();
        read_propertyStart_string_arrow_bytes_arrow();
        read_propertyStart_string_arrow_expression_arrow();
        read_propertyStart_string_arrow_typeObject_arrow();
        read_propertyStart_string_arrow_typeList_arrow();
        read_propertyStart_string_arrow_typeProperty_arrow();
        read_propertyStart_string_arrow_typeString_arrow();
        read_propertyStart_string_arrow_typeInt_arrow();
        read_propertyStart_string_arrow_typeLong_arrow();
        read_propertyStart_string_arrow_typeDouble_arrow();
        read_propertyStart_string_arrow_typeBigInteger_arrow();
        read_propertyStart_string_arrow_typeBigDecimal_arrow();
        read_propertyStart_string_arrow_typeBytes_arrow();
        read_propertyStart_string_arrow_typeExpression_arrow();
        read_propertyStart_string_arrow_typeType_arrow();
        read_propertyStart_string_arrow_typeBoolean_arrow();
        read_propertyStart_string_arrow_typeUndefined_arrow();
        read_propertyStart_string_arrow_false_arrow();
        read_propertyStart_string_arrow_true_arrow();
        read_propertyStart_string_arrow_undefined_arrow();

        read_propertyStart_string_arrow_listStart_listEnd_objectStart();
        read_propertyStart_string_arrow_objectStart_objectEnd_objectStart();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_objectStart();
        read_propertyStart_string_arrow_string_objectStart();
        read_propertyStart_string_arrow_numberInt_objectStart();
        read_propertyStart_string_arrow_numberLong_objectStart();
        read_propertyStart_string_arrow_numberDouble_objectStart();
        read_propertyStart_string_arrow_numberBigInteger_objectStart();
        read_propertyStart_string_arrow_numberBigDecimal_objectStart();
        read_propertyStart_string_arrow_bytes_objectStart();
        read_propertyStart_string_arrow_expression_objectStart();
        read_propertyStart_string_arrow_typeObject_objectStart();
        read_propertyStart_string_arrow_typeList_objectStart();
        read_propertyStart_string_arrow_typeProperty_objectStart();
        read_propertyStart_string_arrow_typeString_objectStart();
        read_propertyStart_string_arrow_typeInt_objectStart();
        read_propertyStart_string_arrow_typeLong_objectStart();
        read_propertyStart_string_arrow_typeDouble_objectStart();
        read_propertyStart_string_arrow_typeBigInteger_objectStart();
        read_propertyStart_string_arrow_typeBigDecimal_objectStart();
        read_propertyStart_string_arrow_typeBytes_objectStart();
        read_propertyStart_string_arrow_typeExpression_objectStart();
        read_propertyStart_string_arrow_typeType_objectStart();
        read_propertyStart_string_arrow_typeBoolean_objectStart();
        read_propertyStart_string_arrow_typeUndefined_objectStart();
        read_propertyStart_string_arrow_false_objectStart();
        read_propertyStart_string_arrow_true_objectStart();
        read_propertyStart_string_arrow_undefined_objectStart();

        read_propertyStart_string_arrow_listStart_listEnd_objectEnd();
        read_propertyStart_string_arrow_objectStart_objectEnd_objectEnd();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_objectEnd();
        read_propertyStart_string_arrow_string_objectEnd();
        read_propertyStart_string_arrow_numberInt_objectEnd();
        read_propertyStart_string_arrow_numberLong_objectEnd();
        read_propertyStart_string_arrow_numberDouble_objectEnd();
        read_propertyStart_string_arrow_numberBigInteger_objectEnd();
        read_propertyStart_string_arrow_numberBigDecimal_objectEnd();
        read_propertyStart_string_arrow_bytes_objectEnd();
        read_propertyStart_string_arrow_expression_objectEnd();
        read_propertyStart_string_arrow_typeObject_objectEnd();
        read_propertyStart_string_arrow_typeList_objectEnd();
        read_propertyStart_string_arrow_typeProperty_objectEnd();
        read_propertyStart_string_arrow_typeString_objectEnd();
        read_propertyStart_string_arrow_typeInt_objectEnd();
        read_propertyStart_string_arrow_typeLong_objectEnd();
        read_propertyStart_string_arrow_typeDouble_objectEnd();
        read_propertyStart_string_arrow_typeBigInteger_objectEnd();
        read_propertyStart_string_arrow_typeBigDecimal_objectEnd();
        read_propertyStart_string_arrow_typeBytes_objectEnd();
        read_propertyStart_string_arrow_typeExpression_objectEnd();
        read_propertyStart_string_arrow_typeType_objectEnd();
        read_propertyStart_string_arrow_typeBoolean_objectEnd();
        read_propertyStart_string_arrow_typeUndefined_objectEnd();
        read_propertyStart_string_arrow_false_objectEnd();
        read_propertyStart_string_arrow_true_objectEnd();
        read_propertyStart_string_arrow_undefined_objectEnd();

        read_propertyStart_string_arrow_listStart_listEnd_listStart();
        read_propertyStart_string_arrow_objectStart_objectEnd_listStart();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_listStart();
        read_propertyStart_string_arrow_string_listStart();
        read_propertyStart_string_arrow_numberInt_listStart();
        read_propertyStart_string_arrow_numberLong_listStart();
        read_propertyStart_string_arrow_numberDouble_listStart();
        read_propertyStart_string_arrow_numberBigInteger_listStart();
        read_propertyStart_string_arrow_numberBigDecimal_listStart();
        read_propertyStart_string_arrow_bytes_listStart();
        read_propertyStart_string_arrow_expression_listStart();
        read_propertyStart_string_arrow_typeObject_listStart();
        read_propertyStart_string_arrow_typeList_listStart();
        read_propertyStart_string_arrow_typeProperty_listStart();
        read_propertyStart_string_arrow_typeString_listStart();
        read_propertyStart_string_arrow_typeInt_listStart();
        read_propertyStart_string_arrow_typeLong_listStart();
        read_propertyStart_string_arrow_typeDouble_listStart();
        read_propertyStart_string_arrow_typeBigInteger_listStart();
        read_propertyStart_string_arrow_typeBigDecimal_listStart();
        read_propertyStart_string_arrow_typeBytes_listStart();
        read_propertyStart_string_arrow_typeExpression_listStart();
        read_propertyStart_string_arrow_typeType_listStart();
        read_propertyStart_string_arrow_typeBoolean_listStart();
        read_propertyStart_string_arrow_typeUndefined_listStart();
        read_propertyStart_string_arrow_false_listStart();
        read_propertyStart_string_arrow_true_listStart();
        read_propertyStart_string_arrow_undefined_listStart();

        read_propertyStart_string_arrow_listStart_listEnd_listEnd();
        read_propertyStart_string_arrow_objectStart_objectEnd_listEnd();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_listEnd();
        read_propertyStart_string_arrow_string_listEnd();
        read_propertyStart_string_arrow_numberInt_listEnd();
        read_propertyStart_string_arrow_numberLong_listEnd();
        read_propertyStart_string_arrow_numberDouble_listEnd();
        read_propertyStart_string_arrow_numberBigInteger_listEnd();
        read_propertyStart_string_arrow_numberBigDecimal_listEnd();
        read_propertyStart_string_arrow_bytes_listEnd();
        read_propertyStart_string_arrow_expression_listEnd();
        read_propertyStart_string_arrow_typeObject_listEnd();
        read_propertyStart_string_arrow_typeList_listEnd();
        read_propertyStart_string_arrow_typeProperty_listEnd();
        read_propertyStart_string_arrow_typeString_listEnd();
        read_propertyStart_string_arrow_typeInt_listEnd();
        read_propertyStart_string_arrow_typeLong_listEnd();
        read_propertyStart_string_arrow_typeDouble_listEnd();
        read_propertyStart_string_arrow_typeBigInteger_listEnd();
        read_propertyStart_string_arrow_typeBigDecimal_listEnd();
        read_propertyStart_string_arrow_typeBytes_listEnd();
        read_propertyStart_string_arrow_typeExpression_listEnd();
        read_propertyStart_string_arrow_typeType_listEnd();
        read_propertyStart_string_arrow_typeBoolean_listEnd();
        read_propertyStart_string_arrow_typeUndefined_listEnd();
        read_propertyStart_string_arrow_false_listEnd();
        read_propertyStart_string_arrow_true_listEnd();
        read_propertyStart_string_arrow_undefined_listEnd();

        read_propertyStart_string_arrow_listStart_listEnd_listStart();
        read_propertyStart_string_arrow_objectStart_objectEnd_listStart();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_listStart();
        read_propertyStart_string_arrow_string_listStart();
        read_propertyStart_string_arrow_numberInt_listStart();
        read_propertyStart_string_arrow_numberLong_listStart();
        read_propertyStart_string_arrow_numberDouble_listStart();
        read_propertyStart_string_arrow_numberBigInteger_listStart();
        read_propertyStart_string_arrow_numberBigDecimal_listStart();
        read_propertyStart_string_arrow_bytes_listStart();
        read_propertyStart_string_arrow_expression_listStart();
        read_propertyStart_string_arrow_typeObject_listStart();
        read_propertyStart_string_arrow_typeList_listStart();
        read_propertyStart_string_arrow_typeProperty_listStart();
        read_propertyStart_string_arrow_typeString_listStart();
        read_propertyStart_string_arrow_typeInt_listStart();
        read_propertyStart_string_arrow_typeLong_listStart();
        read_propertyStart_string_arrow_typeDouble_listStart();
        read_propertyStart_string_arrow_typeBigInteger_listStart();
        read_propertyStart_string_arrow_typeBigDecimal_listStart();
        read_propertyStart_string_arrow_typeBytes_listStart();
        read_propertyStart_string_arrow_typeExpression_listStart();
        read_propertyStart_string_arrow_typeType_listStart();
        read_propertyStart_string_arrow_typeBoolean_listStart();
        read_propertyStart_string_arrow_typeUndefined_listStart();
        read_propertyStart_string_arrow_false_listStart();
        read_propertyStart_string_arrow_true_listStart();
        read_propertyStart_string_arrow_undefined_listStart();

        read_propertyStart_string_arrow_listStart_listEnd_propertyStart();
        read_propertyStart_string_arrow_objectStart_objectEnd_propertyStart();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_propertyStart();
        read_propertyStart_string_arrow_string_propertyStart();
        read_propertyStart_string_arrow_numberInt_propertyStart();
        read_propertyStart_string_arrow_numberLong_propertyStart();
        read_propertyStart_string_arrow_numberDouble_propertyStart();
        read_propertyStart_string_arrow_numberBigInteger_propertyStart();
        read_propertyStart_string_arrow_numberBigDecimal_propertyStart();
        read_propertyStart_string_arrow_bytes_propertyStart();
        read_propertyStart_string_arrow_expression_propertyStart();
        read_propertyStart_string_arrow_typeObject_propertyStart();
        read_propertyStart_string_arrow_typeList_propertyStart();
        read_propertyStart_string_arrow_typeProperty_propertyStart();
        read_propertyStart_string_arrow_typeString_propertyStart();
        read_propertyStart_string_arrow_typeInt_propertyStart();
        read_propertyStart_string_arrow_typeLong_propertyStart();
        read_propertyStart_string_arrow_typeDouble_propertyStart();
        read_propertyStart_string_arrow_typeBigInteger_propertyStart();
        read_propertyStart_string_arrow_typeBigDecimal_propertyStart();
        read_propertyStart_string_arrow_typeBytes_propertyStart();
        read_propertyStart_string_arrow_typeExpression_propertyStart();
        read_propertyStart_string_arrow_typeType_propertyStart();
        read_propertyStart_string_arrow_typeBoolean_propertyStart();
        read_propertyStart_string_arrow_typeUndefined_propertyStart();
        read_propertyStart_string_arrow_false_propertyStart();
        read_propertyStart_string_arrow_true_propertyStart();
        read_propertyStart_string_arrow_undefined_propertyStart();

        read_propertyStart_string_arrow_listStart_listEnd_string();
        read_propertyStart_string_arrow_objectStart_objectEnd_string();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_string();
        read_propertyStart_string_arrow_string_string();
        read_propertyStart_string_arrow_numberInt_string();
        read_propertyStart_string_arrow_numberLong_string();
        read_propertyStart_string_arrow_numberDouble_string();
        read_propertyStart_string_arrow_numberBigInteger_string();
        read_propertyStart_string_arrow_numberBigDecimal_string();
        read_propertyStart_string_arrow_bytes_string();
        read_propertyStart_string_arrow_expression_string();
        read_propertyStart_string_arrow_typeObject_string();
        read_propertyStart_string_arrow_typeList_string();
        read_propertyStart_string_arrow_typeProperty_string();
        read_propertyStart_string_arrow_typeString_string();
        read_propertyStart_string_arrow_typeInt_string();
        read_propertyStart_string_arrow_typeLong_string();
        read_propertyStart_string_arrow_typeDouble_string();
        read_propertyStart_string_arrow_typeBigInteger_string();
        read_propertyStart_string_arrow_typeBigDecimal_string();
        read_propertyStart_string_arrow_typeBytes_string();
        read_propertyStart_string_arrow_typeExpression_string();
        read_propertyStart_string_arrow_typeType_string();
        read_propertyStart_string_arrow_typeBoolean_string();
        read_propertyStart_string_arrow_typeUndefined_string();
        read_propertyStart_string_arrow_false_string();
        read_propertyStart_string_arrow_true_string();
        read_propertyStart_string_arrow_undefined_string();

        read_propertyStart_string_arrow_listStart_listEnd_numberInt();
        read_propertyStart_string_arrow_objectStart_objectEnd_numberInt();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_numberInt();
        read_propertyStart_string_arrow_string_numberInt();
        read_propertyStart_string_arrow_numberInt_numberInt();
        read_propertyStart_string_arrow_numberLong_numberInt();
        read_propertyStart_string_arrow_numberDouble_numberInt();
        read_propertyStart_string_arrow_numberBigInteger_numberInt();
        read_propertyStart_string_arrow_numberBigDecimal_numberInt();
        read_propertyStart_string_arrow_bytes_numberInt();
        read_propertyStart_string_arrow_expression_numberInt();
        read_propertyStart_string_arrow_typeObject_numberInt();
        read_propertyStart_string_arrow_typeList_numberInt();
        read_propertyStart_string_arrow_typeProperty_numberInt();
        read_propertyStart_string_arrow_typeString_numberInt();
        read_propertyStart_string_arrow_typeInt_numberInt();
        read_propertyStart_string_arrow_typeLong_numberInt();
        read_propertyStart_string_arrow_typeDouble_numberInt();
        read_propertyStart_string_arrow_typeBigInteger_numberInt();
        read_propertyStart_string_arrow_typeBigDecimal_numberInt();
        read_propertyStart_string_arrow_typeBytes_numberInt();
        read_propertyStart_string_arrow_typeExpression_numberInt();
        read_propertyStart_string_arrow_typeType_numberInt();
        read_propertyStart_string_arrow_typeBoolean_numberInt();
        read_propertyStart_string_arrow_typeUndefined_numberInt();
        read_propertyStart_string_arrow_false_numberInt();
        read_propertyStart_string_arrow_true_numberInt();
        read_propertyStart_string_arrow_undefined_numberInt();

        read_propertyStart_string_arrow_listStart_listEnd_numberLong();
        read_propertyStart_string_arrow_objectStart_objectEnd_numberLong();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_numberLong();
        read_propertyStart_string_arrow_string_numberLong();
        read_propertyStart_string_arrow_numberInt_numberLong();
        read_propertyStart_string_arrow_numberLong_numberLong();
        read_propertyStart_string_arrow_numberDouble_numberLong();
        read_propertyStart_string_arrow_numberBigInteger_numberLong();
        read_propertyStart_string_arrow_numberBigDecimal_numberLong();
        read_propertyStart_string_arrow_bytes_numberLong();
        read_propertyStart_string_arrow_expression_numberLong();
        read_propertyStart_string_arrow_typeObject_numberLong();
        read_propertyStart_string_arrow_typeList_numberLong();
        read_propertyStart_string_arrow_typeProperty_numberLong();
        read_propertyStart_string_arrow_typeString_numberLong();
        read_propertyStart_string_arrow_typeInt_numberLong();
        read_propertyStart_string_arrow_typeLong_numberLong();
        read_propertyStart_string_arrow_typeDouble_numberLong();
        read_propertyStart_string_arrow_typeBigInteger_numberLong();
        read_propertyStart_string_arrow_typeBigDecimal_numberLong();
        read_propertyStart_string_arrow_typeBytes_numberLong();
        read_propertyStart_string_arrow_typeExpression_numberLong();
        read_propertyStart_string_arrow_typeType_numberLong();
        read_propertyStart_string_arrow_typeBoolean_numberLong();
        read_propertyStart_string_arrow_typeUndefined_numberLong();
        read_propertyStart_string_arrow_false_numberLong();
        read_propertyStart_string_arrow_true_numberLong();
        read_propertyStart_string_arrow_undefined_numberLong();

        read_propertyStart_string_arrow_listStart_listEnd_numberDouble();
        read_propertyStart_string_arrow_objectStart_objectEnd_numberDouble();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_numberDouble();
        read_propertyStart_string_arrow_string_numberDouble();
        read_propertyStart_string_arrow_numberInt_numberDouble();
        read_propertyStart_string_arrow_numberLong_numberDouble();
        read_propertyStart_string_arrow_numberDouble_numberDouble();
        read_propertyStart_string_arrow_numberBigInteger_numberDouble();
        read_propertyStart_string_arrow_numberBigDecimal_numberDouble();
        read_propertyStart_string_arrow_bytes_numberDouble();
        read_propertyStart_string_arrow_expression_numberDouble();
        read_propertyStart_string_arrow_typeObject_numberDouble();
        read_propertyStart_string_arrow_typeList_numberDouble();
        read_propertyStart_string_arrow_typeProperty_numberDouble();
        read_propertyStart_string_arrow_typeString_numberDouble();
        read_propertyStart_string_arrow_typeInt_numberDouble();
        read_propertyStart_string_arrow_typeLong_numberDouble();
        read_propertyStart_string_arrow_typeDouble_numberDouble();
        read_propertyStart_string_arrow_typeBigInteger_numberDouble();
        read_propertyStart_string_arrow_typeBigDecimal_numberDouble();
        read_propertyStart_string_arrow_typeBytes_numberDouble();
        read_propertyStart_string_arrow_typeExpression_numberDouble();
        read_propertyStart_string_arrow_typeType_numberDouble();
        read_propertyStart_string_arrow_typeBoolean_numberDouble();
        read_propertyStart_string_arrow_typeUndefined_numberDouble();
        read_propertyStart_string_arrow_false_numberDouble();
        read_propertyStart_string_arrow_true_numberDouble();
        read_propertyStart_string_arrow_undefined_numberDouble();

        read_propertyStart_string_arrow_listStart_listEnd_numberBigInteger();
        read_propertyStart_string_arrow_objectStart_objectEnd_numberBigInteger();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_numberBigInteger();
        read_propertyStart_string_arrow_string_numberBigInteger();
        read_propertyStart_string_arrow_numberInt_numberBigInteger();
        read_propertyStart_string_arrow_numberLong_numberBigInteger();
        read_propertyStart_string_arrow_numberDouble_numberBigInteger();
        read_propertyStart_string_arrow_numberBigInteger_numberBigInteger();
        read_propertyStart_string_arrow_numberBigDecimal_numberBigInteger();
        read_propertyStart_string_arrow_bytes_numberBigInteger();
        read_propertyStart_string_arrow_expression_numberBigInteger();
        read_propertyStart_string_arrow_typeObject_numberBigInteger();
        read_propertyStart_string_arrow_typeList_numberBigInteger();
        read_propertyStart_string_arrow_typeProperty_numberBigInteger();
        read_propertyStart_string_arrow_typeString_numberBigInteger();
        read_propertyStart_string_arrow_typeInt_numberBigInteger();
        read_propertyStart_string_arrow_typeLong_numberBigInteger();
        read_propertyStart_string_arrow_typeDouble_numberBigInteger();
        read_propertyStart_string_arrow_typeBigInteger_numberBigInteger();
        read_propertyStart_string_arrow_typeBigDecimal_numberBigInteger();
        read_propertyStart_string_arrow_typeBytes_numberBigInteger();
        read_propertyStart_string_arrow_typeExpression_numberBigInteger();
        read_propertyStart_string_arrow_typeType_numberBigInteger();
        read_propertyStart_string_arrow_typeBoolean_numberBigInteger();
        read_propertyStart_string_arrow_typeUndefined_numberBigInteger();
        read_propertyStart_string_arrow_false_numberBigInteger();
        read_propertyStart_string_arrow_true_numberBigInteger();
        read_propertyStart_string_arrow_undefined_numberBigInteger();

        read_propertyStart_string_arrow_listStart_listEnd_numberBigDecimal();
        read_propertyStart_string_arrow_objectStart_objectEnd_numberBigDecimal();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_numberBigDecimal();
        read_propertyStart_string_arrow_string_numberBigDecimal();
        read_propertyStart_string_arrow_numberInt_numberBigDecimal();
        read_propertyStart_string_arrow_numberLong_numberBigDecimal();
        read_propertyStart_string_arrow_numberDouble_numberBigDecimal();
        read_propertyStart_string_arrow_numberBigInteger_numberBigDecimal();
        read_propertyStart_string_arrow_numberBigDecimal_numberBigDecimal();
        read_propertyStart_string_arrow_bytes_numberBigDecimal();
        read_propertyStart_string_arrow_expression_numberBigDecimal();
        read_propertyStart_string_arrow_typeObject_numberBigDecimal();
        read_propertyStart_string_arrow_typeList_numberBigDecimal();
        read_propertyStart_string_arrow_typeProperty_numberBigDecimal();
        read_propertyStart_string_arrow_typeString_numberBigDecimal();
        read_propertyStart_string_arrow_typeInt_numberBigDecimal();
        read_propertyStart_string_arrow_typeLong_numberBigDecimal();
        read_propertyStart_string_arrow_typeDouble_numberBigDecimal();
        read_propertyStart_string_arrow_typeBigInteger_numberBigDecimal();
        read_propertyStart_string_arrow_typeBigDecimal_numberBigDecimal();
        read_propertyStart_string_arrow_typeBytes_numberBigDecimal();
        read_propertyStart_string_arrow_typeExpression_numberBigDecimal();
        read_propertyStart_string_arrow_typeType_numberBigDecimal();
        read_propertyStart_string_arrow_typeBoolean_numberBigDecimal();
        read_propertyStart_string_arrow_typeUndefined_numberBigDecimal();
        read_propertyStart_string_arrow_false_numberBigDecimal();
        read_propertyStart_string_arrow_true_numberBigDecimal();
        read_propertyStart_string_arrow_undefined_numberBigDecimal();

        read_propertyStart_string_arrow_listStart_listEnd_bytes();
        read_propertyStart_string_arrow_objectStart_objectEnd_bytes();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_bytes();
        read_propertyStart_string_arrow_string_bytes();
        read_propertyStart_string_arrow_numberInt_bytes();
        read_propertyStart_string_arrow_numberLong_bytes();
        read_propertyStart_string_arrow_numberDouble_bytes();
        read_propertyStart_string_arrow_numberBigInteger_bytes();
        read_propertyStart_string_arrow_numberBigDecimal_bytes();
        read_propertyStart_string_arrow_bytes_bytes();
        read_propertyStart_string_arrow_expression_bytes();
        read_propertyStart_string_arrow_typeObject_bytes();
        read_propertyStart_string_arrow_typeList_bytes();
        read_propertyStart_string_arrow_typeProperty_bytes();
        read_propertyStart_string_arrow_typeString_bytes();
        read_propertyStart_string_arrow_typeInt_bytes();
        read_propertyStart_string_arrow_typeLong_bytes();
        read_propertyStart_string_arrow_typeDouble_bytes();
        read_propertyStart_string_arrow_typeBigInteger_bytes();
        read_propertyStart_string_arrow_typeBigDecimal_bytes();
        read_propertyStart_string_arrow_typeBytes_bytes();
        read_propertyStart_string_arrow_typeExpression_bytes();
        read_propertyStart_string_arrow_typeType_bytes();
        read_propertyStart_string_arrow_typeBoolean_bytes();
        read_propertyStart_string_arrow_typeUndefined_bytes();
        read_propertyStart_string_arrow_false_bytes();
        read_propertyStart_string_arrow_true_bytes();
        read_propertyStart_string_arrow_undefined_bytes();

        read_propertyStart_string_arrow_listStart_listEnd_expression();
        read_propertyStart_string_arrow_objectStart_objectEnd_expression();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_expression();
        read_propertyStart_string_arrow_string_expression();
        read_propertyStart_string_arrow_numberInt_expression();
        read_propertyStart_string_arrow_numberLong_expression();
        read_propertyStart_string_arrow_numberDouble_expression();
        read_propertyStart_string_arrow_numberBigInteger_expression();
        read_propertyStart_string_arrow_numberBigDecimal_expression();
        read_propertyStart_string_arrow_bytes_expression();
        read_propertyStart_string_arrow_expression_expression();
        read_propertyStart_string_arrow_typeObject_expression();
        read_propertyStart_string_arrow_typeList_expression();
        read_propertyStart_string_arrow_typeProperty_expression();
        read_propertyStart_string_arrow_typeString_expression();
        read_propertyStart_string_arrow_typeInt_expression();
        read_propertyStart_string_arrow_typeLong_expression();
        read_propertyStart_string_arrow_typeDouble_expression();
        read_propertyStart_string_arrow_typeBigInteger_expression();
        read_propertyStart_string_arrow_typeBigDecimal_expression();
        read_propertyStart_string_arrow_typeBytes_expression();
        read_propertyStart_string_arrow_typeExpression_expression();
        read_propertyStart_string_arrow_typeType_expression();
        read_propertyStart_string_arrow_typeBoolean_expression();
        read_propertyStart_string_arrow_typeUndefined_expression();
        read_propertyStart_string_arrow_false_expression();
        read_propertyStart_string_arrow_true_expression();
        read_propertyStart_string_arrow_undefined_expression();

        read_propertyStart_string_arrow_listStart_listEnd_typeObject();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeObject();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeObject();
        read_propertyStart_string_arrow_string_typeObject();
        read_propertyStart_string_arrow_numberInt_typeObject();
        read_propertyStart_string_arrow_numberLong_typeObject();
        read_propertyStart_string_arrow_numberDouble_typeObject();
        read_propertyStart_string_arrow_numberBigInteger_typeObject();
        read_propertyStart_string_arrow_numberBigDecimal_typeObject();
        read_propertyStart_string_arrow_bytes_typeObject();
        read_propertyStart_string_arrow_expression_typeObject();
        read_propertyStart_string_arrow_typeObject_typeObject();
        read_propertyStart_string_arrow_typeList_typeObject();
        read_propertyStart_string_arrow_typeProperty_typeObject();
        read_propertyStart_string_arrow_typeString_typeObject();
        read_propertyStart_string_arrow_typeInt_typeObject();
        read_propertyStart_string_arrow_typeLong_typeObject();
        read_propertyStart_string_arrow_typeDouble_typeObject();
        read_propertyStart_string_arrow_typeBigInteger_typeObject();
        read_propertyStart_string_arrow_typeBigDecimal_typeObject();
        read_propertyStart_string_arrow_typeBytes_typeObject();
        read_propertyStart_string_arrow_typeExpression_typeObject();
        read_propertyStart_string_arrow_typeType_typeObject();
        read_propertyStart_string_arrow_typeBoolean_typeObject();
        read_propertyStart_string_arrow_typeUndefined_typeObject();
        read_propertyStart_string_arrow_false_typeObject();
        read_propertyStart_string_arrow_true_typeObject();
        read_propertyStart_string_arrow_undefined_typeObject();

        read_propertyStart_string_arrow_listStart_listEnd_typeList();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeList();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeList();
        read_propertyStart_string_arrow_string_typeList();
        read_propertyStart_string_arrow_numberInt_typeList();
        read_propertyStart_string_arrow_numberLong_typeList();
        read_propertyStart_string_arrow_numberDouble_typeList();
        read_propertyStart_string_arrow_numberBigInteger_typeList();
        read_propertyStart_string_arrow_numberBigDecimal_typeList();
        read_propertyStart_string_arrow_bytes_typeList();
        read_propertyStart_string_arrow_expression_typeList();
        read_propertyStart_string_arrow_typeObject_typeList();
        read_propertyStart_string_arrow_typeList_typeList();
        read_propertyStart_string_arrow_typeProperty_typeList();
        read_propertyStart_string_arrow_typeString_typeList();
        read_propertyStart_string_arrow_typeInt_typeList();
        read_propertyStart_string_arrow_typeLong_typeList();
        read_propertyStart_string_arrow_typeDouble_typeList();
        read_propertyStart_string_arrow_typeBigInteger_typeList();
        read_propertyStart_string_arrow_typeBigDecimal_typeList();
        read_propertyStart_string_arrow_typeBytes_typeList();
        read_propertyStart_string_arrow_typeExpression_typeList();
        read_propertyStart_string_arrow_typeType_typeList();
        read_propertyStart_string_arrow_typeBoolean_typeList();
        read_propertyStart_string_arrow_typeUndefined_typeList();
        read_propertyStart_string_arrow_false_typeList();
        read_propertyStart_string_arrow_true_typeList();
        read_propertyStart_string_arrow_undefined_typeList();

        read_propertyStart_string_arrow_listStart_listEnd_typeProperty();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeProperty();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeProperty();
        read_propertyStart_string_arrow_string_typeProperty();
        read_propertyStart_string_arrow_numberInt_typeProperty();
        read_propertyStart_string_arrow_numberLong_typeProperty();
        read_propertyStart_string_arrow_numberDouble_typeProperty();
        read_propertyStart_string_arrow_numberBigInteger_typeProperty();
        read_propertyStart_string_arrow_numberBigDecimal_typeProperty();
        read_propertyStart_string_arrow_bytes_typeProperty();
        read_propertyStart_string_arrow_expression_typeProperty();
        read_propertyStart_string_arrow_typeObject_typeProperty();
        read_propertyStart_string_arrow_typeList_typeProperty();
        read_propertyStart_string_arrow_typeProperty_typeProperty();
        read_propertyStart_string_arrow_typeString_typeProperty();
        read_propertyStart_string_arrow_typeInt_typeProperty();
        read_propertyStart_string_arrow_typeLong_typeProperty();
        read_propertyStart_string_arrow_typeDouble_typeProperty();
        read_propertyStart_string_arrow_typeBigInteger_typeProperty();
        read_propertyStart_string_arrow_typeBigDecimal_typeProperty();
        read_propertyStart_string_arrow_typeBytes_typeProperty();
        read_propertyStart_string_arrow_typeExpression_typeProperty();
        read_propertyStart_string_arrow_typeType_typeProperty();
        read_propertyStart_string_arrow_typeBoolean_typeProperty();
        read_propertyStart_string_arrow_typeUndefined_typeProperty();
        read_propertyStart_string_arrow_false_typeProperty();
        read_propertyStart_string_arrow_true_typeProperty();
        read_propertyStart_string_arrow_undefined_typeProperty();

        read_propertyStart_string_arrow_listStart_listEnd_typeString();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeString();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeString();
        read_propertyStart_string_arrow_string_typeString();
        read_propertyStart_string_arrow_numberInt_typeString();
        read_propertyStart_string_arrow_numberLong_typeString();
        read_propertyStart_string_arrow_numberDouble_typeString();
        read_propertyStart_string_arrow_numberBigInteger_typeString();
        read_propertyStart_string_arrow_numberBigDecimal_typeString();
        read_propertyStart_string_arrow_bytes_typeString();
        read_propertyStart_string_arrow_expression_typeString();
        read_propertyStart_string_arrow_typeObject_typeString();
        read_propertyStart_string_arrow_typeList_typeString();
        read_propertyStart_string_arrow_typeProperty_typeString();
        read_propertyStart_string_arrow_typeString_typeString();
        read_propertyStart_string_arrow_typeInt_typeString();
        read_propertyStart_string_arrow_typeLong_typeString();
        read_propertyStart_string_arrow_typeDouble_typeString();
        read_propertyStart_string_arrow_typeBigInteger_typeString();
        read_propertyStart_string_arrow_typeBigDecimal_typeString();
        read_propertyStart_string_arrow_typeBytes_typeString();
        read_propertyStart_string_arrow_typeExpression_typeString();
        read_propertyStart_string_arrow_typeType_typeString();
        read_propertyStart_string_arrow_typeBoolean_typeString();
        read_propertyStart_string_arrow_typeUndefined_typeString();
        read_propertyStart_string_arrow_false_typeString();
        read_propertyStart_string_arrow_true_typeString();
        read_propertyStart_string_arrow_undefined_typeString();

        read_propertyStart_string_arrow_listStart_listEnd_typeInt();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeInt();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeInt();
        read_propertyStart_string_arrow_string_typeInt();
        read_propertyStart_string_arrow_numberInt_typeInt();
        read_propertyStart_string_arrow_numberLong_typeInt();
        read_propertyStart_string_arrow_numberDouble_typeInt();
        read_propertyStart_string_arrow_numberBigInteger_typeInt();
        read_propertyStart_string_arrow_numberBigDecimal_typeInt();
        read_propertyStart_string_arrow_bytes_typeInt();
        read_propertyStart_string_arrow_expression_typeInt();
        read_propertyStart_string_arrow_typeObject_typeInt();
        read_propertyStart_string_arrow_typeList_typeInt();
        read_propertyStart_string_arrow_typeProperty_typeInt();
        read_propertyStart_string_arrow_typeString_typeInt();
        read_propertyStart_string_arrow_typeInt_typeInt();
        read_propertyStart_string_arrow_typeLong_typeInt();
        read_propertyStart_string_arrow_typeDouble_typeInt();
        read_propertyStart_string_arrow_typeBigInteger_typeInt();
        read_propertyStart_string_arrow_typeBigDecimal_typeInt();
        read_propertyStart_string_arrow_typeBytes_typeInt();
        read_propertyStart_string_arrow_typeExpression_typeInt();
        read_propertyStart_string_arrow_typeType_typeInt();
        read_propertyStart_string_arrow_typeBoolean_typeInt();
        read_propertyStart_string_arrow_typeUndefined_typeInt();
        read_propertyStart_string_arrow_false_typeInt();
        read_propertyStart_string_arrow_true_typeInt();
        read_propertyStart_string_arrow_undefined_typeInt();

        read_propertyStart_string_arrow_listStart_listEnd_typeLong();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeLong();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeLong();
        read_propertyStart_string_arrow_string_typeLong();
        read_propertyStart_string_arrow_numberInt_typeLong();
        read_propertyStart_string_arrow_numberLong_typeLong();
        read_propertyStart_string_arrow_numberDouble_typeLong();
        read_propertyStart_string_arrow_numberBigInteger_typeLong();
        read_propertyStart_string_arrow_numberBigDecimal_typeLong();
        read_propertyStart_string_arrow_bytes_typeLong();
        read_propertyStart_string_arrow_expression_typeLong();
        read_propertyStart_string_arrow_typeObject_typeLong();
        read_propertyStart_string_arrow_typeList_typeLong();
        read_propertyStart_string_arrow_typeProperty_typeLong();
        read_propertyStart_string_arrow_typeString_typeLong();
        read_propertyStart_string_arrow_typeInt_typeLong();
        read_propertyStart_string_arrow_typeLong_typeLong();
        read_propertyStart_string_arrow_typeDouble_typeLong();
        read_propertyStart_string_arrow_typeBigInteger_typeLong();
        read_propertyStart_string_arrow_typeBigDecimal_typeLong();
        read_propertyStart_string_arrow_typeBytes_typeLong();
        read_propertyStart_string_arrow_typeExpression_typeLong();
        read_propertyStart_string_arrow_typeType_typeLong();
        read_propertyStart_string_arrow_typeBoolean_typeLong();
        read_propertyStart_string_arrow_typeUndefined_typeLong();
        read_propertyStart_string_arrow_false_typeLong();
        read_propertyStart_string_arrow_true_typeLong();
        read_propertyStart_string_arrow_undefined_typeLong();

        read_propertyStart_string_arrow_listStart_listEnd_typeDouble();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeDouble();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeDouble();
        read_propertyStart_string_arrow_string_typeDouble();
        read_propertyStart_string_arrow_numberInt_typeDouble();
        read_propertyStart_string_arrow_numberLong_typeDouble();
        read_propertyStart_string_arrow_numberDouble_typeDouble();
        read_propertyStart_string_arrow_numberBigInteger_typeDouble();
        read_propertyStart_string_arrow_numberBigDecimal_typeDouble();
        read_propertyStart_string_arrow_bytes_typeDouble();
        read_propertyStart_string_arrow_expression_typeDouble();
        read_propertyStart_string_arrow_typeObject_typeDouble();
        read_propertyStart_string_arrow_typeList_typeDouble();
        read_propertyStart_string_arrow_typeProperty_typeDouble();
        read_propertyStart_string_arrow_typeString_typeDouble();
        read_propertyStart_string_arrow_typeInt_typeDouble();
        read_propertyStart_string_arrow_typeLong_typeDouble();
        read_propertyStart_string_arrow_typeDouble_typeDouble();
        read_propertyStart_string_arrow_typeBigInteger_typeDouble();
        read_propertyStart_string_arrow_typeBigDecimal_typeDouble();
        read_propertyStart_string_arrow_typeBytes_typeDouble();
        read_propertyStart_string_arrow_typeExpression_typeDouble();
        read_propertyStart_string_arrow_typeType_typeDouble();
        read_propertyStart_string_arrow_typeBoolean_typeDouble();
        read_propertyStart_string_arrow_typeUndefined_typeDouble();
        read_propertyStart_string_arrow_false_typeDouble();
        read_propertyStart_string_arrow_true_typeDouble();
        read_propertyStart_string_arrow_undefined_typeDouble();

        read_propertyStart_string_arrow_listStart_listEnd_typeBigInteger();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeBigInteger();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeBigInteger();
        read_propertyStart_string_arrow_string_typeBigInteger();
        read_propertyStart_string_arrow_numberInt_typeBigInteger();
        read_propertyStart_string_arrow_numberLong_typeBigInteger();
        read_propertyStart_string_arrow_numberDouble_typeBigInteger();
        read_propertyStart_string_arrow_numberBigInteger_typeBigInteger();
        read_propertyStart_string_arrow_numberBigDecimal_typeBigInteger();
        read_propertyStart_string_arrow_bytes_typeBigInteger();
        read_propertyStart_string_arrow_expression_typeBigInteger();
        read_propertyStart_string_arrow_typeObject_typeBigInteger();
        read_propertyStart_string_arrow_typeList_typeBigInteger();
        read_propertyStart_string_arrow_typeProperty_typeBigInteger();
        read_propertyStart_string_arrow_typeString_typeBigInteger();
        read_propertyStart_string_arrow_typeInt_typeBigInteger();
        read_propertyStart_string_arrow_typeLong_typeBigInteger();
        read_propertyStart_string_arrow_typeDouble_typeBigInteger();
        read_propertyStart_string_arrow_typeBigInteger_typeBigInteger();
        read_propertyStart_string_arrow_typeBigDecimal_typeBigInteger();
        read_propertyStart_string_arrow_typeBytes_typeBigInteger();
        read_propertyStart_string_arrow_typeExpression_typeBigInteger();
        read_propertyStart_string_arrow_typeType_typeBigInteger();
        read_propertyStart_string_arrow_typeBoolean_typeBigInteger();
        read_propertyStart_string_arrow_typeUndefined_typeBigInteger();
        read_propertyStart_string_arrow_false_typeBigInteger();
        read_propertyStart_string_arrow_true_typeBigInteger();
        read_propertyStart_string_arrow_undefined_typeBigInteger();

        read_propertyStart_string_arrow_listStart_listEnd_typeBigDecimal();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeBigDecimal();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeBigDecimal();
        read_propertyStart_string_arrow_string_typeBigDecimal();
        read_propertyStart_string_arrow_numberInt_typeBigDecimal();
        read_propertyStart_string_arrow_numberLong_typeBigDecimal();
        read_propertyStart_string_arrow_numberDouble_typeBigDecimal();
        read_propertyStart_string_arrow_numberBigInteger_typeBigDecimal();
        read_propertyStart_string_arrow_numberBigDecimal_typeBigDecimal();
        read_propertyStart_string_arrow_bytes_typeBigDecimal();
        read_propertyStart_string_arrow_expression_typeBigDecimal();
        read_propertyStart_string_arrow_typeObject_typeBigDecimal();
        read_propertyStart_string_arrow_typeList_typeBigDecimal();
        read_propertyStart_string_arrow_typeProperty_typeBigDecimal();
        read_propertyStart_string_arrow_typeString_typeBigDecimal();
        read_propertyStart_string_arrow_typeInt_typeBigDecimal();
        read_propertyStart_string_arrow_typeLong_typeBigDecimal();
        read_propertyStart_string_arrow_typeDouble_typeBigDecimal();
        read_propertyStart_string_arrow_typeBigInteger_typeBigDecimal();
        read_propertyStart_string_arrow_typeBigDecimal_typeBigDecimal();
        read_propertyStart_string_arrow_typeBytes_typeBigDecimal();
        read_propertyStart_string_arrow_typeExpression_typeBigDecimal();
        read_propertyStart_string_arrow_typeType_typeBigDecimal();
        read_propertyStart_string_arrow_typeBoolean_typeBigDecimal();
        read_propertyStart_string_arrow_typeUndefined_typeBigDecimal();
        read_propertyStart_string_arrow_false_typeBigDecimal();
        read_propertyStart_string_arrow_true_typeBigDecimal();
        read_propertyStart_string_arrow_undefined_typeBigDecimal();

        read_propertyStart_string_arrow_listStart_listEnd_typeBytes();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeBytes();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeBytes();
        read_propertyStart_string_arrow_string_typeBytes();
        read_propertyStart_string_arrow_numberInt_typeBytes();
        read_propertyStart_string_arrow_numberLong_typeBytes();
        read_propertyStart_string_arrow_numberDouble_typeBytes();
        read_propertyStart_string_arrow_numberBigInteger_typeBytes();
        read_propertyStart_string_arrow_numberBigDecimal_typeBytes();
        read_propertyStart_string_arrow_bytes_typeBytes();
        read_propertyStart_string_arrow_expression_typeBytes();
        read_propertyStart_string_arrow_typeObject_typeBytes();
        read_propertyStart_string_arrow_typeList_typeBytes();
        read_propertyStart_string_arrow_typeProperty_typeBytes();
        read_propertyStart_string_arrow_typeString_typeBytes();
        read_propertyStart_string_arrow_typeInt_typeBytes();
        read_propertyStart_string_arrow_typeLong_typeBytes();
        read_propertyStart_string_arrow_typeDouble_typeBytes();
        read_propertyStart_string_arrow_typeBigInteger_typeBytes();
        read_propertyStart_string_arrow_typeBigDecimal_typeBytes();
        read_propertyStart_string_arrow_typeBytes_typeBytes();
        read_propertyStart_string_arrow_typeExpression_typeBytes();
        read_propertyStart_string_arrow_typeType_typeBytes();
        read_propertyStart_string_arrow_typeBoolean_typeBytes();
        read_propertyStart_string_arrow_typeUndefined_typeBytes();
        read_propertyStart_string_arrow_false_typeBytes();
        read_propertyStart_string_arrow_true_typeBytes();
        read_propertyStart_string_arrow_undefined_typeBytes();

        read_propertyStart_string_arrow_listStart_listEnd_typeExpression();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeExpression();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeExpression();
        read_propertyStart_string_arrow_string_typeExpression();
        read_propertyStart_string_arrow_numberInt_typeExpression();
        read_propertyStart_string_arrow_numberLong_typeExpression();
        read_propertyStart_string_arrow_numberDouble_typeExpression();
        read_propertyStart_string_arrow_numberBigInteger_typeExpression();
        read_propertyStart_string_arrow_numberBigDecimal_typeExpression();
        read_propertyStart_string_arrow_bytes_typeExpression();
        read_propertyStart_string_arrow_expression_typeExpression();
        read_propertyStart_string_arrow_typeObject_typeExpression();
        read_propertyStart_string_arrow_typeList_typeExpression();
        read_propertyStart_string_arrow_typeProperty_typeExpression();
        read_propertyStart_string_arrow_typeString_typeExpression();
        read_propertyStart_string_arrow_typeInt_typeExpression();
        read_propertyStart_string_arrow_typeLong_typeExpression();
        read_propertyStart_string_arrow_typeDouble_typeExpression();
        read_propertyStart_string_arrow_typeBigInteger_typeExpression();
        read_propertyStart_string_arrow_typeBigDecimal_typeExpression();
        read_propertyStart_string_arrow_typeBytes_typeExpression();
        read_propertyStart_string_arrow_typeExpression_typeExpression();
        read_propertyStart_string_arrow_typeType_typeExpression();
        read_propertyStart_string_arrow_typeBoolean_typeExpression();
        read_propertyStart_string_arrow_typeUndefined_typeExpression();
        read_propertyStart_string_arrow_false_typeExpression();
        read_propertyStart_string_arrow_true_typeExpression();
        read_propertyStart_string_arrow_undefined_typeExpression();

        read_propertyStart_string_arrow_listStart_listEnd_typeType();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeType();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeType();
        read_propertyStart_string_arrow_string_typeType();
        read_propertyStart_string_arrow_numberInt_typeType();
        read_propertyStart_string_arrow_numberLong_typeType();
        read_propertyStart_string_arrow_numberDouble_typeType();
        read_propertyStart_string_arrow_numberBigInteger_typeType();
        read_propertyStart_string_arrow_numberBigDecimal_typeType();
        read_propertyStart_string_arrow_bytes_typeType();
        read_propertyStart_string_arrow_expression_typeType();
        read_propertyStart_string_arrow_typeObject_typeType();
        read_propertyStart_string_arrow_typeList_typeType();
        read_propertyStart_string_arrow_typeProperty_typeType();
        read_propertyStart_string_arrow_typeString_typeType();
        read_propertyStart_string_arrow_typeInt_typeType();
        read_propertyStart_string_arrow_typeLong_typeType();
        read_propertyStart_string_arrow_typeDouble_typeType();
        read_propertyStart_string_arrow_typeBigInteger_typeType();
        read_propertyStart_string_arrow_typeBigDecimal_typeType();
        read_propertyStart_string_arrow_typeBytes_typeType();
        read_propertyStart_string_arrow_typeExpression_typeType();
        read_propertyStart_string_arrow_typeType_typeType();
        read_propertyStart_string_arrow_typeBoolean_typeType();
        read_propertyStart_string_arrow_typeUndefined_typeType();
        read_propertyStart_string_arrow_false_typeType();
        read_propertyStart_string_arrow_true_typeType();
        read_propertyStart_string_arrow_undefined_typeType();

        read_propertyStart_string_arrow_listStart_listEnd_typeBoolean();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeBoolean();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeBoolean();
        read_propertyStart_string_arrow_string_typeBoolean();
        read_propertyStart_string_arrow_numberInt_typeBoolean();
        read_propertyStart_string_arrow_numberLong_typeBoolean();
        read_propertyStart_string_arrow_numberDouble_typeBoolean();
        read_propertyStart_string_arrow_numberBigInteger_typeBoolean();
        read_propertyStart_string_arrow_numberBigDecimal_typeBoolean();
        read_propertyStart_string_arrow_bytes_typeBoolean();
        read_propertyStart_string_arrow_expression_typeBoolean();
        read_propertyStart_string_arrow_typeObject_typeBoolean();
        read_propertyStart_string_arrow_typeList_typeBoolean();
        read_propertyStart_string_arrow_typeProperty_typeBoolean();
        read_propertyStart_string_arrow_typeString_typeBoolean();
        read_propertyStart_string_arrow_typeInt_typeBoolean();
        read_propertyStart_string_arrow_typeLong_typeBoolean();
        read_propertyStart_string_arrow_typeDouble_typeBoolean();
        read_propertyStart_string_arrow_typeBigInteger_typeBoolean();
        read_propertyStart_string_arrow_typeBigDecimal_typeBoolean();
        read_propertyStart_string_arrow_typeBytes_typeBoolean();
        read_propertyStart_string_arrow_typeExpression_typeBoolean();
        read_propertyStart_string_arrow_typeType_typeBoolean();
        read_propertyStart_string_arrow_typeBoolean_typeBoolean();
        read_propertyStart_string_arrow_typeUndefined_typeBoolean();
        read_propertyStart_string_arrow_false_typeBoolean();
        read_propertyStart_string_arrow_true_typeBoolean();
        read_propertyStart_string_arrow_undefined_typeBoolean();

        read_propertyStart_string_arrow_listStart_listEnd_typeUndefined();
        read_propertyStart_string_arrow_objectStart_objectEnd_typeUndefined();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_typeUndefined();
        read_propertyStart_string_arrow_string_typeUndefined();
        read_propertyStart_string_arrow_numberInt_typeUndefined();
        read_propertyStart_string_arrow_numberLong_typeUndefined();
        read_propertyStart_string_arrow_numberDouble_typeUndefined();
        read_propertyStart_string_arrow_numberBigInteger_typeUndefined();
        read_propertyStart_string_arrow_numberBigDecimal_typeUndefined();
        read_propertyStart_string_arrow_bytes_typeUndefined();
        read_propertyStart_string_arrow_expression_typeUndefined();
        read_propertyStart_string_arrow_typeObject_typeUndefined();
        read_propertyStart_string_arrow_typeList_typeUndefined();
        read_propertyStart_string_arrow_typeProperty_typeUndefined();
        read_propertyStart_string_arrow_typeString_typeUndefined();
        read_propertyStart_string_arrow_typeInt_typeUndefined();
        read_propertyStart_string_arrow_typeLong_typeUndefined();
        read_propertyStart_string_arrow_typeDouble_typeUndefined();
        read_propertyStart_string_arrow_typeBigInteger_typeUndefined();
        read_propertyStart_string_arrow_typeBigDecimal_typeUndefined();
        read_propertyStart_string_arrow_typeBytes_typeUndefined();
        read_propertyStart_string_arrow_typeExpression_typeUndefined();
        read_propertyStart_string_arrow_typeType_typeUndefined();
        read_propertyStart_string_arrow_typeBoolean_typeUndefined();
        read_propertyStart_string_arrow_typeUndefined_typeUndefined();
        read_propertyStart_string_arrow_false_typeUndefined();
        read_propertyStart_string_arrow_true_typeUndefined();
        read_propertyStart_string_arrow_undefined_typeUndefined();

        read_propertyStart_string_arrow_listStart_listEnd_true();
        read_propertyStart_string_arrow_objectStart_objectEnd_true();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_true();
        read_propertyStart_string_arrow_string_true();
        read_propertyStart_string_arrow_numberInt_true();
        read_propertyStart_string_arrow_numberLong_true();
        read_propertyStart_string_arrow_numberDouble_true();
        read_propertyStart_string_arrow_numberBigInteger_true();
        read_propertyStart_string_arrow_numberBigDecimal_true();
        read_propertyStart_string_arrow_bytes_true();
        read_propertyStart_string_arrow_expression_true();
        read_propertyStart_string_arrow_typeObject_true();
        read_propertyStart_string_arrow_typeList_true();
        read_propertyStart_string_arrow_typeProperty_true();
        read_propertyStart_string_arrow_typeString_true();
        read_propertyStart_string_arrow_typeInt_true();
        read_propertyStart_string_arrow_typeLong_true();
        read_propertyStart_string_arrow_typeDouble_true();
        read_propertyStart_string_arrow_typeBigInteger_true();
        read_propertyStart_string_arrow_typeBigDecimal_true();
        read_propertyStart_string_arrow_typeBytes_true();
        read_propertyStart_string_arrow_typeExpression_true();
        read_propertyStart_string_arrow_typeType_true();
        read_propertyStart_string_arrow_typeBoolean_true();
        read_propertyStart_string_arrow_typeUndefined_true();
        read_propertyStart_string_arrow_false_true();
        read_propertyStart_string_arrow_true_true();
        read_propertyStart_string_arrow_undefined_true();

        read_propertyStart_string_arrow_listStart_listEnd_false();
        read_propertyStart_string_arrow_objectStart_objectEnd_false();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_false();
        read_propertyStart_string_arrow_string_false();
        read_propertyStart_string_arrow_numberInt_false();
        read_propertyStart_string_arrow_numberLong_false();
        read_propertyStart_string_arrow_numberDouble_false();
        read_propertyStart_string_arrow_numberBigInteger_false();
        read_propertyStart_string_arrow_numberBigDecimal_false();
        read_propertyStart_string_arrow_bytes_false();
        read_propertyStart_string_arrow_expression_false();
        read_propertyStart_string_arrow_typeObject_false();
        read_propertyStart_string_arrow_typeList_false();
        read_propertyStart_string_arrow_typeProperty_false();
        read_propertyStart_string_arrow_typeString_false();
        read_propertyStart_string_arrow_typeInt_false();
        read_propertyStart_string_arrow_typeLong_false();
        read_propertyStart_string_arrow_typeDouble_false();
        read_propertyStart_string_arrow_typeBigInteger_false();
        read_propertyStart_string_arrow_typeBigDecimal_false();
        read_propertyStart_string_arrow_typeBytes_false();
        read_propertyStart_string_arrow_typeExpression_false();
        read_propertyStart_string_arrow_typeType_false();
        read_propertyStart_string_arrow_typeBoolean_false();
        read_propertyStart_string_arrow_typeUndefined_false();
        read_propertyStart_string_arrow_false_false();
        read_propertyStart_string_arrow_true_false();
        read_propertyStart_string_arrow_undefined_false();

        read_propertyStart_string_arrow_listStart_listEnd_undefined();
        read_propertyStart_string_arrow_objectStart_objectEnd_undefined();
        read_propertyStart_string_arrow_propertyStart_propertyEnd_undefined();
        read_propertyStart_string_arrow_string_undefined();
        read_propertyStart_string_arrow_numberInt_undefined();
        read_propertyStart_string_arrow_numberLong_undefined();
        read_propertyStart_string_arrow_numberDouble_undefined();
        read_propertyStart_string_arrow_numberBigInteger_undefined();
        read_propertyStart_string_arrow_numberBigDecimal_undefined();
        read_propertyStart_string_arrow_bytes_undefined();
        read_propertyStart_string_arrow_expression_undefined();
        read_propertyStart_string_arrow_typeObject_undefined();
        read_propertyStart_string_arrow_typeList_undefined();
        read_propertyStart_string_arrow_typeProperty_undefined();
        read_propertyStart_string_arrow_typeString_undefined();
        read_propertyStart_string_arrow_typeInt_undefined();
        read_propertyStart_string_arrow_typeLong_undefined();
        read_propertyStart_string_arrow_typeDouble_undefined();
        read_propertyStart_string_arrow_typeBigInteger_undefined();
        read_propertyStart_string_arrow_typeBigDecimal_undefined();
        read_propertyStart_string_arrow_typeBytes_undefined();
        read_propertyStart_string_arrow_typeExpression_undefined();
        read_propertyStart_string_arrow_typeType_undefined();
        read_propertyStart_string_arrow_typeBoolean_undefined();
        read_propertyStart_string_arrow_typeUndefined_undefined();
        read_propertyStart_string_arrow_false_undefined();
        read_propertyStart_string_arrow_true_undefined();
        read_propertyStart_string_arrow_undefined_undefined();
    }

    @Test
    public void notEmptyObjectMissingComma() throws IOException, ModelException {
        read_objectStart_string_arrow_listStart_listEnd_unexpectedChar();
        read_objectStart_string_arrow_objectStart_objectEnd_unexpectedChar();
        read_objectStart_string_arrow_propertyStart_propertyEnd_unexpectedChar();
        read_objectStart_string_arrow_string_unexpectedChar();
        read_objectStart_string_arrow_numberInt_unexpectedChar();
        read_objectStart_string_arrow_numberLong_unexpectedChar();
        read_objectStart_string_arrow_numberDouble_unexpectedChar();
        read_objectStart_string_arrow_numberBigInteger_unexpectedChar();
        read_objectStart_string_arrow_numberBigDecimal_unexpectedChar();
        read_objectStart_string_arrow_bytes_unexpectedChar();
        read_objectStart_string_arrow_expression_unexpectedChar();
        read_objectStart_string_arrow_typeObject_unexpectedChar();
        read_objectStart_string_arrow_typeList_unexpectedChar();
        read_objectStart_string_arrow_typeProperty_unexpectedChar();
        read_objectStart_string_arrow_typeString_unexpectedChar();
        read_objectStart_string_arrow_typeInt_unexpectedChar();
        read_objectStart_string_arrow_typeLong_unexpectedChar();
        read_objectStart_string_arrow_typeDouble_unexpectedChar();
        read_objectStart_string_arrow_typeBigInteger_unexpectedChar();
        read_objectStart_string_arrow_typeBigDecimal_unexpectedChar();
        read_objectStart_string_arrow_typeBytes_unexpectedChar();
        read_objectStart_string_arrow_typeExpression_unexpectedChar();
        read_objectStart_string_arrow_typeType_unexpectedChar();
        read_objectStart_string_arrow_typeBoolean_unexpectedChar();
        read_objectStart_string_arrow_typeUndefined_unexpectedChar();
        read_objectStart_string_arrow_false_unexpectedChar();
        read_objectStart_string_arrow_true_unexpectedChar();
        read_objectStart_string_arrow_undefined_unexpectedChar();

        read_objectStart_string_arrow_listStart_listEnd_unexpectedEOF();
        read_objectStart_string_arrow_objectStart_objectEnd_unexpectedEOF();
        read_objectStart_string_arrow_propertyStart_propertyEnd_unexpectedEOF();
        read_objectStart_string_arrow_string_unexpectedEOF();
        read_objectStart_string_arrow_numberInt_unexpectedEOF();
        read_objectStart_string_arrow_numberLong_unexpectedEOF();
        read_objectStart_string_arrow_numberDouble_unexpectedEOF();
        read_objectStart_string_arrow_numberBigInteger_unexpectedEOF();
        read_objectStart_string_arrow_numberBigDecimal_unexpectedEOF();
        read_objectStart_string_arrow_bytes_unexpectedEOF();
        read_objectStart_string_arrow_expression_unexpectedEOF();
        read_objectStart_string_arrow_typeObject_unexpectedEOF();
        read_objectStart_string_arrow_typeList_unexpectedEOF();
        read_objectStart_string_arrow_typeProperty_unexpectedEOF();
        read_objectStart_string_arrow_typeString_unexpectedEOF();
        read_objectStart_string_arrow_typeInt_unexpectedEOF();
        read_objectStart_string_arrow_typeLong_unexpectedEOF();
        read_objectStart_string_arrow_typeDouble_unexpectedEOF();
        read_objectStart_string_arrow_typeBigInteger_unexpectedEOF();
        read_objectStart_string_arrow_typeBigDecimal_unexpectedEOF();
        read_objectStart_string_arrow_typeBytes_unexpectedEOF();
        read_objectStart_string_arrow_typeExpression_unexpectedEOF();
        read_objectStart_string_arrow_typeType_unexpectedEOF();
        read_objectStart_string_arrow_typeBoolean_unexpectedEOF();
        read_objectStart_string_arrow_typeUndefined_unexpectedEOF();
        read_objectStart_string_arrow_false_unexpectedEOF();
        read_objectStart_string_arrow_true_unexpectedEOF();
        read_objectStart_string_arrow_undefined_unexpectedEOF();

        read_objectStart_string_arrow_listStart_listEnd_arrow();
        read_objectStart_string_arrow_objectStart_objectEnd_arrow();
        read_objectStart_string_arrow_propertyStart_propertyEnd_arrow();
        read_objectStart_string_arrow_string_arrow();
        read_objectStart_string_arrow_numberInt_arrow();
        read_objectStart_string_arrow_numberLong_arrow();
        read_objectStart_string_arrow_numberDouble_arrow();
        read_objectStart_string_arrow_numberBigInteger_arrow();
        read_objectStart_string_arrow_numberBigDecimal_arrow();
        read_objectStart_string_arrow_bytes_arrow();
        read_objectStart_string_arrow_expression_arrow();
        read_objectStart_string_arrow_typeObject_arrow();
        read_objectStart_string_arrow_typeList_arrow();
        read_objectStart_string_arrow_typeProperty_arrow();
        read_objectStart_string_arrow_typeString_arrow();
        read_objectStart_string_arrow_typeInt_arrow();
        read_objectStart_string_arrow_typeLong_arrow();
        read_objectStart_string_arrow_typeDouble_arrow();
        read_objectStart_string_arrow_typeBigInteger_arrow();
        read_objectStart_string_arrow_typeBigDecimal_arrow();
        read_objectStart_string_arrow_typeBytes_arrow();
        read_objectStart_string_arrow_typeExpression_arrow();
        read_objectStart_string_arrow_typeType_arrow();
        read_objectStart_string_arrow_typeBoolean_arrow();
        read_objectStart_string_arrow_typeUndefined_arrow();
        read_objectStart_string_arrow_true_arrow();
        read_objectStart_string_arrow_false_arrow();
        read_objectStart_string_arrow_undefined_arrow();

        read_objectStart_string_arrow_listStart_listEnd_objectStart();
        read_objectStart_string_arrow_objectStart_objectEnd_objectStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_objectStart();
        read_objectStart_string_arrow_string_objectStart();
        read_objectStart_string_arrow_numberInt_objectStart();
        read_objectStart_string_arrow_numberLong_objectStart();
        read_objectStart_string_arrow_numberDouble_objectStart();
        read_objectStart_string_arrow_numberBigInteger_objectStart();
        read_objectStart_string_arrow_numberBigDecimal_objectStart();
        read_objectStart_string_arrow_bytes_objectStart();
        read_objectStart_string_arrow_expression_objectStart();
        read_objectStart_string_arrow_typeObject_objectStart();
        read_objectStart_string_arrow_typeList_objectStart();
        read_objectStart_string_arrow_typeProperty_objectStart();
        read_objectStart_string_arrow_typeString_objectStart();
        read_objectStart_string_arrow_typeInt_objectStart();
        read_objectStart_string_arrow_typeLong_objectStart();
        read_objectStart_string_arrow_typeDouble_objectStart();
        read_objectStart_string_arrow_typeBigInteger_objectStart();
        read_objectStart_string_arrow_typeBigDecimal_objectStart();
        read_objectStart_string_arrow_typeBytes_objectStart();
        read_objectStart_string_arrow_typeExpression_objectStart();
        read_objectStart_string_arrow_typeType_objectStart();
        read_objectStart_string_arrow_typeBoolean_objectStart();
        read_objectStart_string_arrow_typeUndefined_objectStart();
        read_objectStart_string_arrow_false_objectStart();
        read_objectStart_string_arrow_true_objectStart();
        read_objectStart_string_arrow_undefined_objectStart();

        read_objectStart_string_arrow_listStart_listEnd_listStart();
        read_objectStart_string_arrow_objectStart_objectEnd_listStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_listStart();
        read_objectStart_string_arrow_string_listStart();
        read_objectStart_string_arrow_numberInt_listStart();
        read_objectStart_string_arrow_numberLong_listStart();
        read_objectStart_string_arrow_numberDouble_listStart();
        read_objectStart_string_arrow_numberBigInteger_listStart();
        read_objectStart_string_arrow_numberBigDecimal_listStart();
        read_objectStart_string_arrow_bytes_listStart();
        read_objectStart_string_arrow_expression_listStart();
        read_objectStart_string_arrow_typeObject_listStart();
        read_objectStart_string_arrow_typeList_listStart();
        read_objectStart_string_arrow_typeProperty_listStart();
        read_objectStart_string_arrow_typeString_listStart();
        read_objectStart_string_arrow_typeInt_listStart();
        read_objectStart_string_arrow_typeLong_listStart();
        read_objectStart_string_arrow_typeDouble_listStart();
        read_objectStart_string_arrow_typeBigInteger_listStart();
        read_objectStart_string_arrow_typeBigDecimal_listStart();
        read_objectStart_string_arrow_typeBytes_listStart();
        read_objectStart_string_arrow_typeExpression_listStart();
        read_objectStart_string_arrow_typeType_listStart();
        read_objectStart_string_arrow_typeBoolean_listStart();
        read_objectStart_string_arrow_typeUndefined_listStart();
        read_objectStart_string_arrow_false_listStart();
        read_objectStart_string_arrow_true_listStart();
        read_objectStart_string_arrow_undefined_listStart();

        read_objectStart_string_arrow_listStart_listEnd_listEnd();
        read_objectStart_string_arrow_objectStart_objectEnd_listEnd();
        read_objectStart_string_arrow_propertyStart_propertyEnd_listEnd();
        read_objectStart_string_arrow_string_listEnd();
        read_objectStart_string_arrow_numberInt_listEnd();
        read_objectStart_string_arrow_numberLong_listEnd();
        read_objectStart_string_arrow_numberDouble_listEnd();
        read_objectStart_string_arrow_numberBigInteger_listEnd();
        read_objectStart_string_arrow_numberBigDecimal_listEnd();
        read_objectStart_string_arrow_bytes_listEnd();
        read_objectStart_string_arrow_expression_listEnd();
        read_objectStart_string_arrow_typeObject_listEnd();
        read_objectStart_string_arrow_typeList_listEnd();
        read_objectStart_string_arrow_typeProperty_listEnd();
        read_objectStart_string_arrow_typeString_listEnd();
        read_objectStart_string_arrow_typeInt_listEnd();
        read_objectStart_string_arrow_typeLong_listEnd();
        read_objectStart_string_arrow_typeDouble_listEnd();
        read_objectStart_string_arrow_typeBigInteger_listEnd();
        read_objectStart_string_arrow_typeBigDecimal_listEnd();
        read_objectStart_string_arrow_typeBytes_listEnd();
        read_objectStart_string_arrow_typeExpression_listEnd();
        read_objectStart_string_arrow_typeType_listEnd();
        read_objectStart_string_arrow_typeBoolean_listEnd();
        read_objectStart_string_arrow_typeUndefined_listEnd();
        read_objectStart_string_arrow_false_listEnd();
        read_objectStart_string_arrow_true_listEnd();
        read_objectStart_string_arrow_undefined_listEnd();

        read_objectStart_string_arrow_listStart_listEnd_propertyStart();
        read_objectStart_string_arrow_objectStart_objectEnd_propertyStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_propertyStart();
        read_objectStart_string_arrow_string_propertyStart();
        read_objectStart_string_arrow_numberInt_propertyStart();
        read_objectStart_string_arrow_numberLong_propertyStart();
        read_objectStart_string_arrow_numberDouble_propertyStart();
        read_objectStart_string_arrow_numberBigInteger_propertyStart();
        read_objectStart_string_arrow_numberBigDecimal_propertyStart();
        read_objectStart_string_arrow_bytes_propertyStart();
        read_objectStart_string_arrow_expression_propertyStart();
        read_objectStart_string_arrow_typeObject_propertyStart();
        read_objectStart_string_arrow_typeList_propertyStart();
        read_objectStart_string_arrow_typeProperty_propertyStart();
        read_objectStart_string_arrow_typeString_propertyStart();
        read_objectStart_string_arrow_typeInt_propertyStart();
        read_objectStart_string_arrow_typeLong_propertyStart();
        read_objectStart_string_arrow_typeDouble_propertyStart();
        read_objectStart_string_arrow_typeBigInteger_propertyStart();
        read_objectStart_string_arrow_typeBigDecimal_propertyStart();
        read_objectStart_string_arrow_typeBytes_propertyStart();
        read_objectStart_string_arrow_typeExpression_propertyStart();
        read_objectStart_string_arrow_typeType_propertyStart();
        read_objectStart_string_arrow_typeBoolean_propertyStart();
        read_objectStart_string_arrow_typeUndefined_propertyStart();
        read_objectStart_string_arrow_false_propertyStart();
        read_objectStart_string_arrow_true_propertyStart();
        read_objectStart_string_arrow_undefined_propertyStart();

        read_objectStart_string_arrow_listStart_listEnd_propertyEnd();
        read_objectStart_string_arrow_objectStart_objectEnd_propertyEnd();
        read_objectStart_string_arrow_propertyStart_propertyEnd_propertyEnd();
        read_objectStart_string_arrow_string_propertyEnd();
        read_objectStart_string_arrow_numberInt_propertyEnd();
        read_objectStart_string_arrow_numberLong_propertyEnd();
        read_objectStart_string_arrow_numberDouble_propertyEnd();
        read_objectStart_string_arrow_numberBigInteger_propertyEnd();
        read_objectStart_string_arrow_numberBigDecimal_propertyEnd();
        read_objectStart_string_arrow_bytes_propertyEnd();
        read_objectStart_string_arrow_expression_propertyEnd();
        read_objectStart_string_arrow_typeObject_propertyEnd();
        read_objectStart_string_arrow_typeList_propertyEnd();
        read_objectStart_string_arrow_typeProperty_propertyEnd();
        read_objectStart_string_arrow_typeString_propertyEnd();
        read_objectStart_string_arrow_typeInt_propertyEnd();
        read_objectStart_string_arrow_typeLong_propertyEnd();
        read_objectStart_string_arrow_typeDouble_propertyEnd();
        read_objectStart_string_arrow_typeBigInteger_propertyEnd();
        read_objectStart_string_arrow_typeBigDecimal_propertyEnd();
        read_objectStart_string_arrow_typeBytes_propertyEnd();
        read_objectStart_string_arrow_typeExpression_propertyEnd();
        read_objectStart_string_arrow_typeType_propertyEnd();
        read_objectStart_string_arrow_typeBoolean_propertyEnd();
        read_objectStart_string_arrow_typeUndefined_propertyEnd();
        read_objectStart_string_arrow_false_propertyEnd();
        read_objectStart_string_arrow_true_propertyEnd();
        read_objectStart_string_arrow_undefined_propertyEnd();

        read_objectStart_string_arrow_listStart_listEnd_string();
        read_objectStart_string_arrow_objectStart_objectEnd_string();
        read_objectStart_string_arrow_propertyStart_propertyEnd_string();
        read_objectStart_string_arrow_string_string();
        read_objectStart_string_arrow_numberInt_string();
        read_objectStart_string_arrow_numberLong_string();
        read_objectStart_string_arrow_numberDouble_string();
        read_objectStart_string_arrow_numberBigInteger_string();
        read_objectStart_string_arrow_numberBigDecimal_string();
        read_objectStart_string_arrow_bytes_string();
        read_objectStart_string_arrow_expression_string();
        read_objectStart_string_arrow_typeObject_string();
        read_objectStart_string_arrow_typeList_string();
        read_objectStart_string_arrow_typeProperty_string();
        read_objectStart_string_arrow_typeString_string();
        read_objectStart_string_arrow_typeInt_string();
        read_objectStart_string_arrow_typeLong_string();
        read_objectStart_string_arrow_typeDouble_string();
        read_objectStart_string_arrow_typeBigInteger_string();
        read_objectStart_string_arrow_typeBigDecimal_string();
        read_objectStart_string_arrow_typeBytes_string();
        read_objectStart_string_arrow_typeExpression_string();
        read_objectStart_string_arrow_typeType_string();
        read_objectStart_string_arrow_typeBoolean_string();
        read_objectStart_string_arrow_typeUndefined_string();
        read_objectStart_string_arrow_false_string();
        read_objectStart_string_arrow_true_string();
        read_objectStart_string_arrow_undefined_string();

        read_objectStart_string_arrow_listStart_listEnd_numberInt();
        read_objectStart_string_arrow_objectStart_objectEnd_numberInt();
        read_objectStart_string_arrow_propertyStart_propertyEnd_numberInt();
        read_objectStart_string_arrow_string_numberInt();
        read_objectStart_string_arrow_numberInt_numberInt();
        read_objectStart_string_arrow_numberLong_numberInt();
        read_objectStart_string_arrow_numberDouble_numberInt();
        read_objectStart_string_arrow_numberBigInteger_numberInt();
        read_objectStart_string_arrow_numberBigDecimal_numberInt();
        read_objectStart_string_arrow_bytes_numberInt();
        read_objectStart_string_arrow_expression_numberInt();
        read_objectStart_string_arrow_typeObject_numberInt();
        read_objectStart_string_arrow_typeList_numberInt();
        read_objectStart_string_arrow_typeProperty_numberInt();
        read_objectStart_string_arrow_typeString_numberInt();
        read_objectStart_string_arrow_typeInt_numberInt();
        read_objectStart_string_arrow_typeLong_numberInt();
        read_objectStart_string_arrow_typeDouble_numberInt();
        read_objectStart_string_arrow_typeBigInteger_numberInt();
        read_objectStart_string_arrow_typeBigDecimal_numberInt();
        read_objectStart_string_arrow_typeBytes_numberInt();
        read_objectStart_string_arrow_typeExpression_numberInt();
        read_objectStart_string_arrow_typeType_numberInt();
        read_objectStart_string_arrow_typeBoolean_numberInt();
        read_objectStart_string_arrow_typeUndefined_numberInt();
        read_objectStart_string_arrow_false_numberInt();
        read_objectStart_string_arrow_true_numberInt();
        read_objectStart_string_arrow_undefined_numberInt();

        read_objectStart_string_arrow_listStart_listEnd_numberLong();
        read_objectStart_string_arrow_objectStart_objectEnd_numberLong();
        read_objectStart_string_arrow_propertyStart_propertyEnd_numberLong();
        read_objectStart_string_arrow_string_numberLong();
        read_objectStart_string_arrow_numberInt_numberLong();
        read_objectStart_string_arrow_numberLong_numberLong();
        read_objectStart_string_arrow_numberDouble_numberLong();
        read_objectStart_string_arrow_numberBigInteger_numberLong();
        read_objectStart_string_arrow_numberBigDecimal_numberLong();
        read_objectStart_string_arrow_bytes_numberLong();
        read_objectStart_string_arrow_expression_numberLong();
        read_objectStart_string_arrow_typeObject_numberLong();
        read_objectStart_string_arrow_typeList_numberLong();
        read_objectStart_string_arrow_typeProperty_numberLong();
        read_objectStart_string_arrow_typeString_numberLong();
        read_objectStart_string_arrow_typeInt_numberLong();
        read_objectStart_string_arrow_typeLong_numberLong();
        read_objectStart_string_arrow_typeDouble_numberLong();
        read_objectStart_string_arrow_typeBigInteger_numberLong();
        read_objectStart_string_arrow_typeBigDecimal_numberLong();
        read_objectStart_string_arrow_typeBytes_numberLong();
        read_objectStart_string_arrow_typeExpression_numberLong();
        read_objectStart_string_arrow_typeType_numberLong();
        read_objectStart_string_arrow_typeBoolean_numberLong();
        read_objectStart_string_arrow_typeUndefined_numberLong();
        read_objectStart_string_arrow_false_numberLong();
        read_objectStart_string_arrow_true_numberLong();
        read_objectStart_string_arrow_undefined_numberLong();

        read_objectStart_string_arrow_listStart_listEnd_numberDouble();
        read_objectStart_string_arrow_objectStart_objectEnd_numberDouble();
        read_objectStart_string_arrow_propertyStart_propertyEnd_numberDouble();
        read_objectStart_string_arrow_string_numberDouble();
        read_objectStart_string_arrow_numberInt_numberDouble();
        read_objectStart_string_arrow_numberLong_numberDouble();
        read_objectStart_string_arrow_numberDouble_numberDouble();
        read_objectStart_string_arrow_numberBigInteger_numberDouble();
        read_objectStart_string_arrow_numberBigDecimal_numberDouble();
        read_objectStart_string_arrow_bytes_numberDouble();
        read_objectStart_string_arrow_expression_numberDouble();
        read_objectStart_string_arrow_typeObject_numberDouble();
        read_objectStart_string_arrow_typeList_numberDouble();
        read_objectStart_string_arrow_typeProperty_numberDouble();
        read_objectStart_string_arrow_typeString_numberDouble();
        read_objectStart_string_arrow_typeInt_numberDouble();
        read_objectStart_string_arrow_typeLong_numberDouble();
        read_objectStart_string_arrow_typeDouble_numberDouble();
        read_objectStart_string_arrow_typeBigInteger_numberDouble();
        read_objectStart_string_arrow_typeBigDecimal_numberDouble();
        read_objectStart_string_arrow_typeBytes_numberDouble();
        read_objectStart_string_arrow_typeExpression_numberDouble();
        read_objectStart_string_arrow_typeType_numberDouble();
        read_objectStart_string_arrow_typeBoolean_numberDouble();
        read_objectStart_string_arrow_typeUndefined_numberDouble();
        read_objectStart_string_arrow_false_numberDouble();
        read_objectStart_string_arrow_true_numberDouble();
        read_objectStart_string_arrow_undefined_numberDouble();

        read_objectStart_string_arrow_listStart_listEnd_numberBigInteger();
        read_objectStart_string_arrow_objectStart_objectEnd_numberBigInteger();
        read_objectStart_string_arrow_propertyStart_propertyEnd_numberBigInteger();
        read_objectStart_string_arrow_string_numberBigInteger();
        read_objectStart_string_arrow_numberInt_numberBigInteger();
        read_objectStart_string_arrow_numberLong_numberBigInteger();
        read_objectStart_string_arrow_numberDouble_numberBigInteger();
        read_objectStart_string_arrow_numberBigInteger_numberBigInteger();
        read_objectStart_string_arrow_numberBigDecimal_numberBigInteger();
        read_objectStart_string_arrow_bytes_numberBigInteger();
        read_objectStart_string_arrow_expression_numberBigInteger();
        read_objectStart_string_arrow_typeObject_numberBigInteger();
        read_objectStart_string_arrow_typeList_numberBigInteger();
        read_objectStart_string_arrow_typeProperty_numberBigInteger();
        read_objectStart_string_arrow_typeString_numberBigInteger();
        read_objectStart_string_arrow_typeInt_numberBigInteger();
        read_objectStart_string_arrow_typeLong_numberBigInteger();
        read_objectStart_string_arrow_typeDouble_numberBigInteger();
        read_objectStart_string_arrow_typeBigInteger_numberBigInteger();
        read_objectStart_string_arrow_typeBigDecimal_numberBigInteger();
        read_objectStart_string_arrow_typeBytes_numberBigInteger();
        read_objectStart_string_arrow_typeExpression_numberBigInteger();
        read_objectStart_string_arrow_typeType_numberBigInteger();
        read_objectStart_string_arrow_typeBoolean_numberBigInteger();
        read_objectStart_string_arrow_typeUndefined_numberBigInteger();
        read_objectStart_string_arrow_false_numberBigInteger();
        read_objectStart_string_arrow_true_numberBigInteger();
        read_objectStart_string_arrow_undefined_numberBigInteger();

        read_objectStart_string_arrow_listStart_listEnd_numberBigDecimal();
        read_objectStart_string_arrow_objectStart_objectEnd_numberBigDecimal();
        read_objectStart_string_arrow_propertyStart_propertyEnd_numberBigDecimal();
        read_objectStart_string_arrow_string_numberBigDecimal();
        read_objectStart_string_arrow_numberInt_numberBigDecimal();
        read_objectStart_string_arrow_numberLong_numberBigDecimal();
        read_objectStart_string_arrow_numberDouble_numberBigDecimal();
        read_objectStart_string_arrow_numberBigInteger_numberBigDecimal();
        read_objectStart_string_arrow_numberBigDecimal_numberBigDecimal();
        read_objectStart_string_arrow_bytes_numberBigDecimal();
        read_objectStart_string_arrow_expression_numberBigDecimal();
        read_objectStart_string_arrow_typeObject_numberBigDecimal();
        read_objectStart_string_arrow_typeList_numberBigDecimal();
        read_objectStart_string_arrow_typeProperty_numberBigDecimal();
        read_objectStart_string_arrow_typeString_numberBigDecimal();
        read_objectStart_string_arrow_typeInt_numberBigDecimal();
        read_objectStart_string_arrow_typeLong_numberBigDecimal();
        read_objectStart_string_arrow_typeDouble_numberBigDecimal();
        read_objectStart_string_arrow_typeBigInteger_numberBigDecimal();
        read_objectStart_string_arrow_typeBigDecimal_numberBigDecimal();
        read_objectStart_string_arrow_typeBytes_numberBigDecimal();
        read_objectStart_string_arrow_typeExpression_numberBigDecimal();
        read_objectStart_string_arrow_typeType_numberBigDecimal();
        read_objectStart_string_arrow_typeBoolean_numberBigDecimal();
        read_objectStart_string_arrow_typeUndefined_numberBigDecimal();
        read_objectStart_string_arrow_false_numberBigDecimal();
        read_objectStart_string_arrow_true_numberBigDecimal();
        read_objectStart_string_arrow_undefined_numberBigDecimal();

        read_objectStart_string_arrow_listStart_listEnd_bytes();
        read_objectStart_string_arrow_objectStart_objectEnd_bytes();
        read_objectStart_string_arrow_propertyStart_propertyEnd_bytes();
        read_objectStart_string_arrow_string_bytes();
        read_objectStart_string_arrow_numberInt_bytes();
        read_objectStart_string_arrow_numberLong_bytes();
        read_objectStart_string_arrow_numberDouble_bytes();
        read_objectStart_string_arrow_numberBigInteger_bytes();
        read_objectStart_string_arrow_numberBigDecimal_bytes();
        read_objectStart_string_arrow_bytes_bytes();
        read_objectStart_string_arrow_expression_bytes();
        read_objectStart_string_arrow_typeObject_bytes();
        read_objectStart_string_arrow_typeList_bytes();
        read_objectStart_string_arrow_typeProperty_bytes();
        read_objectStart_string_arrow_typeString_bytes();
        read_objectStart_string_arrow_typeInt_bytes();
        read_objectStart_string_arrow_typeLong_bytes();
        read_objectStart_string_arrow_typeDouble_bytes();
        read_objectStart_string_arrow_typeBigInteger_bytes();
        read_objectStart_string_arrow_typeBigDecimal_bytes();
        read_objectStart_string_arrow_typeBytes_bytes();
        read_objectStart_string_arrow_typeExpression_bytes();
        read_objectStart_string_arrow_typeType_bytes();
        read_objectStart_string_arrow_typeBoolean_bytes();
        read_objectStart_string_arrow_typeUndefined_bytes();
        read_objectStart_string_arrow_false_bytes();
        read_objectStart_string_arrow_true_bytes();
        read_objectStart_string_arrow_undefined_bytes();

        read_objectStart_string_arrow_listStart_listEnd_expression();
        read_objectStart_string_arrow_objectStart_objectEnd_expression();
        read_objectStart_string_arrow_propertyStart_propertyEnd_expression();
        read_objectStart_string_arrow_string_expression();
        read_objectStart_string_arrow_numberInt_expression();
        read_objectStart_string_arrow_numberLong_expression();
        read_objectStart_string_arrow_numberDouble_expression();
        read_objectStart_string_arrow_numberBigInteger_expression();
        read_objectStart_string_arrow_numberBigDecimal_expression();
        read_objectStart_string_arrow_bytes_expression();
        read_objectStart_string_arrow_expression_expression();
        read_objectStart_string_arrow_typeObject_expression();
        read_objectStart_string_arrow_typeList_expression();
        read_objectStart_string_arrow_typeProperty_expression();
        read_objectStart_string_arrow_typeString_expression();
        read_objectStart_string_arrow_typeInt_expression();
        read_objectStart_string_arrow_typeLong_expression();
        read_objectStart_string_arrow_typeDouble_expression();
        read_objectStart_string_arrow_typeBigInteger_expression();
        read_objectStart_string_arrow_typeBigDecimal_expression();
        read_objectStart_string_arrow_typeBytes_expression();
        read_objectStart_string_arrow_typeExpression_expression();
        read_objectStart_string_arrow_typeType_expression();
        read_objectStart_string_arrow_typeBoolean_expression();
        read_objectStart_string_arrow_typeUndefined_expression();
        read_objectStart_string_arrow_false_expression();
        read_objectStart_string_arrow_true_expression();
        read_objectStart_string_arrow_undefined_expression();

        read_objectStart_string_arrow_listStart_listEnd_typeObject();
        read_objectStart_string_arrow_objectStart_objectEnd_typeObject();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeObject();
        read_objectStart_string_arrow_string_typeObject();
        read_objectStart_string_arrow_numberInt_typeObject();
        read_objectStart_string_arrow_numberLong_typeObject();
        read_objectStart_string_arrow_numberDouble_typeObject();
        read_objectStart_string_arrow_numberBigInteger_typeObject();
        read_objectStart_string_arrow_numberBigDecimal_typeObject();
        read_objectStart_string_arrow_bytes_typeObject();
        read_objectStart_string_arrow_expression_typeObject();
        read_objectStart_string_arrow_typeObject_typeObject();
        read_objectStart_string_arrow_typeList_typeObject();
        read_objectStart_string_arrow_typeProperty_typeObject();
        read_objectStart_string_arrow_typeString_typeObject();
        read_objectStart_string_arrow_typeInt_typeObject();
        read_objectStart_string_arrow_typeLong_typeObject();
        read_objectStart_string_arrow_typeDouble_typeObject();
        read_objectStart_string_arrow_typeBigInteger_typeObject();
        read_objectStart_string_arrow_typeBigDecimal_typeObject();
        read_objectStart_string_arrow_typeBytes_typeObject();
        read_objectStart_string_arrow_typeExpression_typeObject();
        read_objectStart_string_arrow_typeType_typeObject();
        read_objectStart_string_arrow_typeBoolean_typeObject();
        read_objectStart_string_arrow_typeUndefined_typeObject();
        read_objectStart_string_arrow_false_typeObject();
        read_objectStart_string_arrow_true_typeObject();
        read_objectStart_string_arrow_undefined_typeObject();

        read_objectStart_string_arrow_listStart_listEnd_typeList();
        read_objectStart_string_arrow_objectStart_objectEnd_typeList();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeList();
        read_objectStart_string_arrow_string_typeList();
        read_objectStart_string_arrow_numberInt_typeList();
        read_objectStart_string_arrow_numberLong_typeList();
        read_objectStart_string_arrow_numberDouble_typeList();
        read_objectStart_string_arrow_numberBigInteger_typeList();
        read_objectStart_string_arrow_numberBigDecimal_typeList();
        read_objectStart_string_arrow_bytes_typeList();
        read_objectStart_string_arrow_expression_typeList();
        read_objectStart_string_arrow_typeObject_typeList();
        read_objectStart_string_arrow_typeList_typeList();
        read_objectStart_string_arrow_typeProperty_typeList();
        read_objectStart_string_arrow_typeString_typeList();
        read_objectStart_string_arrow_typeInt_typeList();
        read_objectStart_string_arrow_typeLong_typeList();
        read_objectStart_string_arrow_typeDouble_typeList();
        read_objectStart_string_arrow_typeBigInteger_typeList();
        read_objectStart_string_arrow_typeBigDecimal_typeList();
        read_objectStart_string_arrow_typeBytes_typeList();
        read_objectStart_string_arrow_typeExpression_typeList();
        read_objectStart_string_arrow_typeType_typeList();
        read_objectStart_string_arrow_typeBoolean_typeList();
        read_objectStart_string_arrow_typeUndefined_typeList();
        read_objectStart_string_arrow_false_typeList();
        read_objectStart_string_arrow_true_typeList();
        read_objectStart_string_arrow_undefined_typeList();

        read_objectStart_string_arrow_listStart_listEnd_typeProperty();
        read_objectStart_string_arrow_objectStart_objectEnd_typeProperty();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeProperty();
        read_objectStart_string_arrow_string_typeProperty();
        read_objectStart_string_arrow_numberInt_typeProperty();
        read_objectStart_string_arrow_numberLong_typeProperty();
        read_objectStart_string_arrow_numberDouble_typeProperty();
        read_objectStart_string_arrow_numberBigInteger_typeProperty();
        read_objectStart_string_arrow_numberBigDecimal_typeProperty();
        read_objectStart_string_arrow_bytes_typeProperty();
        read_objectStart_string_arrow_expression_typeProperty();
        read_objectStart_string_arrow_typeObject_typeProperty();
        read_objectStart_string_arrow_typeList_typeProperty();
        read_objectStart_string_arrow_typeProperty_typeProperty();
        read_objectStart_string_arrow_typeString_typeProperty();
        read_objectStart_string_arrow_typeInt_typeProperty();
        read_objectStart_string_arrow_typeLong_typeProperty();
        read_objectStart_string_arrow_typeDouble_typeProperty();
        read_objectStart_string_arrow_typeBigInteger_typeProperty();
        read_objectStart_string_arrow_typeBigDecimal_typeProperty();
        read_objectStart_string_arrow_typeBytes_typeProperty();
        read_objectStart_string_arrow_typeExpression_typeProperty();
        read_objectStart_string_arrow_typeType_typeProperty();
        read_objectStart_string_arrow_typeBoolean_typeProperty();
        read_objectStart_string_arrow_typeUndefined_typeProperty();
        read_objectStart_string_arrow_false_typeProperty();
        read_objectStart_string_arrow_true_typeProperty();
        read_objectStart_string_arrow_undefined_typeProperty();

        read_objectStart_string_arrow_listStart_listEnd_typeString();
        read_objectStart_string_arrow_objectStart_objectEnd_typeString();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeString();
        read_objectStart_string_arrow_string_typeString();
        read_objectStart_string_arrow_numberInt_typeString();
        read_objectStart_string_arrow_numberLong_typeString();
        read_objectStart_string_arrow_numberDouble_typeString();
        read_objectStart_string_arrow_numberBigInteger_typeString();
        read_objectStart_string_arrow_numberBigDecimal_typeString();
        read_objectStart_string_arrow_bytes_typeString();
        read_objectStart_string_arrow_expression_typeString();
        read_objectStart_string_arrow_typeObject_typeString();
        read_objectStart_string_arrow_typeList_typeString();
        read_objectStart_string_arrow_typeProperty_typeString();
        read_objectStart_string_arrow_typeString_typeString();
        read_objectStart_string_arrow_typeInt_typeString();
        read_objectStart_string_arrow_typeLong_typeString();
        read_objectStart_string_arrow_typeDouble_typeString();
        read_objectStart_string_arrow_typeBigInteger_typeString();
        read_objectStart_string_arrow_typeBigDecimal_typeString();
        read_objectStart_string_arrow_typeBytes_typeString();
        read_objectStart_string_arrow_typeExpression_typeString();
        read_objectStart_string_arrow_typeType_typeString();
        read_objectStart_string_arrow_typeBoolean_typeString();
        read_objectStart_string_arrow_typeUndefined_typeString();
        read_objectStart_string_arrow_false_typeString();
        read_objectStart_string_arrow_true_typeString();
        read_objectStart_string_arrow_undefined_typeString();

        read_objectStart_string_arrow_listStart_listEnd_typeInt();
        read_objectStart_string_arrow_objectStart_objectEnd_typeInt();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeInt();
        read_objectStart_string_arrow_string_typeInt();
        read_objectStart_string_arrow_numberInt_typeInt();
        read_objectStart_string_arrow_numberLong_typeInt();
        read_objectStart_string_arrow_numberDouble_typeInt();
        read_objectStart_string_arrow_numberBigInteger_typeInt();
        read_objectStart_string_arrow_numberBigDecimal_typeInt();
        read_objectStart_string_arrow_bytes_typeInt();
        read_objectStart_string_arrow_expression_typeInt();
        read_objectStart_string_arrow_typeObject_typeInt();
        read_objectStart_string_arrow_typeList_typeInt();
        read_objectStart_string_arrow_typeProperty_typeInt();
        read_objectStart_string_arrow_typeString_typeInt();
        read_objectStart_string_arrow_typeInt_typeInt();
        read_objectStart_string_arrow_typeLong_typeInt();
        read_objectStart_string_arrow_typeDouble_typeInt();
        read_objectStart_string_arrow_typeBigInteger_typeInt();
        read_objectStart_string_arrow_typeBigDecimal_typeInt();
        read_objectStart_string_arrow_typeBytes_typeInt();
        read_objectStart_string_arrow_typeExpression_typeInt();
        read_objectStart_string_arrow_typeType_typeInt();
        read_objectStart_string_arrow_typeBoolean_typeInt();
        read_objectStart_string_arrow_typeUndefined_typeInt();
        read_objectStart_string_arrow_false_typeInt();
        read_objectStart_string_arrow_true_typeInt();
        read_objectStart_string_arrow_undefined_typeInt();

        read_objectStart_string_arrow_listStart_listEnd_typeLong();
        read_objectStart_string_arrow_objectStart_objectEnd_typeLong();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeLong();
        read_objectStart_string_arrow_string_typeLong();
        read_objectStart_string_arrow_numberInt_typeLong();
        read_objectStart_string_arrow_numberLong_typeLong();
        read_objectStart_string_arrow_numberDouble_typeLong();
        read_objectStart_string_arrow_numberBigInteger_typeLong();
        read_objectStart_string_arrow_numberBigDecimal_typeLong();
        read_objectStart_string_arrow_bytes_typeLong();
        read_objectStart_string_arrow_expression_typeLong();
        read_objectStart_string_arrow_typeObject_typeLong();
        read_objectStart_string_arrow_typeList_typeLong();
        read_objectStart_string_arrow_typeProperty_typeLong();
        read_objectStart_string_arrow_typeString_typeLong();
        read_objectStart_string_arrow_typeInt_typeLong();
        read_objectStart_string_arrow_typeLong_typeLong();
        read_objectStart_string_arrow_typeDouble_typeLong();
        read_objectStart_string_arrow_typeBigInteger_typeLong();
        read_objectStart_string_arrow_typeBigDecimal_typeLong();
        read_objectStart_string_arrow_typeBytes_typeLong();
        read_objectStart_string_arrow_typeExpression_typeLong();
        read_objectStart_string_arrow_typeType_typeLong();
        read_objectStart_string_arrow_typeBoolean_typeLong();
        read_objectStart_string_arrow_typeUndefined_typeLong();
        read_objectStart_string_arrow_false_typeLong();
        read_objectStart_string_arrow_true_typeLong();
        read_objectStart_string_arrow_undefined_typeLong();

        read_objectStart_string_arrow_listStart_listEnd_typeDouble();
        read_objectStart_string_arrow_objectStart_objectEnd_typeDouble();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeDouble();
        read_objectStart_string_arrow_string_typeDouble();
        read_objectStart_string_arrow_numberInt_typeDouble();
        read_objectStart_string_arrow_numberLong_typeDouble();
        read_objectStart_string_arrow_numberDouble_typeDouble();
        read_objectStart_string_arrow_numberBigInteger_typeDouble();
        read_objectStart_string_arrow_numberBigDecimal_typeDouble();
        read_objectStart_string_arrow_bytes_typeDouble();
        read_objectStart_string_arrow_expression_typeDouble();
        read_objectStart_string_arrow_typeObject_typeDouble();
        read_objectStart_string_arrow_typeList_typeDouble();
        read_objectStart_string_arrow_typeProperty_typeDouble();
        read_objectStart_string_arrow_typeString_typeDouble();
        read_objectStart_string_arrow_typeInt_typeDouble();
        read_objectStart_string_arrow_typeLong_typeDouble();
        read_objectStart_string_arrow_typeDouble_typeDouble();
        read_objectStart_string_arrow_typeBigInteger_typeDouble();
        read_objectStart_string_arrow_typeBigDecimal_typeDouble();
        read_objectStart_string_arrow_typeBytes_typeDouble();
        read_objectStart_string_arrow_typeExpression_typeDouble();
        read_objectStart_string_arrow_typeType_typeDouble();
        read_objectStart_string_arrow_typeBoolean_typeDouble();
        read_objectStart_string_arrow_typeUndefined_typeDouble();
        read_objectStart_string_arrow_false_typeDouble();
        read_objectStart_string_arrow_true_typeDouble();
        read_objectStart_string_arrow_undefined_typeDouble();

        read_objectStart_string_arrow_listStart_listEnd_typeBigInteger();
        read_objectStart_string_arrow_objectStart_objectEnd_typeBigInteger();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeBigInteger();
        read_objectStart_string_arrow_string_typeBigInteger();
        read_objectStart_string_arrow_numberInt_typeBigInteger();
        read_objectStart_string_arrow_numberLong_typeBigInteger();
        read_objectStart_string_arrow_numberDouble_typeBigInteger();
        read_objectStart_string_arrow_numberBigInteger_typeBigInteger();
        read_objectStart_string_arrow_numberBigDecimal_typeBigInteger();
        read_objectStart_string_arrow_bytes_typeBigInteger();
        read_objectStart_string_arrow_expression_typeBigInteger();
        read_objectStart_string_arrow_typeObject_typeBigInteger();
        read_objectStart_string_arrow_typeList_typeBigInteger();
        read_objectStart_string_arrow_typeProperty_typeBigInteger();
        read_objectStart_string_arrow_typeString_typeBigInteger();
        read_objectStart_string_arrow_typeInt_typeBigInteger();
        read_objectStart_string_arrow_typeLong_typeBigInteger();
        read_objectStart_string_arrow_typeDouble_typeBigInteger();
        read_objectStart_string_arrow_typeBigInteger_typeBigInteger();
        read_objectStart_string_arrow_typeBigDecimal_typeBigInteger();
        read_objectStart_string_arrow_typeBytes_typeBigInteger();
        read_objectStart_string_arrow_typeExpression_typeBigInteger();
        read_objectStart_string_arrow_typeType_typeBigInteger();
        read_objectStart_string_arrow_typeBoolean_typeBigInteger();
        read_objectStart_string_arrow_typeUndefined_typeBigInteger();
        read_objectStart_string_arrow_false_typeBigInteger();
        read_objectStart_string_arrow_true_typeBigInteger();
        read_objectStart_string_arrow_undefined_typeBigInteger();

        read_objectStart_string_arrow_listStart_listEnd_typeBigDecimal();
        read_objectStart_string_arrow_objectStart_objectEnd_typeBigDecimal();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeBigDecimal();
        read_objectStart_string_arrow_string_typeBigDecimal();
        read_objectStart_string_arrow_numberInt_typeBigDecimal();
        read_objectStart_string_arrow_numberLong_typeBigDecimal();
        read_objectStart_string_arrow_numberDouble_typeBigDecimal();
        read_objectStart_string_arrow_numberBigInteger_typeBigDecimal();
        read_objectStart_string_arrow_numberBigDecimal_typeBigDecimal();
        read_objectStart_string_arrow_bytes_typeBigDecimal();
        read_objectStart_string_arrow_expression_typeBigDecimal();
        read_objectStart_string_arrow_typeObject_typeBigDecimal();
        read_objectStart_string_arrow_typeList_typeBigDecimal();
        read_objectStart_string_arrow_typeProperty_typeBigDecimal();
        read_objectStart_string_arrow_typeString_typeBigDecimal();
        read_objectStart_string_arrow_typeInt_typeBigDecimal();
        read_objectStart_string_arrow_typeLong_typeBigDecimal();
        read_objectStart_string_arrow_typeDouble_typeBigDecimal();
        read_objectStart_string_arrow_typeBigInteger_typeBigDecimal();
        read_objectStart_string_arrow_typeBigDecimal_typeBigDecimal();
        read_objectStart_string_arrow_typeBytes_typeBigDecimal();
        read_objectStart_string_arrow_typeExpression_typeBigDecimal();
        read_objectStart_string_arrow_typeType_typeBigDecimal();
        read_objectStart_string_arrow_typeBoolean_typeBigDecimal();
        read_objectStart_string_arrow_typeUndefined_typeBigDecimal();
        read_objectStart_string_arrow_false_typeBigDecimal();
        read_objectStart_string_arrow_true_typeBigDecimal();
        read_objectStart_string_arrow_undefined_typeBigDecimal();

        read_objectStart_string_arrow_listStart_listEnd_typeBytes();
        read_objectStart_string_arrow_objectStart_objectEnd_typeBytes();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeBytes();
        read_objectStart_string_arrow_string_typeBytes();
        read_objectStart_string_arrow_numberInt_typeBytes();
        read_objectStart_string_arrow_numberLong_typeBytes();
        read_objectStart_string_arrow_numberDouble_typeBytes();
        read_objectStart_string_arrow_numberBigInteger_typeBytes();
        read_objectStart_string_arrow_numberBigDecimal_typeBytes();
        read_objectStart_string_arrow_bytes_typeBytes();
        read_objectStart_string_arrow_expression_typeBytes();
        read_objectStart_string_arrow_typeObject_typeBytes();
        read_objectStart_string_arrow_typeList_typeBytes();
        read_objectStart_string_arrow_typeProperty_typeBytes();
        read_objectStart_string_arrow_typeString_typeBytes();
        read_objectStart_string_arrow_typeInt_typeBytes();
        read_objectStart_string_arrow_typeLong_typeBytes();
        read_objectStart_string_arrow_typeDouble_typeBytes();
        read_objectStart_string_arrow_typeBigInteger_typeBytes();
        read_objectStart_string_arrow_typeBigDecimal_typeBytes();
        read_objectStart_string_arrow_typeBytes_typeBytes();
        read_objectStart_string_arrow_typeExpression_typeBytes();
        read_objectStart_string_arrow_typeType_typeBytes();
        read_objectStart_string_arrow_typeBoolean_typeBytes();
        read_objectStart_string_arrow_typeUndefined_typeBytes();
        read_objectStart_string_arrow_false_typeBytes();
        read_objectStart_string_arrow_true_typeBytes();
        read_objectStart_string_arrow_undefined_typeBytes();

        read_objectStart_string_arrow_listStart_listEnd_typeExpression();
        read_objectStart_string_arrow_objectStart_objectEnd_typeExpression();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeExpression();
        read_objectStart_string_arrow_string_typeExpression();
        read_objectStart_string_arrow_numberInt_typeExpression();
        read_objectStart_string_arrow_numberLong_typeExpression();
        read_objectStart_string_arrow_numberDouble_typeExpression();
        read_objectStart_string_arrow_numberBigInteger_typeExpression();
        read_objectStart_string_arrow_numberBigDecimal_typeExpression();
        read_objectStart_string_arrow_bytes_typeExpression();
        read_objectStart_string_arrow_expression_typeExpression();
        read_objectStart_string_arrow_typeObject_typeExpression();
        read_objectStart_string_arrow_typeList_typeExpression();
        read_objectStart_string_arrow_typeProperty_typeExpression();
        read_objectStart_string_arrow_typeString_typeExpression();
        read_objectStart_string_arrow_typeInt_typeExpression();
        read_objectStart_string_arrow_typeLong_typeExpression();
        read_objectStart_string_arrow_typeDouble_typeExpression();
        read_objectStart_string_arrow_typeBigInteger_typeExpression();
        read_objectStart_string_arrow_typeBigDecimal_typeExpression();
        read_objectStart_string_arrow_typeBytes_typeExpression();
        read_objectStart_string_arrow_typeExpression_typeExpression();
        read_objectStart_string_arrow_typeType_typeExpression();
        read_objectStart_string_arrow_typeBoolean_typeExpression();
        read_objectStart_string_arrow_typeUndefined_typeExpression();
        read_objectStart_string_arrow_false_typeExpression();
        read_objectStart_string_arrow_true_typeExpression();
        read_objectStart_string_arrow_undefined_typeExpression();

        read_objectStart_string_arrow_listStart_listEnd_typeType();
        read_objectStart_string_arrow_objectStart_objectEnd_typeType();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeType();
        read_objectStart_string_arrow_string_typeType();
        read_objectStart_string_arrow_numberInt_typeType();
        read_objectStart_string_arrow_numberLong_typeType();
        read_objectStart_string_arrow_numberDouble_typeType();
        read_objectStart_string_arrow_numberBigInteger_typeType();
        read_objectStart_string_arrow_numberBigDecimal_typeType();
        read_objectStart_string_arrow_bytes_typeType();
        read_objectStart_string_arrow_expression_typeType();
        read_objectStart_string_arrow_typeObject_typeType();
        read_objectStart_string_arrow_typeList_typeType();
        read_objectStart_string_arrow_typeProperty_typeType();
        read_objectStart_string_arrow_typeString_typeType();
        read_objectStart_string_arrow_typeInt_typeType();
        read_objectStart_string_arrow_typeLong_typeType();
        read_objectStart_string_arrow_typeDouble_typeType();
        read_objectStart_string_arrow_typeBigInteger_typeType();
        read_objectStart_string_arrow_typeBigDecimal_typeType();
        read_objectStart_string_arrow_typeBytes_typeType();
        read_objectStart_string_arrow_typeExpression_typeType();
        read_objectStart_string_arrow_typeType_typeType();
        read_objectStart_string_arrow_typeBoolean_typeType();
        read_objectStart_string_arrow_typeUndefined_typeType();
        read_objectStart_string_arrow_false_typeType();
        read_objectStart_string_arrow_true_typeType();
        read_objectStart_string_arrow_undefined_typeType();

        read_objectStart_string_arrow_listStart_listEnd_typeBoolean();
        read_objectStart_string_arrow_objectStart_objectEnd_typeBoolean();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeBoolean();
        read_objectStart_string_arrow_string_typeBoolean();
        read_objectStart_string_arrow_numberInt_typeBoolean();
        read_objectStart_string_arrow_numberLong_typeBoolean();
        read_objectStart_string_arrow_numberDouble_typeBoolean();
        read_objectStart_string_arrow_numberBigInteger_typeBoolean();
        read_objectStart_string_arrow_numberBigDecimal_typeBoolean();
        read_objectStart_string_arrow_bytes_typeBoolean();
        read_objectStart_string_arrow_expression_typeBoolean();
        read_objectStart_string_arrow_typeObject_typeBoolean();
        read_objectStart_string_arrow_typeList_typeBoolean();
        read_objectStart_string_arrow_typeProperty_typeBoolean();
        read_objectStart_string_arrow_typeString_typeBoolean();
        read_objectStart_string_arrow_typeInt_typeBoolean();
        read_objectStart_string_arrow_typeLong_typeBoolean();
        read_objectStart_string_arrow_typeDouble_typeBoolean();
        read_objectStart_string_arrow_typeBigInteger_typeBoolean();
        read_objectStart_string_arrow_typeBigDecimal_typeBoolean();
        read_objectStart_string_arrow_typeBytes_typeBoolean();
        read_objectStart_string_arrow_typeExpression_typeBoolean();
        read_objectStart_string_arrow_typeType_typeBoolean();
        read_objectStart_string_arrow_typeBoolean_typeBoolean();
        read_objectStart_string_arrow_typeUndefined_typeBoolean();
        read_objectStart_string_arrow_false_typeBoolean();
        read_objectStart_string_arrow_true_typeBoolean();
        read_objectStart_string_arrow_undefined_typeBoolean();

        read_objectStart_string_arrow_listStart_listEnd_typeUndefined();
        read_objectStart_string_arrow_objectStart_objectEnd_typeUndefined();
        read_objectStart_string_arrow_propertyStart_propertyEnd_typeUndefined();
        read_objectStart_string_arrow_string_typeUndefined();
        read_objectStart_string_arrow_numberInt_typeUndefined();
        read_objectStart_string_arrow_numberLong_typeUndefined();
        read_objectStart_string_arrow_numberDouble_typeUndefined();
        read_objectStart_string_arrow_numberBigInteger_typeUndefined();
        read_objectStart_string_arrow_numberBigDecimal_typeUndefined();
        read_objectStart_string_arrow_bytes_typeUndefined();
        read_objectStart_string_arrow_expression_typeUndefined();
        read_objectStart_string_arrow_typeObject_typeUndefined();
        read_objectStart_string_arrow_typeList_typeUndefined();
        read_objectStart_string_arrow_typeProperty_typeUndefined();
        read_objectStart_string_arrow_typeString_typeUndefined();
        read_objectStart_string_arrow_typeInt_typeUndefined();
        read_objectStart_string_arrow_typeLong_typeUndefined();
        read_objectStart_string_arrow_typeDouble_typeUndefined();
        read_objectStart_string_arrow_typeBigInteger_typeUndefined();
        read_objectStart_string_arrow_typeBigDecimal_typeUndefined();
        read_objectStart_string_arrow_typeBytes_typeUndefined();
        read_objectStart_string_arrow_typeExpression_typeUndefined();
        read_objectStart_string_arrow_typeType_typeUndefined();
        read_objectStart_string_arrow_typeBoolean_typeUndefined();
        read_objectStart_string_arrow_typeUndefined_typeUndefined();
        read_objectStart_string_arrow_false_typeUndefined();
        read_objectStart_string_arrow_true_typeUndefined();
        read_objectStart_string_arrow_undefined_typeUndefined();

        read_objectStart_string_arrow_listStart_listEnd_true();
        read_objectStart_string_arrow_objectStart_objectEnd_true();
        read_objectStart_string_arrow_propertyStart_propertyEnd_true();
        read_objectStart_string_arrow_string_true();
        read_objectStart_string_arrow_numberInt_true();
        read_objectStart_string_arrow_numberLong_true();
        read_objectStart_string_arrow_numberDouble_true();
        read_objectStart_string_arrow_numberBigInteger_true();
        read_objectStart_string_arrow_numberBigDecimal_true();
        read_objectStart_string_arrow_bytes_true();
        read_objectStart_string_arrow_expression_true();
        read_objectStart_string_arrow_typeObject_true();
        read_objectStart_string_arrow_typeList_true();
        read_objectStart_string_arrow_typeProperty_true();
        read_objectStart_string_arrow_typeString_true();
        read_objectStart_string_arrow_typeInt_true();
        read_objectStart_string_arrow_typeLong_true();
        read_objectStart_string_arrow_typeDouble_true();
        read_objectStart_string_arrow_typeBigInteger_true();
        read_objectStart_string_arrow_typeBigDecimal_true();
        read_objectStart_string_arrow_typeBytes_true();
        read_objectStart_string_arrow_typeExpression_true();
        read_objectStart_string_arrow_typeType_true();
        read_objectStart_string_arrow_typeBoolean_true();
        read_objectStart_string_arrow_typeUndefined_true();
        read_objectStart_string_arrow_false_true();
        read_objectStart_string_arrow_true_true();
        read_objectStart_string_arrow_undefined_true();

        read_objectStart_string_arrow_listStart_listEnd_false();
        read_objectStart_string_arrow_objectStart_objectEnd_false();
        read_objectStart_string_arrow_propertyStart_propertyEnd_false();
        read_objectStart_string_arrow_string_false();
        read_objectStart_string_arrow_numberInt_false();
        read_objectStart_string_arrow_numberLong_false();
        read_objectStart_string_arrow_numberDouble_false();
        read_objectStart_string_arrow_numberBigInteger_false();
        read_objectStart_string_arrow_numberBigDecimal_false();
        read_objectStart_string_arrow_bytes_false();
        read_objectStart_string_arrow_expression_false();
        read_objectStart_string_arrow_typeObject_false();
        read_objectStart_string_arrow_typeList_false();
        read_objectStart_string_arrow_typeProperty_false();
        read_objectStart_string_arrow_typeString_false();
        read_objectStart_string_arrow_typeInt_false();
        read_objectStart_string_arrow_typeLong_false();
        read_objectStart_string_arrow_typeDouble_false();
        read_objectStart_string_arrow_typeBigInteger_false();
        read_objectStart_string_arrow_typeBigDecimal_false();
        read_objectStart_string_arrow_typeBytes_false();
        read_objectStart_string_arrow_typeExpression_false();
        read_objectStart_string_arrow_typeType_false();
        read_objectStart_string_arrow_typeBoolean_false();
        read_objectStart_string_arrow_typeUndefined_false();
        read_objectStart_string_arrow_false_false();
        read_objectStart_string_arrow_true_false();
        read_objectStart_string_arrow_undefined_false();

        read_objectStart_string_arrow_listStart_listEnd_undefined();
        read_objectStart_string_arrow_objectStart_objectEnd_undefined();
        read_objectStart_string_arrow_propertyStart_propertyEnd_undefined();
        read_objectStart_string_arrow_string_undefined();
        read_objectStart_string_arrow_numberInt_undefined();
        read_objectStart_string_arrow_numberLong_undefined();
        read_objectStart_string_arrow_numberDouble_undefined();
        read_objectStart_string_arrow_numberBigInteger_undefined();
        read_objectStart_string_arrow_numberBigDecimal_undefined();
        read_objectStart_string_arrow_bytes_undefined();
        read_objectStart_string_arrow_expression_undefined();
        read_objectStart_string_arrow_typeObject_undefined();
        read_objectStart_string_arrow_typeList_undefined();
        read_objectStart_string_arrow_typeProperty_undefined();
        read_objectStart_string_arrow_typeString_undefined();
        read_objectStart_string_arrow_typeInt_undefined();
        read_objectStart_string_arrow_typeLong_undefined();
        read_objectStart_string_arrow_typeDouble_undefined();
        read_objectStart_string_arrow_typeBigInteger_undefined();
        read_objectStart_string_arrow_typeBigDecimal_undefined();
        read_objectStart_string_arrow_typeBytes_undefined();
        read_objectStart_string_arrow_typeExpression_undefined();
        read_objectStart_string_arrow_typeType_undefined();
        read_objectStart_string_arrow_typeBoolean_undefined();
        read_objectStart_string_arrow_typeUndefined_undefined();
        read_objectStart_string_arrow_false_undefined();
        read_objectStart_string_arrow_true_undefined();
        read_objectStart_string_arrow_undefined_undefined();
    }

    @Test
    public void notEmptyObjectBrokenAfterComma() throws IOException, ModelException {
        read_objectStart_string_arrow_listStart_listEnd_comma_unexpectedChar();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_unexpectedChar();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_unexpectedChar();
        read_objectStart_string_arrow_string_comma_unexpectedChar();
        read_objectStart_string_arrow_numberInt_comma_unexpectedChar();
        read_objectStart_string_arrow_numberLong_comma_unexpectedChar();
        read_objectStart_string_arrow_numberDouble_comma_unexpectedChar();
        read_objectStart_string_arrow_numberBigInteger_comma_unexpectedChar();
        read_objectStart_string_arrow_numberBigDecimal_comma_unexpectedChar();
        read_objectStart_string_arrow_bytes_comma_unexpectedChar();
        read_objectStart_string_arrow_expression_comma_unexpectedChar();
        read_objectStart_string_arrow_typeObject_comma_unexpectedChar();
        read_objectStart_string_arrow_typeList_comma_unexpectedChar();
        read_objectStart_string_arrow_typeProperty_comma_unexpectedChar();
        read_objectStart_string_arrow_typeString_comma_unexpectedChar();
        read_objectStart_string_arrow_typeInt_comma_unexpectedChar();
        read_objectStart_string_arrow_typeLong_comma_unexpectedChar();
        read_objectStart_string_arrow_typeDouble_comma_unexpectedChar();
        read_objectStart_string_arrow_typeBigInteger_comma_unexpectedChar();
        read_objectStart_string_arrow_typeBigDecimal_comma_unexpectedChar();
        read_objectStart_string_arrow_typeBytes_comma_unexpectedChar();
        read_objectStart_string_arrow_typeExpression_comma_unexpectedChar();
        read_objectStart_string_arrow_typeType_comma_unexpectedChar();
        read_objectStart_string_arrow_typeBoolean_comma_unexpectedChar();
        read_objectStart_string_arrow_typeUndefined_comma_unexpectedChar();
        read_objectStart_string_arrow_false_comma_unexpectedChar();
        read_objectStart_string_arrow_true_comma_unexpectedChar();
        read_objectStart_string_arrow_undefined_comma_unexpectedChar();

        read_objectStart_string_arrow_listStart_listEnd_comma_unexpectedEOF();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_unexpectedEOF();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_unexpectedEOF();
        read_objectStart_string_arrow_string_comma_unexpectedEOF();
        read_objectStart_string_arrow_numberInt_comma_unexpectedEOF();
        read_objectStart_string_arrow_numberLong_comma_unexpectedEOF();
        read_objectStart_string_arrow_numberDouble_comma_unexpectedEOF();
        read_objectStart_string_arrow_numberBigInteger_comma_unexpectedEOF();
        read_objectStart_string_arrow_numberBigDecimal_comma_unexpectedEOF();
        read_objectStart_string_arrow_bytes_comma_unexpectedEOF();
        read_objectStart_string_arrow_expression_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeObject_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeList_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeProperty_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeString_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeInt_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeLong_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeDouble_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeBigInteger_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeBigDecimal_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeBytes_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeExpression_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeType_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeBoolean_comma_unexpectedEOF();
        read_objectStart_string_arrow_typeUndefined_comma_unexpectedEOF();
        read_objectStart_string_arrow_false_comma_unexpectedEOF();
        read_objectStart_string_arrow_true_comma_unexpectedEOF();
        read_objectStart_string_arrow_undefined_comma_unexpectedEOF();

        read_objectStart_string_arrow_listStart_listEnd_comma_comma();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_comma();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_comma();
        read_objectStart_string_arrow_string_comma_comma();
        read_objectStart_string_arrow_numberInt_comma_comma();
        read_objectStart_string_arrow_numberLong_comma_comma();
        read_objectStart_string_arrow_numberDouble_comma_comma();
        read_objectStart_string_arrow_numberBigInteger_comma_comma();
        read_objectStart_string_arrow_numberBigDecimal_comma_comma();
        read_objectStart_string_arrow_bytes_comma_comma();
        read_objectStart_string_arrow_expression_comma_comma();
        read_objectStart_string_arrow_typeObject_comma_comma();
        read_objectStart_string_arrow_typeList_comma_comma();
        read_objectStart_string_arrow_typeProperty_comma_comma();
        read_objectStart_string_arrow_typeString_comma_comma();
        read_objectStart_string_arrow_typeInt_comma_comma();
        read_objectStart_string_arrow_typeLong_comma_comma();
        read_objectStart_string_arrow_typeDouble_comma_comma();
        read_objectStart_string_arrow_typeBigInteger_comma_comma();
        read_objectStart_string_arrow_typeBigDecimal_comma_comma();
        read_objectStart_string_arrow_typeBytes_comma_comma();
        read_objectStart_string_arrow_typeExpression_comma_comma();
        read_objectStart_string_arrow_typeType_comma_comma();
        read_objectStart_string_arrow_typeBoolean_comma_comma();
        read_objectStart_string_arrow_typeUndefined_comma_comma();
        read_objectStart_string_arrow_false_comma_comma();
        read_objectStart_string_arrow_true_comma_comma();
        read_objectStart_string_arrow_undefined_comma_comma();

        read_objectStart_string_arrow_listStart_listEnd_comma_arrow();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_arrow();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_arrow();
        read_objectStart_string_arrow_string_comma_arrow();
        read_objectStart_string_arrow_numberInt_comma_arrow();
        read_objectStart_string_arrow_numberLong_comma_arrow();
        read_objectStart_string_arrow_numberDouble_comma_arrow();
        read_objectStart_string_arrow_numberBigInteger_comma_arrow();
        read_objectStart_string_arrow_numberBigDecimal_comma_arrow();
        read_objectStart_string_arrow_bytes_comma_arrow();
        read_objectStart_string_arrow_expression_comma_arrow();
        read_objectStart_string_arrow_typeObject_comma_arrow();
        read_objectStart_string_arrow_typeList_comma_arrow();
        read_objectStart_string_arrow_typeProperty_comma_arrow();
        read_objectStart_string_arrow_typeString_comma_arrow();
        read_objectStart_string_arrow_typeInt_comma_arrow();
        read_objectStart_string_arrow_typeLong_comma_arrow();
        read_objectStart_string_arrow_typeDouble_comma_arrow();
        read_objectStart_string_arrow_typeBigInteger_comma_arrow();
        read_objectStart_string_arrow_typeBigDecimal_comma_arrow();
        read_objectStart_string_arrow_typeBytes_comma_arrow();
        read_objectStart_string_arrow_typeExpression_comma_arrow();
        read_objectStart_string_arrow_typeType_comma_arrow();
        read_objectStart_string_arrow_typeBoolean_comma_arrow();
        read_objectStart_string_arrow_typeUndefined_comma_arrow();
        read_objectStart_string_arrow_false_comma_arrow();
        read_objectStart_string_arrow_true_comma_arrow();
        read_objectStart_string_arrow_undefined_comma_arrow();

        read_objectStart_string_arrow_listStart_listEnd_comma_objectStart();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_objectStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_objectStart();
        read_objectStart_string_arrow_string_comma_objectStart();
        read_objectStart_string_arrow_numberInt_comma_objectStart();
        read_objectStart_string_arrow_numberLong_comma_objectStart();
        read_objectStart_string_arrow_numberDouble_comma_objectStart();
        read_objectStart_string_arrow_numberBigInteger_comma_objectStart();
        read_objectStart_string_arrow_numberBigDecimal_comma_objectStart();
        read_objectStart_string_arrow_bytes_comma_objectStart();
        read_objectStart_string_arrow_expression_comma_objectStart();
        read_objectStart_string_arrow_typeObject_comma_objectStart();
        read_objectStart_string_arrow_typeList_comma_objectStart();
        read_objectStart_string_arrow_typeProperty_comma_objectStart();
        read_objectStart_string_arrow_typeString_comma_objectStart();
        read_objectStart_string_arrow_typeInt_comma_objectStart();
        read_objectStart_string_arrow_typeLong_comma_objectStart();
        read_objectStart_string_arrow_typeDouble_comma_objectStart();
        read_objectStart_string_arrow_typeBigInteger_comma_objectStart();
        read_objectStart_string_arrow_typeBigDecimal_comma_objectStart();
        read_objectStart_string_arrow_typeBytes_comma_objectStart();
        read_objectStart_string_arrow_typeExpression_comma_objectStart();
        read_objectStart_string_arrow_typeType_comma_objectStart();
        read_objectStart_string_arrow_typeBoolean_comma_objectStart();
        read_objectStart_string_arrow_typeUndefined_comma_objectStart();
        read_objectStart_string_arrow_false_comma_objectStart();
        read_objectStart_string_arrow_true_comma_objectStart();
        read_objectStart_string_arrow_undefined_comma_objectStart();

        read_objectStart_string_arrow_listStart_listEnd_comma_listStart();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_listStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_listStart();
        read_objectStart_string_arrow_string_comma_listStart();
        read_objectStart_string_arrow_numberInt_comma_listStart();
        read_objectStart_string_arrow_numberLong_comma_listStart();
        read_objectStart_string_arrow_numberDouble_comma_listStart();
        read_objectStart_string_arrow_numberBigInteger_comma_listStart();
        read_objectStart_string_arrow_numberBigDecimal_comma_listStart();
        read_objectStart_string_arrow_bytes_comma_listStart();
        read_objectStart_string_arrow_expression_comma_listStart();
        read_objectStart_string_arrow_typeObject_comma_listStart();
        read_objectStart_string_arrow_typeList_comma_listStart();
        read_objectStart_string_arrow_typeProperty_comma_listStart();
        read_objectStart_string_arrow_typeString_comma_listStart();
        read_objectStart_string_arrow_typeInt_comma_listStart();
        read_objectStart_string_arrow_typeLong_comma_listStart();
        read_objectStart_string_arrow_typeDouble_comma_listStart();
        read_objectStart_string_arrow_typeBigInteger_comma_listStart();
        read_objectStart_string_arrow_typeBigDecimal_comma_listStart();
        read_objectStart_string_arrow_typeBytes_comma_listStart();
        read_objectStart_string_arrow_typeExpression_comma_listStart();
        read_objectStart_string_arrow_typeType_comma_listStart();
        read_objectStart_string_arrow_typeBoolean_comma_listStart();
        read_objectStart_string_arrow_typeUndefined_comma_listStart();
        read_objectStart_string_arrow_false_comma_listStart();
        read_objectStart_string_arrow_true_comma_listStart();
        read_objectStart_string_arrow_undefined_comma_listStart();

        read_objectStart_string_arrow_listStart_listEnd_comma_listEnd();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_listEnd();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_listEnd();
        read_objectStart_string_arrow_string_comma_listEnd();
        read_objectStart_string_arrow_numberInt_comma_listEnd();
        read_objectStart_string_arrow_numberLong_comma_listEnd();
        read_objectStart_string_arrow_numberDouble_comma_listEnd();
        read_objectStart_string_arrow_numberBigInteger_comma_listEnd();
        read_objectStart_string_arrow_numberBigDecimal_comma_listEnd();
        read_objectStart_string_arrow_bytes_comma_listEnd();
        read_objectStart_string_arrow_expression_comma_listEnd();
        read_objectStart_string_arrow_typeObject_comma_listEnd();
        read_objectStart_string_arrow_typeList_comma_listEnd();
        read_objectStart_string_arrow_typeProperty_comma_listEnd();
        read_objectStart_string_arrow_typeString_comma_listEnd();
        read_objectStart_string_arrow_typeInt_comma_listEnd();
        read_objectStart_string_arrow_typeLong_comma_listEnd();
        read_objectStart_string_arrow_typeDouble_comma_listEnd();
        read_objectStart_string_arrow_typeBigInteger_comma_listEnd();
        read_objectStart_string_arrow_typeBigDecimal_comma_listEnd();
        read_objectStart_string_arrow_typeBytes_comma_listEnd();
        read_objectStart_string_arrow_typeExpression_comma_listEnd();
        read_objectStart_string_arrow_typeType_comma_listEnd();
        read_objectStart_string_arrow_typeBoolean_comma_listEnd();
        read_objectStart_string_arrow_typeUndefined_comma_listEnd();
        read_objectStart_string_arrow_false_comma_listEnd();
        read_objectStart_string_arrow_true_comma_listEnd();
        read_objectStart_string_arrow_undefined_comma_listEnd();

        read_objectStart_string_arrow_listStart_listEnd_comma_listStart();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_listStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_listStart();
        read_objectStart_string_arrow_string_comma_listStart();
        read_objectStart_string_arrow_numberInt_comma_listStart();
        read_objectStart_string_arrow_numberLong_comma_listStart();
        read_objectStart_string_arrow_numberDouble_comma_listStart();
        read_objectStart_string_arrow_numberBigInteger_comma_listStart();
        read_objectStart_string_arrow_numberBigDecimal_comma_listStart();
        read_objectStart_string_arrow_bytes_comma_listStart();
        read_objectStart_string_arrow_expression_comma_listStart();
        read_objectStart_string_arrow_typeObject_comma_listStart();
        read_objectStart_string_arrow_typeList_comma_listStart();
        read_objectStart_string_arrow_typeProperty_comma_listStart();
        read_objectStart_string_arrow_typeString_comma_listStart();
        read_objectStart_string_arrow_typeInt_comma_listStart();
        read_objectStart_string_arrow_typeLong_comma_listStart();
        read_objectStart_string_arrow_typeDouble_comma_listStart();
        read_objectStart_string_arrow_typeBigInteger_comma_listStart();
        read_objectStart_string_arrow_typeBigDecimal_comma_listStart();
        read_objectStart_string_arrow_typeBytes_comma_listStart();
        read_objectStart_string_arrow_typeExpression_comma_listStart();
        read_objectStart_string_arrow_typeType_comma_listStart();
        read_objectStart_string_arrow_typeBoolean_comma_listStart();
        read_objectStart_string_arrow_typeUndefined_comma_listStart();
        read_objectStart_string_arrow_false_comma_listStart();
        read_objectStart_string_arrow_true_comma_listStart();
        read_objectStart_string_arrow_undefined_comma_listStart();

        read_objectStart_string_arrow_listStart_listEnd_comma_propertyEnd();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_propertyEnd();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_propertyEnd();
        read_objectStart_string_arrow_string_comma_propertyEnd();
        read_objectStart_string_arrow_numberInt_comma_propertyEnd();
        read_objectStart_string_arrow_numberLong_comma_propertyEnd();
        read_objectStart_string_arrow_numberDouble_comma_propertyEnd();
        read_objectStart_string_arrow_numberBigInteger_comma_propertyEnd();
        read_objectStart_string_arrow_numberBigDecimal_comma_propertyEnd();
        read_objectStart_string_arrow_bytes_comma_propertyEnd();
        read_objectStart_string_arrow_expression_comma_propertyEnd();
        read_objectStart_string_arrow_typeObject_comma_propertyEnd();
        read_objectStart_string_arrow_typeList_comma_propertyEnd();
        read_objectStart_string_arrow_typeProperty_comma_propertyEnd();
        read_objectStart_string_arrow_typeString_comma_propertyEnd();
        read_objectStart_string_arrow_typeInt_comma_propertyEnd();
        read_objectStart_string_arrow_typeLong_comma_propertyEnd();
        read_objectStart_string_arrow_typeDouble_comma_propertyEnd();
        read_objectStart_string_arrow_typeBigInteger_comma_propertyEnd();
        read_objectStart_string_arrow_typeBigDecimal_comma_propertyEnd();
        read_objectStart_string_arrow_typeBytes_comma_propertyEnd();
        read_objectStart_string_arrow_typeExpression_comma_propertyEnd();
        read_objectStart_string_arrow_typeType_comma_propertyEnd();
        read_objectStart_string_arrow_typeBoolean_comma_propertyEnd();
        read_objectStart_string_arrow_typeUndefined_comma_propertyEnd();
        read_objectStart_string_arrow_false_comma_propertyEnd();
        read_objectStart_string_arrow_true_comma_propertyEnd();
        read_objectStart_string_arrow_undefined_comma_propertyEnd();

        read_objectStart_string_arrow_listStart_listEnd_comma_propertyStart();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_propertyStart();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_propertyStart();
        read_objectStart_string_arrow_string_comma_propertyStart();
        read_objectStart_string_arrow_numberInt_comma_propertyStart();
        read_objectStart_string_arrow_numberLong_comma_propertyStart();
        read_objectStart_string_arrow_numberDouble_comma_propertyStart();
        read_objectStart_string_arrow_numberBigInteger_comma_propertyStart();
        read_objectStart_string_arrow_numberBigDecimal_comma_propertyStart();
        read_objectStart_string_arrow_bytes_comma_propertyStart();
        read_objectStart_string_arrow_expression_comma_propertyStart();
        read_objectStart_string_arrow_typeObject_comma_propertyStart();
        read_objectStart_string_arrow_typeList_comma_propertyStart();
        read_objectStart_string_arrow_typeProperty_comma_propertyStart();
        read_objectStart_string_arrow_typeString_comma_propertyStart();
        read_objectStart_string_arrow_typeInt_comma_propertyStart();
        read_objectStart_string_arrow_typeLong_comma_propertyStart();
        read_objectStart_string_arrow_typeDouble_comma_propertyStart();
        read_objectStart_string_arrow_typeBigInteger_comma_propertyStart();
        read_objectStart_string_arrow_typeBigDecimal_comma_propertyStart();
        read_objectStart_string_arrow_typeBytes_comma_propertyStart();
        read_objectStart_string_arrow_typeExpression_comma_propertyStart();
        read_objectStart_string_arrow_typeType_comma_propertyStart();
        read_objectStart_string_arrow_typeBoolean_comma_propertyStart();
        read_objectStart_string_arrow_typeUndefined_comma_propertyStart();
        read_objectStart_string_arrow_false_comma_propertyStart();
        read_objectStart_string_arrow_true_comma_propertyStart();
        read_objectStart_string_arrow_undefined_comma_propertyStart();

        read_objectStart_string_arrow_listStart_listEnd_comma_numberInt();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_numberInt();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_numberInt();
        read_objectStart_string_arrow_string_comma_numberInt();
        read_objectStart_string_arrow_numberInt_comma_numberInt();
        read_objectStart_string_arrow_numberLong_comma_numberInt();
        read_objectStart_string_arrow_numberDouble_comma_numberInt();
        read_objectStart_string_arrow_numberBigInteger_comma_numberInt();
        read_objectStart_string_arrow_numberBigDecimal_comma_numberInt();
        read_objectStart_string_arrow_bytes_comma_numberInt();
        read_objectStart_string_arrow_expression_comma_numberInt();
        read_objectStart_string_arrow_typeObject_comma_numberInt();
        read_objectStart_string_arrow_typeList_comma_numberInt();
        read_objectStart_string_arrow_typeProperty_comma_numberInt();
        read_objectStart_string_arrow_typeString_comma_numberInt();
        read_objectStart_string_arrow_typeInt_comma_numberInt();
        read_objectStart_string_arrow_typeLong_comma_numberInt();
        read_objectStart_string_arrow_typeDouble_comma_numberInt();
        read_objectStart_string_arrow_typeBigInteger_comma_numberInt();
        read_objectStart_string_arrow_typeBigDecimal_comma_numberInt();
        read_objectStart_string_arrow_typeBytes_comma_numberInt();
        read_objectStart_string_arrow_typeExpression_comma_numberInt();
        read_objectStart_string_arrow_typeType_comma_numberInt();
        read_objectStart_string_arrow_typeBoolean_comma_numberInt();
        read_objectStart_string_arrow_typeUndefined_comma_numberInt();
        read_objectStart_string_arrow_false_comma_numberInt();
        read_objectStart_string_arrow_true_comma_numberInt();
        read_objectStart_string_arrow_undefined_comma_numberInt();

        read_objectStart_string_arrow_listStart_listEnd_comma_numberLong();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_numberLong();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_numberLong();
        read_objectStart_string_arrow_string_comma_numberLong();
        read_objectStart_string_arrow_numberInt_comma_numberLong();
        read_objectStart_string_arrow_numberLong_comma_numberLong();
        read_objectStart_string_arrow_numberDouble_comma_numberLong();
        read_objectStart_string_arrow_numberBigInteger_comma_numberLong();
        read_objectStart_string_arrow_numberBigDecimal_comma_numberLong();
        read_objectStart_string_arrow_bytes_comma_numberLong();
        read_objectStart_string_arrow_expression_comma_numberLong();
        read_objectStart_string_arrow_typeObject_comma_numberLong();
        read_objectStart_string_arrow_typeList_comma_numberLong();
        read_objectStart_string_arrow_typeProperty_comma_numberLong();
        read_objectStart_string_arrow_typeString_comma_numberLong();
        read_objectStart_string_arrow_typeInt_comma_numberLong();
        read_objectStart_string_arrow_typeLong_comma_numberLong();
        read_objectStart_string_arrow_typeDouble_comma_numberLong();
        read_objectStart_string_arrow_typeBigInteger_comma_numberLong();
        read_objectStart_string_arrow_typeBigDecimal_comma_numberLong();
        read_objectStart_string_arrow_typeBytes_comma_numberLong();
        read_objectStart_string_arrow_typeExpression_comma_numberLong();
        read_objectStart_string_arrow_typeType_comma_numberLong();
        read_objectStart_string_arrow_typeBoolean_comma_numberLong();
        read_objectStart_string_arrow_typeUndefined_comma_numberLong();
        read_objectStart_string_arrow_false_comma_numberLong();
        read_objectStart_string_arrow_true_comma_numberLong();
        read_objectStart_string_arrow_undefined_comma_numberLong();

        read_objectStart_string_arrow_listStart_listEnd_comma_numberDouble();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_numberDouble();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_numberDouble();
        read_objectStart_string_arrow_string_comma_numberDouble();
        read_objectStart_string_arrow_numberInt_comma_numberDouble();
        read_objectStart_string_arrow_numberLong_comma_numberDouble();
        read_objectStart_string_arrow_numberDouble_comma_numberDouble();
        read_objectStart_string_arrow_numberBigInteger_comma_numberDouble();
        read_objectStart_string_arrow_numberBigDecimal_comma_numberDouble();
        read_objectStart_string_arrow_bytes_comma_numberDouble();
        read_objectStart_string_arrow_expression_comma_numberDouble();
        read_objectStart_string_arrow_typeObject_comma_numberDouble();
        read_objectStart_string_arrow_typeList_comma_numberDouble();
        read_objectStart_string_arrow_typeProperty_comma_numberDouble();
        read_objectStart_string_arrow_typeString_comma_numberDouble();
        read_objectStart_string_arrow_typeInt_comma_numberDouble();
        read_objectStart_string_arrow_typeLong_comma_numberDouble();
        read_objectStart_string_arrow_typeDouble_comma_numberDouble();
        read_objectStart_string_arrow_typeBigInteger_comma_numberDouble();
        read_objectStart_string_arrow_typeBigDecimal_comma_numberDouble();
        read_objectStart_string_arrow_typeBytes_comma_numberDouble();
        read_objectStart_string_arrow_typeExpression_comma_numberDouble();
        read_objectStart_string_arrow_typeType_comma_numberDouble();
        read_objectStart_string_arrow_typeBoolean_comma_numberDouble();
        read_objectStart_string_arrow_typeUndefined_comma_numberDouble();
        read_objectStart_string_arrow_false_comma_numberDouble();
        read_objectStart_string_arrow_true_comma_numberDouble();
        read_objectStart_string_arrow_undefined_comma_numberDouble();

        read_objectStart_string_arrow_listStart_listEnd_comma_numberBigInteger();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_numberBigInteger();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_numberBigInteger();
        read_objectStart_string_arrow_string_comma_numberBigInteger();
        read_objectStart_string_arrow_numberInt_comma_numberBigInteger();
        read_objectStart_string_arrow_numberLong_comma_numberBigInteger();
        read_objectStart_string_arrow_numberDouble_comma_numberBigInteger();
        read_objectStart_string_arrow_numberBigInteger_comma_numberBigInteger();
        read_objectStart_string_arrow_numberBigDecimal_comma_numberBigInteger();
        read_objectStart_string_arrow_bytes_comma_numberBigInteger();
        read_objectStart_string_arrow_expression_comma_numberBigInteger();
        read_objectStart_string_arrow_typeObject_comma_numberBigInteger();
        read_objectStart_string_arrow_typeList_comma_numberBigInteger();
        read_objectStart_string_arrow_typeProperty_comma_numberBigInteger();
        read_objectStart_string_arrow_typeString_comma_numberBigInteger();
        read_objectStart_string_arrow_typeInt_comma_numberBigInteger();
        read_objectStart_string_arrow_typeLong_comma_numberBigInteger();
        read_objectStart_string_arrow_typeDouble_comma_numberBigInteger();
        read_objectStart_string_arrow_typeBigInteger_comma_numberBigInteger();
        read_objectStart_string_arrow_typeBigDecimal_comma_numberBigInteger();
        read_objectStart_string_arrow_typeBytes_comma_numberBigInteger();
        read_objectStart_string_arrow_typeExpression_comma_numberBigInteger();
        read_objectStart_string_arrow_typeType_comma_numberBigInteger();
        read_objectStart_string_arrow_typeBoolean_comma_numberBigInteger();
        read_objectStart_string_arrow_typeUndefined_comma_numberBigInteger();
        read_objectStart_string_arrow_false_comma_numberBigInteger();
        read_objectStart_string_arrow_true_comma_numberBigInteger();
        read_objectStart_string_arrow_undefined_comma_numberBigInteger();

        read_objectStart_string_arrow_listStart_listEnd_comma_numberBigDecimal();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_numberBigDecimal();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_numberBigDecimal();
        read_objectStart_string_arrow_string_comma_numberBigDecimal();
        read_objectStart_string_arrow_numberInt_comma_numberBigDecimal();
        read_objectStart_string_arrow_numberLong_comma_numberBigDecimal();
        read_objectStart_string_arrow_numberDouble_comma_numberBigDecimal();
        read_objectStart_string_arrow_numberBigInteger_comma_numberBigDecimal();
        read_objectStart_string_arrow_numberBigDecimal_comma_numberBigDecimal();
        read_objectStart_string_arrow_bytes_comma_numberBigDecimal();
        read_objectStart_string_arrow_expression_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeObject_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeList_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeProperty_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeString_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeInt_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeLong_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeDouble_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeBigInteger_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeBigDecimal_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeBytes_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeExpression_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeType_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeBoolean_comma_numberBigDecimal();
        read_objectStart_string_arrow_typeUndefined_comma_numberBigDecimal();
        read_objectStart_string_arrow_false_comma_numberBigDecimal();
        read_objectStart_string_arrow_true_comma_numberBigDecimal();
        read_objectStart_string_arrow_undefined_comma_numberBigDecimal();

        read_objectStart_string_arrow_listStart_listEnd_comma_bytes();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_bytes();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_bytes();
        read_objectStart_string_arrow_string_comma_bytes();
        read_objectStart_string_arrow_numberInt_comma_bytes();
        read_objectStart_string_arrow_numberLong_comma_bytes();
        read_objectStart_string_arrow_numberDouble_comma_bytes();
        read_objectStart_string_arrow_numberBigInteger_comma_bytes();
        read_objectStart_string_arrow_numberBigDecimal_comma_bytes();
        read_objectStart_string_arrow_bytes_comma_bytes();
        read_objectStart_string_arrow_expression_comma_bytes();
        read_objectStart_string_arrow_typeObject_comma_bytes();
        read_objectStart_string_arrow_typeList_comma_bytes();
        read_objectStart_string_arrow_typeProperty_comma_bytes();
        read_objectStart_string_arrow_typeString_comma_bytes();
        read_objectStart_string_arrow_typeInt_comma_bytes();
        read_objectStart_string_arrow_typeLong_comma_bytes();
        read_objectStart_string_arrow_typeDouble_comma_bytes();
        read_objectStart_string_arrow_typeBigInteger_comma_bytes();
        read_objectStart_string_arrow_typeBigDecimal_comma_bytes();
        read_objectStart_string_arrow_typeBytes_comma_bytes();
        read_objectStart_string_arrow_typeExpression_comma_bytes();
        read_objectStart_string_arrow_typeType_comma_bytes();
        read_objectStart_string_arrow_typeBoolean_comma_bytes();
        read_objectStart_string_arrow_typeUndefined_comma_bytes();
        read_objectStart_string_arrow_false_comma_bytes();
        read_objectStart_string_arrow_true_comma_bytes();
        read_objectStart_string_arrow_undefined_comma_bytes();

        read_objectStart_string_arrow_listStart_listEnd_comma_expression();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_expression();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_expression();
        read_objectStart_string_arrow_string_comma_expression();
        read_objectStart_string_arrow_numberInt_comma_expression();
        read_objectStart_string_arrow_numberLong_comma_expression();
        read_objectStart_string_arrow_numberDouble_comma_expression();
        read_objectStart_string_arrow_numberBigInteger_comma_expression();
        read_objectStart_string_arrow_numberBigDecimal_comma_expression();
        read_objectStart_string_arrow_bytes_comma_expression();
        read_objectStart_string_arrow_expression_comma_expression();
        read_objectStart_string_arrow_typeObject_comma_expression();
        read_objectStart_string_arrow_typeList_comma_expression();
        read_objectStart_string_arrow_typeProperty_comma_expression();
        read_objectStart_string_arrow_typeString_comma_expression();
        read_objectStart_string_arrow_typeInt_comma_expression();
        read_objectStart_string_arrow_typeLong_comma_expression();
        read_objectStart_string_arrow_typeDouble_comma_expression();
        read_objectStart_string_arrow_typeBigInteger_comma_expression();
        read_objectStart_string_arrow_typeBigDecimal_comma_expression();
        read_objectStart_string_arrow_typeBytes_comma_expression();
        read_objectStart_string_arrow_typeExpression_comma_expression();
        read_objectStart_string_arrow_typeType_comma_expression();
        read_objectStart_string_arrow_typeBoolean_comma_expression();
        read_objectStart_string_arrow_typeUndefined_comma_expression();
        read_objectStart_string_arrow_false_comma_expression();
        read_objectStart_string_arrow_true_comma_expression();
        read_objectStart_string_arrow_undefined_comma_expression();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeObject();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeObject();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeObject();
        read_objectStart_string_arrow_string_comma_typeObject();
        read_objectStart_string_arrow_numberInt_comma_typeObject();
        read_objectStart_string_arrow_numberLong_comma_typeObject();
        read_objectStart_string_arrow_numberDouble_comma_typeObject();
        read_objectStart_string_arrow_numberBigInteger_comma_typeObject();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeObject();
        read_objectStart_string_arrow_bytes_comma_typeObject();
        read_objectStart_string_arrow_expression_comma_typeObject();
        read_objectStart_string_arrow_typeObject_comma_typeObject();
        read_objectStart_string_arrow_typeList_comma_typeObject();
        read_objectStart_string_arrow_typeProperty_comma_typeObject();
        read_objectStart_string_arrow_typeString_comma_typeObject();
        read_objectStart_string_arrow_typeInt_comma_typeObject();
        read_objectStart_string_arrow_typeLong_comma_typeObject();
        read_objectStart_string_arrow_typeDouble_comma_typeObject();
        read_objectStart_string_arrow_typeBigInteger_comma_typeObject();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeObject();
        read_objectStart_string_arrow_typeBytes_comma_typeObject();
        read_objectStart_string_arrow_typeExpression_comma_typeObject();
        read_objectStart_string_arrow_typeType_comma_typeObject();
        read_objectStart_string_arrow_typeBoolean_comma_typeObject();
        read_objectStart_string_arrow_typeUndefined_comma_typeObject();
        read_objectStart_string_arrow_false_comma_typeObject();
        read_objectStart_string_arrow_true_comma_typeObject();
        read_objectStart_string_arrow_undefined_comma_typeObject();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeList();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeList();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeList();
        read_objectStart_string_arrow_string_comma_typeList();
        read_objectStart_string_arrow_numberInt_comma_typeList();
        read_objectStart_string_arrow_numberLong_comma_typeList();
        read_objectStart_string_arrow_numberDouble_comma_typeList();
        read_objectStart_string_arrow_numberBigInteger_comma_typeList();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeList();
        read_objectStart_string_arrow_bytes_comma_typeList();
        read_objectStart_string_arrow_expression_comma_typeList();
        read_objectStart_string_arrow_typeObject_comma_typeList();
        read_objectStart_string_arrow_typeList_comma_typeList();
        read_objectStart_string_arrow_typeProperty_comma_typeList();
        read_objectStart_string_arrow_typeString_comma_typeList();
        read_objectStart_string_arrow_typeInt_comma_typeList();
        read_objectStart_string_arrow_typeLong_comma_typeList();
        read_objectStart_string_arrow_typeDouble_comma_typeList();
        read_objectStart_string_arrow_typeBigInteger_comma_typeList();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeList();
        read_objectStart_string_arrow_typeBytes_comma_typeList();
        read_objectStart_string_arrow_typeExpression_comma_typeList();
        read_objectStart_string_arrow_typeType_comma_typeList();
        read_objectStart_string_arrow_typeBoolean_comma_typeList();
        read_objectStart_string_arrow_typeUndefined_comma_typeList();
        read_objectStart_string_arrow_false_comma_typeList();
        read_objectStart_string_arrow_true_comma_typeList();
        read_objectStart_string_arrow_undefined_comma_typeList();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeProperty();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeProperty();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeProperty();
        read_objectStart_string_arrow_string_comma_typeProperty();
        read_objectStart_string_arrow_numberInt_comma_typeProperty();
        read_objectStart_string_arrow_numberLong_comma_typeProperty();
        read_objectStart_string_arrow_numberDouble_comma_typeProperty();
        read_objectStart_string_arrow_numberBigInteger_comma_typeProperty();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeProperty();
        read_objectStart_string_arrow_bytes_comma_typeProperty();
        read_objectStart_string_arrow_expression_comma_typeProperty();
        read_objectStart_string_arrow_typeObject_comma_typeProperty();
        read_objectStart_string_arrow_typeList_comma_typeProperty();
        read_objectStart_string_arrow_typeProperty_comma_typeProperty();
        read_objectStart_string_arrow_typeString_comma_typeProperty();
        read_objectStart_string_arrow_typeInt_comma_typeProperty();
        read_objectStart_string_arrow_typeLong_comma_typeProperty();
        read_objectStart_string_arrow_typeDouble_comma_typeProperty();
        read_objectStart_string_arrow_typeBigInteger_comma_typeProperty();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeProperty();
        read_objectStart_string_arrow_typeBytes_comma_typeProperty();
        read_objectStart_string_arrow_typeExpression_comma_typeProperty();
        read_objectStart_string_arrow_typeType_comma_typeProperty();
        read_objectStart_string_arrow_typeBoolean_comma_typeProperty();
        read_objectStart_string_arrow_typeUndefined_comma_typeProperty();
        read_objectStart_string_arrow_false_comma_typeProperty();
        read_objectStart_string_arrow_true_comma_typeProperty();
        read_objectStart_string_arrow_undefined_comma_typeProperty();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeString();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeString();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeString();
        read_objectStart_string_arrow_string_comma_typeString();
        read_objectStart_string_arrow_numberInt_comma_typeString();
        read_objectStart_string_arrow_numberLong_comma_typeString();
        read_objectStart_string_arrow_numberDouble_comma_typeString();
        read_objectStart_string_arrow_numberBigInteger_comma_typeString();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeString();
        read_objectStart_string_arrow_bytes_comma_typeString();
        read_objectStart_string_arrow_expression_comma_typeString();
        read_objectStart_string_arrow_typeObject_comma_typeString();
        read_objectStart_string_arrow_typeList_comma_typeString();
        read_objectStart_string_arrow_typeProperty_comma_typeString();
        read_objectStart_string_arrow_typeString_comma_typeString();
        read_objectStart_string_arrow_typeInt_comma_typeString();
        read_objectStart_string_arrow_typeLong_comma_typeString();
        read_objectStart_string_arrow_typeDouble_comma_typeString();
        read_objectStart_string_arrow_typeBigInteger_comma_typeString();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeString();
        read_objectStart_string_arrow_typeBytes_comma_typeString();
        read_objectStart_string_arrow_typeExpression_comma_typeString();
        read_objectStart_string_arrow_typeType_comma_typeString();
        read_objectStart_string_arrow_typeBoolean_comma_typeString();
        read_objectStart_string_arrow_typeUndefined_comma_typeString();
        read_objectStart_string_arrow_false_comma_typeString();
        read_objectStart_string_arrow_true_comma_typeString();
        read_objectStart_string_arrow_undefined_comma_typeString();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeInt();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeInt();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeInt();
        read_objectStart_string_arrow_string_comma_typeInt();
        read_objectStart_string_arrow_numberInt_comma_typeInt();
        read_objectStart_string_arrow_numberLong_comma_typeInt();
        read_objectStart_string_arrow_numberDouble_comma_typeInt();
        read_objectStart_string_arrow_numberBigInteger_comma_typeInt();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeInt();
        read_objectStart_string_arrow_bytes_comma_typeInt();
        read_objectStart_string_arrow_expression_comma_typeInt();
        read_objectStart_string_arrow_typeObject_comma_typeInt();
        read_objectStart_string_arrow_typeList_comma_typeInt();
        read_objectStart_string_arrow_typeProperty_comma_typeInt();
        read_objectStart_string_arrow_typeString_comma_typeInt();
        read_objectStart_string_arrow_typeInt_comma_typeInt();
        read_objectStart_string_arrow_typeLong_comma_typeInt();
        read_objectStart_string_arrow_typeDouble_comma_typeInt();
        read_objectStart_string_arrow_typeBigInteger_comma_typeInt();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeInt();
        read_objectStart_string_arrow_typeBytes_comma_typeInt();
        read_objectStart_string_arrow_typeExpression_comma_typeInt();
        read_objectStart_string_arrow_typeType_comma_typeInt();
        read_objectStart_string_arrow_typeBoolean_comma_typeInt();
        read_objectStart_string_arrow_typeUndefined_comma_typeInt();
        read_objectStart_string_arrow_false_comma_typeInt();
        read_objectStart_string_arrow_true_comma_typeInt();
        read_objectStart_string_arrow_undefined_comma_typeInt();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeLong();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeLong();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeLong();
        read_objectStart_string_arrow_string_comma_typeLong();
        read_objectStart_string_arrow_numberInt_comma_typeLong();
        read_objectStart_string_arrow_numberLong_comma_typeLong();
        read_objectStart_string_arrow_numberDouble_comma_typeLong();
        read_objectStart_string_arrow_numberBigInteger_comma_typeLong();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeLong();
        read_objectStart_string_arrow_bytes_comma_typeLong();
        read_objectStart_string_arrow_expression_comma_typeLong();
        read_objectStart_string_arrow_typeObject_comma_typeLong();
        read_objectStart_string_arrow_typeList_comma_typeLong();
        read_objectStart_string_arrow_typeProperty_comma_typeLong();
        read_objectStart_string_arrow_typeString_comma_typeLong();
        read_objectStart_string_arrow_typeInt_comma_typeLong();
        read_objectStart_string_arrow_typeLong_comma_typeLong();
        read_objectStart_string_arrow_typeDouble_comma_typeLong();
        read_objectStart_string_arrow_typeBigInteger_comma_typeLong();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeLong();
        read_objectStart_string_arrow_typeBytes_comma_typeLong();
        read_objectStart_string_arrow_typeExpression_comma_typeLong();
        read_objectStart_string_arrow_typeType_comma_typeLong();
        read_objectStart_string_arrow_typeBoolean_comma_typeLong();
        read_objectStart_string_arrow_typeUndefined_comma_typeLong();
        read_objectStart_string_arrow_false_comma_typeLong();
        read_objectStart_string_arrow_true_comma_typeLong();
        read_objectStart_string_arrow_undefined_comma_typeLong();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeDouble();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeDouble();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeDouble();
        read_objectStart_string_arrow_string_comma_typeDouble();
        read_objectStart_string_arrow_numberInt_comma_typeDouble();
        read_objectStart_string_arrow_numberLong_comma_typeDouble();
        read_objectStart_string_arrow_numberDouble_comma_typeDouble();
        read_objectStart_string_arrow_numberBigInteger_comma_typeDouble();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeDouble();
        read_objectStart_string_arrow_bytes_comma_typeDouble();
        read_objectStart_string_arrow_expression_comma_typeDouble();
        read_objectStart_string_arrow_typeObject_comma_typeDouble();
        read_objectStart_string_arrow_typeList_comma_typeDouble();
        read_objectStart_string_arrow_typeProperty_comma_typeDouble();
        read_objectStart_string_arrow_typeString_comma_typeDouble();
        read_objectStart_string_arrow_typeInt_comma_typeDouble();
        read_objectStart_string_arrow_typeLong_comma_typeDouble();
        read_objectStart_string_arrow_typeDouble_comma_typeDouble();
        read_objectStart_string_arrow_typeBigInteger_comma_typeDouble();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeDouble();
        read_objectStart_string_arrow_typeBytes_comma_typeDouble();
        read_objectStart_string_arrow_typeExpression_comma_typeDouble();
        read_objectStart_string_arrow_typeType_comma_typeDouble();
        read_objectStart_string_arrow_typeBoolean_comma_typeDouble();
        read_objectStart_string_arrow_typeUndefined_comma_typeDouble();
        read_objectStart_string_arrow_false_comma_typeDouble();
        read_objectStart_string_arrow_true_comma_typeDouble();
        read_objectStart_string_arrow_undefined_comma_typeDouble();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeBigInteger();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeBigInteger();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeBigInteger();
        read_objectStart_string_arrow_string_comma_typeBigInteger();
        read_objectStart_string_arrow_numberInt_comma_typeBigInteger();
        read_objectStart_string_arrow_numberLong_comma_typeBigInteger();
        read_objectStart_string_arrow_numberDouble_comma_typeBigInteger();
        read_objectStart_string_arrow_numberBigInteger_comma_typeBigInteger();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeBigInteger();
        read_objectStart_string_arrow_bytes_comma_typeBigInteger();
        read_objectStart_string_arrow_expression_comma_typeBigInteger();
        read_objectStart_string_arrow_typeObject_comma_typeBigInteger();
        read_objectStart_string_arrow_typeList_comma_typeBigInteger();
        read_objectStart_string_arrow_typeProperty_comma_typeBigInteger();
        read_objectStart_string_arrow_typeString_comma_typeBigInteger();
        read_objectStart_string_arrow_typeInt_comma_typeBigInteger();
        read_objectStart_string_arrow_typeLong_comma_typeBigInteger();
        read_objectStart_string_arrow_typeDouble_comma_typeBigInteger();
        read_objectStart_string_arrow_typeBigInteger_comma_typeBigInteger();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeBigInteger();
        read_objectStart_string_arrow_typeBytes_comma_typeBigInteger();
        read_objectStart_string_arrow_typeExpression_comma_typeBigInteger();
        read_objectStart_string_arrow_typeType_comma_typeBigInteger();
        read_objectStart_string_arrow_typeBoolean_comma_typeBigInteger();
        read_objectStart_string_arrow_typeUndefined_comma_typeBigInteger();
        read_objectStart_string_arrow_false_comma_typeBigInteger();
        read_objectStart_string_arrow_true_comma_typeBigInteger();
        read_objectStart_string_arrow_undefined_comma_typeBigInteger();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeBigDecimal();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeBigDecimal();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeBigDecimal();
        read_objectStart_string_arrow_string_comma_typeBigDecimal();
        read_objectStart_string_arrow_numberInt_comma_typeBigDecimal();
        read_objectStart_string_arrow_numberLong_comma_typeBigDecimal();
        read_objectStart_string_arrow_numberDouble_comma_typeBigDecimal();
        read_objectStart_string_arrow_numberBigInteger_comma_typeBigDecimal();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeBigDecimal();
        read_objectStart_string_arrow_bytes_comma_typeBigDecimal();
        read_objectStart_string_arrow_expression_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeObject_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeList_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeProperty_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeString_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeInt_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeLong_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeDouble_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeBigInteger_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeBytes_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeExpression_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeType_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeBoolean_comma_typeBigDecimal();
        read_objectStart_string_arrow_typeUndefined_comma_typeBigDecimal();
        read_objectStart_string_arrow_false_comma_typeBigDecimal();
        read_objectStart_string_arrow_true_comma_typeBigDecimal();
        read_objectStart_string_arrow_undefined_comma_typeBigDecimal();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeBytes();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeBytes();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeBytes();
        read_objectStart_string_arrow_string_comma_typeBytes();
        read_objectStart_string_arrow_numberInt_comma_typeBytes();
        read_objectStart_string_arrow_numberLong_comma_typeBytes();
        read_objectStart_string_arrow_numberDouble_comma_typeBytes();
        read_objectStart_string_arrow_numberBigInteger_comma_typeBytes();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeBytes();
        read_objectStart_string_arrow_bytes_comma_typeBytes();
        read_objectStart_string_arrow_expression_comma_typeBytes();
        read_objectStart_string_arrow_typeObject_comma_typeBytes();
        read_objectStart_string_arrow_typeList_comma_typeBytes();
        read_objectStart_string_arrow_typeProperty_comma_typeBytes();
        read_objectStart_string_arrow_typeString_comma_typeBytes();
        read_objectStart_string_arrow_typeInt_comma_typeBytes();
        read_objectStart_string_arrow_typeLong_comma_typeBytes();
        read_objectStart_string_arrow_typeDouble_comma_typeBytes();
        read_objectStart_string_arrow_typeBigInteger_comma_typeBytes();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeBytes();
        read_objectStart_string_arrow_typeBytes_comma_typeBytes();
        read_objectStart_string_arrow_typeExpression_comma_typeBytes();
        read_objectStart_string_arrow_typeType_comma_typeBytes();
        read_objectStart_string_arrow_typeBoolean_comma_typeBytes();
        read_objectStart_string_arrow_typeUndefined_comma_typeBytes();
        read_objectStart_string_arrow_false_comma_typeBytes();
        read_objectStart_string_arrow_true_comma_typeBytes();
        read_objectStart_string_arrow_undefined_comma_typeBytes();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeExpression();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeExpression();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeExpression();
        read_objectStart_string_arrow_string_comma_typeExpression();
        read_objectStart_string_arrow_numberInt_comma_typeExpression();
        read_objectStart_string_arrow_numberLong_comma_typeExpression();
        read_objectStart_string_arrow_numberDouble_comma_typeExpression();
        read_objectStart_string_arrow_numberBigInteger_comma_typeExpression();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeExpression();
        read_objectStart_string_arrow_bytes_comma_typeExpression();
        read_objectStart_string_arrow_expression_comma_typeExpression();
        read_objectStart_string_arrow_typeObject_comma_typeExpression();
        read_objectStart_string_arrow_typeList_comma_typeExpression();
        read_objectStart_string_arrow_typeProperty_comma_typeExpression();
        read_objectStart_string_arrow_typeString_comma_typeExpression();
        read_objectStart_string_arrow_typeInt_comma_typeExpression();
        read_objectStart_string_arrow_typeLong_comma_typeExpression();
        read_objectStart_string_arrow_typeDouble_comma_typeExpression();
        read_objectStart_string_arrow_typeBigInteger_comma_typeExpression();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeExpression();
        read_objectStart_string_arrow_typeBytes_comma_typeExpression();
        read_objectStart_string_arrow_typeExpression_comma_typeExpression();
        read_objectStart_string_arrow_typeType_comma_typeExpression();
        read_objectStart_string_arrow_typeBoolean_comma_typeExpression();
        read_objectStart_string_arrow_typeUndefined_comma_typeExpression();
        read_objectStart_string_arrow_false_comma_typeExpression();
        read_objectStart_string_arrow_true_comma_typeExpression();
        read_objectStart_string_arrow_undefined_comma_typeExpression();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeType();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeType();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeType();
        read_objectStart_string_arrow_string_comma_typeType();
        read_objectStart_string_arrow_numberInt_comma_typeType();
        read_objectStart_string_arrow_numberLong_comma_typeType();
        read_objectStart_string_arrow_numberDouble_comma_typeType();
        read_objectStart_string_arrow_numberBigInteger_comma_typeType();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeType();
        read_objectStart_string_arrow_bytes_comma_typeType();
        read_objectStart_string_arrow_expression_comma_typeType();
        read_objectStart_string_arrow_typeObject_comma_typeType();
        read_objectStart_string_arrow_typeList_comma_typeType();
        read_objectStart_string_arrow_typeProperty_comma_typeType();
        read_objectStart_string_arrow_typeString_comma_typeType();
        read_objectStart_string_arrow_typeInt_comma_typeType();
        read_objectStart_string_arrow_typeLong_comma_typeType();
        read_objectStart_string_arrow_typeDouble_comma_typeType();
        read_objectStart_string_arrow_typeBigInteger_comma_typeType();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeType();
        read_objectStart_string_arrow_typeBytes_comma_typeType();
        read_objectStart_string_arrow_typeExpression_comma_typeType();
        read_objectStart_string_arrow_typeType_comma_typeType();
        read_objectStart_string_arrow_typeBoolean_comma_typeType();
        read_objectStart_string_arrow_typeUndefined_comma_typeType();
        read_objectStart_string_arrow_false_comma_typeType();
        read_objectStart_string_arrow_true_comma_typeType();
        read_objectStart_string_arrow_undefined_comma_typeType();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeBoolean();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeBoolean();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeBoolean();
        read_objectStart_string_arrow_string_comma_typeBoolean();
        read_objectStart_string_arrow_numberInt_comma_typeBoolean();
        read_objectStart_string_arrow_numberLong_comma_typeBoolean();
        read_objectStart_string_arrow_numberDouble_comma_typeBoolean();
        read_objectStart_string_arrow_numberBigInteger_comma_typeBoolean();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeBoolean();
        read_objectStart_string_arrow_bytes_comma_typeBoolean();
        read_objectStart_string_arrow_expression_comma_typeBoolean();
        read_objectStart_string_arrow_typeObject_comma_typeBoolean();
        read_objectStart_string_arrow_typeList_comma_typeBoolean();
        read_objectStart_string_arrow_typeProperty_comma_typeBoolean();
        read_objectStart_string_arrow_typeString_comma_typeBoolean();
        read_objectStart_string_arrow_typeInt_comma_typeBoolean();
        read_objectStart_string_arrow_typeLong_comma_typeBoolean();
        read_objectStart_string_arrow_typeDouble_comma_typeBoolean();
        read_objectStart_string_arrow_typeBigInteger_comma_typeBoolean();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeBoolean();
        read_objectStart_string_arrow_typeBytes_comma_typeBoolean();
        read_objectStart_string_arrow_typeExpression_comma_typeBoolean();
        read_objectStart_string_arrow_typeType_comma_typeBoolean();
        read_objectStart_string_arrow_typeBoolean_comma_typeBoolean();
        read_objectStart_string_arrow_typeUndefined_comma_typeBoolean();
        read_objectStart_string_arrow_false_comma_typeBoolean();
        read_objectStart_string_arrow_true_comma_typeBoolean();
        read_objectStart_string_arrow_undefined_comma_typeBoolean();

        read_objectStart_string_arrow_listStart_listEnd_comma_typeUndefined();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_typeUndefined();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_typeUndefined();
        read_objectStart_string_arrow_string_comma_typeUndefined();
        read_objectStart_string_arrow_numberInt_comma_typeUndefined();
        read_objectStart_string_arrow_numberLong_comma_typeUndefined();
        read_objectStart_string_arrow_numberDouble_comma_typeUndefined();
        read_objectStart_string_arrow_numberBigInteger_comma_typeUndefined();
        read_objectStart_string_arrow_numberBigDecimal_comma_typeUndefined();
        read_objectStart_string_arrow_bytes_comma_typeUndefined();
        read_objectStart_string_arrow_expression_comma_typeUndefined();
        read_objectStart_string_arrow_typeObject_comma_typeUndefined();
        read_objectStart_string_arrow_typeList_comma_typeUndefined();
        read_objectStart_string_arrow_typeProperty_comma_typeUndefined();
        read_objectStart_string_arrow_typeString_comma_typeUndefined();
        read_objectStart_string_arrow_typeInt_comma_typeUndefined();
        read_objectStart_string_arrow_typeLong_comma_typeUndefined();
        read_objectStart_string_arrow_typeDouble_comma_typeUndefined();
        read_objectStart_string_arrow_typeBigInteger_comma_typeUndefined();
        read_objectStart_string_arrow_typeBigDecimal_comma_typeUndefined();
        read_objectStart_string_arrow_typeBytes_comma_typeUndefined();
        read_objectStart_string_arrow_typeExpression_comma_typeUndefined();
        read_objectStart_string_arrow_typeType_comma_typeUndefined();
        read_objectStart_string_arrow_typeBoolean_comma_typeUndefined();
        read_objectStart_string_arrow_typeUndefined_comma_typeUndefined();
        read_objectStart_string_arrow_false_comma_typeUndefined();
        read_objectStart_string_arrow_true_comma_typeUndefined();
        read_objectStart_string_arrow_undefined_comma_typeUndefined();

        read_objectStart_string_arrow_listStart_listEnd_comma_true();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_true();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_true();
        read_objectStart_string_arrow_string_comma_true();
        read_objectStart_string_arrow_numberInt_comma_true();
        read_objectStart_string_arrow_numberLong_comma_true();
        read_objectStart_string_arrow_numberDouble_comma_true();
        read_objectStart_string_arrow_numberBigInteger_comma_true();
        read_objectStart_string_arrow_numberBigDecimal_comma_true();
        read_objectStart_string_arrow_bytes_comma_true();
        read_objectStart_string_arrow_expression_comma_true();
        read_objectStart_string_arrow_typeObject_comma_true();
        read_objectStart_string_arrow_typeList_comma_true();
        read_objectStart_string_arrow_typeProperty_comma_true();
        read_objectStart_string_arrow_typeString_comma_true();
        read_objectStart_string_arrow_typeInt_comma_true();
        read_objectStart_string_arrow_typeLong_comma_true();
        read_objectStart_string_arrow_typeDouble_comma_true();
        read_objectStart_string_arrow_typeBigInteger_comma_true();
        read_objectStart_string_arrow_typeBigDecimal_comma_true();
        read_objectStart_string_arrow_typeBytes_comma_true();
        read_objectStart_string_arrow_typeExpression_comma_true();
        read_objectStart_string_arrow_typeType_comma_true();
        read_objectStart_string_arrow_typeBoolean_comma_true();
        read_objectStart_string_arrow_typeUndefined_comma_true();
        read_objectStart_string_arrow_false_comma_true();
        read_objectStart_string_arrow_true_comma_true();
        read_objectStart_string_arrow_undefined_comma_true();

        read_objectStart_string_arrow_listStart_listEnd_comma_false();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_false();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_false();
        read_objectStart_string_arrow_string_comma_false();
        read_objectStart_string_arrow_numberInt_comma_false();
        read_objectStart_string_arrow_numberLong_comma_false();
        read_objectStart_string_arrow_numberDouble_comma_false();
        read_objectStart_string_arrow_numberBigInteger_comma_false();
        read_objectStart_string_arrow_numberBigDecimal_comma_false();
        read_objectStart_string_arrow_bytes_comma_false();
        read_objectStart_string_arrow_expression_comma_false();
        read_objectStart_string_arrow_typeObject_comma_false();
        read_objectStart_string_arrow_typeList_comma_false();
        read_objectStart_string_arrow_typeProperty_comma_false();
        read_objectStart_string_arrow_typeString_comma_false();
        read_objectStart_string_arrow_typeInt_comma_false();
        read_objectStart_string_arrow_typeLong_comma_false();
        read_objectStart_string_arrow_typeDouble_comma_false();
        read_objectStart_string_arrow_typeBigInteger_comma_false();
        read_objectStart_string_arrow_typeBigDecimal_comma_false();
        read_objectStart_string_arrow_typeBytes_comma_false();
        read_objectStart_string_arrow_typeExpression_comma_false();
        read_objectStart_string_arrow_typeType_comma_false();
        read_objectStart_string_arrow_typeBoolean_comma_false();
        read_objectStart_string_arrow_typeUndefined_comma_false();
        read_objectStart_string_arrow_false_comma_false();
        read_objectStart_string_arrow_true_comma_false();
        read_objectStart_string_arrow_undefined_comma_false();

        read_objectStart_string_arrow_listStart_listEnd_comma_undefined();
        read_objectStart_string_arrow_objectStart_objectEnd_comma_undefined();
        read_objectStart_string_arrow_propertyStart_propertyEnd_comma_undefined();
        read_objectStart_string_arrow_string_comma_undefined();
        read_objectStart_string_arrow_numberInt_comma_undefined();
        read_objectStart_string_arrow_numberLong_comma_undefined();
        read_objectStart_string_arrow_numberDouble_comma_undefined();
        read_objectStart_string_arrow_numberBigInteger_comma_undefined();
        read_objectStart_string_arrow_numberBigDecimal_comma_undefined();
        read_objectStart_string_arrow_bytes_comma_undefined();
        read_objectStart_string_arrow_expression_comma_undefined();
        read_objectStart_string_arrow_typeObject_comma_undefined();
        read_objectStart_string_arrow_typeList_comma_undefined();
        read_objectStart_string_arrow_typeProperty_comma_undefined();
        read_objectStart_string_arrow_typeString_comma_undefined();
        read_objectStart_string_arrow_typeInt_comma_undefined();
        read_objectStart_string_arrow_typeLong_comma_undefined();
        read_objectStart_string_arrow_typeDouble_comma_undefined();
        read_objectStart_string_arrow_typeBigInteger_comma_undefined();
        read_objectStart_string_arrow_typeBigDecimal_comma_undefined();
        read_objectStart_string_arrow_typeBytes_comma_undefined();
        read_objectStart_string_arrow_typeExpression_comma_undefined();
        read_objectStart_string_arrow_typeType_comma_undefined();
        read_objectStart_string_arrow_typeBoolean_comma_undefined();
        read_objectStart_string_arrow_typeUndefined_comma_undefined();
        read_objectStart_string_arrow_false_comma_undefined();
        read_objectStart_string_arrow_true_comma_undefined();
        read_objectStart_string_arrow_undefined_comma_undefined();
    }

    private void read_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "x" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "=>" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "," );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "}" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "]" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( ")" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\nx" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n=>" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n," );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n}" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n]" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_whitespace_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( " \t\r\n)" );
        assertInitialState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInt_tooBigValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "2147483648" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect integer value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInt_tooSmallValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-2147483649" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect integer value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLong_tooBigValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "9223372036854775808L" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect long value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLong_tooSmallValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-9223372036854775809L" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect long value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenDouble_tooBigValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "9.0E+325" );
        assertInitialState( reader );
        assertNumberState( reader, Double.POSITIVE_INFINITY );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_brokenDouble_tooSmallValue() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-9.0E+325" );
        assertInitialState( reader );
        assertNumberState( reader, Double.NEGATIVE_INFINITY );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_list_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[]_" );
        assertInitialState( reader );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_object_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{}_" );
        assertInitialState( reader );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_property_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=>\"\")_" );
        assertInitialState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "\"\"_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalStringState( reader );
        reader.close();
    }

    private void read_int_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_long_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1L_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_double_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "1.0_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big integer 1_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big decimal 1.0_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalNumberState( reader );
        reader.close();
    }

    private void read_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes{}_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBytesState( reader );
        reader.close();
    }

    private void read_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "expression \"\"_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalExpressionState( reader );
        reader.close();
    }

    private void read_listType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LIST_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_objectType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "OBJECT_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_propertyType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "PROPERTY_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_stringType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "STRING_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_intType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "INT_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_longType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LONG_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_doubleType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "DOUBLE_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bigIntegerType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_INTEGER_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bigDecimalType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_DECIMAL_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_bytesType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BYTES_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_expressionType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "EXPRESSION_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "TYPE_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_booleanType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BOOLEAN_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_undefinedType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "UNDEFINED_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalTypeState( reader );
        reader.close();
    }

    private void read_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "false_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBooleanState( reader );
        reader.close();
    }

    private void read_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "true_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalBooleanState( reader );
        reader.close();
    }

    private void read_undefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "undefined_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected content following the DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bI" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'I' while reading DMR big or bytes token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "b" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "biG" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'G' while reading DMR big token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bi" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger5_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'I' while reading DMR decimal or integer token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger5_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big " );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger6_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big iN" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'N' while reading DMR integer token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger6_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big i" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger_brokenValue_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big integer a" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect big integer value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigInteger_brokenValue_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big integer" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect big integer value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bI" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'I' while reading DMR big or bytes token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "b" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "biG" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'G' while reading DMR big token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bi" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal5_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big D" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'D' while reading DMR decimal or integer token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal5_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big " );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal6_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big dE" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'E' while reading DMR decimal token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal6_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big d" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal_brokenValue_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big decimal a" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect big decimal value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimal_brokenValue_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "big decimal" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect big decimal value" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "\"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bY" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'Y' while reading DMR big or bytes token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "b" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "byT" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'T' while reading DMR bytes token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "by" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_missingLeftBracket() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes 0" );
        assertInitialState( reader );
        assertModelException( reader, "Incorrect bytes value. It must start with: '{'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_missingRightBracket() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {z" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_wrongNumberValue1() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {a" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_wrongHexedNumberValue1() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {0xz" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_wrongNumberValue2() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {0, a" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_wrongHexedNumberValue2() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {0, 0xz" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_missingComma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {0 1" );
        assertInitialState( reader );
        assertModelException( reader, "Missing ',' separator inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytes_duplicatedComma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "bytes {0,, 1" );
        assertInitialState( reader );
        assertModelException( reader, "Expected integer or hexed integer value inside bytes section" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "eX" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'X' while reading DMR expression token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "e" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "expression \"" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenObjectType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Ob" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'b' while reading DMR OBJECT token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenObjectType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "O" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenObjectType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "OBj" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'j' while reading DMR OBJECT token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenObjectType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "OB" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenListType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Li" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'i' while reading DMR LIST or LONG token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenListType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "L" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenListType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LIs" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 's' while reading DMR LIST token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenListType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LI" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPropertyType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Pr" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'r' while reading DMR PROPERTY token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPropertyType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "P" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPropertyType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "PRo" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'o' while reading DMR PROPERTY token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPropertyType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "PR" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenStringType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "St" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 't' while reading DMR STRING token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenStringType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "S" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenStringType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "STr" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'r' while reading DMR STRING token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenStringType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "ST" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenIntType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'z' while reading DMR INT or Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenIntType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenIntType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "INt" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 't' while reading DMR INT token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenIntType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "IN" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLongType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Lo" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'o' while reading DMR LIST or LONG token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLongType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "L" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLongType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LOn" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR LONG token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenLongType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "LO" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenDoubleType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Do" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'o' while reading DMR DOUBLE token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenDoubleType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "D" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenDoubleType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "DOu" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'u' while reading DMR DOUBLE token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenDoubleType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "DO" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Bi" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'i' while reading DMR BIG_DECIMAL or BIG_INTEGER or BOOLEAN or BYTES token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "B" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIg" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BI" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType5_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_i" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected fifth character 'i' while reading DMR BIG_DECIMAL or BIG_INTEGER token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType5_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType6_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR BIG_INTEGER token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigIntegerType6_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Bi" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'i' while reading DMR BIG_DECIMAL or BIG_INTEGER or BOOLEAN or BYTES token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "B" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIg" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BI" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType5_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_d" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected fifth character 'd' while reading DMR BIG_DECIMAL or BIG_INTEGER token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType5_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType6_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_De" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'e' while reading DMR BIG_DECIMAL token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBigDecimalType6_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BIG_D" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytesType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "By" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'y' while reading DMR BIG_DECIMAL or BIG_INTEGER or BOOLEAN or BYTES token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytesType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "B" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytesType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BYt" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 't' while reading DMR BYTES token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBytesType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BY" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBooleanType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Bo" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'o' while reading DMR BIG_DECIMAL or BIG_INTEGER or BOOLEAN or BYTES token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBooleanType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "B" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBooleanType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BOo" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'o' while reading DMR BOOLEAN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenBooleanType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "BO" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Ex" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR EXPRESSION token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "E" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "EXp" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'p' while reading DMR EXPRESSION token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenExpressionType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "EX" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTypeType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Ty" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'y' while reading DMR TYPE token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTypeType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "T" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTypeType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "TYp" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'p' while reading DMR TYPE token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTypeType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "TY" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefinedType2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Un" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR UNDEFINED token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefinedType2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "U" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefinedType3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "UNd" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'd' while reading DMR UNDEFINED token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefinedType3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "UN" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected second character 'z' while reading DMR INT or Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'z' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Iz" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'z' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+I" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+InF" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'F' while reading DMR Infinity token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveInfinity3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+In" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenNegativeNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "-Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN1_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+A" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected first character 'A' while reading DMR Infinity or NaN or number token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN1_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN2_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+NA" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'A' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN2_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+N" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN3_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Nan" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'n' while reading DMR NaN token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenPositiveNaN3_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "+Na" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTrue_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "trua" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'a' while reading DMR true token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenTrue_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "tru" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenFalse_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "falsa" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'a' while reading DMR false token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenFalse_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "fals" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "undefineD" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character 'D' while reading DMR undefined token" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "undef" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenArrow_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "=<" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected character '<' while reading DMR arrow" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_brokenArrow_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "=" );
        assertInitialState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{x" );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{=>" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{," );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{]" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{[" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{{" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{)" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{(" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{1L" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{1.0" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{big integer 1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{big decimal 1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{bytes {}" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{expression \"env.JAVA_HOME\"" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{OBJECT" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{LIST" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{PROPERTY" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{STRING" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{INT" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{LONG" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{DOUBLE" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{BIG_INTEGER" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{BIG_DECIMAL" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{BOOLEAN" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{BYTES" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{EXPRESSION" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{TYPE" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{UNDEFINED" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{false" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{true" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{undefined" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nx" );
        assertObjectStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n=>" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n," );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n]" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n[" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n{" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n)" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n(" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n1L" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\n1.0" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nbig integer 1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nbig decimal 1" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nbytes {}" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nexpression \"env.JAVA_HOME\"" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nOBJECT" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nLIST" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nPROPERTY" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nSTRING" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nINT" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nLONG" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nDOUBLE" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nBIG_INTEGER" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nBIG_DECIMAL" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nBOOLEAN" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nBYTES" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nEXPRESSION" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nTYPE" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nUNDEFINED" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nfalse" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\ntrue" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_whitespace_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{ \t\r\nundefined" );
        assertObjectStartState( reader );
        assertModelException( reader, "Expecting OBJECT_END or STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[x" );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }
    private void read_listStart_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[=>" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[," );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[}" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[)" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nx" );
        assertListStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n=>" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n," );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n}" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_whitespace_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n)" );
        assertListStartState( reader );
        assertModelException( reader, "Expecting LIST_END or OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(x" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(=>" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(," );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(}" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "({" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(]" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "([" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "()" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "((" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(1L" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(1.0" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(big integer 1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(big decimal 1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(bytes {}" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(expression \"env.JAVA_HOME\"" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(OBJECT" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(LIST" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(PROPERTY" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(STRING" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(INT" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(LONG" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(DOUBLE" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(BIG_INTEGER" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(BIG_DECIMAL" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(BOOLEAN" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(BYTES" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(EXPRESSION" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(TYPE" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(UNDEFINED" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(false" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(true" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(undefined" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nx" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n=>" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n," );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n}" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n{" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n]" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n[" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n)" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n(" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n1L" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n1.0" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nbig integer 1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nbig decimal 1" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nbytes {}" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nexpression \"env.JAVA_HOME\"" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nOBJECT" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nLIST" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nPROPERTY" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nSTRING" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nINT" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nLONG" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nDOUBLE" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nBIG_INTEGER" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nBIG_DECIMAL" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nBOOLEAN" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nBYTES" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nEXPRESSION" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nTYPE" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nUNDEFINED" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nfalse" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\ntrue" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_whitespace_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\nundefined" );
        assertPropertyStartState( reader );
        assertModelException( reader, "Expecting STRING" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[])" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{})" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\")" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[][" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}[" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"[" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n[" );
        assertListStartState( reader );
        assertNumberState( reader, +13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-13 \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, -13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n{" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n(" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n1" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n1L" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n1.0" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\nbig integer 1" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n big decimal 1" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n bytes{}" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n expression \"\"" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n OBJECT" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n LIST" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n PROPERTY" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n STRING" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n INT" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n LONG" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n DOUBLE" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n BIG_INTEGER" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n BIG_DECIMAL" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n BOOLEAN" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n BYTES" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n EXPRESSION" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n TYPE" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n UNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n false" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n true" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n13 \t\r\n undefined" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+17L \t\r\n[" );
        assertListStartState( reader );
        assertNumberState( reader, +17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n17L \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, 17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n{" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n(" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n1" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n1L" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\n1.0" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nbig integer 1" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nbig decimal 1" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nbytes{}" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nexpression \"\"" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nOBJECT" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nLIST" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nPROPERTY" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nSTRING" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nINT" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nLONG" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nDOUBLE" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nBIG_INTEGER" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nBIG_DECIMAL" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nBOOLEAN" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nBYTES" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nEXPRESSION" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nTYPE" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nUNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nUNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\ntrue" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-17L \t\r\nundefined" );
        assertListStartState( reader );
        assertNumberState( reader, -17L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+19.0 \t\r\n[" );
        assertListStartState( reader );
        assertNumberState( reader, +19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n19.0 \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, 19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n{" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n(" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n1" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n1L" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\n1.0" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nbig integer 1" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nbig decimal 1" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nbytes{}" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nexpression \"\"" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nOBJECT" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nLIST" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nPROPERTY" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nSTRING" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nINT" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nLONG" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nDOUBLE" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nBIG_INTEGER" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nBIG_DECIMAL" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nBOOLEAN" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nBYTES" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nEXPRESSION" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nTYPE" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nUNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nUNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\ntrue" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-19.0 \t\r\nundefined" );
        assertListStartState( reader );
        assertNumberState( reader, -19.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n+23[" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n23}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23{" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n23)" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23(" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 1L" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 1.0" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 big integer 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 big decimal 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 bytes{}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 expression \"\"" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 OBJECT" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 LIST" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 PROPERTY" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 STRING" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 INT" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 LONG" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 DOUBLE" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 BIG_INTEGER" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 BIG_DECIMAL" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 BOOLEAN" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 BYTES" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 EXPRESSION" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 TYPE" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 UNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 false" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 true" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ninteger \t\r\n-23 undefined" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-23" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n+29[" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n[" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     [" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     [" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     [" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     [" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     [" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     }" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     }" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     }" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     }" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     }" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     {" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     {" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     {" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     {" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     {" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     )" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     )" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     )" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     )" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     )" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     (" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     (" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     (" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     (" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     (" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     \"\"" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     1" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     \"\"" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     1" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     1" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     1" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     1L" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     1L" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     1L" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     1L" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     1L" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     1.0" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     1.0" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     1.0" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     1.0" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     1.0" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     big integer 1" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     big decimal 1" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     bytes{}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     expression \"\"" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     OBJECT" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     LIST" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     LIST" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     LIST" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     LIST" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     LIST" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     PROPERTY" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     STRING" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     STRING" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     STRING" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     STRING" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     STRING" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     INT" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     INT" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     INT" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     INT" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     INT" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     LONG" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     LONG" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     LONG" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     LONG" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     LONG" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     DOUBLE" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     BIG_INTEGER" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     BIG_DECIMAL" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     BOOLEAN" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     BYTES" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     BYTES" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     BYTES" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     BYTES" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     BYTES" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     EXPRESSION" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     TYPE" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     TYPE" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     TYPE" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     TYPE" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     TYPE" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     UNDEFINED" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     false" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     false" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     false" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     false" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     false" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     true" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[OBJECTtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[LISTtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[PROPERTYtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[STRINGtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[INTtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[LONGtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[DOUBLEtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BIG_INTEGERtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BIG_DECIMALtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BYTEStrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[EXPRESSIONtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[TYPEtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[UNDEFINEDtrue" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[truetrue" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[falsetrue" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[undefinedtrue" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     undefined" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[OBJECTundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[LISTundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[PROPERTYundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[STRINGundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[INTundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[LONGundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[DOUBLEundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BIG_INTEGERundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BIG_DECIMALundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[BYTESundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[EXPRESSIONundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[TYPEundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[UNDEFINEDundefined" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[trueundefined" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[falseundefined" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[undefinedundefined" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n{" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n)" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n(" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n\"\"" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n1" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n1L" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n1.0" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nbig integer 1" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nbig decimal 1" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nbytes{}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nexpression \"\"" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nOBJECT" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nLIST" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nPROPERTY" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nSTRING" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nINT" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nLONG" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nDOUBLE" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nBIG_INTEGER" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nBIG_DECIMAL" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nBOOLEAN" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nBYTES" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nEXPRESSION" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nTYPE" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nUNDEFINED" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nfalse" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\ntrue" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes \t\r\n{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\nundefined" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n29}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29{" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n29)" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29(" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29\"\"" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 1L" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 1.0" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 big integer 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 big decimal 1" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 bytes{}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 expression \"\"" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 OBJECT" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 LIST" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 PROPERTY" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 STRING" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 INT" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 LONG" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 DOUBLE" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 BIG_INTEGER" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 BIG_DECIMAL" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 BOOLEAN" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 BYTES" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 EXPRESSION" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 TYPE" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 UNDEFINED" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 false" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 true" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig  \t\ndecimal \t\r\n-29 undefined" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( "-29" ) );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]{" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[](" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}{" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}(" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"{" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"(" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]\"\"" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]1" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]1L" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]1.0" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]big decimal 1.0" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]big integer 10" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]bytes{}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]expression \"\"" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]OBJECT" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]LIST" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]PROPERTY" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]STRING" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]INT" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]LONG" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]DOUBLE" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]BIG_INTEGER" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]BIG_DECIMAL" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]BOOLEAN" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]BYTES" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]EXPRESSION" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]TYPE" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]UNDEFINED" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}\"\"" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}1" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}1L" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}1.0" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}big decimal 1.0" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}big integer 10" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}bytes{}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}expression \"\"" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}OBJECT" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}LIST" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}PROPERTY" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}STRING" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}INT" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}LONG" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}DOUBLE" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}BIG_INTEGER" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}BIG_DECIMAL" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}BOOLEAN" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}BYTES" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}EXPRESSION" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}TYPE" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}UNDEFINED" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"\"\"" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"1" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"1L" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"1.0" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"big integer 1" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"big decimal 1" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"bytes{}" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"expression \"\"" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"OBJECT" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"LIST" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"PROPERTY" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"STRING" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"INT" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"LONG" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"DOUBLE" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"BIG_INTEGER" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"BIG_DECIMAL" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"BOOLEAN" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"BYTES" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"EXPRESSION" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"TYPE" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"UNDEFINED" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]true" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}true" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"true" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]false" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}false" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"false" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]undefined" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}undefined" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"undefined" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]x" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]=>" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}x" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}=>" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyUnexpectedChar\"=>0L)x" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyUnexpectedChar" );
        assertNumberState( reader, 0L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyUnexpectedEOF\"=>big integer 1)" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyUnexpectedEOF" );
        assertNumberState( reader, BigInteger.ONE );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyArrow\"=>big decimal 10)=>" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyArrow" );
        assertNumberState( reader, BigDecimal.TEN );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyListStart\"=>big decimal 0)[" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyListStart" );
        assertNumberState( reader, BigDecimal.ZERO );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyObjectEnd\"=>1.0e+1)}" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyObjectEnd" );
        assertNumberState( reader, 10.0 );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyObjectStart\"=>10L){" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyObjectStart" );
        assertNumberState( reader, 10L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyObjectEnd\"=>1.0e+1))" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyObjectEnd" );
        assertNumberState( reader, 10.0 );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyObjectStart\"=>10L)(" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyObjectStart" );
        assertNumberState( reader, 10L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyString\"=>-10L)\"\"" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyString" );
        assertNumberState( reader, -10L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyNumberInt\"=>+127)0" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyNumberInt" );
        assertNumberState( reader, 127 );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyNumberLong\"=>+12L)0L" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyNumberLong" );
        assertNumberState( reader, +12L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyNumberDouble\"=>+11.0)1.0" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyNumberDouble" );
        assertNumberState( reader, +11.0 );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyNumberBigDecimal\"=>big decimal -10.0)big decimal 1" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyNumberBigDecimal" );
        assertNumberState( reader, new BigDecimal( "-10.0" ) );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyNumberBigInteger\"=>+10L)big integer 1" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyNumberBigInteger" );
        assertNumberState( reader, +10L );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyBytes\"=>expression \"expressionValue\")bytes{}" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyBytes" );
        assertExpressionState( reader, "expressionValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyExpression\"=>bytes{})expression \"someExpression\"" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyExpression" );
        assertBytesState( reader, new byte[] {} );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeObject\"=>OBJECT)OBJECT" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeObject" );
        assertTypeState( reader, ModelType.OBJECT );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeList\"=>LIST)LIST" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeList" );
        assertTypeState( reader, ModelType.LIST );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeProperty\"=>PROPERTY)PROPERTY" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeProperty" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeString\"=>STRING)STRING" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeString" );
        assertTypeState( reader, ModelType.STRING );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeInt\"=>INT)INT" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeInt" );
        assertTypeState( reader, ModelType.INT );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeLong\"=>LONG)LONG" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeLong" );
        assertTypeState( reader, ModelType.LONG );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeDouble\"=>DOUBLE)DOUBLE" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeDouble" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeBigInteger\"=>BIG_INTEGER)BIG_INTEGER" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeBigInteger" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeBigDecimal\"=>BIG_DECIMAL)BIG_DECIMAL" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeBigDecimal" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeBoolean\"=>BOOLEAN)BOOLEAN" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeBoolean" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeType\"=>TYPE)TYPE" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeType" );
        assertTypeState( reader, ModelType.TYPE );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeUndefined\"=>UNDEFINED)UNDEFINED" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeUndefined" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyFalse\"=>false)false" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyFalse" );
        assertBooleanState( reader, false );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTrue\"=>true)true" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTrue" );
        assertBooleanState( reader, true );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyUndefined\"=>undefined)undefined" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyUndefined" );
        assertUndefinedState( reader );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeBytes\"=>BYTES)BYTES" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeBytes" );
        assertTypeState( reader, ModelType.BYTES );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[(\"keyTypeExpression\"=>EXPRESSION)EXPRESSION" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "keyTypeExpression" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"x" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1x" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n1L \r\t\n    x" );
        assertListStartState( reader );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n1.0 \r\t\nx" );
        assertListStartState( reader );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[big \r\t\ninteger \r\t\n    1x" );
        assertListStartState( reader );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[big \r\t\ndecimal \r\t\n    10     x" );
        assertListStartState( reader );
        assertNumberState( reader, BigDecimal.TEN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n     x" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     x" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     x" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     x" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     x" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     x" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n1 \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\n1L \r\t\n" );
        assertListStartState( reader );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n1L \t\r\n     " );
        assertListStartState( reader );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig \t\r\ninteger \t\r\n    1" );
        assertListStartState( reader );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n   big \t\t\r\r\n\n decimal \t\r\n   10   \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, BigDecimal.TEN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     " );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     " );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     " );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     " );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     " );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"\"=>" );
        assertListStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1=>" );
        assertListStartState( reader );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1L=>" );
        assertListStartState( reader );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[1.0=>" );
        assertListStartState( reader );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[big integer 1=>" );
        assertListStartState( reader );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[big decimal 1=>" );
        assertListStartState( reader );
        assertNumberState( reader, BigDecimal.ONE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nbytes{ \t\r\n127 , \t\r\n0x0a \t\n} \r\t\n=>" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 127, 10 } );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nexpression \t\r\n\"env.JAVA_HOME\"\t\r\n     =>" );
        assertListStartState( reader );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nOBJECT\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLIST\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nPROPERTY\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nSTRING\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nINT\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nLONG\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nDOUBLE\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_INTEGER\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBIG_DECIMAL\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nBYTES\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nEXPRESSION\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nTYPE\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nUNDEFINED\t\r\n     =>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\ntrue\t\r\n     =>" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nfalse\t\r\n     =>" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \r\t\nundefined\t\r\n     =>" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or LIST_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],}" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},}" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",}" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\n}" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],)" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},)" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",)" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\n)" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],=>" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},=>" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",=>" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\n=>" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],," );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},," );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",," );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\n," );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[],x" );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{},x" );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\",x" );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\nx" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_listStart_listEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[[]," );
        assertListStartState( reader );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_objectStart_objectEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[{}," );
        assertListStartState( reader );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_propertyStart_propertyEnd_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[( \t\r\n\"propKey\" \t\r\n=> \t\r\n\"propValue\" \t\r\n) \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertPropertyStartState( reader );
        assertStringState( reader, "propKey" );
        assertStringState( reader, "propValue" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_string_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[\"some\\\\\\\"String\\\"\"," );
        assertListStartState( reader );
        assertStringState( reader, "some\\\"String\"" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberInt_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, 13 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberLong_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n-133L \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, -133L );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberDouble_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\n+13.e+1 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, +130. );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig integer -17 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, new BigInteger( "-17" ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_numberBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbig decimal +11 \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertNumberState( reader, new BigDecimal( 11 ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_bytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nbytes \t\r\n{ \t\r\n 0x10 \t\r\n, \t\r\n 0x0A \t\r\n,} \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBytesState( reader, new byte[] { 16, 10 } );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_expression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nexpression \"expression\\\\Value\" \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertExpressionState( reader, "expression\\Value" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeObject_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nOBJECT \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeList_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLIST \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeProperty_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nPROPERTY \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeString_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nSTRING \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeInt_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nINT \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeLong_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nLONG \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeDouble_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nDOUBLE \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigInteger_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_INTEGER \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBigDecimal_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBIG_DECIMAL \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBytes_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBYTES \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeExpression_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nEXPRESSION \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeType_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nTYPE \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeBoolean_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nBOOLEAN \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_typeUndefined_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nUNDEFINED \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_true_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\ntrue \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_false_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nfalse \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_listStart_undefined_comma_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "[ \t\r\nundefined \t\r\n, \t\r\n" );
        assertListStartState( reader );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\")" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"(" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\n1L" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nbig integer 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"big decimal 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nbytes{}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nexpression \t\r\n\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nOBJECT" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nLIST" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nPROPERTY" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nSTRING" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nINT" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nLONG" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nDOUBLE" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nBIG_INTEGER" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nBIG_DECIMAL" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nBOOLEAN" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nBYTES" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nEXPRESSION" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nTYPE" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\" \t\r\nUNDEFINED" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"true" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"false" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"undefined" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"x" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"," );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"\"\"" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"1" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_numberLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\n1L" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_numberDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"1.0" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_numberBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nbig integer 1" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_numberBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"big decimal 1" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_bytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nbytes{}" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_expression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nexpression \t\r\n\"\"" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeObject() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nOBJECT" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeList() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nLIST" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeProperty() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nPROPERTY" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeString() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nSTRING" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nINT" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeLong() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nLONG" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeDouble() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nDOUBLE" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeBigInteger() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nBIG_INTEGER" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeBigDecimal() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nBIG_DECIMAL" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeBoolean() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nBOOLEAN" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeBytes() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nBYTES" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeExpression() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nEXPRESSION" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeType() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nTYPE" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_typeUndefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\" \t\r\nUNDEFINED" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_true() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"true" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_false() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"false" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_undefined() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"undefined" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"]" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"[" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"}" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"{" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\")" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"(" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting '=>'" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_brokenArrow_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"= >" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character ' ' while reading DMR arrow" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_brokenArrow_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_brokenArrow_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"= >" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character ' ' while reading DMR arrow" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_brokenArrow_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>)" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>," );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_objectEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=>}" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=>]" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=>)" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "(\"\"=>=>" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_comma() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n\"\" \t\r\n=> \t\r\n," );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Expecting OBJECT_START or LIST_START or PROPERTY_START or STRING or EXPRESSION or BYTES or NUMBER or BOOLEAN or TYPE or UNDEFINED" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n\"\" \t\r\n=> \t\r\nx" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_propertyStart_string_arrow_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "( \t\r\n\"\" \t\r\n=> \t\r\n" );
        assertPropertyStartState( reader );
        assertStringState( reader, "" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")x" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1Lx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"x" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LISTx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECTx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTYx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRINGx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INTx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONGx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLEx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGERx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMALx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTESx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSIONx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPEx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEANx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINEDx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>truex" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>falsex" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_unexpectedChar() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefinedx" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected character 'x' while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_unexpectedEOF() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Unexpected EOF while reading DMR stream" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_arrow() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined=>" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\"){" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_objectStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined{" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[][" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")[" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_listStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined[" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")]" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_listEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined]" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[](" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}(" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")(" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_propertyStart() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined(" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[])" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{})" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\"))" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\")" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{})" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\")" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_propertyEnd() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined)" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[]\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\")\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\"\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{}\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\"\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeUndefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>UNDEFINED\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.UNDEFINED );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_true_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>true\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, true );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_false_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>false\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBooleanState( reader, false );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_undefined_string() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>undefined\"\"" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertUndefinedState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_listStart_listEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>[] 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertListStartState( reader );
        assertListEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_objectStart_objectEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>{} 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertObjectStartState( reader );
        assertObjectEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_propertyStart_propertyEnd_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"\"=>(\"key\"=>\"value\") 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "" );
        assertPropertyStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertPropertyEndState( reader );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_string_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>\"value\" 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertStringState( reader, "value" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberInt_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberLong_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1L 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1L );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberDouble_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>1.0 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, 1.0 );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigInteger_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big integer1 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, BigInteger.ONE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_numberBigDecimal_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>big decimal1 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertNumberState( reader, new BigDecimal( 1 ) );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_bytes_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>bytes{} 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertBytesState( reader, new byte[] {} );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_expression_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>expression\"env.JAVA_HOME\" 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertExpressionState( reader, "env.JAVA_HOME" );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeList_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LIST 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LIST );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeObject_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>OBJECT 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.OBJECT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeProperty_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>PROPERTY 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.PROPERTY );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeString_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>STRING 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.STRING );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeInt_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>INT 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.INT );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeLong_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>LONG 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.LONG );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeDouble_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>DOUBLE 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.DOUBLE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigInteger_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_INTEGER 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_INTEGER );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBigDecimal_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BIG_DECIMAL 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BIG_DECIMAL );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBytes_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BYTES 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BYTES );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeExpression_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>EXPRESSION 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.EXPRESSION );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeType_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>TYPE 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.TYPE );
        assertModelException( reader, "Expecting ',' or OBJECT_END" );
        assertFinalState( reader );
        reader.close();
    }

    private void read_objectStart_string_arrow_typeBoolean_numberInt() throws IOException, ModelException {
        final ModelReader reader = getModelReader( "{\"key\"=>BOOLEAN 1" );
        assertObjectStartState( reader );
        assertStringState( reader, "key" );
        assertTypeState( reader, ModelType.BOOLEAN );
        assertModelException( reader, "Expecting 