/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.reflect.plugins.bytecode.accessor.generated;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.security.ProtectionDomain;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import javassist.bytecode.DuplicateMemberException;

import org.jboss.reflect.plugins.bytecode.BytecodeConstructorInfo;
import org.jboss.reflect.plugins.bytecode.BytecodeFieldInfo;
import org.jboss.reflect.plugins.bytecode.BytecodeMethodInfo;
import org.jboss.reflect.plugins.bytecode.BytecodeTypeInfo;
import org.jboss.reflect.plugins.bytecode.SignatureKey;
import org.jboss.reflect.plugins.bytecode.accessor.ConstructorAccessor;
import org.jboss.reflect.plugins.bytecode.accessor.FieldAccessor;
import org.jboss.reflect.plugins.bytecode.accessor.MethodAccessor;
import org.jboss.reflect.plugins.bytecode.bytes.BytecodePrimitive;

/**
 * <p>Abstract base class to create implementations of the {@link MethodAccessor}, {@link ConstructorAccessor} 
 * and {@link FieldAccessor} interfaces.</p>
 * <p>This implementation generates raw bytecode to avoid the overhead of compilation via javassist. If 
 * <code>sun.reflect.MagicAccessorImpl</code> is used as the <code>superClass</code> field the implementation
 * classes can access private and protected members of the target class.</p>
 * 
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class GeneratedMemberAccessorFactory
{
   /** Object class name */
   protected static final String OBJECT_NAME = Object.class.getName();

   /** Array of exceptions containing jvm name of Throwable */
   protected final static String[] THROWABLE_EXCEPTIONS = new String[] {"java/lang/Throwable"};
   
   /** The method class counter */
   protected static final AtomicInteger counter = new AtomicInteger(0);

   /** The super class to use for the implementation */
   private final Class<?> superClass;
   
   /** If true the class bytes are output to the file system so they can be inspected with a decompiler/javap */
   private final boolean debug;
   
   /** Cached string representation of the signature of the accessed member */
   private String accessedMember;
   
   /**
    * In AS the GeneratedMethod, -Constructor and -Field classloaders are not deployed in 
    * the system classpath. When generating an accessor for something from the system classpath
    * we need to make sure it happens at the level that can see the implemented interfaces
    * 
    */
   private static final ParentLoaderHandler PARENT_LOADER_HANDLER;
   
   static
   {
      PARENT_LOADER_HANDLER = AccessController.doPrivileged(new PrivilegedAction<ParentLoaderHandler>()
      {
         public ParentLoaderHandler run()
         {
            ClassLoader reflectLoader = MethodAccessor.class.getClassLoader();
            if (reflectLoader == null)
               reflectLoader = ClassLoader.getSystemClassLoader();
            ClassLoader loader = reflectLoader.getParent();
            Set<ClassLoader> parents = loader == null ? null : new HashSet<ClassLoader>(); 
            while (loader != null)
            {
               parents.add(loader);
               loader = loader.getParent();
            }
            return new ParentLoaderHandler(reflectLoader, parents);
         }
      });
   }

   /**
    * Constructor
    * 
    * @param superClass the super class to use for the implementation
    * @param check whether the methods implemented from the interfaces should check the parameters
    * @param debug true to cause the class bytes to be output to the file system so they can be inspected with a decompiler/javap
    */      
   GeneratedMemberAccessorFactory(Class<?> superClass, boolean debug)
   {
      this.superClass = superClass;
      this.debug = debug;
   }
   
   /**
    * Creates a new GeneratedMethod implementation for a given method
    * 
    * @param superClass the super class of the GeneratedMethod implementation
    * @param method the BytecodeMethodInfo for which we want to create a GeneratedMethod implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static MethodAccessor generateMethodAccessor(Class<?> superClass, BytecodeMethodInfo method, boolean debug)
   {
      GeneratedMemberAccessorFactory factory = new GeneratedMethodAccessorFactory(superClass, method, debug);
      Class<MethodAccessor> member = factory.makeClass(MethodAccessor.class, method.getDeclaringClass());
      return ErrorCheckingMemberFactory.wrapInErrorChecker(factory.instantiate(member), method);
   }

   /**
    * Creates a new GeneratedConstructor implementation for a given constructor
    * 
    * @param superClass the super class of the GeneratedConstructor implementation
    * @param constructor the BytecodeConstructorInfo for which we want to create a GeneratedConstructor implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static ConstructorAccessor generateConstructorAccessor(Class<?> superClass, BytecodeConstructorInfo constructor, boolean debug)
   {
      GeneratedMemberAccessorFactory factory = new GeneratedConstructorAccessorFactory(superClass, constructor, debug);
      Class<ConstructorAccessor> member = factory.makeClass(ConstructorAccessor.class, constructor.getDeclaringClass());
      return ErrorCheckingMemberFactory.wrapInErrorChecker(factory.instantiate(member), constructor);
   }

   /**
    * Creates a new GeneratedField implementation for a given field
    * 
    * @param superClass the super class of the GeneratedField implementation
    * @param field the BytecodeFieldInfo for which we want to create a GeneratedField implementation
    * @param debug true if the bytecode should be output to file
    * @return the generated class
    * @throws RuntimeException if an error ocurred
    */
   public static FieldAccessor generateFieldAccessor(Class<?> superClass, BytecodeFieldInfo field, boolean debug)
   {
      GeneratedMemberAccessorFactory factory = new GeneratedFieldAccessorFactory(superClass, field, debug);
      Class<FieldAccessor> member = factory.makeClass(FieldAccessor.class, field.getDeclaringClass());
      return ErrorCheckingMemberFactory.wrapInErrorChecker(factory.instantiate(member), field);
   }

   /**
    * Instantiates the class
    * 
    * @param clazz the class
    * @return the instance
    * @throws RuntimeException if an error ocurred
    */
   protected <T> T instantiate(Class<T> clazz)
   {
      try
      {
         return clazz.newInstance();
      }
      catch (Exception e)
      {
         // AutoGenerated
         throw new RuntimeException(e);
      }
   }
   
   
   /**
    * Creates a Method-/Field-/ConstructorAccessor implementation
    * 
    * @return the generated class
    */
   protected <T> Class<T> makeClass(Class<T> expected, BytecodeTypeInfo target)
   {
      //Create a new public class
      String name = getGeneratedClassName();
      final ClassFileWriterContext<T> cfwc = new ClassFileWriterContext<T>(name, superClass.getName(), expected, getInterfaceNames());

      try
      {
         //implement the methods from the interfaces
         implementMethods(cfwc);
         
         
         if (debug)
         {
            debugWriteFile(cfwc);
         }
         
         
      }
      catch(Exception e)
      {
         throw new RuntimeException("Error creating " + expected.getSimpleName() + " for " + target.getName(), e);
      }

      ;
      ClassLoader cl = target.getClassLoaderInternal();
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();
      
      return toClass(cfwc, target, cl, null);
   }
   
   /**
    * Create a class from the class file writer context 
    * 
    *  @param cfwc the class file writer context
    *  @param target the target class
    *  @param cl the classloader
    *  @param domain the protection domain
    *  @return the class
    * 
    */
   private <T> Class<T> toClass(ClassFileWriterContext<T> cfwc, final BytecodeTypeInfo target, ClassLoader cl, ProtectionDomain domain)
   {
      final ClassLoader actualLoader = PARENT_LOADER_HANDLER.getActualLoader(cl);
      Throwable t = null;
      
      try
      {
         return cfwc.toClass(actualLoader, domain);
      }
      catch (InvocationTargetException e)
      {
         t = e;
      }
      catch (IllegalAccessException e)
      {
         t = e;
      }
      throw new RuntimeException("Error creating " + cfwc.getSimpleType() + " for " + target.getName() + " with classloader " + actualLoader + "(" + cl + ")", t);
   }
   
   /**
    * Implements the methods of the interface
    * 
    * @param cfcw the class file of the class we are creating
    */
   void implementMethods(ClassFileWriterContext<?> cfcw) throws DuplicateMemberException
   {
      int i = 0;
      while (true)
      {
         if (!implementMethod(i++, cfcw))
            break;
      }
   }
   
   /**
    * Returns the signature string of the accessed member
    * 
    * @return the signature
    */
   String getAccessedMember()
   {
      if (accessedMember == null)
      {
         accessedMember = initAccessedMember();
      }
      return accessedMember;
   }
   
   /**
    * Create the method 
    * 
    * @param index the current index (starting at 0)
    * @param cfwc the class file writer context
    * @return true, or false if there was no method with that index
    */
   abstract boolean implementMethod(int index, ClassFileWriterContext<?> cfwc);
   
   /**
    * Get the names of the interfaces to implement
    */
   abstract String[] getInterfaceNames();
   
   /**
    * Get the name of the class we are creating
    * 
    * @return the class name
    */
   abstract String getGeneratedClassName();
   
   /**
    * Initialize the signature string of the accessed member
    * 
    *  @return the signature
    */
   abstract String initAccessedMember();
   
   /**
    * Counts the maximum stack size of the methods being implemented
    * 
    * @param offset the offset
    * @param signatureKey the signature key of the parameters being loaded
    * @return the max stack size
    */
   int countParameterStackSize(int offset, SignatureKey signatureKey)
   {
      int stacksize = offset;
         
      String[] params = signatureKey.getParams();
      int n = params.length;
      for (int i = 0; i < n; ++i)
      {
         stacksize++;
         if (signatureKey.isDouble(i) || signatureKey.isLong(i))
            stacksize++;
      }

      return stacksize;
   }
   
   /**
    * Writes the class file bytes to the local file system so they can be inspected
    * with javap or another decompiler.
    * 
    * @param cfwc the class file writer context
    * @throws IOException if an error occurred writing the file
    */
   void debugWriteFile(final ClassFileWriterContext<?> cfwc) throws IOException
   {
      try
      {
         AccessController.doPrivileged(new PrivilegedExceptionAction<Object>()
         {

            public Object run() throws Exception
            {
               FileOutputStream fout = new FileOutputStream(cfwc.getName() + ".class");
               BufferedOutputStream out = new BufferedOutputStream(fout);
               try 
               {
                  out.write(cfwc.getBytes());
               }
               finally 
               {
                  try
                  {
                     out.close();
                  }
                  catch (Exception ignore)
                  {
                  }
               }
               return null;
            }
         });
      }
      catch (PrivilegedActionException e)
      {
         if (e.getCause() instanceof IOException)
            throw (IOException)e.getCause();
         if (e.getCause() instanceof RuntimeException)
            throw (RuntimeException)e.getCause();
         throw new RuntimeException(e.getCause());
      }
   }

   /**
    * Get the boxed type
    * 
    * TODO JBMICROCONT-119 integer progression?
    * @param type the type to box
    * @return the boxed type name
    */
   String getBoxedType(String type)
   {
      if (type.startsWith("["))
      {
         return type.replace('.', '/');
      }
      Boxing boxing = Boxing.getUnboxer(type);
      if (boxing != null)
      {
         return boxing.getClassName();
      }
      return type;
   }

   /**
    * Casts the value currently on the stack to the target type, and if a primitive
    * unboxes it. The byte code instructions to do this are added to the 
    * <code>cfwc</code> parameter.
    * 
    * @param cfcw the class file writer context of the method/constructor we are creating
    * @param type the target type we want to cast to
    */
   void castAndUnbox(ClassFileWriterContext<?> cfwc, String type)
   {
      if (type.equals(OBJECT_NAME))
         return;

      cfwc.addCheckcast(ClassFileWriterContext.jvmClassName(getBoxedType(type)));
      Boxing unboxer = Boxing.getUnboxer(type);
      if (unboxer != null)
      {
         cfwc.addInvokeVirtual(unboxer.getClassName(), unboxer.getUnboxMethodName(), unboxer.getUnboxMethodDescriptor());
         
      }
   }
   
   /**
    * Adds the byte code instructions to the <code>cfcw</code> paramter to box the value currently on the stack to the target type.
    * 
    * @param cfcw the class file writer context of the method/constructor we are creating
    * @param type the type we want to cast to
    */
   void boxReturnValue(ClassFileWriterContext<?> cfwc, String type)
   {
      Boxing boxing = Boxing.BOXERS.get(type);
      if (boxing != null)
      {
         cfwc.addInvokeStatic(boxing.getClassName(), boxing.getBoxMethodName(), boxing.getBoxMethodDescriptor());
      }
   }

   /**
    * In AS the GeneratedMethod, -Constructor and -Field classloaders are not deployed in 
    * the system classpath. When generating an accessor for something from the system classpath
    * we need to make sure it happens at the level that can see the implemented interfaces
    * 
    */
   private static class ParentLoaderHandler
   {
      final ClassLoader reflectLoader;
      final Set<ClassLoader> parentLoaders;
      
      public ParentLoaderHandler(ClassLoader reflectLoader, Set<ClassLoader> parentLoaders)
      {
         if (reflectLoader == null)
            throw new IllegalArgumentException("Null reflect loader");
         this.reflectLoader = reflectLoader;
         this.parentLoaders = parentLoaders;
      }
      
      private ClassLoader getActualLoader(ClassLoader loader)
      {
         if (parentLoaders != null)
            if (parentLoaders.contains(loader))
               return reflectLoader;
         return loader;
      }
   }

   private static class Boxing
   {
      static final Map<String , Boxing> BOXERS;
      static
      {
         Map<String , Boxing> map = new HashMap<String, Boxing>();
         addEntry(map, BytecodePrimitive.BOOLEAN);
         addEntry(map, BytecodePrimitive.BYTE);
         addEntry(map, BytecodePrimitive.CHAR);
         addEntry(map, BytecodePrimitive.DOUBLE);
         addEntry(map, BytecodePrimitive.FLOAT);
         addEntry(map, BytecodePrimitive.INT);
         addEntry(map, BytecodePrimitive.LONG);
         addEntry(map, BytecodePrimitive.SHORT);
         BOXERS = Collections.unmodifiableMap(map);
      }
      
      static void addEntry(Map<String, Boxing> map, BytecodePrimitive primitive)
      {
         String jvmClassName = ClassFileWriterContext.jvmClassName(primitive.getWrapperClassName()); 
         Boxing boxing = new Boxing(
               jvmClassName, 
               primitive.getValueMethodName(), 
               "()" + primitive.getArrayComponentName(),
               "(" + primitive.getArrayComponentName() + ")L" + jvmClassName + ";");
         map.put(primitive.getName(), boxing);
         map.put(primitive.getArrayComponentName(), boxing);
      }
      
      private String className;
      private String unboxMethodName;
      private String unboxMethodDescriptor;
      private String boxMethodDescriptor;
      
      static Boxing getUnboxer(String primitive)
      {
         return BOXERS.get(primitive);
      }
      
      public Boxing(String className, String methodName, String unboxMethodDescriptor, String boxMethodDescriptor)
      {
         this.className = className;
         this.unboxMethodName = methodName;
         this.unboxMethodDescriptor = unboxMethodDescriptor;
         this.boxMethodDescriptor = boxMethodDescriptor;
      }

      String getClassName()
      {
         return className;
      }
      
      String getUnboxMethodName()
      {
         return unboxMethodName;
      }
   
      String getUnboxMethodDescriptor()
      {
         return unboxMethodDescriptor;
      }
      
      String getBoxMethodName()
      {
         return "valueOf";
      }
      
      String getBoxMethodDescriptor()
      {
         return boxMethodDescriptor;
      }
   }
   
   
}