/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.bytecode;

import java.lang.annotation.Annotation;
import java.lang.ref.WeakReference;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javassist.bytecode.Descriptor;
import javassist.bytecode.SignatureAttribute;
import javassist.bytecode.SignatureAttribute.ArrayType;
import javassist.bytecode.SignatureAttribute.BaseType;
import javassist.bytecode.SignatureAttribute.ClassType;
import javassist.bytecode.SignatureAttribute.ObjectType;
import javassist.bytecode.SignatureAttribute.TypeArgument;

import org.jboss.reflect.plugins.AnnotationAttributeImpl;
import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.plugins.AnnotationValueFactory;
import org.jboss.reflect.plugins.AnnotationValueImpl;
import org.jboss.reflect.plugins.EnumConstantInfoImpl;
import org.jboss.reflect.plugins.GenericsUtil;
import org.jboss.reflect.plugins.TypeVariableAware;
import org.jboss.reflect.plugins.bytecode.bytes.BytecodePrimitive;
import org.jboss.reflect.plugins.bytecode.bytes.ClassBytes;
import org.jboss.reflect.plugins.bytecode.bytes.ClassBytesFactory;
import org.jboss.reflect.plugins.bytecode.bytes.FieldBytes;
import org.jboss.reflect.plugins.bytecode.bytes.MemberBytes;
import org.jboss.reflect.plugins.bytecode.bytes.MethodBytes;
import org.jboss.reflect.plugins.bytecode.bytes.asm.AsmClassBytesFactory;
import org.jboss.reflect.plugins.introspection.IntrospectionTypeInfoFactory;
import org.jboss.reflect.spi.AnnotationInfo;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.ArrayInfo;
import org.jboss.reflect.spi.ClassInfo;
import org.jboss.reflect.spi.DelegateClassInfo;
import org.jboss.reflect.spi.NumberInfo;
import org.jboss.reflect.spi.PrimitiveInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;
import org.jboss.util.collection.WeakClassCache;

/**
 * A javassist type factory.
 * TODO: need to fix the cl stuff
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 *
 * @version $Revision: 105101 $
 */
public class BytecodeTypeInfoFactoryImpl extends WeakClassCache<TypeInfo> implements TypeInfoFactory, AnnotationHelper
{
   private final ClassBytesFactory classBytesFactory = AsmClassBytesFactory.INSTANCE;
   
   /** Tmp invocation results */
   private static ThreadLocal<Map<ClassBytes, TypeInfo>> results = new ThreadLocal<Map<ClassBytes, TypeInfo>>();

   private volatile ClassLoaderFinder finder;
   
   public BytecodeTypeInfoFactoryImpl()
   {
      setClassLoaderFinder(null);
   }
   
   static final AnnotationValue[] NO_ANNOTATIONS = new AnnotationValue[0];
   

   ClassBytesFactory getClassBytesFactory()
   {
      return classBytesFactory;
   }
   
   public ClassLoaderFinder getClassLoaderFinder()
   {
      return finder;  
   }
   
   public void setClassLoaderFinder(ClassLoaderFinder finder)
   {
      if (finder == null)
      {
         this.finder = ClassLoaderFinder.LoadClass.INSTANCE;
//         this.finder = CachingLoadClassClassLoaderFinder.INSTANCE;
         return;
      }
      this.finder = finder;
   }
   
   private ClassLoader findClassLoaderForClass(ClassLoader initiating, String name)
   {
      int start = 0;
      while (name.charAt(start) == '[')
         start++;
      
      int end = name.length() - 1;
      while (end > start && name.charAt(end) == ']' && name.charAt(end - 1) == '[')
         end -= 2;
      
      if (start > 0 || end < name.length() - 1)
         name = name.substring(0, end + 1);
      
      BytecodePrimitive primitive = BytecodePrimitive.valueOf(name);
      if (primitive != null)
         return SecurityActions.getSystemClassLoader();
      
      return finder.getLoaderForClass(initiating, name);
   }

   @Override
   @SuppressWarnings("unchecked")
   protected TypeInfo instantiate(Class clazz)
   {
      ClassBytes classBytes = loadClassBytes(getClassLoader(clazz), clazz.getName());
      return instantiate(clazz.getName(), classBytes, clazz);
   }
   
   protected TypeInfo instantiate(String name, ClassBytes classBytes, Class<?> clazz)
   {
      boolean start = false;
      Map<ClassBytes, TypeInfo> tmp = results.get();
      if (tmp == null)
      {
         start = true;
         tmp = new HashMap<ClassBytes, TypeInfo>();
         results.set(tmp);
      }

      // if we return here, it means we're already in a loop,
      // hence no need to cleanup
      TypeInfo cached = tmp.get(classBytes);
      if (cached != null)
         return cached;

      try
      {
         if (classBytes.getComponentType() != null)
         {
            try
            {
               TypeInfo componentType = getTypeInfo(classBytes.getComponentType().getTypeInfoName(), classBytes.getClassLoader());
               BytecodeArrayInfoImpl arrayInfo = new BytecodeArrayInfoImpl(this, classBytes, clazz, componentType);
               tmp.put(classBytes, arrayInfo);
               return arrayInfo;
            }
            catch (ClassNotFoundException e)
            {
               throw new RuntimeException(e);
            }
         }
         //TODO handle arrays
         else if (classBytes.isAnnotation())
         {
            BytecodeAnnotationInfo result = new BytecodeAnnotationInfo(this, name, classBytes, clazz);
            tmp.put(classBytes, result);

            MethodBytes[] methods = classBytes.getDeclaredMethodBytes();
            AnnotationAttributeImpl[] atttributes = new AnnotationAttributeImpl[methods.length];
            for (int i = 0 ; i < methods.length ; i++)
            {
               try
               {
                  atttributes[i] = new AnnotationAttributeImpl(methods[i].getName(), getTypeInfo(SignatureKey.getReturnType(methods[i]), classBytes.getClassLoader()), null);
               }
               catch (ClassNotFoundException e)
               {
                  throw new RuntimeException("Error reading attribute " + methods[i].getName() + " for " + classBytes.getJvmName());
               }
            }
            result.setAttributes(atttributes);

            return result;

         }
         else if (classBytes.isEnum())
         {
            BytecodeEnumInfo enumInfo = new BytecodeEnumInfo(this, name, classBytes, clazz);
            tmp.put(classBytes, enumInfo);

            FieldBytes[] fields = classBytes.getDeclaredFieldBytes();
            List<EnumConstantInfoImpl> constants = new ArrayList<EnumConstantInfoImpl>();
            for (FieldBytes field : fields)
            {
               if (field.isEnumConstant())
               {
                  AnnotationValue[] annotations = getAnnotations(field);
                  constants.add(new EnumConstantInfoImpl(field.getName(), enumInfo, annotations));
               }
            }
            enumInfo.setEnumConstants(constants.toArray(new EnumConstantInfoImpl[constants.size()]));

            return enumInfo;
         }
         else
         {
            BytecodeTypeInfo typeInfo = new BytecodeTypeInfo(this, name, classBytes, clazz);
            tmp.put(classBytes, typeInfo);
            return typeInfo;
         }
      }
      finally
      {
         if (start)
            results.remove();
      }
   }
   

   @Override
   @SuppressWarnings("unchecked")
   protected void generate(Class clazz, TypeInfo result)
   {
      // Everything is done lazily
   }

   public TypeInfo getTypeInfo(Class<?> clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null class");

      return get(clazz);
   }

   
   /**
    * Gets the type info for a class
    * 
    * @param type the type
    * @return the type info
    */
   public TypeInfo getTypeInfo(String name, ClassLoader cl) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null class name");
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();

      return get(name, cl);
   }
   
   /**
    * Gets the type info for a reflect type
    * 
    * @param type the type
    * @return the type info
    */
   public TypeInfo getTypeInfo(Type type)
   {
      if (type instanceof Class)
         return getTypeInfo((Class<?>) type);
      else if (type instanceof ParameterizedType)
         return getParameterizedType((ParameterizedType)type);
      else if (type instanceof WildcardType)
         return getWildcardType((WildcardType)type);
      else if (type instanceof GenericArrayType)
         return getGenericArrayType((GenericArrayType)type);
      else if (type instanceof TypeVariable)
         return getTypeVariable((TypeVariable<?>) type);
      else
         throw new UnsupportedOperationException("Unknown type: " + type + " class=" + type.getClass());
   }

   /**
    * Get the information for a class
    * 
    * @param name the name
    * @param cl the classloader
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   @Override
   public TypeInfo get(String name, ClassLoader cl) throws ClassNotFoundException
   {
      return get(name, cl, null);
   }

   /**
    * Get the information for a class
    * 
    * 
    * @param clazz the class
    * @return the info
    */
   @SuppressWarnings("unchecked")
   @Override
   public TypeInfo get(Class clazz)
   {
      try
      {
         ClassLoader cl = SecurityActions.getClassLoader(clazz);
         if(cl == null)
            cl = SecurityActions.getContextClassLoader();
         return get(clazz.getName(), cl, clazz);
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException("Class not found: "+e.getMessage());
      }
   }

   /**
    * Get the information for a class
    * 
    * @param name the name
    * @param cl the classloader
    * @param clazz the class
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   protected TypeInfo get(String name, ClassLoader cl, Class<?> clazz) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      if (cl == null)
         throw new IllegalArgumentException("Null classloader");
   
      return get(null, clazz, name, cl, true);
   }

   /**
    * Get the information for a class
    * 
    * @param classBytes the class bytes, may be null
    * @param clazz the class, may be null
    * @param name the name of the class
    * @param cl the class loader initiating the lookup
    * @return the info
    * @throws IllegalArgumentException if either name or cl are null
    */
   protected TypeInfo get(ClassBytes classBytes, Class<?> clazz, String name, ClassLoader cl, boolean delegateToReflectIfNotFound) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null name");
      if (cl == null)
         throw new IllegalArgumentException("Null class loader");
      
      TypeInfo primitive = PrimitiveInfo.valueOf(name);
      if (primitive != null)
         return primitive;

      NumberInfo number = NumberInfo.valueOf(name);
      if (number != null)
      {
         synchronized (number)
         {
            if (number.getPhase() != NumberInfo.Phase.INITIALIZING)
            {
               if (number.getPhase() != NumberInfo.Phase.COMPLETE)
               {
                  number.initializing();
                  Class<?> useClass = clazz;
                  try
                  {
                     if (useClass == null)
                        useClass = cl.loadClass(name);
                  }
                  catch (ClassNotFoundException e)
                  {
                     throw new RuntimeException(e);
                  }
                  number.setDelegate(get(useClass));
               }
               return number;
            }
         }
      }
      
      //Handle array names
      if (name.charAt(0) == '[')
         name = Descriptor.toClassName(name);
      
      Map<String, WeakReference<TypeInfo>> classLoaderCache = getClassLoaderCache(cl);
      Map<String, WeakReference<TypeInfo>> originalCache = classLoaderCache;
      TypeInfo result = getFromCache(name, classLoaderCache);
      if (result != null)
         return result;
      
      ClassLoader real = cl;
      if (classBytes == null)
      {
         real = findClassLoaderForClass(cl, name);
         if (real == null)
            real = cl;
         else if (real != cl)
         {
            //Make sure that we check the cache of the classloader actually containing the class
            classLoaderCache = getClassLoaderCache(real);
            result = getFromCache(name, classLoaderCache);
            if (result != null)
            {
               originalCache.put(name, new WeakReference<TypeInfo>(result));
               return result;
            }
         }

         classBytes = loadClassBytes(real, name);
      }
      
      if (classBytes == null)
      {
         if (clazz == null)
            throw new ClassNotFoundException("Could not load " + name + " in loader " + cl);
         else
            return delegateToIntrospectionImplementation(cl, name);
      }
      result = instantiate(name, classBytes, clazz);


      WeakReference<TypeInfo>weak = new WeakReference<TypeInfo>(result);
      classLoaderCache.put(name, weak);
      if (originalCache != classLoaderCache)
         originalCache.put(name, weak);
      
//      we just ignore generate(..) since it doesnt do anything atm
//      generate(clazz, result);
      
      return result;
   }
   
   private ClassLoader getClassLoader(Class<?> clazz)
   {
      ClassLoader loader = SecurityActions.getClassLoader(clazz);
      if (loader != null)
         return loader;
      return SecurityActions.getSystemClassLoader();
   }
   
   private ClassLoader getClassLoader(final TypeInfo info)
   {
      if (System.getSecurityManager() == null)
         return info.getClassLoader();
      
      if (info instanceof BytecodeTypeInfo)
         return ((BytecodeTypeInfo)info).getClassLoaderInternal();
      
      return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>()
      {
         public ClassLoader run()
         {
            return info.getClassLoader();
         }
      });
   }

   /**
    * Proxies, whether
    * <ul>
    * <li>JDK dynamic proxies</li>
    * <li>javassist ProxyFactory proxies - created by calling ClassLoader.defineClass()</li>
    * </ul> 
    * are not visible to the javassist classpools, and neither will classes generated by cglib or other
    * frameworks, so try to load up the class from the reflect implementation.
    * 
    * @param cl the classloader
    * @param name the name of the class
    * @return the info
    * @throws ClassNotFoundException when the class cannot be found
    */
   private TypeInfo delegateToIntrospectionImplementation(ClassLoader cl, String name) throws ClassNotFoundException
   {
      try
      {
         IntrospectionTypeInfoFactory factory = new IntrospectionTypeInfoFactory();
         return factory.getTypeInfo(name, cl);
      }
      catch (ClassNotFoundException e)
      {
         throw e;
      }
   }
   
   /**
    * Reads the type info from the cache
    * 
    * @param name the class name
    * @param classLoaderCache The cached type infos for the loader
    */
   private TypeInfo getFromCache(String name, Map<String, WeakReference<TypeInfo>> classLoaderCache)
   {
      WeakReference<TypeInfo> weak = classLoaderCache.get(name);
      if (weak != null)
      {
         return weak.get();
      }
      return null;
   }

   public AnnotationValue[] getAnnotations(Object obj)
   {
      try
      {
         Object[] annotations;
         if (obj instanceof ClassBytes)
         {
            annotations = ((ClassBytes)obj).getAnnotations();
         }
         else if (obj instanceof MemberBytes)
         {
            annotations = ((MemberBytes)obj).getAnnotations();
         }
         else
         {
            throw new RuntimeException("Attempt was made to read annotations from unsupported type " + obj.getClass().getName() + ": " + obj);
         }

         if (annotations == null || annotations.length == 0)
         {
            return NO_ANNOTATIONS;
         }

         AnnotationValue[] annotationValues = new AnnotationValueImpl[annotations.length];
         for (int i = 0 ; i < annotations.length ; i++)
         {
            Class<?> clazz = ((Annotation)annotations[i]).annotationType();
            
            AnnotationInfo info = (AnnotationInfo)getTypeInfo(clazz);
            annotationValues[i] = AnnotationValueFactory.createAnnotationValue(this, this, info, annotations[i]);
         }
         return annotationValues;
      }
      catch (Throwable t)
      {
         throw new RuntimeException(t);
      }
   }

   public AnnotationValue createAnnotationValue(AnnotationInfo info, Object ann)
   {
      return AnnotationValueFactory.createAnnotationValue(this, this, info, ann);
   }
   

   /**
    * Creates a type info from a reflect generic paramemeterized type
    * 
    * @param type the parameterized type
    * @return the type info
    */
   protected TypeInfo getParameterizedType(ParameterizedType type)
   {
      //Create the parameterized type info
      Class<?> rawType = (Class<?>) type.getRawType();
      ClassInfo raw = (ClassInfo)getTypeInfo(rawType);
      Type[] types = type.getActualTypeArguments();
      
      return new BytecodeParameterizedClassInfo(this, raw, types);
   }

   /**
    * Gets the type info for a javassist type
    * 
    * @param loader the class loader
    * @param type the class type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo getTypeInfo(ClassLoader loader, SignatureAttribute.Type type, BytecodeTypeVariableSpy spy)
   {
      if (type instanceof ClassType)
         return getTypeInfo(loader, (ClassType)type, spy);
      else if (type instanceof ArrayType)
         return getGenericArrayType(loader, (ArrayType)type, spy);
      else if (type instanceof BaseType)
      {
         String s = String.valueOf(((BaseType)type).getDescriptor());
         Class<?> clazz = PrimitiveInfo.getPrimativeArrayComponentType(s);
         return PrimitiveInfo.valueOf(clazz.getName()); 
      }
      else if (type instanceof javassist.bytecode.SignatureAttribute.TypeVariable)
      {
         TypeInfo typeInfo = getTypeInfo(loader, spy.getTypeBound((javassist.bytecode.SignatureAttribute.TypeVariable)type), spy);
         if (typeInfo instanceof TypeVariableAware)
            ((TypeVariableAware)typeInfo).setTypeVariable(((javassist.bytecode.SignatureAttribute.TypeVariable)type).getName());
         return typeInfo;
      }
      
      //Should not happen
      throw new IllegalArgumentException("Bad type " + type + " - " + type.getClass().getName());
   }
   
   /**
    * Gets the type info for a javassist generic class type
    * 
    * @param loader the class loader of the raw class
    * @param parameterloaders the class of the
    * @param type the class type
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo getTypeInfo(ClassLoader loader, ClassType type, BytecodeTypeVariableSpy spy)
   {
      //If it is not a parameterized type, just return the type
      boolean isParameterized = type.getTypeArguments() != null && type.getTypeArguments().length > 0;
      if (!isParameterized)
      {
         try
         {
            return get(BytecodeGenericsHelper.getClassNameForGenericType(type), loader);
         }
         catch (ClassNotFoundException e)
         {
            throw new IllegalStateException(e);
         }
      }
      
      //Look in the cache first
      String genericName = BytecodeGenericsHelper.getGenericName(type, spy);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(loader);
      TypeInfo info = getFromCache(genericName, cache);
      if (info != null)
      {
         if (info instanceof ClassInfo == false)
            throw new IllegalStateException("Not a ClassInfo " + info);
         return info;
      }
      
      //Create the parameterized type info
      try
      {
         ClassInfo delegate = (ClassInfo)get(BytecodeGenericsHelper.getClassNameForGenericType(type), loader);
         if (!isParameterized)
            return delegate;
         info = new BytecodeParameterizedClassInfo(this, delegate, loader, type.getTypeArguments(), spy);
      }
      catch (ClassNotFoundException e)
      {
         throw new IllegalStateException(e);
      }

      //Cache the parameterized type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));
      
      return info;
   }
   
   /**
    * Gets the type info for a javassist type argument
    * 
    * @param arg the type argument
    * @param loader the class loader
    * @param spy used to determine actual types of type variables
    * @return the type info
    */
   protected TypeInfo createTypeInfoForTypeArgument(TypeArgument arg, ClassLoader loader, BytecodeTypeVariableSpy spy)
   {
      ObjectType type = arg.getType();
      if (type == null)
      {
         try
         {
            return get(Object.class.getName(), loader, null);
         }
         catch(ClassNotFoundException e)
         {
            throw new RuntimeException(e);
         }
      }
      else if (type instanceof ClassType)
      {
         return getTypeInfo(loader, (ClassType) type, spy);
      }
      else if (type instanceof javassist.bytecode.SignatureAttribute.TypeVariable)
      {
         return getTypeInfo(loader, spy.getTypeBound((javassist.bytecode.SignatureAttribute.TypeVariable)type), spy);
      }
      throw new IllegalStateException("Unhandled type " + type);
   }

   /**
    * Gets the type info for a reflect wildcard type
    * 
    * @param type the wildcard type
    * @return the type info
    */
   protected TypeInfo getWildcardType(WildcardType type)
   {
      //Create the wildcard type info
      Type bound = type.getLowerBounds().length > 0 ? type.getLowerBounds()[0] : type.getUpperBounds()[0];
      return getTypeInfo(bound);
   }
   
   /**
    * Gets the classloader cache for the passed in classloader. 
    * 
    * @param cl the classloader. If null is used, the Thread context classloader will be used instead
    * @return the cache of type infos by name
    */
   @Override
   protected Map<String, WeakReference<TypeInfo>> getClassLoaderCache(ClassLoader cl)
   {
      if (cl == null)
         cl = SecurityActions.getContextClassLoader();
      return super.getClassLoaderCache(cl);
   }

   /**
    * Get the type info for a reflect generic array type
    * 
    * @param type the array type
    * @return the info
    */
   protected TypeInfo getGenericArrayType(GenericArrayType type)
   {
      //Look in the cache first
      String genericName = GenericsUtil.getGenericName(type);
      ClassLoader cl = GenericsUtil.findClassLoader(type);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(cl);
      TypeInfo info = getFromCache(genericName, cache);
      if (info != null)
         return info;
      
      //Create the wildcard type info
      Type compType = type.getGenericComponentType();
      TypeInfo componentType = getTypeInfo(compType);
      
      String arrayName = getArrayName(componentType);
      
      ClassBytes clazz = classBytesFactory.loadClassBytes(cl, arrayName + "[]");
      info = new BytecodeArrayInfoImpl(this, clazz, null, componentType);
      
      //Cache the wildcard type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));
      
      return info;
   }

   /**
    * Get the type info for a javassist array type
    * 
    * @param cl the classloader
    * @param type the array type
    * @param spy used to determine actual types of type variables
    * @return the type info 
    */
   protected ArrayInfo getGenericArrayType(ClassLoader cl, ArrayType type, BytecodeTypeVariableSpy spy)
   {
      ModifiableArrayType wrapped = new ModifiableArrayType(type);
      return getGenericArrayType(cl, wrapped, spy);
   }
   
   /**
    * Recursively get the type info for a javassist array type
    * 
    * @param cl the classloader
    * @param type the array type
    * @param spy used to determine actual types of type variables
    * @return the type info 
    */
   protected ArrayInfo getGenericArrayType(ClassLoader cl, ModifiableArrayType type, BytecodeTypeVariableSpy spy)
   {
      //Look in the cache first
      String genericName = BytecodeGenericsHelper.getGenericName(type, spy);
      Map<String, WeakReference<TypeInfo>> cache = getClassLoaderCache(cl);
      ArrayInfo info = (ArrayInfo)getFromCache(genericName, cache);
      if (info != null)
         return info;

      type.decrement();
      TypeInfo componentType = type.getDimension() > 0 ?
         getGenericArrayType(cl, type, spy) : getTypeInfo(cl, type.getComponentType(), spy);
      
      String arrayName = getArrayName(componentType);
      ClassBytes clazz = classBytesFactory.loadClassBytes(cl, arrayName + "[]");
      info = new BytecodeArrayInfoImpl(this, clazz, null, componentType);

      
      //Cache the wildcard type info
      cache.put(genericName, new WeakReference<TypeInfo>(info));
      
      return info;
   }
   
   private String getArrayName(TypeInfo componentType)
   {
      ClassBytes componentBytes = getClassBytes(componentType, true);
      if (!componentType.isArray())
         return componentBytes.getJvmName();
      else
         return componentBytes.getTypeInfoName().replace('.', '/');
   }
   
   /**
    * Gets the ClassBytes from a TypeInfo if it is a bytecode one
    * 
    * @param typeInfo the type info
    * @param error if true and the type can not be determined, throw an error
    * @return null if the type could not be determined and <code>error</code> is false
    * @throws IllegalArgumentException if <code>error</code> is true and the type can not be determined
    */
   private ClassBytes getClassBytes(TypeInfo typeInfo, boolean error)
   {
      if (typeInfo instanceof PrimitiveInfo)
         return getPrimitiveClassBytes(typeInfo.getName());
      if (typeInfo instanceof BytecodeTypeInfo)
         return ((BytecodeTypeInfo)typeInfo).getClassBytes();
      if (typeInfo instanceof DelegateClassInfo)
         return getClassBytes(((DelegateClassInfo) typeInfo).getDelegate(), error);

      if (error)
         throw new IllegalArgumentException(typeInfo + " is not a JavassistType info, a PrimitiveTypeInfo or a JavassistParameterizedType");
      
      return null; 
   }
   
   private ClassBytes getPrimitiveClassBytes(String name)
   {
      return classBytesFactory.loadClassBytes(SecurityActions.getSystemClassLoader(), name);
   }
   
   ClassBytes loadClassBytes(ClassLoader loader, String name)
   {
      return classBytesFactory.loadClassBytes(loader, name.replace('.', '/'));
   }
   
   /**
    * Get the type info for a reflect TypeVariable
    * 
    * @param type the type variable
    * @return the type info 
    * 
    */
   protected TypeInfo getTypeVariable(TypeVariable<?> type)
   {
      return getTypeInfo(type.getBounds()[0]);
   }
   
   /**
    * Wrapper around {@link ArrayType} to make it easier to 
    * create the name used to look up the array type infos 
    * in the cache 
    */
   protected static class ModifiableArrayType extends ArrayType
   {
      int dims;
      
      ModifiableArrayType(ArrayType original)
      {
         super(original.getDimension(), original.getComponentType());
         dims = original.getDimension();
      }
      
      @Override
      public int getDimension()
      {
         return dims;
      }

      void decrement()
      {
         dims--;
      }
   }
}
