/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.aop.asintegration.jboss5.temp;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.jboss.aop.AspectManager;
import org.jboss.aop.Domain;
import org.jboss.aop.classpool.AOPClassLoaderScopingPolicy;
import org.jboss.aop.domain.DomainInitializer;
import org.jboss.aop.microcontainer.beans.metadata.AspectManagerAwareBeanMetaDataFactory;
import org.jboss.beans.metadata.plugins.AbstractClassLoaderMetaData;
import org.jboss.beans.metadata.plugins.AbstractValueMetaData;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.BeanMetaDataFactory;
import org.jboss.beans.metadata.spi.ClassLoaderMetaData;
import org.jboss.beans.metadata.spi.ValueMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.deployer.AbstractSimpleVFSRealDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.plugins.dependency.AbstractKernelControllerContext;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;

/**
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class AbstractAopMetaDataDeployer<T> extends AbstractSimpleVFSRealDeployer<T>
{
   private AspectManager aspectManager;
   private KernelController controller;
   private MyBeanMetaDataDeployer beanMetaDataDeployer = new MyBeanMetaDataDeployer();
   private static int sequence;
   
   
   public AbstractAopMetaDataDeployer(Class<T> input)
   {
      super(input);
      super.setStage(DeploymentStages.POST_CLASSLOADER);
      super.setOutput(AopMetaDataDeployerOutput.class);
   }

   /**
    * Get the aspectManager.
    * 
    * @return the aspectManager.
    */
   public AspectManager getAspectManager()
   {
      return aspectManager;
   }

   /**
    * Set the aspectManager.
    * 
    * @param aspectManager the aspectManager.
    */
   public void setAspectManager(AspectManager aspectManager)
   {
      this.aspectManager = aspectManager;
   }
   
   /**
    * Set the kernel.
    * 
    * @param kernel the kernel
    */
   public void setKernel(Kernel kernel)
   {
      this.controller = kernel.getController();
   }
   
   /**
    * Method for subclasses to call upon deployment 
    */
   @Override
   public void deploy(VFSDeploymentUnit unit, T deployment) throws DeploymentException
   {
      log.debug("Deploying " + unit + " " + deployment);
      
      AopMetaDataDeployerOutput output = new AopMetaDataDeployerOutput();
      unit.getTransientManagedObjects().addAttachment(AopMetaDataDeployerOutput.class, output);
      
      if (extractAopBeanMetaDataFactories(unit, deployment, output))
      {
         AspectManager correctManager = getCorrectManager(unit, output);
         if (correctManager != aspectManager)
         {
            registerScopedManagerBean(unit, correctManager, output);
            massageScopedDeployment(unit, deployment, output);
         }
      }
      
      try
      {
         deployBeans(unit, output);
      }
      catch (Throwable t)
      {
         unregisterScopedManagerBean(output.getScopedAspectManagerBeanName(), false);
         if (t instanceof DeploymentException)
         {
            throw (DeploymentException)t;
         }
         else
         {
            throw new DeploymentException(t);
         }
      }
   }

   
   /**
    * Method for subclasses to call upon undeployment 
    */
   @Override
   public void undeploy(VFSDeploymentUnit unit, T deployment)
   {
      log.debug("Undeploying " + unit + " " + deployment);
      
      AopMetaDataDeployerOutput output = unit.getTransientManagedObjects().getAttachment(AopMetaDataDeployerOutput.class);
      
      undeployBeans(unit, output);
      
      unregisterScopedManagerBean(output.getScopedAspectManagerBeanName(), true);
   }

   protected abstract List<BeanMetaDataFactory> getFactories(T deployment);
   
   private boolean extractAopBeanMetaDataFactories(VFSDeploymentUnit unit, T deployment, AopMetaDataDeployerOutput output)
   {
      log.debug("Extracting aop bean metadata factories for  " + unit);
      List<AspectManagerAwareBeanMetaDataFactory> aopFactories = new ArrayList<AspectManagerAwareBeanMetaDataFactory>();
      
      List<BeanMetaDataFactory> factories = getFactories(deployment);
      if (factories != null && factories.size() > 0)
      {
         for (Iterator<BeanMetaDataFactory> it = factories.iterator() ; it.hasNext() ; )
         {
            BeanMetaDataFactory factory = it.next();
            if (factory instanceof AspectManagerAwareBeanMetaDataFactory)
            {
               it.remove();
               aopFactories.add((AspectManagerAwareBeanMetaDataFactory)factory);
            }
         }     
      }
      
      if (aopFactories.size() > 0)
      {
         output.setFactories(aopFactories);
         return true;
      }
      return false;
   }
   
   private AspectManager getCorrectManager(VFSDeploymentUnit unit, AopMetaDataDeployerOutput output) throws DeploymentException
   {
      //Scoped AOP deployments are only available when deployed as part of a scoped sar, ear etc.
      //It can contain an aop.xml file, or it can be part of a .aop file
      //Linking a standalone -aop.xml file onto a scoped deployment is not possible at the moment
      AOPClassLoaderScopingPolicy policy = AspectManager.getClassLoaderScopingPolicy();

      Domain domain = null;
      if (policy != null)
      {
         if (policy instanceof DomainInitializer == false)
         {
            throw new RuntimeException(policy + " must implement DomainInitializer");
         }
         DomainInitializer initializer = (DomainInitializer)policy;
         domain = initializer.initializeDomain(unit); 
      }
      
      if (domain != null)
      {
         log.debug("Deployment is scoped to " + domain);
         return domain;
      }
      
      return aspectManager;
   }
   
   private void registerScopedManagerBean(VFSDeploymentUnit unit, AspectManager scopedManager, AopMetaDataDeployerOutput output) throws DeploymentException
   {
      String name = "ScopedManager_" + getSequence() + "_" + unit.getName();
      BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(name, scopedManager.getClass().getName());
      
      try
      {
         controller.install(builder.getBeanMetaData(), scopedManager);
         output.setScopedAspectManagerBeanName(name);
      }
      catch (Throwable e)
      {
         unregisterScopedManagerBean(name, false);
         throw new DeploymentException("Error registering scoped manager" + name + " " + scopedManager, e);
      }
   }
 
   private synchronized int getSequence()
   {
      return ++sequence;
   }
   
   private void unregisterScopedManagerBean(String name, boolean logError)
   {
      if (name == null)
      {
         return;
      }
      
      try
      {
         controller.uninstall(name);
      }
      catch(Throwable t)
      {
         if (logError)
         {
            log.debug("Error unregistering scoped aspect manager " + name, t);
         }
      }
   }
   
   private void massageScopedDeployment(VFSDeploymentUnit unit, T deployment, AopMetaDataDeployerOutput output) throws DeploymentException
   {
      if (output.getScopedAspectManagerBeanName() == null)
      {
         return;
      }
      
      log.debug("Massaging scoped deployment " + unit);
      List<AspectManagerAwareBeanMetaDataFactory> aopFactories = output.getFactories();

      if (aopFactories != null && aopFactories.size() > 0)
      {
         for (AspectManagerAwareBeanMetaDataFactory factory : aopFactories)
         {
            factory.setManagerBean(output.getScopedAspectManagerBeanName());
            factory.setManagerProperty(null);
         }
      }
   }
   
   private void deployBeans(VFSDeploymentUnit unit, AopMetaDataDeployerOutput output) throws DeploymentException
   {
      List<AspectManagerAwareBeanMetaDataFactory> aopFactories = output.getFactories();
      List<BeanMetaData> done = new ArrayList<BeanMetaData>();
      try
      {
         if (aopFactories != null && aopFactories.size() > 0)
         {
            for (AspectManagerAwareBeanMetaDataFactory factory : aopFactories)
            {
               List<BeanMetaData> beans = factory.getBeans();
               if (beans != null && beans.size() > 0)
               {
                  for (BeanMetaData bean : beans)
                  {
                     beanMetaDataDeployer.deploy(unit, bean);
                     done.add(bean);
                  }
               }
            }
         }
      }
      catch (Throwable t)
      {
         for (int i = done.size() - 1 ; i >= 0 ; i--)
         {
            try
            {
               beanMetaDataDeployer.undeploy(unit, done.get(i));
               controller.uninstall(done.get(i));
            }
            catch (Throwable e)
            {
               log.debug("Error undeploying " + done.get(i) + " for " + unit);
            }
         }
         throw new DeploymentException(t);
      }
   }
   
   private void undeployBeans(VFSDeploymentUnit unit, AopMetaDataDeployerOutput output)
   {
      List<AspectManagerAwareBeanMetaDataFactory> aopFactories = output.getFactories();
      if (aopFactories != null && aopFactories.size() > 0)
      {
         for (int i = aopFactories.size() - 1 ; i >= 0 ; i--)
         {
            AspectManagerAwareBeanMetaDataFactory factory = aopFactories.get(i);
            List<BeanMetaData> beans = factory.getBeans();
            if (beans != null && beans.size() > 0)
            {
               for (int j = beans.size() - 1 ; j >= 0 ; j--)
               {
                  BeanMetaData bean = beans.get(j);
                  beanMetaDataDeployer.undeploy(unit, bean);
               }
            }
         }
      }      
   }
   
   private class MyBeanMetaDataDeployer
   {   
      /**
       * Copied from the real org.jboss.deployers.vfs.deployer.kernel.BeanMetaDataDeployer
       */
      private void deploy(DeploymentUnit unit, BeanMetaData deployment) throws DeploymentException
      {
         // No explicit classloader, use the deployment's classloader
         if (deployment.getClassLoader() == null)
         {
            try
            {
               // Check the unit has a classloader
               unit.getClassLoader();
               // TODO clone the metadata?
               deployment.setClassLoader(new DeploymentClassLoaderMetaData(unit));
            }
            catch (Exception e)
            {
               log.debug("Unable to retrieve classloader for deployment: " + unit.getName() + " reason=" + e.toString());
            }
         }
         KernelControllerContext context = new AbstractKernelControllerContext(null, deployment, null);
         try
         {
            controller.install(context);
         }
         catch (Throwable t)
         {
            throw DeploymentException.rethrowAsDeploymentException("Error deploying: " + deployment.getName(), t);
         }
      }
   
      /**
       * Copied from the real org.jboss.deployers.vfs.deployer.kernel.BeanMetaDataDeployer
       */
      private void undeploy(DeploymentUnit unit, BeanMetaData deployment)
      {
         try
         {
            controller.uninstall(deployment.getName());
            
            // Remove any classloader metadata we added (not necessary if we clone above)
            ClassLoaderMetaData classLoader = deployment.getClassLoader();
            if (classLoader instanceof DeploymentClassLoaderMetaData)
               deployment.setClassLoader(null);
         }
         catch(Throwable t)
         {
            log.info("Error undeploying " + deployment + " for " + unit);
         }
      }
   }
   
   /**
    * Copied from BeanMetaDataDeployer
    */
   private static class DeploymentClassLoaderMetaData extends AbstractClassLoaderMetaData
   {
      /** The serialVersionUID */
      private static final long serialVersionUID = 1L;
      
      /** The deployment unit */
      private DeploymentUnit unit;

      /**
       * Create a new DeploymentClassLoaderMetaData.
       * 
       * @param unit the deployment unit
       */
      public DeploymentClassLoaderMetaData(DeploymentUnit unit)
      {
         if (unit == null)
            throw new IllegalArgumentException("Null unit");
         this.unit = unit;
      }
      
      @Override
      public ValueMetaData getClassLoader()
      {
         return new AbstractValueMetaData(unit.getClassLoader());
      }
   }
}
