/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.ha.framework.server;

import java.net.InetAddress;
import java.rmi.dgc.VMID;
import java.rmi.server.UID;

import org.jboss.logging.Logger;
import org.jboss.system.ServiceMBean;
import org.jboss.system.server.ServerConfigUtil;
import org.jgroups.Channel;
import org.jgroups.Event;
import org.jgroups.mux.MuxChannel;
import org.jgroups.stack.IpAddress;

/**
 * Extension to the JGroups JChannelFactory that supports the addition
 * of "additional_data" to the channel config.
 * 
 * @author <a href="mailto://brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision$
 */
public class JChannelFactory extends org.jgroups.JChannelFactory
      implements org.jgroups.jmx.JChannelFactoryMBean
{
   protected static Logger log = Logger.getLogger(JChannelFactory.class);
   
   private static final int CREATED = ServiceMBean.CREATED;
   private static final int STARTING = ServiceMBean.STARTING;
   private static final int STARTED = ServiceMBean.STARTED;
   private static final int STOPPING = ServiceMBean.STOPPING;
   private static final int STOPPED = ServiceMBean.STOPPED;
   private static final int DESTROYED = ServiceMBean.DESTROYED;
   private static final int FAILED = ServiceMBean.FAILED;
   
   private InetAddress nodeAddress;
   private String nodeName;
   private int namingServicePort = -1;
   private int state;
   private boolean assignLogicalAddresses = true;

   /**
    * Overrides the superclass version by generating a unique node id
    * and passing it down the Channel as additional_data and by suppressing the 
    * <code>register_for_state_transfer</code> param.
    * <p/>
    * The additional data is the value of {@link #getNodeName()} which either
    * can be injected as part of configuration or will be calculated 
    * the first time a channel is created.
    * <p/>
    * Suppressing the <code>register_for_state_transfer</code> param is necessary
    * to avoid issues that occur with JGroups when it is <code>true</code>.
    * In JBoss AS there is little benefit to setting this to true, as the 
    * single-threaded nature of deployments in the AS ensures that services 
    * always connect channels and request state transfer in series, with no
    * interleaving. 
    * 
    * @param register_for_state_transfer  ignored; always treated as <code>false</code>
    */
   @Override
   public Channel createMultiplexerChannel(String stack_name, String id, boolean register_for_state_transfer, String substate_id) throws Exception
   {
      // Due to problems with coordinated state transfer, we suppress register_for_state_transfer usage.
      // TODO revert to normal if coordinated state transfer is fixed
//      Channel channel = super.createMultiplexerChannel(stack_name, id, register_for_state_transfer, substate_id);
      Channel channel = super.createMultiplexerChannel(stack_name, id, false, null);
      
      if (assignLogicalAddresses)
         setChannelUniqueId(channel);
      
      return channel;
   }
   
   /**
    * Overrides the superclass version by generating a unique node id
    * and passing it down the Channel as additional_data.
    */
   @Override
   public Channel createMultiplexerChannel(String stack_name, String id) throws Exception
   {
      return createMultiplexerChannel(stack_name, id, false, null);
   }

   public InetAddress getNodeAddress()
   {
      return nodeAddress;
   }


   public void setNodeAddress(InetAddress nodeAddress)
   {
      this.nodeAddress = nodeAddress;
   }


   public String getNodeName()
   {
      return nodeName;
   }


   public void setNodeName(String nodeName)
   {
      this.nodeName = nodeName;
   }

   public int getNamingServicePort()
   {
      return namingServicePort;
   }

   public void setNamingServicePort(int jndiPort)
   {
      this.namingServicePort = jndiPort;
   }
   
   /**
    * Gets whether this factory should create a "logical address" (or use
    * one set via {@link #setNodeName(String)} and assign it to
    * any newly created <code>Channel</code> as JGroups "additional_data".
    * 
    * @see #setAssignLogicalAddresses(boolean)
    */
   public boolean getAssignLogicalAddresses()
   {
      return assignLogicalAddresses;
   }

   /**
    * Sets whether this factory should create a "logical address" (or use
    * one set via {@link #setNodeName(String)} and assign it to
    * any newly created <code>Channel</code> as JGroups "additional_data".
    * <p>
    * Any such logical address will be used by <code>HAPartition</code>
    * to assign a name to the <code>ClusterNode</code> object representing 
    * this node. If a logical address is not set, the <code>ClusterNode</code> 
    * will use the address and port JGroups is using to receive messages to
    * create its name.
    * </p>
    * <p>
    * Default is <code>true</code>.
    * </p>
    */
   public void setAssignLogicalAddresses(boolean logicalAddresses)
   {
      this.assignLogicalAddresses = logicalAddresses;
   }

   @Override
   public void create() throws Exception
   {

      if (state == CREATED || state == STARTING || state == STARTED
         || state == STOPPING || state == STOPPED)
      {
         log.debug("Ignoring create call; current state is " + getStateString());
         return;
      }
      
      log.debug("Creating JChannelFactory");
      
      try
      {
         super.create();
         state = CREATED;
      }
      catch (Exception e)
      {
         log.debug("Initialization failed JChannelFactory", e);
         throw e;
      }
      
      log.debug("Created JChannelFactory");
      super.create();
   }

   @Override
   public void start() throws Exception
   {
      if (state == STARTING || state == STARTED || state == STOPPING)
      {
         log.debug("Ignoring start call; current state is " + getStateString());
         return;
      }
      
      if (state != CREATED && state != STOPPED && state != FAILED)
      {
         log.debug("Start requested before create, calling create now");         
         create();
      }
      
      state = STARTING;
      log.debug("Starting JChannelFactory");

      try
      {
         super.start();
      }
      catch (Exception e)
      {
         state = FAILED;
         log.debug("Starting failed JChannelFactory", e);
         throw e;
      }

      state = STARTED;
      log.debug("Started JChannelFactory");
      
   }

   @Override
   public void stop()
   {
      if (state != STARTED)
      {
         log.debug("Ignoring stop call; current state is " + getStateString());
         return;
      }
      
      state = STOPPING;
      log.debug("Stopping JChannelFactory");

      try
      {
         super.stop();
      }
      catch (Throwable e)
      {
         state = FAILED;
         log.warn("Stopping failed JChannelFactory", e);
         return;
      }
      
      state = STOPPED;
      log.debug("Stopped JChannelFactory");
   }

   @Override
   public void destroy()
   {
      if (state == DESTROYED)
      {
         log.debug("Ignoring destroy call; current state is " + getStateString());
         return;
      }
      
      if (state == STARTED)
      {
         log.debug("Destroy requested before stop, calling stop now");
         stop();
      }
      
      log.debug("Destroying JChannelFactory");
      
      try
      {
         super.destroy();
      }
      catch (Throwable t)
      {
         log.warn("Destroying failed JChannelFactory", t);
      }
      state = DESTROYED;
      log.debug("Destroyed JChannelFactory");
   }

   private void setChannelUniqueId(Channel channel) throws Exception
   {
      IpAddress address = (IpAddress) channel.getLocalAddress();
      if (address == null)
      {
         // We push the independent name in the protocol stack before connecting to the cluster
         if (this.nodeName == null || "".equals(this.nodeName)) {
            this.nodeName = generateUniqueNodeName();
         }
         
         log.debug("Passing unique node id " + nodeName + " to the channel as additional data");
         
         java.util.HashMap staticNodeName = new java.util.HashMap();
         staticNodeName.put("additional_data", this.nodeName.getBytes());
         channel.down(new Event(Event.CONFIG, staticNodeName));
         
      }
      else if (address.getAdditionalData() == null)
      {
         Channel testee = channel;
         if (channel instanceof MuxChannel)
         {
            testee = ((MuxChannel) channel).getChannel();
         }
         
         if (testee.isConnected())
         {
            throw new IllegalStateException("Underlying JChannel was " +
                    "connected before additional_data was set");
         }
      }
      else if (this.nodeName == null || "".equals(this.nodeName))
      {         
         this.nodeName = new String(address.getAdditionalData());
         log.warn("Field nodeName was not set but mux channel already had " +
                "additional data -- setting nodeName to " + nodeName);
      }
   }
   
   private String getStateString()
   {
      return ServiceMBean.states[state];
   }

   private String generateUniqueNodeName () throws Exception
   {
      // we first try to find a simple meaningful name:
      // 1st) "local-IP:JNDI_PORT" if JNDI is running on this machine
      // 2nd) "local-IP:JMV_GUID" otherwise
      // 3rd) return a fully GUID-based representation
      //

      // Before anything we determine the local host IP (and NOT name as this could be
      // resolved differently by other nodes...)

      // But use the specified node address for multi-homing

      String hostIP = null;
      InetAddress address = ServerConfigUtil.fixRemoteAddress(nodeAddress);
      if (address == null)
      {
         log.debug ("unable to create a GUID for this cluster, check network configuration is correctly setup (getLocalHost has returned an exception)");
         log.debug ("using a full GUID strategy");
         return new VMID().toString();
      }
      else
      {
         hostIP = address.getHostAddress();
      }

      // 1st: is JNDI up and running?
      //
      if (namingServicePort > 0)
      {
         // we can proceed with the JNDI trick!
         return hostIP + ":" + namingServicePort;
      }
      else
      {
         log.debug("JNDI has been found but the service wasn't started so we cannot " +
                   "be entirely sure we are the only one that wants to use this PORT " +
                   "as a GUID on this host.");
      }

      // 2nd: host-GUID strategy
      //
      String uid = new UID().toString();
      return hostIP + ":" + uid;
   }
   
}
