/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ha.jndi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameClassPair;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;
import javax.naming.NotContextException;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.logging.Logger;
import org.jnp.interfaces.Naming;
import org.jnp.interfaces.NamingContext;
import org.jnp.interfaces.NamingParser;

/**
 *  This class utilizes JBossCache to provide a distributed JNDI implementation.
 *  Lookups will look for Names in HAJNDI then delegate to the local InitialContext.
 *
 *  @author <a href="mailto:jgauthier@novell.com">Jerry Gauthier</a>
 *  @version $Revision: 64282 $
 */
public class TreeHead
   implements org.jnp.interfaces.Naming
{
   static final long serialVersionUID = 6342802270002172451L;

   private static final NamingParser parser = new NamingParser();
   private final Fqn m_root;
   
   // Attributes --------------------------------------------------------
   private static Logger log = Logger.getLogger(TreeHead.class);

   private Cache m_cache;
   
   private transient HAPartition partition;
   private transient HAJNDI father;

   // Constructor --------------------------------------------------------
  
   public TreeHead (Cache cache, Fqn root)
      throws NamingException
   {
      super();
      m_cache = cache;
      m_root = root;
   }

   // Public --------------------------------------------------------

   public void init() throws Exception
   {
      log.debug("HAJNDI registering RPC Handler with HAPartition");
      partition.registerRPCHandler("HAJNDI", this);
      log.debug("initializing HAJNDITreeCache root");
      putTreeRoot();
   }

   public void stop() throws Exception
   {
      
   }

   public void destroy() throws Exception
   {
      log.debug("HAJNDI unregistering RPCHandler with HAPartition");
      partition.unregisterRPCHandler("HAJNDI", this);
   }

   public void setPartition (HAPartition partition)
   {
      this.partition = partition;
   }

   public void setHARMIHead (HAJNDI father)
   {
      this.father = father;
   }

   // Naming implementation -----------------------------------------
  
   public void bind(Name name, Object obj, String className)
      throws NamingException
   {
      if( log.isTraceEnabled() )
         log.trace("bind, name="+name);
      
      internalBind(name, obj, className, false);
   }
   
   public void rebind(Name name, Object obj, String className)
      throws NamingException
   {
      if( log.isTraceEnabled() )
         log.trace("rebind, name="+name);

      internalBind(name, obj, className, true);
   }

   public void unbind(Name name)
      throws NamingException
   {
      if( log.isTraceEnabled() )
         log.trace("unbind, name="+name);
      if (name.isEmpty())
      {
         // Empty names are not allowed
         throw new InvalidNameException();
      }
      
      // is the name a context?
      try
      {
         Fqn temp = new Fqn(m_root, Fqn.fromString(name.toString()));
         // TODO why not jst call remove -- why hasChild first?
         if (m_cache.getRoot().hasChild(temp))
         {
            m_cache.removeNode(temp);
            return;
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
      
      int size = name.size(); 
      
      // get the context and key
      Fqn ctx;
      String key = name.get(size - 1);
      if (size > 1) // find subcontext to which the key is bound
      {  
         String prefix = name.getPrefix(size - 1).toString();
         Fqn fqn = Fqn.fromString(prefix);
         ctx = new Fqn(m_root, fqn); 
      }
      else
         ctx = m_root;
      
      try
      {
         Object removed = m_cache.remove(ctx, key);
         if (removed == null)
         {
            if (!m_cache.getRoot().hasChild(ctx))
            {
               throw new NotContextException(name.getPrefix(size - 1).toString() + " not a context");
            }
            else
            {
               throw new NameNotFoundException(key + " not bound");
            }
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
   }
   
   public Object lookup(Name name)
      throws NamingException
   {
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("lookup, name="+name);
   
      Object result = null;
      if (name.isEmpty())
      {
         // Return this      
         return new NamingContext(null, (Name)(parser.parse("")), getRoot());
      }

      // is the name a context?
      try
      {
         Node n = m_cache.getRoot().getChild(new Fqn(m_root, Fqn.fromString(name.toString())));
         if (n != null)
         {
            Name fullName = (Name)(name.clone());
            return new NamingContext(null, fullName, getRoot());
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
   
      int size = name.size(); 
   
      // get the context and key
      Fqn ctx;
      String key = name.get(size - 1);
      if (size > 1) // find subcontext to which the key is bound
      {  
         String prefix = name.getPrefix(size - 1).toString();
         Fqn fqn = Fqn.fromString(prefix);
         ctx = new Fqn(m_root, fqn); 
      }
      else
         ctx = m_root;
   
      try
      {
         Binding b = (Binding)m_cache.get(ctx, key);
         if (b != null)
         {
            result = b.getObject();
         }
         else
         {
            // key not in cache, try local naming server
            result = internalLookupLocally(name);
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
      return result;
   }

   public Object lookupLocally(Name name) throws NamingException
   {
      boolean trace = log.isTraceEnabled();
      if( trace )
         log.trace("lookupLocally, name="+name);

      // TODO: This is a really big hack here
      // We cannot do InitialContext().lookup(name) because
      // we get ClassNotFound errors and ClassLinkage errors.
      // So, what we do is cheat and get the static localServer variable
      try
      {
         if (NamingContext.localServer != null)
         {
            return NamingContext.localServer.lookup(name);
         }
         else
         {
            InitialContext ctx = new InitialContext();
            return ctx.lookup(name);
         }
      }
      catch (NamingException e)
      {
         if( trace )
            log.trace("lookupLocally failed, name=" + name, e);
         throw e;
      }
      catch (java.rmi.RemoteException e)
      {
         NamingException ne = new NamingException("unknown remote exception");
         ne.setRootCause(e);
         if( trace )
            log.trace("lookupLocally failed, name=" + name, e);
         throw ne;
      }
      catch (RuntimeException e)
      {
         if( trace )
            log.trace("lookupLocally failed, name=" + name, e);
         throw e;
      }
   }
   
   public Collection list(Name name)
      throws NamingException
   {
      if( log.isTraceEnabled() )
         log.trace("list, name="+name);
      Collection result = null;
   
      // get the context
      Fqn ctx;
      String ctxName = "";
      int size = name.size();
      if (size >= 1)
      {  
         ctxName = name.toString();
         Fqn fqn = Fqn.fromString(ctxName);
         ctx = new Fqn(m_root, fqn); 
      }
      else
         ctx = m_root;
      
      boolean exists = m_cache.getRoot().hasChild(ctx);
      if (!exists)
      {  
         try
         {
            return enum2list(new InitialContext().list(name));
         } catch (NamingException e)
         {
            throw new NotContextException(ctxName+ " not a context");
         }
      }
         
      Vector list = null;
      try
      {
         list = new Vector();

         Node base = m_cache.getRoot().getChild(ctx);
         if (base != null)
         {
            Map data = base.getData();
            for (Iterator it = data.values().iterator(); it.hasNext();)
            {               
               Binding b = (Binding) it.next();
               list.addElement(new NameClassPair(b.getName(),b.getClassName(),true));
            }
            
            Set children = base.getChildrenNames();
            if (children != null && !children.isEmpty())
            {
               Iterator iter2 = children.iterator();
               while (iter2.hasNext()) {
                   String node = (String)iter2.next();
                   Name fullName = (Name)(name.clone());
                   fullName.add(node);
                   list.addElement(new NameClassPair(node, NamingContext.class.getName(),true));
               }
            }
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
      return list;
   }

   public Collection listBindings(Name name)
      throws NamingException
   {
      if( log.isTraceEnabled() )
         log.trace("listBindings, name="+name);
      
      // get the context
      Fqn ctx;
      String ctxName = "";
      int size = name.size();
      if (size >= 1)
      {  
         ctxName = name.toString();
         Fqn fqn = Fqn.fromString(ctxName);
         ctx = new Fqn(m_root, fqn); 
      }
      else
         ctx = m_root;
      
      boolean exists = m_cache.getRoot().hasChild(ctx);
      if (!exists)
      {
         // not found in global jndi, look in local.
         try
         {
            return enum2list(new InitialContext().listBindings(name));
         } catch (NamingException e)
         {
            throw new NotContextException(ctxName+ " not a context");
         }
      }
         
      Vector list = null;
      try
      {
         list = new Vector();
         
         Node node = m_cache.getRoot().getChild(ctx);
         if (node != null)
         {
            Map data = node.getData();
            if (data != null && data.size() > 0)
            {
               list.addAll(data.values());
            }
            
            Set children = node.getChildrenNames();
            if (children != null && !children.isEmpty())
            {
               Iterator iter2 = children.iterator();
               while (iter2.hasNext()) {
                   String child = (String)iter2.next();
                   Name fullName = (Name)(name.clone());
                   fullName.add(child);
                   NamingContext subCtx = new NamingContext(null, fullName, getRoot());
                   Binding b = new Binding(child, NamingContext.class.getName(), subCtx, true);
                   list.addElement(b);
               }
            }
         }
      } catch (CacheException ce)
      {
         // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
         NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
         ne.setStackTrace(ce.getStackTrace());
         throw ne;
      }
      return list;
   }

   public javax.naming.Context createSubcontext(Name name)
      throws NamingException
   {
      
      if( log.isTraceEnabled() )
         log.trace("createSubcontext, name="+name);
      int size = name.size();
      if( size == 0 )
         throw new InvalidNameException("Cannot pass an empty name to createSubcontext");

     Context subCtx = null;
     
     // does the new context already exist?
     String str = name.toString();
     Fqn fqn = Fqn.fromString(str);
     Fqn ctx = new Fqn(m_root, fqn);
     if (m_cache.getRoot().hasChild(ctx))
     {
        throw new NameAlreadyBoundException();
     }
     
     // does the prefix context already exist?
     Fqn pctx;
     String newctx = name.get(size - 1);
     if (size > 1) // find subcontext to which the context will be added
     {  
        String prefix = name.getPrefix(size - 1).toString();
        Fqn fqn2 = Fqn.fromString(prefix);
        pctx = new Fqn(m_root, fqn2); 
     }
     else
        pctx = m_root;
     
     boolean exists = m_cache.getRoot().hasChild(pctx);
     if (!exists)
     {
        throw new NotContextException(name.getPrefix(size - 1).toString());
     }

     Fqn newf = new Fqn(pctx, Fqn.fromString(newctx));
     try
     {
        m_cache.put(newf, new HashMap());
     } catch (CacheException ce)
     {
        // don't chain CacheException since JBoss Cache may not be on remote client's classpath 
        NamingException ne = new NamingException(ce.getClass().getName() + ": " + ce.getMessage());
        ne.setStackTrace(ce.getStackTrace());
        throw ne;
     }
   
     Name fullName = (Name) parser.parse("");
     fullName.addAll(name);
     subCtx = new NamingContext(null, fullName, getRoot());

     return subCtx;
   }

   public Naming getRoot ()
   {
      return father.getHAStub();
   }
   
   private void putTreeRoot()
      throws CacheException
   {         
      if (!m_cache.getRoot().hasChild(m_root))
      {   
         m_cache.put(m_root, null);
      }
   }   
  
   private void internalBind(Name name, Object obj, String className, boolean rebind)
      throws NamingException
   {
      if (name.isEmpty())
      {  // Empty names are not allowed
         throw new InvalidNameException();
      }

      int size = name.size();
     
      // get the context and key
      Fqn ctx;
      String key = name.get(size - 1);
      if (size > 1) // find subcontext to which the key will be added
      {  
         String prefix = name.getPrefix(size - 1).toString();
         Fqn fqn = Fqn.fromString(prefix);
         ctx = new Fqn(m_root, fqn); 
      }
      else
         ctx = m_root;
   
      boolean exists = m_cache.getRoot().hasChild(ctx);
      if (!exists)
      {
         throw new NotContextException(name.getPrefix(size - 1).toString() + " not a context");
         // note - NamingServer throws a CannotProceedException if the client attempts to bind 
         //        to a Reference object having an "nns" address.  This implementation simply
         //        throws the NotContextException that's used when "nns" isn't present.
      }
      if (!rebind)
      {
         Node node = m_cache.getRoot().getChild(ctx);
         if (node != null && (node.get(key) != null))
         {
            throw new NameAlreadyBoundException(key);
         }
      }
      try {
         m_cache.put(ctx, key, new Binding(key, className, obj, true));
      }
      catch (Exception e)
      {
         System.out.println(e.toString());
      }     
   }

   private Object internalLookupLocally(Name name)
      throws NamingException
   {
      boolean trace = log.isTraceEnabled();
      Object result = null;
      try
      {
         // not found in global jndi, look in local.
         result = lookupLocally(name);
      }
      catch (NameNotFoundException nnfe)
      {
         // if we get here, this means we need to try on every node.
         Object[] args = new Object[1];
         args[0] = name;
         List rsp = null;
         Exception cause = null;
         try
         {
            if( trace )
               log.trace("calling lookupLocally(" + name + ") on HAJNDI cluster");
            rsp = partition.callMethodOnCluster("HAJNDI", "lookupLocally", args, new Class[]{Name.class}, true);
         }
         catch (Exception ignored)
         {
            if( trace )               
               log.trace("Clustered lookupLocally("+name+") failed", ignored);
            cause = ignored;
         }

         if( trace )
            log.trace("Returned results size: "+ (rsp != null ? rsp.size() : 0));
         if (rsp == null || rsp.size() == 0)
         {
            NameNotFoundException nnfe2 = new NameNotFoundException(name.toString());
            nnfe2.setRootCause(cause);
            throw nnfe2;
         }

         for (int i = 0; i < rsp.size(); i++)
         {
            result = rsp.get(i);
            if( trace )
            {
               String type = (result != null ? result.getClass().getName() : "null");
               log.trace("lookupLocally, i="+i+", value="+result+", type="+type);
            }
            // Ignore null and Exception return values
            if ( result != null && !(result instanceof Exception) )
               return result;
         }
         throw nnfe;
      }
      return result;
   }
   
   private ArrayList enum2list (javax.naming.NamingEnumeration en)
   {
      ArrayList rtn = new ArrayList();
      try
      {
         while (en.hasMore())
         {
            rtn.add(en.next());
         }
         en.close();
      }
      catch (NamingException ignored) {}
      return rtn;
   }
}
