/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.singleton;

import java.net.URI;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.StringTokenizer;

import org.jboss.deployers.client.spi.main.MainDeployer;
import org.jboss.deployers.spi.management.deploy.DeploymentManager;
import org.jboss.deployers.spi.management.deploy.DeploymentProgress;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.system.server.profileservice.DeploymentPhaseVFSScanner;
import org.jboss.virtual.VirtualFile;

/**
 * VFSScanner that exposes methods whereby an HASingletonController
 * can trigger the deployment or undeployment.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1.1 $
 */
public class HASingletonDeploymentScanner extends DeploymentPhaseVFSScanner implements HASingletonDeploymentScannerMBean
{   
   private static final DeploymentPhase DEPLOYMENT_PHASE = DeploymentPhase.APPLICATION;
   
   private MainDeployer mainDeployer;
   
   private DeploymentManager deploymentManager;
   
   private boolean deployed;
   
   private ProfileKey activeProfile;
   
   private Set<String> uploadedNames = new LinkedHashSet<String>();
   
   /**
    * Create a new HASingletonDeploymentScanner.
    *
    */
   public HASingletonDeploymentScanner()
   {
      super(DEPLOYMENT_PHASE);
   }
   
   // ----------------------------------------------------------  Properties
   
   public MainDeployer getMainDeployer()
   {
      return mainDeployer;
   }

   public void setMainDeployer(MainDeployer mainDeployer)
   {
      this.mainDeployer = mainDeployer;
   }

   @Override
   public void setProfileService(ProfileService profileService)
   {
      super.setProfileService(profileService);
      if (profileService != null)
      {
         this.deploymentManager = profileService.getDeploymentManager();
      }
      else
      {
         this.deploymentManager = null;
      }
   }

   // -----------------------------------------------------------------  Public
   
   /**
    * Deploy the content under the given URIs. Called by the 
    * HASingletonController when we become the singleton master.
    * 
    * @param uris comma delimited list of URIs containing content to deploy
    */
   public synchronized void deploySingletons(String uris) throws Exception
   {
      if (!this.deployed)
      {         
         if (this.deploymentManager == null && this.mainDeployer == null)
         {
            throw new IllegalStateException("Configuration must either provide " +
            		"a ProfileService with a DeploymentManager or a MainDeployer");
         }
         
         StringTokenizer tokenizer = new StringTokenizer(uris, ",");
         while (tokenizer.hasMoreTokens())
         {
            String uri = tokenizer.nextToken();
            uri = uri.trim();
            addURI(new URI(uri));
         }
         
         scan();
         
         if (this.deploymentManager == null)
         {
            this.mainDeployer.process();
            // We don't call mainDeployer.checkComplete() as we may be in the
            // middle of the initial startup deployments; i.e. not expected to be complete 
         }
         else
         {
            String[] allUploaded = getAllUploadedNames();
            DeploymentProgress progress = this.deploymentManager.start(DEPLOYMENT_PHASE, allUploaded);
            progress.run();
         }
         
         this.deployed = true;
      }
   }

   /**
    * Undeploy the content under the given URIs. Called by the 
    * HASingletonController when we are no longer the singleton master.
    * 
    * @param uris comma delimited list of URIs containing content to deploy
    */
   public synchronized void undeploySingletons(String uris) throws Exception
   {
      if (this.deployed)
      {         
         if (this.deploymentManager == null && this.mainDeployer == null)
         {
            throw new IllegalStateException("Configuration must either provide " +
                    "a ProfileService with a DeploymentManager or a MainDeployer");
         }
         
         StringTokenizer tokenizer = new StringTokenizer(uris, ",");
         while (tokenizer.hasMoreTokens())
         {
            String uri = tokenizer.nextToken();
            uri = uri.trim();
            removeURI(new URI(uri));
         }      
         
         scan();
         
         if (this.deploymentManager == null)
         {
            this.mainDeployer.process();
            // We don't call mainDeployer.checkComplete() as we may be in the
            // middle of the main shutdown startup deployments; i.e. not expected to be complete 
         }
         else
         {
            String[] allUploaded = getAllUploadedNames();
            
            this.uploadedNames.clear();
            
            DeploymentProgress progress = this.deploymentManager.stop(DEPLOYMENT_PHASE, allUploaded);
            progress.run();
            progress = this.deploymentManager.undeploy(DEPLOYMENT_PHASE, allUploaded);
            progress.run();
         }
         
         this.deployed = false;
      }
   }
   
   // ----------------------------------------------------  Protected Overrides

   @Override
   protected VFSDeployment add(Profile profile, VirtualFile file) throws Exception
   {
      VFSDeployment deployment = createDeployment(file);
      if (this.deploymentManager == null)
      {         
         if(profile.hasDeployment(deployment.getName(), DEPLOYMENT_PHASE) == false)
         {
            profile.addDeployment(deployment, DEPLOYMENT_PHASE);
            this.mainDeployer.addDeployment(deployment);
         }
      }
      else 
      {
         activateProfile(profile.getKey());
         
         String name = file.getPathName();
         DeploymentProgress progress = this.deploymentManager.distribute(name, DEPLOYMENT_PHASE, file.toURL());
         progress.run();
         
         String[] repoNames = progress.getDeploymentID().getRepositoryNames();
         if (repoNames != null)
         {
            for (String repoName : repoNames)
            {
               uploadedNames.add(repoName);
            }
         }
      }
      return deployment;
   }

   @Override
   protected void remove(Profile profile, String name) throws Exception
   {
      if(this.deploymentManager == null && profile.hasDeployment(name, DEPLOYMENT_PHASE))
      {
         profile.removeDeployment(name, DEPLOYMENT_PHASE);
         this.mainDeployer.removeDeployment(name);
      }
      // else we deal with everything at the end of undeploySingletons
   }
   
   // ----------------------------------------------------------------  Private

   private void activateProfile(ProfileKey key) throws Exception
   {
      if (activeProfile != key)
      {
         this.deploymentManager.loadProfile(key);
         activeProfile = key;
      }      
   }
   
   private String[] getAllUploadedNames()
   {
      String[] result = new String[this.uploadedNames.size()];
      return this.uploadedNames.toArray(result);
   }
   
}
