/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ha.singleton;

import java.util.List;

import org.jboss.ha.framework.interfaces.ClusterNode;
import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.system.ServiceMBeanSupport;

/**
 * A base class for policy service that decides which node in the cluster should be 
 * the master node to run certain HASingleton service.
 * 
 * @author <a href="mailto:afu@novell.com">Alex Fu</a>
 * @author <a href="mailto:galder.zamarreno@jboss.com">Galder Zamarreno</a> 
 * @version $Revision: 69774 $
 */
public abstract class HASingletonElectionPolicyBase 
   extends ServiceMBeanSupport
   implements HASingletonElectionPolicyMBean
{
   private String mSingletonName;
   private HAPartition mPartition;
   private String mPreferredMaster;
   
   /**
    * @see HASingletonElectionPolicy#setSingletonName
    */
   public void setSingletonName(String singletonName)
   {
      this.mSingletonName = singletonName;
      log.debug("set singleton name to " + this.mSingletonName);
   }   
   
   /**
    * @see HASingletonElectionPolicy#getSingletonName
    */
   public String getSingletonName()
   {
      return this.mSingletonName;
   }
   
   /**
    * @see HASingletonElectionPolicy#setPreferredMaster(ClusterNode)
    */
   public void setPreferredMaster(String node)
   {
      this.mPreferredMaster = node;
   }
   
   /**
    * @see HASingletonElectionPolicy#getPreferredMaster()
    */
   public String getPreferredMaster()
   {
      return this.mPreferredMaster;
   }
   
   /**
    * @see HASingletonElectionPolicy#setHAPartition(HAPartition)
    */
   public void setHAPartition(HAPartition partition)
   {
      this.mPartition = partition;
   }

   /**
    * @see HASingletonElectionPolicy#getHAPartition()
    */
   public HAPartition getHAPartition()
   {
      return this.mPartition;
   }
      
   /**
    * @see HASingletonElectionPolicy#elect()
    */
   public ClusterNode elect() 
   {
      List<ClusterNode> candidates = getCandidates();
      if (candidates == null)
      {
         log.debug("list of cluster node candidates where to run the singleton is null");
         return null;
      }
             
      return elect(candidates);    
   }
   
   /**
    * Get the list of candidate {@link ClusterNode} instances where the 
    * singleton could be deployed.
    * 
    * @return List of {@link ClusterNode} instances.
    */
   protected List<ClusterNode> getCandidates() 
   {
      return getHAPartition().getDistributedReplicantManager().lookupReplicantsNodes(getSingletonName());
   }
   
   /**
    * Given a List of candidate {@link ClusterNode} instances, return the 
    * elected node where the singleton should run.
    * 
    * @param candidates List of {@link ClusterNode}.
    * @return {@link ClusterNode} instance.
    */
   protected abstract ClusterNode elect(List<ClusterNode> candidates);
}
