/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.resource.deployment;

import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Iterator;

import javax.resource.spi.ResourceAdapter;

import org.jboss.deployment.DeploymentException;
import org.jboss.resource.metadata.ConfigPropertyMetaData;
import org.jboss.resource.metadata.ConnectorMetaData;
import org.jboss.resource.metadata.RARDeploymentMetaData;

/**
 * A resource adapter factory
 *
 * @author  <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="weston.price@jboss.com">Weston Price</a>
 * @version $Revision: 65366 $
 */
public class ResourceAdapterFactory
{
   /** The dummy resource adapter for old deployment */
   public static final String DUMMY_RA_CLASS = DummyResourceAdapter.class.getName();
   
   /**
    * Create a new ResourceAdapter
    * 
    * @param ramd the ResourceAdapterMetaData
    * @return the resource adapter
    * @throws Exception
    */
   public static ResourceAdapter createResourceAdapter(RARDeploymentMetaData ramd) throws Exception
   {
      ResourceAdapter adapter = createResourceAdapter(ramd.getConnectorMetaData());
      
      for(Iterator iter = ramd.getRaXmlMetaData().getProperties().iterator(); iter.hasNext();)
      {
         ConfigPropertyMetaData cpmd = (ConfigPropertyMetaData)iter.next();
         applyProperty(cpmd, adapter.getClass(), adapter);
         
      }
            
      return adapter;
   }
   
   /**
    * Create a new resource adapter
    * 
    * @param cmd the connector meta data
    * @throws Exception for any error
    */
   public static ResourceAdapter createResourceAdapter(ConnectorMetaData cmd) throws Exception
   {
      // Determine the resource adapter class
      String className = cmd.getRAClass();
      if (className == null)
         className = DUMMY_RA_CLASS;
      
      // Load the class
      Class raClass = Thread.currentThread().getContextClassLoader().loadClass(className);
      if (ResourceAdapter.class.isAssignableFrom(raClass) == false)
         throw new DeploymentException(raClass.getName() + " is not a resource adapter class");
      ResourceAdapter result = (ResourceAdapter) raClass.newInstance();
      
      // Apply the properties
      for (Iterator i = cmd.getProperties().iterator(); i.hasNext();)
      {
         ConfigPropertyMetaData cpmd = (ConfigPropertyMetaData) i.next();
         applyProperty(cpmd, raClass, result);
         
         
      }
      
      return result;
   }

   private static void applyProperty(ConfigPropertyMetaData cpmd, Class clz, ResourceAdapter ra) throws Exception
   {
      
      String name = cpmd.getName();
      String type = cpmd.getType();
      String value = cpmd.getValue();
      
      Class clazz = Thread.currentThread().getContextClassLoader().loadClass(type);
      PropertyEditor editor = PropertyEditorManager.findEditor(clazz);
      if (editor == null)
         throw new IllegalArgumentException("No property editor found for property " + cpmd);
      editor.setAsText(value);
      Object object = editor.getValue();
      
      try
      {
         String setter = "set" + Character.toUpperCase(name.charAt(0));
         if (name.length() > 1)
            setter = setter.concat(name.substring(1));
         Method method = clz.getMethod(setter, new Class[] { clazz });
         method.invoke(ra, new Object[] { object });
      }
      catch (InvocationTargetException e)
      {
         DeploymentException.rethrowAsDeploymentException("Error for resource adapter class " + clz.getName() + " setting property " + cpmd, e.getTargetException());
      }
      catch (Throwable t)
      {
         DeploymentException.rethrowAsDeploymentException("Error for resource adapter class " + clz.getName() + " accessing property setter " + cpmd, t);
      }
      
   }

}
