/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.hibernate.cache;

import java.util.Properties;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.transaction.TransactionManager;

import org.hibernate.cache.Cache;
import org.hibernate.cache.CacheException;
import org.hibernate.cache.CacheProvider;
import org.jboss.cache.jmx.CacheJmxWrapperMBean;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.tm.TransactionManagerLocator;

/**
 * A Hibernate CacheProvider implementation which knows how to
 * obtain a deployed JBossCache via its JMX ObjectName.
 *
 * @version <tt>$Revision: 58543 $</tt>
 * @author <a href="mailto:steve@hibernate.org">Steve Ebersole</a>
 */
public class DeployedTreeCacheProvider implements CacheProvider
{
   public static final String OBJECT_NAME_PROP = "hibernate.treecache.objectName";
   public static final String DEFAULT_OBJECT_NAME = "jboss.cache:service=HibernateTreeCache";

   private org.jboss.cache.Cache deployedTreeCache;

   public void start(Properties properties) throws CacheException
   {
      // Determine the TreeCache MBean ObjectName.
      String configObjectName = properties.getProperty( OBJECT_NAME_PROP, DEFAULT_OBJECT_NAME );
      ObjectName objectName;
      try
      {
         objectName = new ObjectName( configObjectName );
      }
      catch( Throwable t )
      {
         throw new CacheException( "Malformed TreeCache ObjectName");
      }

      CacheJmxWrapperMBean mbean;
      try
      {
         MBeanServer server = MBeanServerLocator.locateJBoss();
         mbean = (CacheJmxWrapperMBean) MBeanProxyExt.create(CacheJmxWrapperMBean.class, objectName, server);
         deployedTreeCache = mbean.getCache();
      }
      catch( Throwable t )
      {
         throw new CacheException( "Unable to locate TreeCache MBean under object name [" + configObjectName + "]" );
      }
   }

   public void stop()
   {
      deployedTreeCache = null;
   }

   public boolean isMinimalPutsEnabledByDefault()
   {
      return true;
   }

   /**
    * Called by Hibernate in order to build the given named cache "region".
    *
    * @param name The cache "region" name.
    * @param properties The configuration properties.
    * @return The constructed Cache wrapper around the jndi-deployed TreeCache.
    * @throws CacheException Generally indicates a problem locating the TreeCache.
    */
   public Cache buildCache(String name, Properties properties) throws CacheException
   {
      TransactionManager tm = TransactionManagerLocator.getInstance().locate(); 
      return new JBCCache( deployedTreeCache, name, tm );
   }

   public long nextTimestamp()
   {
		return System.currentTimeMillis() / 100;
   }
}
