/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.integration;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.concurrent.ConcurrentHashMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.auth.callback.CallbackHandler;

import org.jboss.logging.Logger;
import org.jboss.security.AuthenticationManager;
import org.jboss.security.AuthorizationManager;
import org.jboss.security.ISecurityManagement;
import org.jboss.security.SecurityConstants;
import org.jboss.security.audit.AuditManager;
import org.jboss.security.auth.callback.SecurityAssociationHandler;
import org.jboss.security.config.SecurityConfiguration;
import org.jboss.security.identitytrust.IdentityTrustManager;
import org.jboss.security.mapping.MappingManager;
import org.jboss.security.plugins.JaasSecurityDomain;
import org.jboss.security.plugins.SecurityDomainContext;
import org.jboss.util.CachePolicy;
import org.jboss.util.TimedCachePolicy;
 
/**
 *  JNDI Based Security Management
 *  @author Anil.Saldhana@redhat.com
 *  @since  Sep 9, 2007 
 *  @version $Revision$
 */ 
public class JNDIBasedSecurityManagement implements ISecurityManagement
{ 
   private static final long serialVersionUID = 1L;

   protected static Logger log = Logger.getLogger(JNDIBasedSecurityManagement.class);
   
   static transient ConcurrentHashMap<String,SecurityDomainContext> securityMgrMap = new ConcurrentHashMap<String,SecurityDomainContext>();
    
   protected String BASE_CTX = SecurityConstants.JAAS_CONTEXT_ROOT; 
   
   protected String authenticationMgrClass = "org.jboss.security.plugins.JaasSecurityManager";
   
   protected String authorizationMgrClass = "org.jboss.security.plugins.JBossAuthorizationManager";
   
   protected String auditMgrClass = "org.jboss.security.plugins.audit.JBossAuditManager";
   
   protected String identityTrustMgrClass = "org.jboss.security.plugins.identitytrust.JBossIdentityTrustManager";
   
   protected String mappingMgrClass = "org.jboss.security.plugins.mapping.JBossMappingManager";
   
   protected CallbackHandler callBackHandler = new SecurityAssociationHandler(); 
   
   /** Enable the IdentityTrust feature */
   protected boolean enableIdentity = false;
   
   /** Enable the Audit feature */
   protected boolean enableAudit = true; 
   
   private CachePolicy cachePolicy = null;
   
   private transient ConcurrentHashMap<String,AuthenticationManager> authMgrMap = null;
   private transient ConcurrentHashMap<String,AuthorizationManager> authzMgrMap = null;
   private transient ConcurrentHashMap<String,MappingManager> mappingMgrMap = null;
   private transient ConcurrentHashMap<String,AuditManager> auditMgrMap = null;
   private transient ConcurrentHashMap<String,IdentityTrustManager> idmMgrMap = null;
   
   public JNDIBasedSecurityManagement()
   {   
      initializeMaps();
   } 
   
   public AuditManager getAuditManager(String securityDomain)
   { 
      initializeMaps();
      AuditManager auditManager = null;
      try
      { 
         if(this.enableAudit)
         {
            auditManager = this.auditMgrMap.get(securityDomain);
            if(auditManager == null)
            {
               auditManager = (AuditManager) lookUpJNDI(securityDomain + "/auditMgr");
               this.auditMgrMap.put(securityDomain, auditManager); 
            } 
         }  
      }
      catch(Exception e)
      {
         log.trace("Exception in getting audit mgr", e); 
      }
      return auditManager;
   }

   public AuthenticationManager getAuthenticationManager(String securityDomain)
   {
      initializeMaps();
      AuthenticationManager am = null;
      try
      {
         am = this.authMgrMap.get(securityDomain);
         if(am == null)
         {
            am = (AuthenticationManager) lookUpJNDI(securityDomain + "/authenticationMgr");
            this.authMgrMap.put(securityDomain, am); 
         }
      }
      catch(Exception e)
      {
         log.trace("Exception in getting authentication mgr "
               + " for domain="+securityDomain , e );
      }
      return am;
   }

   public AuthorizationManager getAuthorizationManager(String securityDomain)
   {
      initializeMaps();
      AuthorizationManager am = null;
      try
      {
         am = this.authzMgrMap.get(securityDomain);
         if(am == null)
         {
            am = (AuthorizationManager) lookUpJNDI(securityDomain + "/authorizationMgr");
            this.authzMgrMap.put(securityDomain, am);
         }
      }
      catch(Exception e)
      {
         log.trace("Exception in getting authorization mgr", e);
      }
      return am;
   }

   public IdentityTrustManager getIdentityTrustManager(String securityDomain)
   {
      initializeMaps();
      IdentityTrustManager am = null;
      try
      {
         if(this.enableIdentity)
         {
            am = this.idmMgrMap.get(securityDomain);
            if(am == null)
            {
               am = (IdentityTrustManager) lookUpJNDI(securityDomain + "/identityTrustMgr");
               this.idmMgrMap.put(securityDomain, am); 
            } 
         }
      }
      catch(Exception e)
      {
         log.trace("Exception in getting IdentityTrustManager", e);
      }
      return am;
   }

   public MappingManager getMappingManager(String securityDomain)
   {
      initializeMaps();
      MappingManager am = null;
      try
      {
         am = this.mappingMgrMap.get(securityDomain);
         if(am == null)
         {
            am = (MappingManager) lookUpJNDI(securityDomain + "/mappingMgr");
            this.mappingMgrMap.put(securityDomain, am); 
         }
      }
      catch(Exception e)
      {
         log.trace("Exception in getting MappingManager", e);
      }
      return am;
   }
       
   public void setAuthenticationMgrClass(String authenticationMgrClass)
   {
      this.authenticationMgrClass = authenticationMgrClass;
   }

   public void setAuthorizationMgrClass(String authorizationMgrClass)
   {
      this.authorizationMgrClass = authorizationMgrClass;
   }

   public void setAuditMgrClass(String auditMgrClass)
   {
      this.auditMgrClass = auditMgrClass;
   } 

   public void setEnableAudit(boolean enableAudit)
   {
      this.enableAudit = enableAudit;
   }
   
   public void setEnableIdentity(boolean enableIdentity)
   {
      this.enableIdentity = enableIdentity;
   }

   public void setIdentityTrustMgrClass(String identityTrustMgrClass)
   {
      this.identityTrustMgrClass = identityTrustMgrClass;
   }

   public void setMappingMgrClass(String mappingMgrClass)
   {
      this.mappingMgrClass = mappingMgrClass;
   }

   public void setCallBackHandler(CallbackHandler callBackHandler)
   {
      this.callBackHandler = callBackHandler;
   }

   public void setBaseContext(String ctx)
   {
      if(ctx == null)
         throw new IllegalArgumentException("ctx is null");
      this.BASE_CTX = ctx;
   }
     
   public void setCachePolicy(CachePolicy cp)
   {
      this.cachePolicy = cp;
   } 
   
   /** Set the indicated security domain cache timeout. This only has an
   effect if the security domain is using the default jboss TimedCachePolicy
   implementation.

   @param securityDomain the name of the security domain cache
   @param timeoutInSecs - the cache timeout in seconds.
   @param resInSecs - resolution of timeouts in seconds.
   */
  public static void setCacheTimeout(String securityDomain, int timeoutInSecs, int resInSecs)
  {
     SecurityDomainContext securityDomainCtx = (SecurityDomainContext) securityMgrMap.get(securityDomain);
     if(securityDomainCtx == null)
     {
      try
      {
         String lookupStr = SecurityConstants.JAAS_CONTEXT_ROOT + "/" + securityDomain;
         securityDomainCtx = (SecurityDomainContext) new InitialContext().lookup(lookupStr);
         securityMgrMap.put(securityDomain, securityDomainCtx);
      }
      catch (NamingException e)
      {
         log.trace("SetCacheTimeOut:Failed to look up SecurityDomainCtx:"+securityDomain);
      }  
     }
     if(securityDomainCtx != null)
     {
        CachePolicy cache = securityDomainCtx.getAuthenticationCache(); 
        if( cache != null && cache instanceof TimedCachePolicy )
        {
           TimedCachePolicy tcp = (TimedCachePolicy) cache;
           synchronized( tcp )
           {
              tcp.setDefaultLifetime(timeoutInSecs);
              tcp.setResolution(resInSecs);
           }
        }
        else
        {
           log.warn("Failed to find cache policy for securityDomain='"
              + securityDomain + "'");
        } 
     }
  } 
   
   public static void setDefaultCacheTimeout(int defaultCacheTimeout)
   {
      SecurityConstantsBridge.defaultCacheTimeout = defaultCacheTimeout;
   }

   public static void setDefaultCacheResolution(int defaultCacheResolution)
   {
      SecurityConstantsBridge.defaultCacheResolution = defaultCacheResolution;
   }

   public SecurityDomainContext createSecurityDomainContext(String domain) throws Exception
   {   
      log.debug("Creating SDC for domain="+domain);
      AuthenticationManager am = createAuthenticationManager(domain);
      if(cachePolicy == null)
      {
         cachePolicy = createDefaultCachePolicy();
      }
      //Set security cache if the auth manager implementation supports it
      setSecurityDomainCache(am, cachePolicy);
      //Set DeepCopySubject option if supported
      if(SecurityConfiguration.isDeepCopySubjectMode())
      {
        setDeepCopySubjectMode(am);  
      }
      
      SecurityDomainContext securityDomainContext = new SecurityDomainContext(am, cachePolicy); 
      
      securityDomainContext.setAuthorizationManager(createAuthorizationManager(domain));
      securityDomainContext.setAuditMgr(createAuditManager(domain));
      securityDomainContext.setIdentityTrustMgr(createIdentityTrustManager(domain));
      securityDomainContext.setMappingMgr(createMappingManager(domain));
      return securityDomainContext;
   }
   
   /**
    * Legacy registration of JaasSecurityDomain instance with the JNDI
    * Object Factory internal hashmap
    * @param domain
    * @param jsd
    * @throws Exception
    */
   public void registerJaasSecurityDomainInstance(String domain, JaasSecurityDomain jsd) throws Exception
   {
      SecurityDomainContext sdc = (SecurityDomainContext) securityMgrMap.get(domain);
      if(sdc != null)
      {
         sdc.setAuthenticationManager(jsd);
      }
      else
      {
         sdc = createSecurityDomainContext(domain);
         sdc.setAuthenticationManager(jsd);
      }
      securityMgrMap.put(domain, sdc);
   }  
   
   /**
    * Legacy deregistration of JaasSecurityDomain instance with the JNDI
    * Object Factory internal hashmap
    * @param domain
    * @param jsd
    * @throws Exception
    */
   public void deregisterJaasSecurityDomainInstance(String domain, 
         JaasSecurityDomain jsd)
   {
      securityMgrMap.remove(domain); 
   }  
   
   // Private Methods
 
   private Object lookUpJNDI(String ctxName) 
   {
      Object result = null;
      try
      { 
         Context ctx = new InitialContext();
         if(ctxName.startsWith(BASE_CTX))
            result = ctx.lookup(ctxName);
         else
            result = ctx.lookup(BASE_CTX + "/" + ctxName);  
      }
      catch(Exception e)
      {
         log.trace("Look up of JNDI for " + ctxName + " failed with "+ e.getLocalizedMessage());
         return null;
      }
      return result;
   }
   
   private AuthenticationManager createAuthenticationManager(String securityDomain) throws Exception
   {
      Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(authenticationMgrClass);
      Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class, CallbackHandler.class});
      return (AuthenticationManager) ctr.newInstance(new Object[]{ securityDomain, callBackHandler});
   }
   
   private AuthorizationManager createAuthorizationManager(String securityDomain) throws Exception
   {
      Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(authorizationMgrClass);
      Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class});
      return (AuthorizationManager) ctr.newInstance(new Object[]{ securityDomain});
   }
   
   private AuditManager createAuditManager(String securityDomain) throws Exception
   {
      Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(auditMgrClass);
      Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class});
      return (AuditManager) ctr.newInstance(new Object[]{ securityDomain});
   }
   
   private MappingManager createMappingManager(String securityDomain) throws Exception
   {
      Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(mappingMgrClass);
      Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class});
      return (MappingManager) ctr.newInstance(new Object[]{ securityDomain});
   }
   
   private IdentityTrustManager createIdentityTrustManager(String securityDomain) throws Exception
   {
      Class<?> clazz = SecurityActions.getContextClassLoader().loadClass(identityTrustMgrClass);
      Constructor<?> ctr = clazz.getConstructor(new Class[] { String.class});
      return (IdentityTrustManager) ctr.newInstance(new Object[]{ securityDomain});
   }
   
   /** Use reflection to attempt to set the authentication cache on the
    * securityMgr argument.
    * @param securityMgr the security manager
    * @param cachePolicy the cache policy implementation
    */
   private static void setSecurityDomainCache(AuthenticationManager securityMgr,
      CachePolicy cachePolicy)
   {
      try
      {
         Class<?>[] setCachePolicyTypes = {CachePolicy.class};
         Method m = securityMgr.getClass().getMethod("setCachePolicy", setCachePolicyTypes);
         Object[] setCachePolicyArgs = {cachePolicy};
         m.invoke(securityMgr, setCachePolicyArgs);
         log.debug("setCachePolicy, c="+setCachePolicyArgs[0]);
      }
      catch(Exception e2)
      {    
         if(log.isTraceEnabled())
            log.trace("Optional setCachePolicy failed" + e2.getLocalizedMessage());
      }
   }
   
   /** Use reflection to attempt to set the authentication cache on the
    * securityMgr argument.
    * @param securityMgr the security manager
    * @param cachePolicy the cache policy implementation
    */
   private static void setDeepCopySubjectMode(AuthenticationManager securityMgr)
   {
      try
      {
         Class<?>[] argsType = {Boolean.class};
         Method m = securityMgr.getClass().getMethod("setDeepCopySubjectOption", argsType);
         Object[] deepCopyArgs = {Boolean.TRUE};
         m.invoke(securityMgr, deepCopyArgs);
         log.trace("setDeepCopySubjectOption, option="+deepCopyArgs[0]);
      }
      catch(Exception e2)
      {    
         if(log.isTraceEnabled())
            log.trace("Optional setDeepCopySubjectMode failed" + e2.getLocalizedMessage());
      }
   }
   
   /**
    * Create a Default Cache Policy
    * @return
    */
   private CachePolicy createDefaultCachePolicy()
   {
      TimedCachePolicy cachePolicy = 
          new TimedCachePolicy(SecurityConstantsBridge.defaultCacheTimeout,
                               true, 
                               SecurityConstantsBridge.defaultCacheResolution);
      cachePolicy.create();
      cachePolicy.start();
      return cachePolicy; 
   } 
   
   /**
    * Since the maps are transient, initialize them
    */
   private void initializeMaps()
   {
      if(authMgrMap == null)
         authMgrMap = new ConcurrentHashMap<String,AuthenticationManager>();
      if(authzMgrMap == null)
         authzMgrMap = new ConcurrentHashMap<String,AuthorizationManager>();
      if(mappingMgrMap == null)
         mappingMgrMap = new ConcurrentHashMap<String,MappingManager>();
      if(auditMgrMap == null)
         auditMgrMap = new ConcurrentHashMap<String,AuditManager>();
      if(idmMgrMap == null)
         idmMgrMap = new ConcurrentHashMap<String,IdentityTrustManager>(); 
   }
}