/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployment;

import java.io.IOException;
import java.lang.reflect.AnnotatedElement;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeployer;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.annotation.creator.client.ApplicationClient5MetaDataCreator;
import org.jboss.metadata.annotation.creator.ejb.EjbJar30Creator;
import org.jboss.metadata.annotation.creator.web.Web25MetaDataCreator;
import org.jboss.metadata.annotation.finder.AnnotationFinder;
import org.jboss.metadata.annotation.finder.DefaultAnnotationFinder;
import org.jboss.metadata.client.jboss.JBossClientMetaData;
import org.jboss.metadata.client.spec.ApplicationClient5MetaData;
import org.jboss.metadata.client.spec.ApplicationClientMetaData;
import org.jboss.metadata.ejb.spec.EjbJar30MetaData;
import org.jboss.metadata.ejb.spec.EjbJar3xMetaData;
import org.jboss.metadata.ejb.spec.EjbJarMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.metadata.web.spec.Web25MetaData;
import org.jboss.metadata.web.spec.WebMetaData;
import org.jboss.virtual.VirtualFile;

/**
 * A POST_CLASSLOADER deployer which generates metadata from
 * annotations
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 69052 $
 */
public class AnnotationMetaDataDeployer extends AbstractDeployer
{
   public static final String EJB_ANNOTATED_ATTACHMENT_NAME = "annotated."+EjbJarMetaData.class.getName();
   public static final String CLIENT_ANNOTATED_ATTACHMENT_NAME = "annotated."+ApplicationClientMetaData.class.getName();
   public static final String WEB_ANNOTATED_ATTACHMENT_NAME = "annotated."+WebMetaData.class.getName();

   private boolean metaDataCompleteIsDefault = true;

   public AnnotationMetaDataDeployer()
   {
      setStage(DeploymentStages.POST_CLASSLOADER);
      addOutput(EJB_ANNOTATED_ATTACHMENT_NAME);
      addOutput(CLIENT_ANNOTATED_ATTACHMENT_NAME);
      addOutput(WEB_ANNOTATED_ATTACHMENT_NAME);
   }

   
   public boolean isMetaDataCompleteIsDefault()
   {
      return metaDataCompleteIsDefault;
   }
   public void setMetaDataCompleteIsDefault(boolean metaDataCompleteIsDefault)
   {
      this.metaDataCompleteIsDefault = metaDataCompleteIsDefault;
   }

   public void deploy(DeploymentUnit unit) throws DeploymentException
   {
      if (unit instanceof VFSDeploymentUnit == false)
         return;
      
      VFSDeploymentUnit vfsDeploymentUnit = (VFSDeploymentUnit) unit;
      deploy(vfsDeploymentUnit);
   }

   public void undeploy(DeploymentUnit unit)
   {
      if (unit instanceof VFSDeploymentUnit == false)
         return;
      
      VFSDeploymentUnit vfsDeploymentUnit = (VFSDeploymentUnit) unit;
      undeploy(vfsDeploymentUnit);
   }

   /**
    * Deploy a vfs deployment
    * 
    * @param unit the unit
    * @throws DeploymentException for any error
    */
   protected void deploy(VFSDeploymentUnit unit)
      throws DeploymentException
   {
      // Ignore any metadata complete deployments
      boolean isComplete = metaDataCompleteIsDefault;
      /* Since we haven't merged yet, JBossMetaData is not the authority on isMetadataComplete
      JBossMetaData jbossMetaData = unit.getAttachment(JBossMetaData.class);
      if(jbossMetaData != null)
         isComplete &= jbossMetaData.isMetadataComplete();
      */
      EjbJarMetaData ejbJarMetaData = unit.getAttachment(EjbJarMetaData.class);
      if(ejbJarMetaData != null && ejbJarMetaData instanceof EjbJar3xMetaData)
         isComplete &= ((EjbJar3xMetaData) ejbJarMetaData).isMetadataComplete();
      JBossWebMetaData jbossWebMetaData = unit.getAttachment(JBossWebMetaData.class);
      if(jbossWebMetaData != null)
         isComplete &= jbossWebMetaData.isMetadataComplete();
      JBossClientMetaData jbossClientMetaData = unit.getAttachment(JBossClientMetaData.class);
      if(jbossClientMetaData != null)
         isComplete &= jbossClientMetaData.isMetadataComplete();
      if(isComplete)
         return;

      VirtualFile root = unit.getRoot();
      boolean isLeaf = true;
      try
      {
         isLeaf = root.isLeaf();
      }
      catch(IOException ignore)
      {
      }
      if(isLeaf == true)
         return;

      ClassLoader loader = unit.getClassLoader();
      List<VirtualFile> classpath = unit.getClassPath();
      if(classpath == null || classpath.isEmpty())
         return;

      boolean trace = log.isTraceEnabled();
      if (trace)
         log.trace("Deploying annotations for unit: " + unit + ", classpath: " + classpath);

      try
      {
         Map<VirtualFile, Class<?>> classpathClasses = new HashMap<VirtualFile, Class<?>>();
         for(VirtualFile path : classpath)
         {
            AnnotatedClassFilter classVisitor = new AnnotatedClassFilter(unit, loader, path);
            path.visit(classVisitor);
            Map<VirtualFile, Class<?>> classes = classVisitor.getAnnotatedClasses();
            if(classes != null && classes.size() > 0)
            {
               if(trace)
                  log.trace("Annotated classes: "+classes);
               classpathClasses.putAll(classes);
            }
         }

         if(classpathClasses.size() > 0)
         {
            AnnotationFinder<AnnotatedElement> finder = new DefaultAnnotationFinder<AnnotatedElement>();
            if(jbossWebMetaData != null)
               processJBossWebMetaData(unit, finder, classpathClasses);
            else if(jbossClientMetaData != null)
               processJBossClientMetaData(unit, finder, classpathClasses);
            else
               processJBossMetaData(unit, finder, classpathClasses);
         }
      }
      catch(IOException e)
      {
         throw new DeploymentException(e);
      }
   }

   /**
    * Undeploy a vfs deployment
    * 
    * @param unit the unit
    */
   protected void undeploy(VFSDeploymentUnit unit)
   {
      // Nothing
   }

   /**
    * Process annotations.
    *
    * @param unit the deployment unit
    * @param finder the annotation finder
    * @param classes the candidate classes
    */
   protected void processJBossMetaData(VFSDeploymentUnit unit,
         AnnotationFinder<AnnotatedElement> finder, Map<VirtualFile, Class<?>> classes)
   {
      // Create the metadata model from the annotations
      EjbJar30Creator creator = new EjbJar30Creator(finder);
      EjbJar30MetaData annotationMetaData = creator.create(classes.values());
      if(annotationMetaData != null)
         unit.addAttachment(EJB_ANNOTATED_ATTACHMENT_NAME, annotationMetaData, EjbJarMetaData.class);
   }

   /**
    * Process annotations.
    *
    * @param unit the deployment unit
    * @param finder the annotation finder
    * @param classes the candidate classes
    */
   protected void processJBossWebMetaData(VFSDeploymentUnit unit,
         AnnotationFinder<AnnotatedElement> finder, Map<VirtualFile, Class<?>> classes)
   {
      Web25MetaDataCreator creator = new Web25MetaDataCreator(finder);
      Web25MetaData annotationMetaData = creator.create(classes.values());
      if(annotationMetaData != null)
         unit.addAttachment(WEB_ANNOTATED_ATTACHMENT_NAME, annotationMetaData, WebMetaData.class);
   }

   /**
    * Process annotations.
    *
    * @param unit the deployment unit
    * @param finder the annotation finder
    * @param classes the candidate classes
    */
   protected void processJBossClientMetaData(VFSDeploymentUnit unit,
         AnnotationFinder<AnnotatedElement> finder, Map<VirtualFile, Class<?>> classes)
   {
      ApplicationClient5MetaDataCreator creator = new ApplicationClient5MetaDataCreator(finder);
      ApplicationClient5MetaData annotationMetaData = creator.create(classes.values());
      if(annotationMetaData != null)
         unit.addAttachment(CLIENT_ANNOTATED_ATTACHMENT_NAME, annotationMetaData, ApplicationClientMetaData.class);      
   }
}

