/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.deployment.security;

import javax.security.jacc.PolicyConfiguration;
import javax.security.jacc.PolicyConfigurationFactory;
import javax.security.jacc.PolicyContextException;

import org.jboss.logging.Logger;

//$Id$

/**
 *  A Service Bean representing the JACC Policy for the top level deployment
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Dec 11, 2006 
 *  @version $Revision$
 */
public class JaccPolicy implements JaccPolicyMBean
{ 
   private static Logger log = Logger.getLogger(JaccPolicy.class);
   private boolean trace = log.isTraceEnabled();
   
   private PolicyConfiguration parentPC = null;
   private String contextID = null;  
    
   private Object metaData; 
   
   private Boolean standaloneDeployment = Boolean.FALSE;
    
   
   /**
    * Ctr
    * @param id Jacc Context Id for the top level deployment
    * @throws IllegalArgumentException if id passed is null
    */
   public JaccPolicy(String id)
   {
      if(id == null)
         throw new IllegalArgumentException("Jacc Context Id passed is null");
      this.contextID = id;  
   }
   
   public JaccPolicy(String id, Object metaData, Boolean standaloneDeployment)
   {
      this(id);
      this.metaData = metaData; 
      this.standaloneDeployment = standaloneDeployment;
   }
   
   /**
    * @see JaccPolicyMBean#create()
    */
   public void create()
   { 
      try
      {
         createPolicyConfiguration();
      }
      catch (Exception e)
      {
         throw new RuntimeException(e);
      }
      if(this.standaloneDeployment == Boolean.TRUE)
      {
         try
         {
            if (metaData != null)
               JaccPolicyUtil.createPermissions(parentPC, metaData);
            else
               log.warn("Cannot create permissions with 'null' metaData for id=" + contextID);
         }
         catch (PolicyContextException e)
         {

            throw new RuntimeException("Cannot create permissions:",e);
         }
      }
   }
   
   /**
    * @see JaccPolicyMBean#destroy()
    */
   public void destroy()
   {  
      parentPC= null; 
   }

   /**
    * @see JaccPolicyMBean#start()
    */
   public void start()
   {  
      //All the sub deployments have started
      try
      {
         parentPC.commit();
      }
      catch (PolicyContextException e)
      {
         log.error("Cannot commit Parent Policy Configuration:",e);
      }
   }

   /**
    * @see JaccPolicyMBean#stop()
    */
   public void stop()
   {  
      try
      {
         this.parentPC.delete();
      }
      catch (PolicyContextException e)
      {
         throw new RuntimeException(e);
      }
   } 
   
   
   /**
    * @see JaccPolicyMBean#link(PolicyConfiguration)
    */
   public void link(PolicyConfiguration pc) throws PolicyContextException
   {  
      if(trace)
         log.trace("Linking " + pc + " to parent pc=" +  parentPC);
      if(pc == null)
         throw new IllegalArgumentException("pc is null");
      try
      {
         createPolicyConfiguration();
      }
      catch (ClassNotFoundException e)
      {
         throw new RuntimeException(e);
      }
      parentPC.linkConfiguration(pc); 
   } 
   
   private void createPolicyConfiguration() throws PolicyContextException, ClassNotFoundException 
   {
      if(parentPC == null)
      {
         PolicyConfigurationFactory pcf = PolicyConfigurationFactory.getPolicyConfigurationFactory();
         parentPC = pcf.getPolicyConfiguration(contextID, true);
      }
   }
}
