/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.deployment.security;
  
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.dependency.spi.ControllerState;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.deployer.AbstractVFSRealDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.ejb.jboss.JBossMetaData;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.system.metadata.ServiceAttributeMetaData;
import org.jboss.system.metadata.ServiceConstructorMetaData;
import org.jboss.system.metadata.ServiceDependencyMetaData;
import org.jboss.system.metadata.ServiceDeployment;
import org.jboss.system.metadata.ServiceInjectionValueMetaData;
import org.jboss.system.metadata.ServiceMetaData;

//$Id: SecurityDeployer.java 69657 2008-02-06 16:01:33Z anil.saldhana@jboss.com $

/**
 *  Security Deployer that does Jacc initialization
 *
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @author Scott.Stark@jboss.org
 *  @author adrian@jboss.org
 *  @since  Dec 11, 2006 
 *  @version $Revision: 69657 $
 */
public class SecurityDeployer extends AbstractVFSRealDeployer 
{   
   private HashSet<String> ignoreSuffixes = null;
   
   String BASE_OBJECT_NAME = "jboss:service=jacc,id=";
   
   public SecurityDeployer()
   { 
      setTopLevelOnly(true);
      
      // see getMetaData
      setInputs(JBossWebMetaData.class.getName(),
            JBossMetaData.class.getName(),
            JBossEnterpriseBeanMetaData.class.getName(), 
            //TODO: EJB3 hack
            JaccPolicyUtil.IGNORE_ME_NAME);
      
      setOutputs("jboss.jacc");
   }
   
   public void setIgnoreSuffixes(HashSet<String> suffixSet)
   {
      this.ignoreSuffixes = suffixSet;
   }  
   
   public void deploy(VFSDeploymentUnit unit) throws DeploymentException
   { 
      //Ignore some of the extensions
      String contextId = unit.getSimpleName(); 
      
      for(String suffix : ignoreSuffixes)
      {
         String dotSuffix = "." + suffix;
         if(contextId.endsWith(dotSuffix))
         {
            log.debug("Ignoring: "+contextId);
            return;
         }
      }
      
      //Need to get an estimate of all the deployments underneath that are jacc capable
      List<String> jaccDeployments = JaccPolicyUtil.getJaccDeployments(unit, ignoreSuffixes);
      
      if(jaccDeployments.size() == 0)
      {
         boolean ejbJar = (contextId.endsWith("jar") && JaccPolicyUtil.isEJBDeployment(unit));
         if((contextId.endsWith("war")) || ejbJar) 
         { 
            createTopLevelServiceBeanWithMetaData(contextId,unit);
            return;
         }
         return; //No need to process if there are no jacc deployments
      }
      
      //Create a top level Jacc Policy Bean
      if(getMetaData(unit) != null)
         this.createTopLevelServiceBeanWithMetaData(contextId, unit);
      else
         this.createTopLevelServiceBean(contextId, unit);
      
      //Now for each of the deployments underneath, create a policy configuration facade
      for(String str:jaccDeployments)
      {
         //get the deployment unit for str
         DeploymentUnit childDU = getChildDeployment(unit,str);
         
         Object metadata = getMetaData(childDU);
         if(metadata == null)
         {
            log.debug("meta data is null for child deployment:" + childDU.getSimpleName() 
                  + "; No Jacc policy created.");
            continue;
         }
         
         ServiceMetaData subjaccPolicy = new ServiceMetaData();
         subjaccPolicy.setCode(PolicyConfigurationFacade.class.getName());
         try
         {
            subjaccPolicy.setObjectName(new ObjectName(BASE_OBJECT_NAME  + str  + ",parent=" + childDU.getParent().getSimpleName()) );
         }
         catch (Exception e)
         {  
            throw new RuntimeException(e);
         }
         //Provide a constructor for the service bean 
         ServiceConstructorMetaData constructor = new ServiceConstructorMetaData();
         constructor.setSignature(new String[] { String.class.getName(), Object.class.getName()});
         constructor.setParameters(new Object[] {str, getMetaData(childDU)});
         subjaccPolicy.setConstructor(constructor);
         
         ArrayList<ServiceMetaData> services = new ArrayList<ServiceMetaData>();
         services.add(subjaccPolicy); 
         childDU.addAttachment("jboss.jacc", subjaccPolicy, ServiceMetaData.class ); 
         
         //Add a dependence into the top level jacc policy bean
         ServiceMetaData parentJacc = (ServiceMetaData)childDU.getParent().getAttachment("jboss.jacc");
         if(parentJacc == null)
            throw new IllegalStateException("Parent JACC Policy is null");
         
         List<ServiceDependencyMetaData> dependencies = new ArrayList<ServiceDependencyMetaData>();
         ServiceDependencyMetaData sdmd = new ServiceDependencyMetaData();
         sdmd.setIDependOnObjectName(subjaccPolicy.getObjectName());
         dependencies.add(sdmd);
         parentJacc.setDependencies(dependencies); 
         
         //Inject the top level jacc policy bean 
         List<ServiceAttributeMetaData> attributes = new ArrayList<ServiceAttributeMetaData>();
         ServiceAttributeMetaData attribute = new ServiceAttributeMetaData();
         attribute.setName("JaccPolicyMBean");
         String parentObjectName = parentJacc.getObjectName().getCanonicalName();
         ServiceInjectionValueMetaData sivmd = new ServiceInjectionValueMetaData(parentObjectName);
         sivmd.setDependentState(ControllerState.CONFIGURED);
         attribute.setValue(sivmd);
         attributes.add(attribute);
         attributes.addAll(subjaccPolicy.getAttributes());
         subjaccPolicy.setAttributes(attributes); 
      } 
   }
   

   @Override
   public void undeploy(VFSDeploymentUnit unit)
   {
      unit.removeAttachment("jboss.jaccpolicy", ServiceDeployment.class); 
      List<String> jaccDeployments = JaccPolicyUtil.getJaccDeployments(unit, ignoreSuffixes);
      for(String deployment: jaccDeployments)
      {
         DeploymentUnit childDU = this.getChildDeployment(unit, deployment);
         if(childDU != null)
            childDU.removeAttachment("jboss.jaccpolicy", ServiceDeployment.class); 
      }
   } 
    
   
   //Private Methods
   
   private DeploymentUnit getChildDeployment(DeploymentUnit parent, String id)
   {
      List<DeploymentUnit> children = parent.getChildren();
      for(DeploymentUnit child: children)
      {
         if (id.equals(child.getSimpleName()))
            return child; 
      }
      
      return null;
   }
   
   private void createTopLevelServiceBeanWithMetaData(String contextId, DeploymentUnit unit)
   { 
      //Provide a constructor for the service bean 
      ServiceConstructorMetaData constructor = new ServiceConstructorMetaData();
      constructor.setSignature(new String[] { String.class.getName(), 
            Object.class.getName(),Boolean.class.getName()});
      constructor.setParameters(new Object[] {contextId,getMetaData(unit), Boolean.TRUE});
      createJaccPolicyBean(constructor, unit);
   }
   
   private void createTopLevelServiceBean(String contextId, DeploymentUnit unit)
   { 
      //Provide a constructor for the service bean 
      ServiceConstructorMetaData constructor = new ServiceConstructorMetaData();
      constructor.setSignature(new String[] { String.class.getName()});
      constructor.setParameters(new Object[] {contextId}); 
      
      createJaccPolicyBean(constructor, unit); 
   }
   
   private void createJaccPolicyBean(ServiceConstructorMetaData constructor, DeploymentUnit unit)
   {
      //Create a Service Bean for the JACC Policy 
      ServiceMetaData jaccPolicy = new ServiceMetaData();
      jaccPolicy.setCode(JaccPolicy.class.getName());  
      try
      {
         jaccPolicy.setObjectName(new ObjectName(BASE_OBJECT_NAME + unit.getSimpleName()));
      }
      catch (Exception e)
      { 
         throw new RuntimeException(e);
      }
      //Provide a constructor for the service bean  
      jaccPolicy.setConstructor(constructor);
      ArrayList<ServiceMetaData> services = new ArrayList<ServiceMetaData>();
      services.add(jaccPolicy);
      
      unit.addAttachment("jboss.jacc", jaccPolicy, ServiceMetaData.class); 
   }
   
   private Object getMetaData(DeploymentUnit du)
   {
      Object md = null;
      //Look for WebMetaData
      md = du.getAttachment(JBossWebMetaData.class.getName());
      //Look for ApplicationMetaData
      if(md == null)
         md = du.getAttachment(JBossMetaData.class.getName());
      //Look for BeanMetaData??
      if(md == null)
         md = du.getAttachment(JBossEnterpriseBeanMetaData.class.getName());
      return md;
   }
}
