/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployment;

import java.util.Set;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.jboss.aop.microcontainer.aspects.jmx.JMX;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.deployer.SchemaResolverDeployer;
import org.jboss.metadata.common.jboss.LoaderRepositoryConfigMetaData;
import org.jboss.metadata.common.jboss.LoaderRepositoryMetaData;
import org.jboss.metadata.ear.jboss.JBossAppMetaData;
import org.jboss.metadata.ear.spec.EarMetaData;
import org.jboss.mx.loading.LoaderRepositoryFactory;
import org.jboss.mx.loading.LoaderRepositoryFactory.LoaderRepositoryConfig;

/**
 * An ObjectModelFactoryDeployer for translating jboss-app.xml descriptors into
 * JBossAppMetaData instances.
 * 
 * @author Scott.Stark@jboss.org
 * @author Anil.Saldhana@redhat.com
 * @author adrian@jboss.org
 * @version $Revision: 73419 $
 */
@JMX(name="jboss.j2ee:service=EARDeployer", exposedInterface=JBossAppParsingDeployerMBean.class)
public class JBossAppParsingDeployer extends SchemaResolverDeployer<JBossAppMetaData> implements JBossAppParsingDeployerMBean
{
   private boolean isolated = false;

   private boolean callByValue = false;
   
   private String unauthenticatedIdentity = null;

   /**
    * Create a new JBossAppParsingDeployer.
    */
   public JBossAppParsingDeployer()
   {
      super(JBossAppMetaData.class);
      //setInput(JBoss50Aporg.jboss.metadata.ear.spec.EarMetaDatapMetaData.class);
      setName("jboss-app.xml");
   }

   /**
    * Get the virtual file path for the application descriptor in the
    * DeploymentContext.getMetaDataPath.
    * 
    * @return the current virtual file path for the application descriptor
    */
   public String getAppXmlPath()
   {
      return getName();
   }
   /**
    * Set the virtual file path for the application descriptor in the
    * DeploymentContext.getMetaDataLocation. The standard path is jboss-app.xml
    * to be found in the META-INF metdata path.
    * 
    * @param appXmlPath - new virtual file path for the application descriptor
    */
   public void setAppXmlPath(String appXmlPath)
   {
      setName(appXmlPath);
   }

   /**
    * @return whether ear deployments should be isolated
    */
   public boolean isIsolated()
   {
      return isolated;
   }
   
   /**
    * @param isolated whether ear deployments should be isolated
    */
   public void setIsolated(boolean isolated)
   {
      this.isolated = isolated;
   }
   
   /**
    * @return whether ear deployments should be call by value
    */
   public boolean isCallByValue()
   {
      return callByValue;
   }
   
   /**
    * @param callByValue whether ear deployments should be call by value
    */
   public void setCallByValue(boolean callByValue)
   {
      this.callByValue = callByValue;
   }
  
   /**
    * Obtain an unauthenticated identity
    * 
    * @return the unauthenticated identity
    */
   public String getUnauthenticatedIdentity()
   {
      return unauthenticatedIdentity;
   }

   /**
    * Specify an unauthenticated identity
    * @param unauthenticatedIdentity
    */
   public void setUnauthenticatedIdentity(String unauthenticatedIdentity)
   {
      this.unauthenticatedIdentity = unauthenticatedIdentity;
   }

   
   // FIXME This should all be in a seperate deployer
   @Override
   protected void createMetaData(DeploymentUnit unit, String name, String suffix) throws DeploymentException
   {
      super.createMetaData(unit, name, suffix);
      EarMetaData specMetaData = unit.getAttachment(EarMetaData.class);
      JBossAppMetaData metaData = unit.getAttachment(JBossAppMetaData.class);
      if(specMetaData == null && metaData == null)
         return;

      LoaderRepositoryConfig loaderCfg = null;

      // If there no JBossMetaData was created from a jboss-app.xml, create one
      if (metaData == null)
      {
         metaData = new JBossAppMetaData();
      }
      else
      {
         LoaderRepositoryMetaData lrmd = metaData.getLoaderRepository();
         if (lrmd != null)
         {
            loaderCfg = new LoaderRepositoryConfig();
            loaderCfg.repositoryClassName = lrmd.getLoaderRepositoryClass();
            if(lrmd.getName() != null)
            {
               try
               {
                  loaderCfg.repositoryName = new ObjectName(lrmd.getName());
               }
               catch (MalformedObjectNameException e)
               {
                  throw new DeploymentException("Invalid object name: " + lrmd.getName(), e);
               }
            }
            Set<LoaderRepositoryConfigMetaData> configs = lrmd.getLoaderRepositoryConfig();
            if (configs != null && configs.isEmpty() == false)
            {
               LoaderRepositoryConfigMetaData config = configs.iterator().next();
               loaderCfg.configParserClassName = config.getConfigParserClass();
               loaderCfg.repositoryConfig = config.getConfig();
            }
         }
      }

      if (loaderCfg == null && isolated)
      {
         // If no config was given and isolation is on, create an isolated config
         loaderCfg = initLoaderRepository(metaData, unit);
      }

      // Add the loader repository config
      if (loaderCfg != null)
         unit.addAttachment(LoaderRepositoryConfig.class, loaderCfg);

      // Create a merged view
      JBossAppMetaData mergedMetaData = new JBossAppMetaData();
      mergedMetaData.merge(metaData, specMetaData);
      // Set the merged as the output
      unit.getTransientManagedObjects().addAttachment(JBossAppMetaData.class, mergedMetaData);
      // Keep the raw parsed metadata as well
      unit.addAttachment("Raw"+JBossAppMetaData.class.getName(), metaData, JBossAppMetaData.class);
      // Pass the ear callByValue setting
      if (isCallByValue())
         unit.addAttachment("EAR.callByValue", Boolean.TRUE, Boolean.class);
      //Pass the unauthenticated identity
      if (this.unauthenticatedIdentity != null)
         unit.addAttachment("EAR.unauthenticatedIdentity", this.unauthenticatedIdentity, String.class);
   }

   /** Build the ear scoped repository
   *
   * @param metaData the deployment info passed to deploy
   * @param unit the deployment unit
   * @return LoaderRepositoryConfig to use
   * @throws DeploymentException for any error
   */
  protected LoaderRepositoryConfig initLoaderRepository(JBossAppMetaData metaData, DeploymentUnit unit)
     throws DeploymentException
  {
     try
     {
        String name = EARDeployment.getJMXName(metaData, unit) + ",extension=LoaderRepository";
        ObjectName objectName = new ObjectName(name);
        LoaderRepositoryConfig config = new LoaderRepositoryFactory.LoaderRepositoryConfig();
        config.repositoryName = objectName;
        return config;
     }
     catch(Exception e)
     {
        throw new DeploymentException("Failed to create LoaderRepositoryConfig", e);
     }
  }
}
