/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployment;

import java.util.Set;

import javax.management.ObjectName;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.deployer.SchemaResolverDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.common.jboss.LoaderRepositoryConfigMetaData;
import org.jboss.metadata.common.jboss.LoaderRepositoryMetaData;
import org.jboss.metadata.ear.jboss.JBossAppMetaData;
import org.jboss.metadata.ear.spec.EarMetaData;
import org.jboss.metadata.web.jboss.ClassLoadingMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.metadata.web.spec.WebMetaData;
import org.jboss.mx.loading.LoaderRepositoryFactory.LoaderRepositoryConfig;
import org.jboss.virtual.VirtualFile;

/**
 * An ObjectModelFactoryDeployer for translating jboss-web.xml descriptors into
 * WebMetaData instances.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @version $Revision:$
 */
public class JBossWebAppParsingDeployer extends SchemaResolverDeployer<JBossWebMetaData>
{
   /**
    * Create a new JBossWebAppParsingDeployer.
    */
   public JBossWebAppParsingDeployer()
   {
      super(JBossWebMetaData.class);
      setInput(WebMetaData.class);
      setName("jboss-web.xml");
   }

   /**
    * Get the virtual file path for the jboss-web descriptor in the
    * DeploymentContext.getMetaDataPath.
    * 
    * @return the current virtual file path for the web-app descriptor
    */
   public String getWebXmlPath()
   {
      return getName();
   }
   /**
    * Set the virtual file path for the jboss-web descriptor in the
    * DeploymentContext.getMetaDataLocation. The standard path is jboss-web.xml
    * to be found in the WEB-INF metdata path.
    * 
    * @param webXmlPath - new virtual file path for the web-app descriptor
    */
   public void setWebXmlPath(String webXmlPath)
   {
      setName(webXmlPath);
   }

   @Override
   protected void createMetaData(DeploymentUnit unit, String name, String suffix) throws DeploymentException
   {
      super.createMetaData(unit, name, suffix);
      // Merge the spec metadata
      WebMetaData specMetaData = unit.getAttachment(WebMetaData.class);
      JBossWebMetaData metaData = unit.getAttachment(JBossWebMetaData.class);
      if(specMetaData == null && metaData == null)
         return;

      // If there no JBossWebMetaData was created from a jboss-web.xml, create one
      if (metaData == null)
      {
         metaData = new JBossWebMetaData();
      }
      // Create a merged view
      JBossWebMetaData mergedMetaData = new JBossWebMetaData();
      mergedMetaData.merge(metaData, specMetaData);
      // Set the merged as the output
      unit.getTransientManagedObjects().addAttachment(JBossWebMetaData.class, mergedMetaData);
      // Keep the raw parsed metadata as well
      unit.addAttachment("Raw"+JBossWebMetaData.class.getName(), metaData, JBossWebMetaData.class);
   }

   @Override
   protected void init(VFSDeploymentUnit unit, JBossWebMetaData metaData, VirtualFile file) throws Exception
   {
      // Add the loader repository config
      ClassLoadingMetaData classLoading = metaData.getClassLoading();
      LoaderRepositoryMetaData lrmd = null;
      if(classLoading != null)
         lrmd = classLoading.getLoaderRepository();
      if (lrmd != null)
      {
         LoaderRepositoryConfig loaderConfig = new LoaderRepositoryConfig();
         
         loaderConfig.repositoryClassName = lrmd.getLoaderRepositoryClass();
         if(lrmd.getName() != null)
            loaderConfig.repositoryName = new ObjectName(lrmd.getName());
         Set<LoaderRepositoryConfigMetaData> configs = lrmd.getLoaderRepositoryConfig();
         if (configs != null && configs.isEmpty() == false)
         {
            LoaderRepositoryConfigMetaData config = configs.iterator().next();
            loaderConfig.configParserClassName = config.getConfigParserClass();
            loaderConfig.repositoryConfig = config.getConfig();
         }
         unit.addAttachment(LoaderRepositoryConfig.class, loaderConfig);
      }      
   }

   /**
    * Make sure we always have a JBossWebMetaData object attached, even if there is no jboss-web.xml
    * in the deployment
    */
   @Override
   protected void createMetaData(DeploymentUnit unit, String name, String suffix, String key) throws DeploymentException
   {
      super.createMetaData(unit, name, suffix, key);
      
      WebMetaData wmd = unit.getTransientManagedObjects().getAttachment(WebMetaData.class);
      JBossWebMetaData result = unit.getTransientManagedObjects().getAttachment(getOutput());
      if (result == null && wmd != null)
      {
         result = new JBossWebMetaData();
         result.merge(null, wmd);
         unit.getTransientManagedObjects().addAttachment(key, result, getOutput());
      }
   }
}
