/*
  * JBoss, Home of Professional Open Source
  * Copyright 2007, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.ejb.plugins.security; 

import java.lang.reflect.Method;

import javax.ejb.TimedObject;
import javax.ejb.Timer;

import org.jboss.ejb.Container;
import org.jboss.ejb.plugins.AbstractInterceptor;
import org.jboss.invocation.Invocation;
import org.jboss.security.AuthenticationManager;
import org.jboss.security.SecurityContext;
import org.jboss.security.SecurityIdentity;

//$Id$

/**
 *  Interceptor that performs the initialization required for 
 *  the security interceptor. This interceptor performs
 *  Security Context establishment and other initialization required
 *  - The Outgoing run-as from the security context is pushed to incoming run-as
 *  - For EJB Local Invocations, the security context is obtained from the
 *  SecurityContextAssocation threadlocal and placed on the invocation.
 *  @author Anil.Saldhana@redhat.com
 *  @since  Apr 30, 2007 
 *  @version $Revision$
 */
public class PreSecurityInterceptor extends AbstractInterceptor
{ 
   private String securityDomain = null;
   
   private String timedObjectMethod = null;
   
   @Override
   public void setContainer(Container container)
   { 
      super.setContainer(container);
      if (container != null)
      { 
         AuthenticationManager am = container.getSecurityManager();
         if(am != null)
         {
            securityDomain = am.getSecurityDomain();
         } 
      }
      try
      {
         timedObjectMethod = TimedObject.class.getMethod("ejbTimeout", new Class[]{Timer.class})
                               .getName();
      }
      catch (Exception e)
      {
         log.trace("Exception in creating TimedObject method:",e);
      } 
   }

   @Override
   public Object invoke(Invocation mi) throws Exception
   { 
      //No Security in the absence of SecurityDomain
      if(securityDomain == null)
         return getNext().invoke(mi);
      
      
      SecurityIdentity si = null;
      Method m = mi.getMethod();
      boolean isEjbTimeOutMethod =  m!= null && m.getName().equals(timedObjectMethod);
      //For local ejb invocations
      if(mi.isLocal() && !isEjbTimeOutMethod)
      {
         //Cache the security context
         SecurityContext sc = SecurityActions.getSecurityContext();
         if(sc != null)
           si = sc.getUtil().getSecurityIdentity(); 

         //Set the security context on the invocation
         mi.setSecurityContext(sc); 
      }
      else
      {
         establishSecurityContext(mi); 
      } 
      
      try
      { 
         //Establish the run-as on the SC as the caller SC
         SecurityActions.pushCallerRunAsIdentity(SecurityActions.getSecurityContext().getOutgoingRunAs());
         Object returnValue = getNext().invoke(mi);
         return returnValue;
      }
      finally
      { 
         SecurityActions.popCallerRunAsIdentity();
         if(mi.isLocal() && si != null)
            SecurityActions.getSecurityContext().getUtil().setSecurityIdentity(si);
      }
   }

   @Override
   public Object invokeHome(Invocation mi) throws Exception
   { 
      //No Security in the absence of SecurityDomain
      if(securityDomain == null)
         return getNext().invokeHome(mi);
      
      SecurityIdentity si = null;
      Method m = mi.getMethod();
      boolean isEjbTimeOutMethod =  m!= null && m.getName().equals(timedObjectMethod);
      //For local ejb invocations
      if(mi.isLocal() && !isEjbTimeOutMethod)
      {  
         //Cache the security context
         SecurityContext sc = SecurityActions.getSecurityContext();
         if(sc != null)
            si = sc.getUtil().getSecurityIdentity();
         //Set the security context on the invocation
         mi.setSecurityContext(sc); 
      }
      else
      {
         establishSecurityContext(mi); 
      }
      try
      { 
         //Establish the run-as on the SC as the caller SC
         SecurityActions.pushCallerRunAsIdentity(SecurityActions.getSecurityContext().getOutgoingRunAs());
         Object returnValue = getNext().invokeHome(mi);
         return returnValue;
      }
      finally
      { 
         SecurityActions.popCallerRunAsIdentity();
         if(mi.isLocal() && si != null)
            SecurityActions.getSecurityContext().getUtil().setSecurityIdentity(si);
      }
   }
   
   private void establishSecurityContext(Invocation mi) throws Exception
   { 
      //For Local EJB invocations, the security context needs
      //to be obtained from the thread local. For remote ejb
      //invocations, the SC is obtained in the invocation
      SecurityContext sc = mi.getSecurityContext(); 
      SecurityContext newSC = SecurityActions.createAndSetSecurityContext(securityDomain,
            container.getSecurityContextClassName());  
      
      if(sc != null)
      {
         //Get the run-as, principal, cred etc from the invocation and set it on the context
         SecurityActions.setSecurityIdentity(newSC,
                      sc.getUtil().getSecurityIdentity());
      }
      else
      { 
         //Local EJB Invocation or some one created the Invocation object on the server side
         mi.setSecurityContext(newSC);
      }
      //Set the SecurityManagement on the context
      newSC.setSecurityManagement(container.getSecurityManagement());
   }
}
