/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.proxy;

import java.security.Principal; 

import org.jboss.invocation.Invocation; 
import org.jboss.invocation.InvocationKey;
import org.jboss.security.RunAs; 
import org.jboss.security.SecurityContext;   

/**
* The client-side proxy for an EJB Home object.
*      
* @author <a href="mailto:marc.fleury@jboss.org">Marc Fleury</a>
* @author Anil.Saldhana@redhat.com
* @version $Revision: 69871 $
*/
public class SecurityInterceptor
   extends Interceptor
{
   /** Serial Version Identifier. @since 1.4.2.1 */
   private static final long serialVersionUID = -4206940878404525061L;
   
   /**
   * No-argument constructor for externalization.
   */
   public SecurityInterceptor()
   {
   }

   // Public --------------------------------------------------------
   
   public Object invoke(Invocation invocation)
      throws Throwable
   {
      // Get Principal and credentials
      SecurityActions sa = SecurityActions.UTIL.getSecurityActions();

      Principal principal = sa.getPrincipal();
      if (principal != null)
      {
         invocation.setPrincipal(principal);
      }

      Object credential = sa.getCredential();
      if (credential != null)
      {
         invocation.setCredential(credential);
      }
 
      SecurityContext sc  = sa.getSecurityContext();
      RunAs callerRAI =  sa.getCallerRunAsIdentity();
      SecurityContext newSc = createSecurityContext(invocation);
      //Push the caller run-as identity onto the security context 
      if(callerRAI != null)
      {
         newSc.setOutgoingRunAs(callerRAI);
         newSc.setIncomingRunAs(callerRAI);
      }
      /**
       * Push the security context on the invocation
       */
      invocation.setSecurityContext(newSc);
      
      try
      { 
         return getNext().invoke(invocation); 
      }
      finally
      { 
         if(sc != null)
            sa.setSecurityContext(sc); 
      }
   }
   
   /**
    * Return loaded Security Context to be passed on the invocation
    * @param invocation invocation instance
    * @return
    */
   private SecurityContext createSecurityContext(Invocation invocation) throws Exception
   { 
      SecurityActions sa = SecurityActions.UTIL.getSecurityActions();

      //There may be principal set on the invocation
      Principal p = invocation.getPrincipal();
      Object cred = invocation.getCredential(); 
      
      //Create a new SecurityContext
      String domain = (String) invocation.getInvocationContext().getValue(InvocationKey.SECURITY_DOMAIN);
      if(domain == null)
         domain = "CLIENT_PROXY";
      return sa.createSecurityContext(p,cred, domain);
   } 
}
