/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.deployers;

import java.io.File;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.management.Attribute;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;

import org.apache.catalina.Lifecycle;
import org.apache.catalina.connector.Connector;
import org.apache.catalina.startup.CatalinaProperties;
import org.apache.tomcat.util.modeler.Registry;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.web.jboss.JBoss50WebMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.metadata.web.jboss.ReplicationConfig;
import org.jboss.metadata.web.spec.Web25MetaData;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.security.ISecurityManagement;
import org.jboss.security.plugins.JaasSecurityManagerServiceMBean;
import org.jboss.system.server.Server;
import org.jboss.system.server.ServerImplMBean;
import org.jboss.virtual.VirtualFile;
import org.jboss.web.deployers.AbstractWarDeployer;
import org.jboss.web.deployers.AbstractWarDeployment;
import org.jboss.web.tomcat.security.HttpServletRequestPolicyContextHandler;
import org.jboss.web.tomcat.service.session.SessionIDGenerator;
import org.jboss.xb.binding.JBossXBException;
import org.jboss.xb.binding.Unmarshaller;
import org.jboss.xb.binding.UnmarshallerFactory;
import org.jboss.xb.binding.sunday.unmarshalling.SchemaBinding;
import org.jboss.xb.builder.JBossXBBuilder;


/**
 * A concrete implementation of the AbstractWarDeployer that creates
 * {@link #TomcatDeployment} instances as the web application bean
 * representation.
 * 
 * @see #getDeployment(VFSDeploymentUnit, WebMetaData)
 * 
 * @author Scott.Stark@jboss.org
 * @author Costin Manolache
 * @author Wonne.Keysers@realsoftware.be
 * @author Dimitris.Andreadis@jboss.org
 * @author adrian@jboss.org
 * @version $Revision: 56809 $
 * @see org.jboss.web.deployers.AbstractWarDeployer
 */
public class TomcatDeployer extends AbstractWarDeployer
   implements NotificationListener, TomcatDeployerMBean
{
   // Constants -----------------------------------------------------
   public static final String NAME = "TomcatDeployer";

   /** The web app context implementation class */
   private String contextClassName =
      "org.apache.catalina.core.StandardContext";

   /**
    * Unmarshall factory used for parsing shared web.xml.
    */
   private static final UnmarshallerFactory factory = UnmarshallerFactory.newInstance();
   
   /**
    * Configurable map of tomcat authenticators
    * Keyed in by the http auth method that gets 
    * plugged into the Context Config and then into the StandardContext 
    */
   private Properties authenticators = null;

   /**
    * Domain for tomcat6 mbeans
    */
   private String catalinaDomain = "Catalina";

   /**
    * The fully qualified name of the class that will be used for session
    * management if <tt>distributable</tt> is set to true.
    */
   protected String managerClass = "org.jboss.web.tomcat.service.session.JBossCacheManager";

   /**
    * A flag indicating if the JBoss Loader should be used
    */
   private boolean useJBossWebLoader = true;
   
   /**
    * JBAS-3358: Work directory shouldn't be deleted on Context Destroy
    */
   private boolean deleteWorkDirOnContextDestroy = false;
   
   /**
    * JBAS-2283: Provide custom header based auth support
    */
   private String httpHeaderForSSOAuth = null;
   private String sessionCookieForSSOAuth = null;
   
   /**
    * Shared metaData.
    */
   private JBossWebMetaData sharedMetaData = null;


   /**
    * The server xml configuration file name
    */
   private String serverConfigFile = "server.xml";

   /**
    * Get the request attribute name under which the JAAS Subject is store
    */
   private String subjectAttributeName = null;

   /**
    * Flag indicating whether web-app specific context xmls may set the privileged flag.
    */
   private boolean allowSelfPrivilegedWebApps = false;

   /** The service used to flush authentication cache on session invalidation. */
   private JaasSecurityManagerServiceMBean secMgrService;
   /** The AbstractWarDeployment implementation class */
   private Class deploymentClass = TomcatDeployment.class; 
   
   /** The JBoss Security Manager Wrapper */
   private ISecurityManagement securityManagement;
   /** FQN of the SecurityContext Class */
   private String securityContextClassName; 

   /** */
   private String[] filteredPackages;

   private boolean runtimeLifecycleCoupled = false;
   
   public TomcatDeployer()
   {
   }

   public String getName()
   {
      return NAME;
   }


   public String getManagerClass()
   {
      return managerClass;
   }

   public void setManagerClass(String managerClass)
   {
      this.managerClass = managerClass;
   }


   public String getDomain()
   {
      return this.catalinaDomain;
   }

   public Properties getAuthenticators()
   {
      return this.authenticators;
   }

   public void setAuthenticators(Properties prop)
   {
      this.authenticators = prop;
      log.debug("Passed set of authenticators=" + prop);
   }

   /**
    * The most important atteribute - defines the managed domain.
    * A catalina instance (engine) corresponds to a JMX domain, that's
    * how we know where to deploy webapps.
    *
    * @param catalinaDomain the domain portion of the JMX ObjectNames
    */
   public void setDomain(String catalinaDomain)
   {
      this.catalinaDomain = catalinaDomain;
   }

   public void setContextMBeanCode(String className)
   {
      this.contextClassName = className;
   }

   public String getContextMBeanCode()
   {
      return contextClassName;
   }

   public boolean getDeleteWorkDirOnContextDestroy()
   {
      return deleteWorkDirOnContextDestroy;
   }

   public void setDeleteWorkDirOnContextDestroy(boolean deleteFlag)
   {
      this.deleteWorkDirOnContextDestroy = deleteFlag;
   } 
   
   public String getHttpHeaderForSSOAuth()
   {
      return httpHeaderForSSOAuth;
   }
    
   public void setHttpHeaderForSSOAuth(String httpHeader)
   {
       this.httpHeaderForSSOAuth = httpHeader;
   }
   
   public String getSessionCookieForSSOAuth()
   {
       return sessionCookieForSSOAuth;
   }
   
   public void setSessionCookieForSSOAuth(String sessionC)
   {
      this.sessionCookieForSSOAuth = sessionC;
   }
   
   /**
    * The SessionIdAlphabet is the set of characters used to create a session Id
    */
   public void setSessionIdAlphabet(String sessionIdAlphabet)
   {
       SessionIDGenerator.getInstance().setSessionIdAlphabet(sessionIdAlphabet);
   }

   /**
    * The SessionIdAlphabet is the set of characters used to create a session Id
    */
   public String getSessionIdAlphabet()
   {
       return SessionIDGenerator.getInstance().getSessionIdAlphabet();
   }

   public boolean getUseJBossWebLoader()
   {
      return useJBossWebLoader;
   }

   public void setUseJBossWebLoader(boolean flag)
   {
      this.useJBossWebLoader = flag;
   }

   public String getConfigFile()
   {
      return serverConfigFile;
   }

   public void setConfigFile(String configFile)
   {
      this.serverConfigFile = configFile;
   }

   public String getSubjectAttributeName()
   {
      return this.subjectAttributeName;
   }

   public void setSubjectAttributeName(String name)
   {
      this.subjectAttributeName = name;
   }

   public boolean isAllowSelfPrivilegedWebApps()
   {
      return allowSelfPrivilegedWebApps;
   }

   public void setAllowSelfPrivilegedWebApps(boolean allowSelfPrivilegedWebApps)
   {
      this.allowSelfPrivilegedWebApps = allowSelfPrivilegedWebApps;
   } 

   public void setSecurityManagerService(JaasSecurityManagerServiceMBean mgr)
   {
      this.secMgrService = mgr;
   } 
  
   public void setSecurityManagement(ISecurityManagement securityManagement)
   {
      this.securityManagement = securityManagement;
   } 

   public void setSecurityContextClassName(String securityContextClassName)
   {
      this.securityContextClassName = securityContextClassName;
   }

   public String[] getFilteredPackages()
   {
      return filteredPackages;
   }
   public void setFilteredPackages(String[] pkgs)
   {
      this.filteredPackages = pkgs;
   }

   public Class getDeploymentClass()
   {
      return deploymentClass;
   }
   public void setDeploymentClass(Class deploymentClass)
   {
      this.deploymentClass = deploymentClass;
   }   
   
   /**
    * Gets whether this object should start/stop the JBoss Web runtime
    * during execution of its own start/stop lifecycle callbacks.
    * 
    * @return <code>true</code> if a call to {@link #start()} should
    *        trigger a call to {@link #startWebServer()} and a call to
    *        {@link #stop()} should trigger a call to {@link #stopWebServer()};
    *        <code>false</code> if the webserver runtime lifecycle will
    *        be separately managed.  Default is <code>false</code>.
    */
   public boolean isRuntimeLifecycleCoupled()
   {
      return runtimeLifecycleCoupled;
   }

   /**
    * Sets whether this object should start/stop the JBoss Web runtime
    * during execution of its own start/stop lifecycle callbacks.
    * 
    * @param coupled <code>true</code> if a call to {@link #start()} should
    *        trigger a call to {@link #startWebServer()} and a call to
    *        {@link #stop()} should trigger a call to {@link #stopWebServer()};
    *        <code>false</code> if the webserver runtime lifecycle will
    *        be separately managed. Default is <code>false</code>.
    */
   public void setRuntimeLifecycleCoupled(boolean coupled)
   {
      runtimeLifecycleCoupled = coupled;
   }

   /**
    * Start the deployer. This sets up the tomcat core.
    */
   public void start()
      throws Exception
   {
      super.start();
      
      // Parse shared web.xml
      Unmarshaller unmarshaller = factory.newUnmarshaller();
      URL webXml = this.getClass().getClassLoader().getResource("web.xml");
      if (webXml == null)
      {
         webXml = this.getClass().getClassLoader().getResource("conf/web.xml");;
      }
      if (webXml == null)
         throw new IllegalStateException("Unable to find shared web.xml or conf/web.xml");
      SchemaBinding schema = JBossXBBuilder.build(Web25MetaData.class);
      Web25MetaData confWebMD = (Web25MetaData) unmarshaller.unmarshal(webXml.toString(), schema);
      sharedMetaData = new JBoss50WebMetaData();
      sharedMetaData.merge(null, confWebMD);
      
      if (isRuntimeLifecycleCoupled())
         startWebServer();
   }

   /**
    * Initializes and starts the webserver runtime. 
    */
   public void startWebServer() throws Exception
   {
      // Load Catalina properties
      CatalinaProperties.getProperty("");
      
      log.debug("Starting tomcat deployer");
      MBeanServer server = super.getServer();
      System.setProperty("catalina.ext.dirs",
         (System.getProperty("jboss.server.home.dir")
         + File.separator + "lib"));
      
      String objectNameS = catalinaDomain + ":type=server";
      ObjectName objectName = new ObjectName(objectNameS);

      // Set the modeler Registry MBeanServer to the that of the tomcat service
      Registry.getRegistry().setMBeanServer(server);

      Registry.getRegistry().registerComponent(Class.forName("org.apache.catalina.startup.Catalina").newInstance(), 
              objectName, "org.apache.catalina.startup.Catalina");
      /*
      server.createMBean("org.apache.tomcat.util.modeler.BaseModelMBean",
         objectName,
         new Object[]{"org.apache.catalina.startup.Catalina"},
         new String[]{"java.lang.String"});
         */

      server.setAttribute(objectName, new Attribute
         ("catalinaHome",
            System.getProperty("jboss.server.home.dir")));
      server.setAttribute(objectName, new Attribute
         ("configFile", serverConfigFile));
      server.setAttribute(objectName, new Attribute
         ("useNaming", new Boolean(false)));
      server.setAttribute(objectName, new Attribute
         ("useShutdownHook", new Boolean(false)));
      server.setAttribute(objectName, new Attribute
         ("await", new Boolean(false)));
      server.setAttribute(objectName, new Attribute
         ("redirectStreams", new Boolean(false)));

      server.invoke(objectName, "create", new Object[]{},
         new String[]{});

      server.invoke(objectName, "start", new Object[]{},
         new String[]{});
      
      //Set up the authenticators in JNDI such that they can be configured for web apps
      InitialContext ic = new InitialContext();
      try
      {
        ic.bind("TomcatAuthenticators", this.authenticators); 
      }
      catch(NamingException ne)
      {
        if(log.isTraceEnabled())
           log.trace("Binding Authenticators to JNDI failed",ne);
      }
      finally
      { 
          try{ ic.close();}catch(NamingException nee){}
      }

      // Register the web container JACC PolicyContextHandlers
      HttpServletRequestPolicyContextHandler handler = new HttpServletRequestPolicyContextHandler();
      PolicyContext.registerHandler(HttpServletRequestPolicyContextHandler.WEB_REQUEST_KEY,
         handler, true);

//      // Parse shared web.xml
//      Unmarshaller unmarshaller = factory.newUnmarshaller();
//      URL webXml = this.getClass().getClassLoader().getResource("web.xml");
//      if (webXml == null)
//      {
//         webXml = this.getClass().getClassLoader().getResource("conf/web.xml");;
//      }
//      if (webXml == null)
//         throw new IllegalStateException("Unable to find shared web.xml or conf/web.xml");
//      SchemaBinding schema = JBossXBBuilder.build(Web25MetaData.class);
//      Web25MetaData confWebMD = (Web25MetaData) unmarshaller.unmarshal(webXml.toString(), schema);
//      sharedMetaData = new JBoss50WebMetaData();
//      sharedMetaData.merge(null, confWebMD);

      // If we are hot-deployed *after* the overall server is started
      // we'll never receive Server.START_NOTIFICATION_TYPE, so check
      // with the Server and start the connectors immediately, if this is the case.
      // Otherwise register to receive the server start-up notification.
      Boolean started = (Boolean)server.getAttribute(ServerImplMBean.OBJECT_NAME, "Started");
      if (started.booleanValue() == true)
      {
         log.debug("Server '" + ServerImplMBean.OBJECT_NAME +
               "' already started, starting connectors now");

         startConnectors();
      }
      else
      {
         // Register for notification of the overall server startup
         log.debug("Server '" + ServerImplMBean.OBJECT_NAME +
               "' not started, registering for start-up notification");

         server.addNotificationListener(ServerImplMBean.OBJECT_NAME, this, null, null);
      }
   }


   public void stop()
      throws Exception
   {
      if (isRuntimeLifecycleCoupled())
         stopWebServer();
      
      super.stop();
   }

   /**
    * Stops the webserver runtime.
    */
   public void stopWebServer() throws Exception
   {
      MBeanServer server = super.getServer();
      String objectNameS = catalinaDomain + ":type=server";
      ObjectName objectName = new ObjectName(objectNameS);

      server.invoke(objectName, "stop", new Object[]{},
         new String[]{});

      server.invoke(objectName, "destroy", new Object[]{},
         new String[]{});

      server.unregisterMBean(objectName);

      MBeanServer server2 = server;

      // Unregister any remaining jboss.web or Catalina MBeans
      ObjectName queryObjectName = new ObjectName
         (catalinaDomain + ":*");
      Iterator iterator =
         server2.queryMBeans(queryObjectName, null).iterator();
      while (iterator.hasNext())
      {
         ObjectInstance oi = (ObjectInstance) iterator.next();
         ObjectName toRemove = oi.getObjectName();
         // Exception: Don't unregister the service right now
         if (!"WebServer".equals(toRemove.getKeyProperty("service")))
         {
            if (server2.isRegistered(toRemove))
            {
               server2.unregisterMBean(toRemove);
            }
         }
      }
      queryObjectName = new ObjectName("Catalina:*");
      iterator = server2.queryMBeans(queryObjectName, null).iterator();
      while (iterator.hasNext())
      {
         ObjectInstance oi = (ObjectInstance) iterator.next();
         ObjectName name = oi.getObjectName();
         server2.unregisterMBean(name);
      }
   }

   /**
    * Start the tomcat service connectors.
    */
   public void startConnectors() throws Exception
   {
      MBeanServer server = super.getServer();
      ObjectName service = new ObjectName(catalinaDomain + ":type=Service,serviceName=jboss.web");
      Object[] args = {};
      String[] sig = {};
      Connector[] connectors = (Connector[]) server.invoke(service,
         "findConnectors", args, sig);
      for (int n = 0; n < connectors.length; n++)
      {
         Lifecycle lc = (Lifecycle) connectors[n];
         lc.start();
      }
      /* TODO:
      // Notify listeners that connectors have started processing requests
      sendNotification(new Notification(TOMCAT_CONNECTORS_STARTED,
            this, getNextNotificationSequenceNumber()));
      */
   }

   /**
    * Stop the tomcat service connectors.
    */
   public void stopConnectors() throws Exception
   {
      MBeanServer server = super.getServer();
      ObjectName service = new ObjectName(catalinaDomain + ":type=Service,serviceName=jboss.web");
      Object[] args = {};
      String[] sig = {};
      Connector[] connectors = (Connector[]) server.invoke(service,
         "findConnectors", args, sig);
      for (int n = 0; n < connectors.length; n++)
      {
         Lifecycle lc = (Lifecycle) connectors[n];
         lc.stop();
      }
   }

   /**
    * Used to receive notification of the server start msg so the tomcat
    * connectors can be started after all web apps are deployed.
    */
   public void handleNotification(Notification msg, Object handback)
   {
      String type = msg.getType();
      if (type.equals(Server.START_NOTIFICATION_TYPE))
      {
         log.debug("Saw " + type + " notification, starting connectors");
         try
         {
            startConnectors();
         }
         catch (Exception e)
         {
            log.warn("Failed to startConnectors", e);
         }
      }
   }

   /**
    * A trival extension of URLClassLoader that uses an empty URL[] as its
    * classpath so that all work is delegated to its parent.
    */
   static class ENCLoader extends URLClassLoader
   {
      private boolean parentFirst;

      ENCLoader(URL[] urls, ClassLoader parent, boolean parentFirst)
      {
         super(urls, parent);
         this.parentFirst = parentFirst;
      }

      protected synchronized Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException
      {
         if (parentFirst) return super.loadClass(name, resolve);
         try
         {
            Class clazz = findClass(name);
            if (resolve) {
                resolveClass(clazz);
            }
            return clazz;
         }
         catch (ClassNotFoundException e)
         {
            return getParent().loadClass(name);
         }
      }

   }

   public static ClassLoader getTmpLoader(URL warURL, ClassLoader parent, boolean parentFirst) throws Exception
   {

      String path = warURL.getFile();
      ArrayList<URL> list = new ArrayList<URL>();
      File classesDir = new File(path, "WEB-INF/classes");
      if (classesDir.exists())
      {
         list.add(classesDir.toURL());
      }
      File libDir = new File(path, "WEB-INF/lib");
      if (libDir.exists())
      {
         File[] jars = libDir.listFiles();
         int length = jars != null ? jars.length : 0;
         for (int j = 0; j < length; j++)
         {
            File jar = jars[j];
            if(jar.getAbsolutePath().endsWith(".jar"))
            {
               list.add(jar.toURL());
            }
         }
      }
      ENCLoader loader = new ENCLoader(list.toArray(new URL[list.size()]), parent, parentFirst);
      return loader;

   }

   /**
    * Create a tomcat war deployment bean for the deployment unit/metaData.
    * @param unit - the current web app deployment unit
    * @param metaData - the parsed metdata for the web app deployment
    * @return TomcatDeployment instnace
    */
   @Override
   public AbstractWarDeployment getDeployment(VFSDeploymentUnit unit, JBossWebMetaData metaData)
      throws Exception
   {
      AbstractWarDeployment deployment = (AbstractWarDeployment) deploymentClass.newInstance();
      
      DeployerConfig config = new DeployerConfig();
      config.setDefaultSecurityDomain(this.defaultSecurityDomain);
      config.setSubjectAttributeName(this.subjectAttributeName);
      config.setServiceClassLoader(getClass().getClassLoader());
      config.setManagerClass(this.managerClass);
      config.setJava2ClassLoadingCompliance(this.java2ClassLoadingCompliance);
      config.setUnpackWars(this.unpackWars);
      config.setLenientEjbLink(this.lenientEjbLink);
      config.setCatalinaDomain(catalinaDomain);
      config.setContextClassName(contextClassName);
      config.setServiceName(null);
      config.setSubjectAttributeName(this.subjectAttributeName);
      config.setUseJBossWebLoader(this.useJBossWebLoader);
      config.setAllowSelfPrivilegedWebApps(this.allowSelfPrivilegedWebApps); 
      config.setSecurityManagerService(this.secMgrService);
      config.setFilteredPackages(filteredPackages);
      config.setSharedMetaData(sharedMetaData);
      
      config.setSecurityContextClassName(securityContextClassName);
      config.setSecurityManagement(securityManagement);

      //Check if there are any xacml policy files
      VirtualFile vf = unit.getMetaDataFile("jboss-xacml-policy.xml");
      if(vf != null)
        config.setXacmlPolicyURL(vf.toURL());
      
      // Add a dependency on the webserver itself
      List<String> depends = metaData.getDepends();
      if (depends == null)
         depends = new ArrayList<String>();
      depends.add(TomcatDeployerMBean.OBJECT_NAME.getCanonicalName());
      metaData.setDepends(depends);     
      
      deployment.setServer(super.getServer());
      deployment.init(config);

      return deployment;
   }
   
   public void create() throws Exception
   {
//      MBeanServer server = MBeanServerLocator.locateJBoss();
//      if (server != null)
//         server.registerMBean(this, OBJECT_NAME);
   }

   public void destroy() throws Exception
   {
//      MBeanServer server = MBeanServerLocator.locateJBoss();
//      if (server != null)
//         server.unregisterMBean(OBJECT_NAME);
   } 
}
