/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session;

import java.util.Map;

import org.jboss.cache.Fqn;
import org.jboss.cache.buddyreplication.BuddyManager;
import org.jboss.cache.notifications.annotation.CacheStarted;
import org.jboss.cache.notifications.annotation.CacheStopped;
import org.jboss.cache.notifications.annotation.NodeModified;
import org.jboss.cache.notifications.annotation.NodeRemoved;
import org.jboss.cache.notifications.event.CacheStartedEvent;
import org.jboss.cache.notifications.event.CacheStoppedEvent;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.jboss.cache.notifications.event.NodeRemovedEvent;
import org.jboss.logging.Logger;
import org.jboss.metadata.web.jboss.ReplicationGranularity;

@org.jboss.cache.notifications.annotation.CacheListener
public class CacheListener extends CacheListenerBase
{
   // Element within an FQN that is the root of a Pojo attribute map
   private static final int POJO_ATTRIBUTE_FQN_INDEX = SESSION_ID_FQN_INDEX + 1;
   // Element within an FQN that is the root of an individual Pojo attribute
   private static final int POJO_KEY_FQN_INDEX = POJO_ATTRIBUTE_FQN_INDEX + 1;
   // Size of an Fqn that points to the root of a session
   private static final int POJO_KEY_FQN_SIZE = POJO_KEY_FQN_INDEX + 1;
   private static Logger log_ = Logger.getLogger(CacheListener.class);
   private boolean fieldBased_;

   CacheListener(JBossCacheWrapper wrapper, JBossCacheManager manager, String hostname, String webapp)
   {      
      super(manager, hostname, webapp);
      fieldBased_ = (manager_.getReplicationGranularity() == ReplicationGranularity.FIELD);
   }

   // --------------- TreeCacheListener methods ------------------------------------

   @NodeRemoved
   public void nodeRemoved(NodeRemovedEvent event)
   {      
      if (event.isPre() || event.isOriginLocal())
         return;
      
      Fqn fqn = event.getFqn();
      boolean isBuddy = isBuddyFqn(fqn);
      
      // Potential removal of a Pojo where we need to unregister as an Observer.
      if (fieldBased_ 
            && isFqnPojoKeySized(fqn, isBuddy)
            && isFqnForOurWebapp(fqn, isBuddy))
      {
         String sessId = getIdFromFqn(fqn, isBuddy);
         String attrKey = getPojoKeyFromFqn(fqn, isBuddy);
         manager_.processRemoteAttributeRemoval(sessId, attrKey);
      }
      else if(isFqnSessionRootSized(fqn, isBuddy) 
                  && isFqnForOurWebapp(fqn, isBuddy))
      {
         // A session has been invalidated from another node;
         // need to inform manager
         String sessId = getIdFromFqn(fqn, isBuddy);
         manager_.processRemoteInvalidation(sessId);
      }
   }
   
   @NodeModified
   public void nodeModified(NodeModifiedEvent event)
   {          
      if (event.isPre() || event.isOriginLocal())
         return;
      
      Fqn fqn = event.getFqn();
      boolean isBuddy = isBuddyFqn(fqn);      
      // We only care if there is a chance this is for a session root
      if (!isFqnSessionRootSized(fqn, isBuddy))
         return;
      
      // We only care if this is for our webapp
      if (!isFqnForOurWebapp(fqn, isBuddy))
         return;

      // Query if we have version value in the distributed cache. 
      // If we have a version value, compare the version and invalidate if necessary.
      Integer version = (Integer) event.getData().get(JBossCacheService.VERSION_KEY);
      if(version != null)
      {
         String realId = getIdFromFqn(fqn, isBuddy);
         
         ClusteredSession session = manager_.findLocalSession(realId);
         if (session == null)
         {
            String owner = isBuddy ? getBuddyOwner(fqn) : null;
            // Notify the manager that an unloaded session has been updated
            manager_.unloadedSessionChanged(realId, owner);
         }
         else if (session.isNewData(version.intValue()))
         {
            // Need to invalidate the loaded session
            session.setOutdatedVersion(version.intValue());
            if(log_.isTraceEnabled())
            {
               log_.trace("nodeDirty(): session in-memory data is " +
                          "invalidated with id: " + realId + " and version: " +
                          version.intValue());
            }
         }
         else if (!isBuddy)
         {
            log_.warn("Possible concurrency problem: Replicated version id " + 
                      version + " matches in-memory version for session " + realId); 
         }
         /*else 
         {
            We have a local session but got a modification for the buddy tree.
            This means another node is in the process of taking over the session;
            we don't worry about it
         }
          */
      }
      else
      {
//         log_.warn("No VERSION_KEY attribute found in " + fqn);
      }
   }

   @CacheStarted
   public void cacheStarted(CacheStartedEvent event)
   {
      // TODO will need to synchronize this with local sessions
      // BES -- 2007/07/03 Not sure what this means
   }

   @CacheStopped
   public void cacheStopped(CacheStoppedEvent event)
   {
      // TODO will need to synchronize this with local sessions
      // BES -- 2007/07/03 Not sure what this means
   }
   
   public static boolean isFqnPojoKeySized(Fqn fqn, boolean isBuddy)
   {
      return fqn.size() == (isBuddy ? BUDDY_BACKUP_ROOT_OWNER_SIZE + POJO_KEY_FQN_SIZE : POJO_KEY_FQN_SIZE);
   }
   
   public static String getPojoKeyFromFqn(Fqn fqn, boolean isBuddy)
   {
      return (String) fqn.get(isBuddy ? BUDDY_BACKUP_ROOT_OWNER_SIZE + POJO_KEY_FQN_INDEX: POJO_KEY_FQN_INDEX);
   }
}
