/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session;

import org.jboss.invocation.MarshalledValueInputStream;
import org.jboss.invocation.MarshalledValueOutputStream;
import org.jboss.invocation.MarshalledValue;
import org.jboss.logging.Logger;
import org.jboss.serial.io.JBossObjectInputStreamSharedTree;
import org.jboss.serial.io.JBossObjectOutputStreamSharedTree;
import org.jboss.serial.io.MarshalledObject;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;

/**
 * Factory class for creating object output and input streams, 
 * switching between JDK Serialization and JBoss Serialization classes. 
 * Using MarshalledValue to replace Serializable used inside different 
 * web app class loader context. Assuming that the caller classes are already
 * handling the switch between web app class loader context. 
 *  
 * 
 * @author <a href="hmesha@novell.com">Hany Mesha</a>
 */
public class SessionSerializationFactory
{
   static Logger log_ = Logger.getLogger(SessionSerializationFactory.class);
   static boolean useJBossSerialization = false;

   static
   {
       String useJBossSerializationStr = System.getProperty("session.serialization.jboss", "false");
       useJBossSerialization = Boolean.valueOf(useJBossSerializationStr).booleanValue();
       try
       {
          if (useJBossSerialization)
          {
             log_.debug("Using JBossSerialization for web session replication");
          }
       }
       catch (Throwable ignored) 
       {
          // we don't fail a static initializer due to a debug log stmt
       }
   }

   public static ObjectOutputStream createObjectOutputStream(OutputStream out) throws IOException
   {
      return useJBossSerialization ? new JBossObjectOutputStreamSharedTree(out) : new MarshalledValueOutputStream(out);
   }

   public static ObjectInputStream createObjectInputStream(byte[] bytes) throws IOException
   {
      ByteArrayInputStream in = new ByteArrayInputStream(bytes);
      return useJBossSerialization ? new JBossObjectInputStreamSharedTree(in) : new MarshalledValueInputStream(in);
   }
   
   public static MarshalledValue createMarshalledValue(Object o) throws IOException
   {
      return new MarshalledValue (o);
   }
   
   public static MarshalledObject createMarshalledObject(Object o) throws IOException
   {
      return new MarshalledObject(o);
   }

   public static boolean useJBossSerialization()
   {
       return useJBossSerialization;
   }

}
