/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service;

// $Id: $

import java.lang.annotation.Annotation;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import org.apache.InstanceManager;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.ejb3.Container;
import org.jboss.ejb3.DependencyPolicy;
import org.jboss.ejb3.DeploymentScope;
import org.jboss.ejb3.deployers.JBoss5DependencyPolicy;
import org.jboss.ejb3.enc.DeploymentPersistenceUnitResolver;
import org.jboss.ejb3.entity.PersistenceUnitDeployment;
import org.jboss.ejb3.javaee.AbstractJavaEEComponent;
import org.jboss.ejb3.javaee.SimpleJavaEEModule;
import org.jboss.injection.DependsHandler;
import org.jboss.injection.EJBHandler;
import org.jboss.injection.EncInjector;
import org.jboss.injection.InjectionContainer;
import org.jboss.injection.InjectionHandler;
import org.jboss.injection.InjectionUtil;
import org.jboss.injection.Injector;
import org.jboss.injection.PersistenceContextHandler;
import org.jboss.injection.PersistenceUnitHandler;
import org.jboss.injection.WebServiceRefHandler;
import org.jboss.logging.Logger;
import org.jboss.metadata.javaee.spec.Environment;
import org.jboss.metadata.web.jboss.JBossServletsMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.metadata.web.spec.FilterMetaData;
import org.jboss.metadata.web.spec.FiltersMetaData;
import org.jboss.metadata.web.spec.ListenerMetaData;
import org.jboss.metadata.web.spec.ServletMetaData;
import org.jboss.virtual.VirtualFile;
import org.jboss.web.WebApplication;
import org.jboss.web.tomcat.service.injection.WebResourceHandler;

/**
 * Comment
 *
 * @author <a href="mailto:bill@jboss.org">Bill Burke</a>
 * @author adrian@jboss.org
 * @author Scott.Stark@jboss.org
 * @version $Revision: 1.17 $
 */
public class TomcatInjectionContainer extends AbstractJavaEEComponent implements InjectionContainer, InstanceManager
{
   private static final Logger log = Logger.getLogger(TomcatInjectionContainer.class);

   private static class EncMap extends HashMap<String, EncInjector>
   {
      private HashMap<String, EncInjector> added;

      public void recordAdded()
      {
         added = new HashMap<String, EncInjector>();
      }

      public void clearAdded()
      {
         added = null;
      }

      public Map<String, EncInjector> getAdded()
      {
         return added;
      }

      @Override
      public EncInjector put(String key, EncInjector value)
      {
         if (added != null)
            added.put(key, value);
         return super.put(key, value);
      }

      @Override
      public void putAll(Map<? extends String, ? extends EncInjector> m)
      {
         if (added != null)
            added.putAll(m);
         super.putAll(m);
      }
   }

   protected EncMap encInjectors = new EncMap();
   protected Map<String, Map<AccessibleObject, Injector>> encInjections = new HashMap<String, Map<AccessibleObject, Injector>>();
   protected Map<String, Map<AccessibleObject, Injector>> resolvedClassInjections = new HashMap<String, Map<AccessibleObject, Injector>>();

   protected List<PersistenceUnitDeployment> persistenceUnitDeployments = new ArrayList<PersistenceUnitDeployment>();
   protected DeploymentPersistenceUnitResolver persistenceUnitResolver;
   protected DependencyPolicy dependencyPolicy = new JBoss5DependencyPolicy(this);
   protected Collection<InjectionHandler<Environment>> handlers;
   protected VFSDeploymentUnit unit;
   protected ClassLoader webLoader;
   protected WebApplication appInfo;
   protected JBossWebMetaData webDD;
   protected org.apache.catalina.Context catalinaContext;

   public TomcatInjectionContainer(WebApplication appInfo, VFSDeploymentUnit unit, org.apache.catalina.Context catalinaContext)
   {
      super(new SimpleJavaEEModule(appInfo.getName()));
      
      this.unit = unit;
      DeploymentScope deploymentScope = null;
      this.appInfo = appInfo;
      this.catalinaContext = catalinaContext;
      LinkedHashMap empty = new LinkedHashMap(0);
      persistenceUnitResolver = new DeploymentPersistenceUnitResolver(persistenceUnitDeployments, deploymentScope, empty);
      
      this.webDD = unit.getAttachment(JBossWebMetaData.class);
      assert this.webDD != null : "webDD is null (no JBossWebMetaData attachment in VFSDeploymentUnit)";
   }

   public Environment getEnvironmentRefGroup()
   {
      return webDD.getJndiEnvironmentRefsGroup();
   }

   public Object newInstance(String className)
      throws IllegalAccessException, InvocationTargetException, NamingException,
      InstantiationException, ClassNotFoundException
   {
      ClassLoader loader = catalinaContext.getLoader().getClassLoader();
      Class<?> clazz = loader.loadClass(className);
      Object instance = clazz.newInstance();
	   if (!catalinaContext.getIgnoreAnnotations())
	   {
		   processAnnotations(instance);
		   postConstruct(instance);
	   }
       return instance;
   }

   public Object newInstance(String className, ClassLoader classLoader) throws IllegalAccessException, InvocationTargetException, NamingException, InstantiationException, ClassNotFoundException {
       Class<?> clazz = classLoader.loadClass(className);
       Object instance = clazz.newInstance();
	   if (!catalinaContext.getIgnoreAnnotations())
	   {
		   processAnnotations(instance);
		   postConstruct(instance);
	   }
       return instance;
   }

   public void newInstance(Object instance) throws IllegalAccessException, InvocationTargetException, NamingException {
	   if (!catalinaContext.getIgnoreAnnotations())
	   {
		   processAnnotations(instance);
		   postConstruct(instance);
	   }
   }

   public void destroyInstance(Object instance) throws IllegalAccessException, InvocationTargetException {
	   if (!catalinaContext.getIgnoreAnnotations())
	   {
		   preDestroy(instance);
	   }
   }

   public void postConstruct(Object object) throws IllegalAccessException, InvocationTargetException
   {
	   // ignore for now
   }

   public void preDestroy(Object object) throws IllegalAccessException, InvocationTargetException
   {
      // ignore for now
   }

   /**
    * When we get here, we are assuming that any XML defined ENC has been set up.  We will set up more here
    * if the class being processed is a JSP
    *
    *
    * @param object
    * @throws IllegalAccessException
    * @throws InvocationTargetException
    * @throws NamingException
    */
   public void processAnnotations(Object object) throws IllegalAccessException, InvocationTargetException, NamingException
   {
	  if (log.isTraceEnabled()) {
		 log.trace("**************** Processing annotations for: " + object.getClass().getName());
	  }
      Map<AccessibleObject, Injector> injectors = resolvedClassInjections.get(object.getClass().getName());
      if (injectors == null)
      {
    	  if (log.isTraceEnabled())
    	  {
    		  log.trace("-- there was no prior annotation preprocessing done");
    	  }
         encInjectors.recordAdded();
         // let's assume this is a JSP or some other artifact that cannot be found within XML
         injectors = InjectionUtil.processAnnotations(this, handlers, object.getClass());
         resolvedClassInjections.put(object.getClass().getName(), injectors);

         // only execute injectors that were added additionally
         if (encInjectors.getAdded().size() > 0)
         {
            for (EncInjector encInjector : encInjectors.getAdded().values())
            {
               encInjector.inject(this);
            }
            encInjectors.clearAdded();
         }
      }
      if (injectors == null || injectors.size() == 0)
      {
    	  if (log.isTraceEnabled()) 
    	  {
    		  log.trace("-- no injectors found: " + injectors);
    	  }
         return;
      }

	  if (log.isTraceEnabled())
	  {
		  log.trace("-- doing injections");
	  }
      for (Injector injector : injectors.values())
      {
         injector.inject(object);
      }
   }

   public void populateEnc(ClassLoader loader)
   {
      for (EncInjector injector : encInjectors.values())
      {
         injector.inject(this);
      }
   }

   private void processClass(String className, ClassLoader webLoader) throws ClassNotFoundException
   {
      if (resolvedClassInjections.containsKey(className))
         return;
      Class<?> cls = webLoader.loadClass(className);
      Map<AccessibleObject, Injector> tmp = InjectionUtil.processAnnotations(this, handlers, cls);
      resolvedClassInjections.put(className, tmp);
   }
   
   private void processFilters(FiltersMetaData filters, ClassLoader webLoader)
   {
      if(filters == null) return;
      for (FilterMetaData filter : filters)
      {
         try
         {
            processClass(filter.getFilterClass(), webLoader);
         }
         catch (ClassNotFoundException e)
         {
            throw new RuntimeException("could not find filter class in classpath", e);
         }
      }
   }
   
   private void processListeners(List<ListenerMetaData> listeners, ClassLoader webLoader)
   {
      if(listeners == null) return;
      for (ListenerMetaData listener : listeners)
      {
         try
         {
            processClass(listener.getListenerClass(), webLoader);
         }
         catch (ClassNotFoundException e)
         {
            throw new RuntimeException("could not find listener class in classpath", e);
         }
      }
   }
   
   /**
    * introspects EJB container to find all dependencies
    * and initialize any extra metadata.
    * <p/>
    * This must be called before container is registered with any microcontainer
    *
    * @param dependencyPolicy
    */
   public void processMetadata()
   {
      // XML must be done first so that any annotation overrides are initialized

      // todo injection handlers should be pluggable from XML
      handlers = new ArrayList<InjectionHandler<Environment>>();
      handlers.add(new EJBHandler<Environment>());
      handlers.add(new DependsHandler<Environment>());
      // FIXME: is this allowed in servlets?
      //handlers.add(new JndiInjectHandler<Environment>());
      handlers.add(new PersistenceContextHandler<Environment>());
      handlers.add(new PersistenceUnitHandler<Environment>());
      handlers.add(new WebResourceHandler<Environment>());
      handlers.add(new WebServiceRefHandler<Environment>());

      ClassLoader old = Thread.currentThread().getContextClassLoader();
      ClassLoader webLoader = getClassloader();
      Thread.currentThread().setContextClassLoader(webLoader);
      try
      {
         for (InjectionHandler<Environment> handler : handlers)
            handler.loadXml(webDD.getJndiEnvironmentRefsGroup(), this);

         processServlets(webDD.getServlets(), webLoader);
         processFilters(webDD.getFilters(), webLoader);
         processListeners(webDD.getListeners(), webLoader);
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }
   }

   private void processServlets(JBossServletsMetaData servlets, ClassLoader webLoader)
   {
      if(servlets == null) return;
      for (ServletMetaData servlet : servlets)
      {
         try
         {
            if (servlet.getServletClass() == null)
               continue; // jsp
            processClass(servlet.getServletClass(), webLoader);
         }
         catch (ClassNotFoundException e)
         {
            log.warn("could not find servlet class " + servlet.getServletClass() + " in classpath when processing annotations.");
         }
      }
   }
   
   public Map<String, EncInjector> getEncInjectors()
   {
      return encInjectors;
   }

   public Map<String, Map<AccessibleObject, Injector>> getEncInjections()
   {
      return encInjections;
   }

   // EncInjectors/Handlers may need to add extra instance injectors
   public List<Injector> getInjectors()
   {
      return new ArrayList<Injector>(); // no equivalent in WAR
   }

   public VirtualFile getRootFile()
   {
      return unit.getRoot();
   }

   public String getIdentifier()
   {
      return unit.getSimpleName();
   }

   public String getDeploymentDescriptorType()
   {
      return "web.xml";
   }

   public ClassLoader getClassloader()
   {
      return webLoader;
   }

   public void setClassLoader(ClassLoader loader)
   {
      this.webLoader = loader;
   }

   public Context getEnc()
   {
      ClassLoader old = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(getClassloader());
         try
         {
            return (Context)new InitialContext().lookup("java:comp");
         }
         catch (NamingException e)
         {
            throw new RuntimeException(e);
         }
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }
   }

   public PersistenceUnitDeployment getPersistenceUnitDeployment(String unitName) throws NameNotFoundException
   {
      return persistenceUnitResolver.getPersistenceUnitDeployment(unitName);
   }

   public boolean hasJNDIBinding(String jndiName)
   {
      return false;
   }
   
   public Container resolveEjbContainer(String link, Class<?> businessIntf)
   {
      return null;
   }

   public Container resolveEjbContainer(Class<?> businessIntf) throws NameNotFoundException
   {
      return null;
   }

   public String getEjbJndiName(Class<?> businessIntf) throws NameNotFoundException
   {
      throw new IllegalStateException("Resolution should not happen via injection container");

   }

   public String getEjbJndiName(String link, Class<?> businessIntf)
   {
      throw new IllegalStateException("Resolution should not happen via injection container");
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType, Class<?> clazz)
   {
      return clazz.getAnnotation(annotationType);
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType, Class<?> clazz, Method method)
   {
      return method.getAnnotation(annotationType);
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType, Method method)
   {
      return method.getAnnotation(annotationType);
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType, Class<?> clazz, Field field)
   {
      return field.getAnnotation(annotationType);
   }

   public <T extends Annotation> T getAnnotation(Class<T> annotationType, Field field)
   {
      return field.getAnnotation(annotationType);
   }

   public DependencyPolicy getDependencyPolicy()
   {
      return dependencyPolicy;
   }

   public String resolveMessageDestination(String link)
   {
      throw new IllegalStateException("Resolution should not happen via injection container");
   }
}