/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session;

import java.util.HashMap;
import java.util.Map;


/**
 * Implementation of a clustered session for the JBossCacheManager. The replication granularity
 * level is session based; that is, we replicate per whole session object.
 * We use JBossCache for our internal replicated data store.
 * The internal structure in JBossCache is as follows:
 * <pre>
 * /JSESSION
 *    /hostname
 *       /web_app_path    (path + session id is unique)
 *          /id    Map(id, session)
 *                    (VERSION_KEY, version)  // Used for version tracking. version is an Integer.
 * </pre>
 * <p/>
 * Note that the isolation level of the cache dictates the
 * concurrency behavior.</p>
 *
 * @author Ben Wang
 * @author Brian Stansberry
 * 
 * @version $Revision: 56542 $
 */
class SessionBasedClusteredSession
   extends JBossCacheClusteredSession
{
   static final long serialVersionUID = 3200976125245487256L;

   /**
    * Descriptive information describing this Session implementation.
    */
   protected static final String info = "SessionBasedClusteredSession/1.0";

   public SessionBasedClusteredSession(JBossCacheManager manager)
   {
      super(manager);
   }

   // ---------------------------------------------- Overridden Public Methods

   /**
    * Return a string representation of this object.
    */
   public String toString()
   {
      StringBuffer sb = new StringBuffer();
      sb.append("SessionBasedClusteredSession[");
      sb.append(super.toString());
      sb.append("]");
      return (sb.toString());

   }

   public void removeMyself()
   {
      proxy_.removeSession(realId, false);
   }

   public void removeMyselfLocal()
   {
      proxy_.removeSessionLocal(realId, false);
   }

   // ----------------------------------------------HttpSession Public Methods
   
   /**
    * Does nothing -- all attributes are populated already
    */
   protected void populateAttributes()
   {
      // no-op
   }

   protected Object getJBossInternalAttribute(String name)
   {
      Object result = attributes.get(name);

      // Do dirty check even if result is null, as w/ SET_AND_GET null
      // still makes us dirty (ensures timely replication w/o using ACCESS)
      if (isGetDirty(result))
      {
         sessionAttributesDirty();
      }

      return result;

   }

   protected Object removeJBossInternalAttribute(String name, 
                                                 boolean localCall, 
                                                 boolean localOnly)
   {
      if (localCall)
         sessionAttributesDirty();
      return attributes.remove(name);
   }

   protected Map getJBossInternalAttributes()
   {
      return attributes;
   }

   protected Object setJBossInternalAttribute(String name, Object value)
   {
      sessionAttributesDirty();
      return attributes.put(name, value);
   }

   @Override
   public Map<String, Object> getSessionAttributeMap()
   {
      Map attrs = new HashMap(attributes);
      removeExcludedAttributes(attrs);
      return attrs;
   }

   protected void update(Integer version, SessionTimestamp timestamp, 
         SessionMetadata metadata, Map updatedAttrs)
   {
      // A new session may not have sent over any attributes
      if (updatedAttrs == null)
      {
         updatedAttrs = new HashMap();
      }

      super.update(version, timestamp, metadata, updatedAttrs);
      
      Map excluded = removeExcludedAttributes(attributes);
      if (excluded != null)
         updatedAttrs.putAll(excluded);
      
      this.attributes = updatedAttrs;
   }

}
