/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.spi.tracker;

import java.util.Set;

import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;

/**
 * Query contexts.
 *
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 */
public interface ContextQueries
{
   /**
    * Get a context that has reached at least the passed in state
    *
    * @param name the name of the component
    * @param state the state (pass null for any state)
    * @return the context
    */
   ControllerContext getContext(Object name, ControllerState state);

   /**
    * Get an installed context
    *
    * @param name the name of the component
    * @return the context
    */
   ControllerContext getInstalledContext(Object name);

   /**
    * Get the contexts not installed
    *
    * @return Set<ControllerContext>
    */
   Set<ControllerContext> getNotInstalled();

   /**
    * Get the contexts in certain state
    *
    * @param state controller state to get contexts for
    * @return set of contexts in certain state
    */
   Set<ControllerContext> getContextsByState(ControllerState state);

   /**
    * Get all instantiated contexts of a given type
    *
    * @param clazz the type
    * @return the contexts
    */
   Set<ControllerContext> getInstantiatedContexts(Class<?> clazz);

   /**
    * Get all contexts of a type which are in the given state or above.
    * Note: state must not be null
    *
    * @param clazz the type
    * @param state the required state
    * @return the contexts
    * @throws IllegalArgumentException for null state
    */
   Set<ControllerContext> getContexts(Class<?> clazz, ControllerState state);

   /**
    * Get an instantiated context that is of the type passed in.
    * If zero or multiple instances match class clazz
    * a warning is issued, but no throwable is thrown
    *
    * @param clazz the type
    * @return context whose target is instance of this class clazz param or null if zero or multiple such instances
    */
   ControllerContext getContextByClass(Class<?> clazz);

   /**
    * Filter existing contexts.
    *
    * @param contexts the contexts
    * @param filter the filter
    * @return filtered contexts
    */
   Set<ControllerContext> filter(Iterable<ControllerContext> contexts, ContextFilter filter);

   /**
    * Get the exposed classes for context.
    *
    * @param context the context
    * @return exposed classes
    */
   Set<Class<?>> getExposedClasses(ControllerContext context);
}