/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.dependency.plugins;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;

import org.jboss.dependency.spi.CallbackItem;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.ControllerStateModel;
import org.jboss.dependency.spi.DependencyInfo;
import org.jboss.dependency.spi.DependencyItem;
import org.jboss.dependency.spi.LifecycleCallbackItem;
import org.jboss.util.JBossObject;
import org.jboss.util.JBossStringBuilder;
import org.jboss.util.collection.ConcurrentSet;

/**
 * A DependencyInfo.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.com">Ales Justin</a>
 * @version $Revision: 104825 $
 */
public class AbstractDependencyInfo extends JBossObject implements TrackingDependencyInfo
{
   /** My dependencies */
   private volatile Set<DependencyItem> iDependOn;

   /** Dependencies referencing me */
   private volatile Set<DependencyItem> dependsOnMe;

   /** Install callbacks */
   private volatile Set<CallbackItem<?>> installCallbacks;

   /** Uninstall callbacks */
   private volatile Set<CallbackItem<?>> uninstallCallbacks;
   
   /** Lifecycle callbacks */
   private volatile List<LifecycleCallbackItem> lifecycleCallbacks;
   
   /** Whether this is an autowire candidate */
   private volatile boolean autowireCandidate = true;
   
   /** Whether we are tracking */
   volatile boolean tracking = true;
   
   /** The unresolved dependencies by state */
   private volatile Map<ControllerState, Set<DependencyItem>> unresolved;
   
   /** The semiresolved dependencies by state */
   private volatile Map<ControllerState, Set<DependencyItem>> semiResolved;

   /** Whether we have been modified */
   private volatile boolean modified = false;
   
   /**
    * Create an abstract dependency info
    */
   public AbstractDependencyInfo()
   {
   }

   private void removeUnresolved(DependencyItem dependency, ControllerState whenRequired)
   {
      Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
      if (unresolved != null)
      {
         Set<DependencyItem> items = unresolved.get(whenRequired);
         if (items != null)
         {
            items.remove(dependency);
            if (items.isEmpty())
               unresolved.remove(whenRequired);
         }
      }
   }

   private void removeSemiResolved(DependencyItem item, ControllerState whenRequired)
   {
      Map<ControllerState, Set<DependencyItem>> semiResolved = this.semiResolved;
      if (semiResolved != null)
      {
         Set<DependencyItem> items = semiResolved.get(whenRequired);
         if (items != null)
         {
            items.remove(item);
            if (items.isEmpty())
               semiResolved.remove(whenRequired);
         }
      }
   }

   public void resolved(DependencyItem item)
   {
      if (tracking)
      {
         ControllerState whenRequired = item.getWhenRequired();
         removeUnresolved(item, whenRequired);
         removeSemiResolved(item, whenRequired);
      }
   }

   public void unresolved(DependencyItem item)
   {
      if (tracking)
      {
         ControllerState whenRequired = item.getWhenRequired();
         Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
         if (unresolved == null)
         {
            this.unresolved = new ConcurrentHashMap<ControllerState, Set<DependencyItem>>();
            unresolved = this.unresolved;
         }
         Set<DependencyItem> items = unresolved.get(whenRequired);
         if (items == null)
         {
            items = new ConcurrentSet<DependencyItem>();
            unresolved.put(whenRequired, items);
         }
         items.add(item);
         removeSemiResolved(item, whenRequired);
      }
   }

   public void semiResolved(DependencyItem item)
   {
      if (tracking)
      {
         ControllerState whenRequired = item.getWhenRequired();
         Map<ControllerState, Set<DependencyItem>> semiResolved = this.semiResolved;
         if (semiResolved == null)
         {
            this.semiResolved = new ConcurrentHashMap<ControllerState, Set<DependencyItem>>();
            semiResolved = this.semiResolved;
         }
         Set<DependencyItem> items = semiResolved.get(whenRequired);
         if (items == null)
         {
            items = new ConcurrentSet<DependencyItem>();
            semiResolved.put(whenRequired, items);
         }
         items.add(item);
         removeUnresolved(item, whenRequired);
      }
   }

   public ResolvedState semiResolve(TrackingDependencyItem item)
   {
      if (item == null)
         throw new IllegalArgumentException("Null item");

      // Should this be an error?
      ControllerContext other = item.getSemiResolvedContext();
      if (other == null)
         return ResolvedState.UNRESOLVED;

      // Something strange?
      Controller controller = other.getController();
      if (controller == null)
         return ResolvedState.UNRESOLVED;
      
      // It is already in the correct state?
      ControllerState dependentState = item.getDependentState();
      if (dependentState == null)
         dependentState = ControllerState.INSTALLED;
      ControllerStateModel states = controller.getStates();
      ControllerState otherState = other.getState();
      if (states.isBeforeState(otherState, dependentState) == false)
         return ResolvedState.RESOLVED;
      
      // Can't do a full semi-resolve
      if (tracking == false)
         return ResolvedState.UNRESOLVED;
      
      // Can't be semi-resolved if the context is not in the previous state
      ControllerState previousState = states.getPreviousState(dependentState);
      if (otherState.equals(previousState) == false)
         return ResolvedState.UNRESOLVED;

      // It is at least in the semi-resolved state
      item.setResolved(ResolvedState.SEMI_RESOLVED);
      
      // Now check whether the other contexts are transitively semi-resolved
      Set<ControllerContext> alreadyChecked = new HashSet<ControllerContext>();
      return checkSemiResolvedIsFullyResolved(other, item.getDependentState(), alreadyChecked);
   }

   /**
    * Checks whether the context and its transitive contexts are fully or semi-resolved
    * 
    * @param context the context to check
    * @param dependentState the dependentState of the item
    * @param alreadyChecked the contexts already checked to avoid recursion
    * @return the resolved state
    */
   private ResolvedState checkSemiResolvedIsFullyResolved(ControllerContext context, ControllerState dependentState, Set<ControllerContext> alreadyChecked)
   {
      // Avoid recursion
      alreadyChecked.add(context);

      // Something strange going on?
      Controller controller = context.getController();
      if (controller == null)
         return ResolvedState.UNRESOLVED;
      
      // Work out its state
      ControllerStateModel states = controller.getStates();
      ControllerState state = context.getState();
      
      // Has it been resolved?
      if (dependentState == null)
         dependentState = ControllerState.INSTALLED;
      if (states.isBeforeState(state, dependentState) == false)
         return ResolvedState.RESOLVED;
      
      // Its not going to move
      ControllerState nextState = states.getNextState(state);
      ControllerState requiredState = context.getRequiredState();
      if (states.isBeforeState(requiredState, nextState))
         return ResolvedState.SEMI_RESOLVED;
      
      DependencyInfo info = context.getDependencyInfo();
      // No dependencies so we are done
      if (info == null)
         return ResolvedState.RESOLVED;

      // It has unresolved dependencies
      if (info.getUnresolvedDependencies(nextState).isEmpty() == false)
         return ResolvedState.SEMI_RESOLVED;

      // Its not tracking and no unresolved dependencies so we are done
      if (info instanceof TrackingDependencyInfo == false)
         return ResolvedState.RESOLVED;
      
      // Now transitively check the semi resolved dependencies
      TrackingDependencyInfo trackingInfo = (TrackingDependencyInfo) info;
      Set<DependencyItem> items = trackingInfo.getSemiResolvedDependencies(requiredState);
      if (items.isEmpty() == false)
      {
         for (DependencyItem item : items)
         {
            // Shouldn't really happen?
            if (item instanceof TrackingDependencyItem == false)
               return ResolvedState.UNRESOLVED;
            
            TrackingDependencyItem trackingItem = (TrackingDependencyItem) item;
            ControllerContext other = trackingItem.getSemiResolvedContext();
            // This shouldn't really happen either?
            if (other == null)
               return ResolvedState.UNRESOLVED;
            
            // Only check if we haven't already checked
            if (alreadyChecked.contains(other) == false)
            {
               ResolvedState result = checkSemiResolvedIsFullyResolved(other, item.getDependentState(), alreadyChecked);
               if (result != ResolvedState.RESOLVED)
                  return result;
            }
         }
      }
      
      // All is good
      return ResolvedState.RESOLVED;
   }
   
   public Set<DependencyItem> getIDependOn(Class<?> type)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
         iDependOn = Collections.emptySet();
      if (type == null || iDependOn.isEmpty())
         return iDependOn;
      else
      {
         HashSet<DependencyItem> set = new HashSet<DependencyItem>();
         for (DependencyItem item : iDependOn)
         {
            if (type.isInstance(item))
               set.add(item);
         }
         return set;
      }
   }
   
   public void addIDependOn(DependencyItem dependency)
   {
      modified = true;
      
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
      {
         this.iDependOn = new CopyOnWriteArraySet<DependencyItem>();
         iDependOn = this.iDependOn;
      }
      iDependOn.add(dependency);
      flushJBossObjectCache();
      if (dependency instanceof TrackingDependencyItem)
      {
         ((TrackingDependencyItem) dependency).setDependencyInfo(this);
         if (tracking)
         {
            if (dependency.isResolved() == false)
            {
               ControllerState whenRequired = dependency.getWhenRequired();
               Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
               if (unresolved == null)
               {
                  this.unresolved = new ConcurrentHashMap<ControllerState, Set<DependencyItem>>();
                  unresolved = this.unresolved;
               }
               Set<DependencyItem> items = unresolved.get(whenRequired);
               if (items == null)
               {
                  items = new ConcurrentSet<DependencyItem>();
                  unresolved.put(whenRequired, items);
               }
               items.add(dependency);
            }
         }
      }
      else
      {
         if (tracking)
         {
            log.debug("Cannot track " + dependency);
            tracking = false;
            unresolved = null;
            semiResolved = null;
         }
      }
   }

   public void removeIDependOn(DependencyItem dependency)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null)
         return;
      iDependOn.remove(dependency);
      flushJBossObjectCache();
      if (dependency instanceof TrackingDependencyItem)
      {
         ((TrackingDependencyItem) dependency).setDependencyInfo(null);
         if (tracking)
         {
            ControllerState whenRequired = dependency.getWhenRequired();
            removeUnresolved(dependency, whenRequired);
         }
      }
   }

   public Set<DependencyItem> getDependsOnMe(Class<?> type)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
         dependsOnMe = Collections.emptySet();
      
      if (type == null || dependsOnMe.isEmpty())
         return dependsOnMe;
      else
      {
         HashSet<DependencyItem> set = new HashSet<DependencyItem>();
         for (DependencyItem item : dependsOnMe)
         {
            if (type.isInstance(item))
               set.add(item);
         }
         return set;
      }
   }
   
   public void addDependsOnMe(DependencyItem dependency)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
      {
         this.dependsOnMe = new CopyOnWriteArraySet<DependencyItem>();
         dependsOnMe = this.dependsOnMe;
      }
      dependsOnMe.add(dependency);
      flushJBossObjectCache();
   }

   public void removeDependsOnMe(DependencyItem dependency)
   {
      Set<DependencyItem> dependsOnMe = this.dependsOnMe;
      if (dependsOnMe == null)
         return;
      dependsOnMe.remove(dependency);
      flushJBossObjectCache();
   }
   
   public boolean resolveDependencies(Controller controller, ControllerState state)
   {
      modified = false;
      if (internalResolveDependencies(controller, state) == false)
         return false;

      while (modified)
      {
         modified = false;
         if (internalResolveDependencies(controller, state) == false)
            return false;
      }
      return true;
   }
   
   protected boolean internalResolveDependencies(Controller controller, ControllerState state)
   {
      boolean resolved = true;
      Set<DependencyItem> items = getUnresolvedDependencies(state);
      if (items.isEmpty() == false)
      {
         for (DependencyItem item : items)
         {
            if (item.resolve(controller) == false)
               resolved = false;
         }
      }
      if (tracking)
      {
         items = getSemiResolvedDependencies(state);
         if (items.isEmpty() == false)
         {
            for (DependencyItem item : items)
            {
               if (item.resolve(controller) == false)
                  resolved = false;
            }
         }
      }
      return resolved;
   }

   public Set<DependencyItem> getUnresolvedDependencies(ControllerState state)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn == null || iDependOn.isEmpty())
         return Collections.emptySet();
      
      Set<DependencyItem> result = null;

      if (tracking && state != null)
      {
         Map<ControllerState, Set<DependencyItem>> unresolved = this.unresolved;
         if (unresolved != null)
            result = unresolved.get(state);
         if (result == null)
            result = Collections.emptySet();
         return result;
      }
      
      for (DependencyItem item : iDependOn)
      {
         if (state == null || state.equals(item.getWhenRequired()))
         {
            if (item.isResolved() == false)
            {
               if (result == null)
                  result = new HashSet<DependencyItem>();
               result.add(item);
            }
         }
      }
      if (result == null)
         return Collections.emptySet();
      return result;
   }

   /**
    * Get the semi resolved dependencies
    * 
    * @param state the state
    * @return the dependencies or an empty set if there is no state or not tracking dependencies
    */
   public Set<DependencyItem> getSemiResolvedDependencies(ControllerState state)
   {
      Set<DependencyItem> result = null;

      if (tracking && state != null)
      {
         Map<ControllerState, Set<DependencyItem>> semiResolved = this.semiResolved;
         if (semiResolved != null)
            result = semiResolved.get(state);
      }
      if (result == null)
         return Collections.emptySet();
      return result;
   }

   public <T> void addInstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
      {
         this.installCallbacks = new CopyOnWriteArraySet<CallbackItem<?>>();
         installCallbacks = this.installCallbacks;
      }
      installCallbacks.add(callbackItem);
      flushJBossObjectCache();
   }

   public <T> void removeInstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
         return;
      installCallbacks.remove(callbackItem);
      flushJBossObjectCache();
   }

   public Set<CallbackItem<?>> getInstallItems()
   {
      Set<CallbackItem<?>> installCallbacks = this.installCallbacks;
      if (installCallbacks == null)
         return Collections.emptySet();
      return installCallbacks;
   }

   public <T> void addUninstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
      {
         this.uninstallCallbacks = new CopyOnWriteArraySet<CallbackItem<?>>();
         uninstallCallbacks = this.uninstallCallbacks;
      }
      uninstallCallbacks.add(callbackItem);
      flushJBossObjectCache();
   }

   public <T> void removeUninstallItem(CallbackItem<T> callbackItem)
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
         return;
      uninstallCallbacks.remove(callbackItem);
      flushJBossObjectCache();
   }

   public Set<CallbackItem<?>> getUninstallItems()
   {
      Set<CallbackItem<?>> uninstallCallbacks = this.uninstallCallbacks;
      if (uninstallCallbacks == null)
         return Collections.emptySet();
      return uninstallCallbacks;
   }

   public void addLifecycleCallback(LifecycleCallbackItem lifecycleCallbackItem)
   {
      List<LifecycleCallbackItem> lifecycleCallbacks = this.lifecycleCallbacks;
      if (lifecycleCallbacks == null)
      {
         this.lifecycleCallbacks = new CopyOnWriteArrayList<LifecycleCallbackItem>();
         lifecycleCallbacks = this.lifecycleCallbacks;
      }
      lifecycleCallbacks.add(lifecycleCallbackItem);
   }
   
   public List<LifecycleCallbackItem> getLifecycleCallbacks()
   {
      List<LifecycleCallbackItem> lifecycleCallbacks = this.lifecycleCallbacks;
      if (lifecycleCallbacks == null)
         return Collections.emptyList();
      return lifecycleCallbacks;
   }
   
   public boolean isAutowireCandidate()
   {
      return autowireCandidate;
   }

   public void setAutowireCandidate(boolean candidate)
   {
      this.autowireCandidate = candidate;
   }

   public void toString(JBossStringBuilder buffer)
   {
      Set<DependencyItem> iDependOn = this.iDependOn;
      if (iDependOn != null)
         buffer.append("idependOn=").append(iDependOn);
      try
      {
         Set<DependencyItem> unresolved = getUnresolvedDependencies(null);
         if (unresolved.isEmpty() == false)
            buffer.append(" unresolved=").append(unresolved);
      }
      catch (Throwable ignored)
      {
         buffer.append(" unresolved=" + ignored);
      }
   }
}
