/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.microcontainer.jmx;

import javax.management.ObjectName;

import org.jboss.aop.microcontainer.aspects.jmx.JMX;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.logging.Logger;
import org.jboss.system.ServiceController;

/**
 * ServiceControllerStartStopLifecycleCallback. Intended for use when installing to/uninstalling from
 * the {@link org.jboss.dependency.spi.ControllerState#START} state. Example from AS bootstrap:
 * 
 * <pre>
 *    &lt;lifecycle-start xmlns="urn:jboss:aop-beans:1.0"
 *               name="JMXRegistrationAdvice"
 *               class="org.jboss.system.microcontainer.jmx.ServiceControllerStartStopLifecycleCallback"
 *               classes="@org.jboss.aop.microcontainer.aspects.jmx.JMX"
 *               manager-bean="AspectManager"
 *               manager-property="aspectManager"&gt;
 *      &lt;property name="serviceController"&gt;&lt;inject bean="JMXKernel" property="serviceController"/&gt;&lt;/property&gt;
 *   &lt;/lifecycle-start&gt;
 * </pre>
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 89990 $
 */
public class ServiceControllerStartStopLifecycleCallback extends AbstractServiceControllerLifecycleCallback
{
   /** The log */
   private static final Logger log = Logger.getLogger(ServiceControllerStartStopLifecycleCallback.class);
   
   /**
    * Tells the {@link ServiceController} to invoke the <code>start</code> lifecycle method on the MBean created
    * by {@link ServiceControllerRegistrationLifecycleCallback}. The ObjectName is given in a bean's @{@link JMX}
    * annotation. 
    * 
    * @param context the context being installed
    */
   public void install(ControllerContext context) throws Exception
   {
      ObjectName objectName = determineObjectName(context); 
      getServiceController().start(objectName);
   }
   
   /**
    * Tells the {@link ServiceController} to invoke the <code>stop</code> lifecycle method on the MBean created
    * by {@link ServiceControllerRegistrationLifecycleCallback}. The ObjectName is given in a bean's @{@link JMX}
    * annotation. 
    * 
    * @param context the context being uninstalled
    */
   public void uninstall(ControllerContext context) throws Exception
   {
      ObjectName objectName = determineObjectName(context); 
      try
      {
         getServiceController().stop(objectName);
      }
      catch(Exception e)
      {
         log.debug("Error stopping mbean", e);
      }
   }
}