/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.dependency;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;

import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.enterprise.inject.spi.InjectionTarget;

import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.DependencyMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.kernel.plugins.dependency.AbstractMetaDataVisitor;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.kernel.weld.plugins.metadata.WeldDependencyMetaData;
import org.jboss.kernel.weld.spi.annotated.MDRAnnotatedTypeFactory;
import org.jboss.logging.Logger;

/**
 * 
 * @param <T> the type of the injection target
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class WeldInjector<T> 
{
   final static Logger log = Logger.getLogger(WeldInjector.class);

   //TODO should cache the annotated types and injection targets, taking into account metadata at instance level and below

   /**
    * The web beans kernel controller context 'owning' this injector
    */
   WeldKernelControllerContext context;

   /**
    * The annotated type of the bean class
    */
   AnnotatedType<T> type;
   
   /**
    * injection target of this bean used to inject WB->MC
    */
   private final InjectionTarget<T> it;
   
   /**
    * The creational context for WB->MC injection
    */
   private CreationalContext<T> creationalContext;
   
   /**
    * True if web beans should construct this object
    */
   private boolean createInWeld;
   
   /**
    * True if web beans should construct this object
    */
   private boolean injectInWeld;

   /**
    * Constrcutor
    * 
    * @param context The controller context 'owning' this injector
    * @param clazz the class of this bean
    * @throws IllegalArgumentException if any of the parameters are null
    */
   WeldInjector(WeldKernelControllerContext context, Class<T> clazz) 
   {
      if (context == null)
         throw new IllegalArgumentException("Null context");
      if (clazz == null)
         throw new IllegalArgumentException("Null clazz");
      
      this.context = context;
      creationalContext = context.getManager().createCreationalContext(null);

      type = createMdrDecoratedAnnotatedType(clazz);
      it = getInjectionTarget(clazz);
   }

   /**
    * Gets the injection target
    */
   private InjectionTarget<T> getInjectionTarget(Class<T> clazz)
   {
      return context.getManager().createInjectionTarget(type);
   }
   
   /**
    * Check if the bean should be constructed by web beans
    * 
    * @return true if the bean should be constructed by web beans
    */
   boolean createInWeld()
   {
      return createInWeld;
   }

   /**
    * Method to be called during DESCRIBE state. Sets the annotated type in the context if
    * this bean should be registered in web beans. If the bean should have web beans injection
    * performed on it, this injector is set in the context, and WeldDependencyMetaData is
    * created for each web beans injection point.
    */
   void describe()
   {
      AnnotationMetaDataVisitor annotationsVisitor = new AnnotationMetaDataVisitor(context);
      annotationsVisitor.before();
      try
      {
         if (it.getInjectionPoints().size() > 0)
         {
            injectInWeld = true;
            BeanMetaData metaData = context.getBeanMetaData();
            for (InjectionPoint injectionPoint : it.getInjectionPoints())
            {
               BeanMetaDataBuilder builder = BeanMetaDataBuilder.createBuilder(metaData);
               
               DependencyMetaData dependency = new WeldDependencyMetaData(injectionPoint);
               if (!containsDependency(dependency))
               {
                  builder.addDependency(dependency);
                  
                  dependency.describeVisit(annotationsVisitor);
                  dependency.initialVisit(annotationsVisitor);
               }
               
               if (injectionPoint.getMember() instanceof Constructor<?>)
               {
                  createInWeld = true;
               }
            }
         }
      }
      finally
      {
         annotationsVisitor.after();
      }
   }
   
   /**
    * Instantiates the bean using web beans
    * 
    * @return the created instance
    * @throws IllegalStateException if this bean should not be created using web beans
    */
   T instantiate()
   {
      if (!createInWeld)
         throw new IllegalStateException(context.getName() + "  should not be created using web beans");
      
      T t = it.produce(creationalContext);
      return t;
   }
   
   /**
    * Performs injection from web beans into the bean
    */
   @SuppressWarnings("unchecked")
   void inject()
   {
      if (injectInWeld)
         it.inject((T)context.getTarget(), creationalContext);
   }
   
   /**
    * Unseta all fields that were set by web beans
    */
   void unconfigure()
   {
      if (injectInWeld)
      {
         Object tgt = context.getTarget();
         for (InjectionPoint ip : it.getInjectionPoints())
         {
            //TODO Any point in creating ClassInfo here?
            if (ip.getMember() instanceof Method)
            {
               Method m = (Method)ip.getMember();
               Object[] params = new Object[m.getParameterTypes().length];
               try
               {
                  m.invoke(tgt, params);
               }
               catch(Exception e)
               {
                  log.warn("Error unsetting values for method " + m.getName() + " in bean " + context.getName(), e);
               }
            }
            else if (ip.getMember() instanceof Field)
            {
               Field f = (Field)ip.getMember();
               try
               {
                  f.set(tgt, null);
               }
               catch (Exception e)
               {
                  log.warn("Error unsetting values for field " + f.getName() + " in bean " + context.getName(), e);
               }
            }
         }      
      }
   }
   
   /**
    * Invokes the @PostConstruct method if any
    * 
    *  @param instance the instance
    *  @throws IllegalArgumentException if the instance is null
    */
   @SuppressWarnings("unchecked")
   void postConstruct(Object instance) {
      if (instance == null)
         throw new IllegalArgumentException("Null instance");
      
      it.postConstruct((T)instance);
   }

   /**
    * Invokes the @PreDestroy method if any
    * 
    *  @param instance the instance
    *  @throws IllegalArgumentException if the instance is null
    */
   @SuppressWarnings("unchecked")
   void preDestroy(Object instance) {
      if (instance == null)
         throw new IllegalArgumentException("Null instance");

      it.preDestroy((T)instance);
      creationalContext.release();
   }
   
   /**
    * Checks if the context contains the passed in dependency
    * 
    * @param dependency the dependency to check
    * @return true if the context contains the dependency
    */
   private boolean containsDependency(DependencyMetaData dependency)
   {
      return context.getBeanMetaData().getDepends() != null && context.getBeanMetaData().getDepends().contains(dependency);
   }
   
   /**
    * Decorates an annotated type with the contexts MDR metadata
    * 
    * @param clazz
    * @return the decorated annotated type
    */
   private AnnotatedType<T> createMdrDecoratedAnnotatedType(Class<T> clazz)
   {
      AnnotatedType<T> type = context.getManager().createAnnotatedType(clazz);
      return MDRAnnotatedTypeFactory.getInstance().decorateAnnotatedType(type, context);
   }
   
   /**
    * Get the annotated type
    * @return the annotated type
    */
   public AnnotatedType<T> getType()
   {
      return type;
   }



   class AnnotationMetaDataVisitor extends AbstractMetaDataVisitor
   {
      public AnnotationMetaDataVisitor(KernelControllerContext context)
      {
         super(context.getBeanMetaData(), context);
      }

      // push bean meta data as first node
      public void before()
      {
         visitorNodeStack.push(bmd);
      }

      // remove bean meta data
      public void after()
      {
         visitorNodeStack.pop();
         visitorNodeStack = null;
      }
   }
}
