/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.weld;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.BeanManager;

import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;
import org.jboss.kernel.spi.dependency.KernelControllerContext;
import org.jboss.kernel.spi.metadata.KernelMetaDataRepository;
import org.jboss.kernel.weld.metadata.api.annotations.WeldEnabled;
import org.jboss.kernel.weld.plugins.dependency.WeldKernelControllerContext;
import org.jboss.kernel.weld.spi.annotated.MDRAnnotatedTypeFactory;
import org.jboss.metadata.spi.MetaData;
import org.jboss.util.collection.ConcurrentSet;

/**
 * Singleton registry of MC beans that should be accessible from web beans
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class WeldFromMcRegistry
{
   private final static WeldFromMcRegistry INSTANCE = new WeldFromMcRegistry();
   
   /** 
    * Map of context and annotated type wrapper
    */
   private final Map<KernelControllerContext, AnnotatedTypeWrapper> typesByContext = new ConcurrentHashMap<KernelControllerContext, AnnotatedTypeWrapper>();
   
   /**
    * Map of contexts that have not yet been initialized with a type
    */
   private final Set<KernelControllerContext> contextsWithNoType = new ConcurrentSet<KernelControllerContext>();
   
   /** 
    * Map of annotated type wrapper and context
    */
   private final Map<AnnotatedTypeWrapper, KernelControllerContext> contextsByType = new ConcurrentHashMap<AnnotatedTypeWrapper, KernelControllerContext>();  
   
   /**
    * Get the singleton
    * @return the singleton instance
    */
   public static WeldFromMcRegistry getInstance()
   {
      return INSTANCE;
   }
   
   /**
    * Check if a context has the @WeldEnabled annotation, and if so register it as a web bean.
    * 
    * @param context the context to register in web beans
    */
   public void registerBean(KernelControllerContext context)
   {
      if (context instanceof WeldKernelControllerContext)
         registerBean(context, ((WeldKernelControllerContext)context).getWeldInjector().getType());
      else
         registerBean(context, null);
   }
   
   /**
    * Check if a context has the @WeldEnabled annotation, and if so register it as a web bean.
    * 
    * @param context the context to register in web beans
    * @param type the annotated type of the context's bean
    */
   private void registerBean(KernelControllerContext context, AnnotatedType<?> type)
   {
      if (!hasWeldEnabledAnnotation(context, type))
         return;

      if (type != null)
      {
         AnnotatedTypeWrapper typeWrapper = new AnnotatedTypeWrapper(type);
         typesByContext.put(context, typeWrapper);
         contextsByType.put(typeWrapper, context);
      }
      else
      {
         contextsWithNoType.add(context);
      }
   }
   
   /**
    * Check if the bean has the WeldEnabled annotation
    * 
    * @param context the kernel controller context
    * @param type the annotated type
    * @return true if bean has web beans annotations
    */
   private boolean hasWeldEnabledAnnotation(KernelControllerContext context, AnnotatedType<?> type)
   {
      if (type != null)
         return type.getAnnotation(WeldEnabled.class) != null;
      
      KernelController controller = (KernelController)context.getController();
      Kernel kernel = controller.getKernel();
      KernelMetaDataRepository repository = kernel.getMetaDataRepository();
      MetaData metaData = repository.getMetaData(context);
      return metaData.isAnnotationPresent(WeldEnabled.class);
   }
   
   /**
    * Unregister an mc context as a web bean
    * 
    * @param context the context to register in web beans
    */
   public void unregisterBean(KernelControllerContext context)
   {
      AnnotatedTypeWrapper typeWrapper = typesByContext.remove(context);
      if (typeWrapper != null)
         contextsByType.remove(typeWrapper);
      contextsWithNoType.remove(context);
   }
   
   public void initializeTypes(BeanManager beanManager)
   {
      for (Iterator<KernelControllerContext> it = contextsWithNoType.iterator() ; it.hasNext() ; )
      {
         KernelControllerContext ctx = it.next();
         it.remove();
         AnnotatedType<?> type = beanManager.createAnnotatedType(ctx.getBeanInfo().getClassInfo().getType());
         type = MDRAnnotatedTypeFactory.getInstance().decorateAnnotatedType(type, ctx);
         registerBean(ctx, type);
      }
   }
   
   /**
    * Get a copy of all the MC beans that should be registered as web beans
    * 
    * @return the collection of beans
    */
   public Collection<AnnotatedTypeWrapper> getTypes()
   {
      return Collections.unmodifiableCollection(typesByContext.values());
   }
   
   public KernelControllerContext getContext(AnnotatedType<?> type)
   {
      return contextsByType.get(new AnnotatedTypeWrapper(type));
   }
   
}
