/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.annotated;

import java.util.ArrayList;
import java.util.List;

import javax.enterprise.inject.spi.Annotated;
import javax.enterprise.inject.spi.AnnotatedCallable;
import javax.enterprise.inject.spi.AnnotatedParameter;

import org.jboss.metadata.spi.MetaData;
import org.jboss.metadata.spi.signature.ParametersSignature;

/**
 * Decorator that understands MDR metadata for a method or constructor
 * 
 * @param <X> the declaring type
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public abstract class MDRAnnotatedCallable<X> extends MDRAnnotatedMember<X> implements AnnotatedCallable<X>
{
   private final List<AnnotatedParameter<X>> parameters;
   private MetaData mainMetaData;

   /**
    * Constructor
    *
    * @param owner the annotated type containing this callable
    * @param delegate the delegate
    * @param metaData the MDR metadata for the method or constructor
    * @param mainMetaData the MDR metadata for the class containing this method or constructor
    */
   public MDRAnnotatedCallable(MDRAnnotatedType<X> owner, Annotated delegate, MetaData metaData, MetaData mainMetaData)
   {
      super(owner, delegate, metaData);
      this.mainMetaData = mainMetaData;
      parameters = initialiseParameters();
   }

   /**
    * Gets the delegate
    * 
    * @return the delegate
    */
   @Override
   AnnotatedCallable<X> getDelegate()
   {
      return (AnnotatedCallable<X>) super.getDelegate();
   }
   
   /**
    * initialises the parameters for this constructor or method
    * 
    * @return the list of parameters, wrapped in decorators if there is MDR metadata
    */
   private List<AnnotatedParameter<X>> initialiseParameters()
   {
      if (getMetaData() == null)
         return getDelegate().getParameters();
      
      int size = getDelegate().getParameters().size();
      List<AnnotatedParameter<X>> params = new ArrayList<AnnotatedParameter<X>>(size);
      if (size > 0)
      {
         int i = 0;
         for (AnnotatedParameter<X> param : getDelegate().getParameters())
         {
            MetaData parameterMetaData = mainMetaData.getComponentMetaData(getParametersSignature(i++));
            params.add(new MDRAnnotatedParameter<X>(param, this, parameterMetaData));
         }
      }
      return params;
   }

   /**
    * Gets the MDR parameters signature for a parameter
    * 
    * @param i the index of the parameter
    * @return the parameter signature
    */
   abstract ParametersSignature getParametersSignature(int i);
 
   /**
    * Gets the parameters
    * 
    * @return the parameters
    */
   public List<AnnotatedParameter<X>> getParameters()
   {
      return parameters;
   }
}
