/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.metadata;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;

import javax.enterprise.inject.InjectionException;
import javax.enterprise.inject.spi.InjectionPoint;

import org.jboss.beans.metadata.plugins.AbstractDependencyMetaData;
import org.jboss.beans.metadata.spi.MetaDataVisitor;
import org.jboss.dependency.plugins.AbstractDependencyItem;
import org.jboss.dependency.spi.Controller;
import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.kernel.weld.plugins.dependency.WeldKernelControllerContext;
import org.jboss.kernel.weld.plugins.dependency.UnsatisfiedDependencyRegistry;

/**
 * DependencyMetaData representing a dependency on a web bean
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class WeldDependencyMetaData extends AbstractDependencyMetaData
{
   private static final long serialVersionUID = 1L;
   
   /** The web beans injection point */
   private final InjectionPoint injectionPoint;
   
   /** The web beans kernel controller context of the bean containing the dependency */
   private WeldKernelControllerContext context;
   
   /** When the dependency is required. If the injection point is a constructor, this will be ControllerState.INSTATIATED, 
    * otherwise it will be ControllerState.CONFIGURED
    */
   private final ControllerState whenRequired;

   
   /** 
    * Constructor 
    * 
    * @param injectionPoint the injection point
    */
   @SuppressWarnings("unchecked")
   public WeldDependencyMetaData(InjectionPoint injectionPoint)
   {
      super(injectionPoint);
      this.injectionPoint = injectionPoint;
      
      if (injectionPoint.getMember() instanceof Constructor)
         whenRequired = ControllerState.INSTANTIATED;
      else
         whenRequired = ControllerState.CONFIGURED;
         
   }
   
   public void initialVisit(MetaDataVisitor visitor)
   {
      ControllerContext context = visitor.getControllerContext();
      if (context instanceof WeldKernelControllerContext == false)
         throw new IllegalStateException("Context is not a WeldKernelControllerContext " + context.getName());
      this.context = (WeldKernelControllerContext)context;
      
      visitor.addDependency(new WeldDependencyItem(context.getName()));
   }

   @Override
   protected int getHashCode()
   {
      int i = 17;
      i = 31 * i + injectionPoint.getMember().hashCode();
      for (Annotation ann : injectionPoint.getQualifiers())
      {
         i = 31 * i + ann.hashCode();
      }
      return i;
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj instanceof WeldDependencyMetaData == false)
         return false;
      return equals(injectionPoint.getMember(), ((WeldDependencyMetaData)obj).injectionPoint.getMember()) &&
         equals(injectionPoint.getQualifiers(), ((WeldDependencyMetaData)obj).injectionPoint.getQualifiers());
   }

   /**
    * DependencyItem representing a web beans injection point 
    */
   private class WeldDependencyItem extends AbstractDependencyItem
   {
      public WeldDependencyItem(Object name)
      {
         super(name, null, whenRequired, null);
      }

      public boolean resolve(Controller controller)
      {
         try
         {
            context.getManager().validate(injectionPoint);
            setIDependOn(injectionPoint);
            setResolved(true);
         }
         catch(InjectionException e)
         {
            UnsatisfiedDependencyRegistry.getInstance().addHungContext(context);
            setResolved(false);
         }

         return isResolved();
      }

      @Override
      public void unresolved()
      {
         setIDependOn(null);
         setResolved(false);
      }
   }
}
