/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.kernel.weld.plugins.annotated;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.enterprise.inject.spi.Annotated;

import org.jboss.metadata.spi.MetaData;

/**
 * Decorator that understands MDR metadata
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class MDRAnnotated implements Annotated
{
   /** The delegate */
   private Annotated delegate;
   
   /** The MDR metadata */
   private MetaData metaData;
   
   /**
    * Constructor
    * 
    * @param delegate the delegate
    * @param metaData the MDR metadata 
    */
   public MDRAnnotated(Annotated delegate, MetaData metaData)
   {
      if (delegate == null)
         throw new IllegalArgumentException("Null delegate");
      this.delegate = delegate;
      this.metaData = metaData;
   }

   /**
    * Gets the delegate
    * 
    * @return the delegate
    */
   Annotated getDelegate()
   {
      return delegate;
   }
   
   /**
    * Gets the MDR metadata
    * 
    * @return the MDR metadata
    */
   MetaData getMetaData()
   {
      return metaData;
   }
   
   /**
    * Gets the base type from the delegate
    * 
    * @return the base type
    */
   public Type getBaseType()
   {
      return delegate.getBaseType();
   }

   /**
    * Gets the type closure from the delegate
    * 
    * @return the type closure
    */
   public Set<Type> getTypeClosure()
   {
      return delegate.getTypeClosure();
   }

   /**
    * Gets an annotation
    * 
    * @param annotationType the annotation type to get
    * @return the annotations from the delegate and the MDR metadata
    */
   public <T extends Annotation> T getAnnotation(Class<T> annotationType)
   {
      if (metaData != null)
      {
         T t = metaData.getAnnotation(annotationType);
         if (t != null)
            return t;
      }      
      return getDelegate().getAnnotation(annotationType);
   }

   /**
    * Gets the annotations
    * 
    * @return the annotations from the delegate and the MDR metadata
    */
   public Set<Annotation> getAnnotations()
   {
      Set<Annotation> annotations = new HashSet<Annotation>(getDelegate().getAnnotations());
      if (metaData == null)
         return trimInjectIfWeldAbsent(annotations);

      Annotation[] metaDataAnnotations = metaData.getAnnotations();
      if (metaDataAnnotations.length == 0)
         return trimInjectIfWeldAbsent(annotations);
      
      if (annotations.size() == 0)
      {
         Set<Annotation> metaDataAnns = new HashSet<Annotation>();
         for (Annotation ann : metaDataAnnotations)
         {
            metaDataAnns.add(ann);
         }
         return trimInjectIfWeldAbsent(metaDataAnns);
      }
      
      Map<Class<?>, Annotation> done = new HashMap<Class<?>, Annotation>();
      
      for (Annotation annotation : annotations)
         done.put(annotation.annotationType(), annotation);
      for (Annotation annotation : metaDataAnnotations)
      {
         done.put(annotation.annotationType(), annotation);
      }
      return trimInjectIfWeldAbsent(new HashSet<Annotation>(done.values()));
   }

   /**
    * Checks if an annotation is present
    * 
    * @return true if the annotation is present in the delegate or the MDR metadata
    */
   public boolean isAnnotationPresent(Class<? extends Annotation> annotationType)
   {
      if (metaData != null)
      {
         if (metaData.isAnnotationPresent(annotationType))
            return true;
      }
      return getDelegate().isAnnotationPresent(annotationType);
   }
   
   protected Set<Annotation> trimInjectIfWeldAbsent(Set<Annotation> annotations)
   {
      //Empty, only implement for members
      return annotations;
   }
}
