/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.managed.plugins;

import java.io.IOException;
import java.io.InvalidObjectException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.managed.api.Fields;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.annotation.ActivationPolicy;
import org.jboss.managed.api.annotation.ManagementProperty;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;

/**
 * ManagedProperty.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author Scott.Stark@jboss.org
 * @version $Revision: 89193 $
 */
public class ManagedPropertyImpl implements ManagedProperty
{
   /** The serialVersionUID */
   private static final long serialVersionUID = 2;
   /* writeObject format:
    * - int version
    * - Fields fields
    * - ManagedObject managedObject
    * - ManagedObject targetManagedObject
    */
   private static final int VERSION1 = 1;
   /** The serialization version used by writeObject */
   private static final int STREAM_VERSION = VERSION1;

   /** The managed object */
   private ManagedObject managedObject;
   /** The managed object target for a ManagementObjectRef */
   private ManagedObject targetManagedObject;
   
   /** The fields */
   private Fields fields;

   /** The property name */
   private transient String name;
   /** The transient attachments map */
   private transient Map<String, Object> transientAttachments;

   /**
    * Create a new ManagedProperty that is not associated to
    * a ManagedObject.
    *
    * @param name the managed property name
    * @throws IllegalArgumentException for null fields or
    *    missing Fields.NAME
    */
   public ManagedPropertyImpl(String name)
   {
      this(null, new DefaultFieldsImpl(name));
   }

   /**
    * Create a new ManagedProperty that is not associated to
    * a ManagedObject.
    * 
    * @param fields the fields
    * @throws IllegalArgumentException for null fields or
    *    missing Fields.NAME
    */
   public ManagedPropertyImpl(Fields fields)
   {
      this(null, fields);
   }

   /**
    * Create a new ManagedProperty.
    * 
    * @param managedObject the managed object, may be null
    * @param fields the fields
    * @throws IllegalArgumentException for null fields or
    *    missing Fields.NAME
    */
   public ManagedPropertyImpl(ManagedObject managedObject, Fields fields)
   {
      init(managedObject, fields);
   }
   
   public ManagedObject getManagedObject()
   {
      return managedObject;
   }

   /**
    * Set managed object
    * 
    * @param managedObject the managed object
    */
   public void setManagedObject(ManagedObject managedObject)
   {
      this.managedObject = managedObject;
   }

   public ManagedObject getTargetManagedObject()
   {
      return targetManagedObject;
   }

   public void setTargetManagedObject(ManagedObject target)
   {
      this.targetManagedObject = target;
   }

   public Fields getFields()
   {
      return fields;
   }
   
   // TODO general reconstruction code for metatypes
   @SuppressWarnings("unchecked")
   public <T> T getField(String fieldName, Class<T> expected)
   {
      if (fieldName == null)
         throw new IllegalArgumentException("Null field name");
      if (expected == null)
         throw new IllegalArgumentException("Null expected type");
      
      Object field = getFields().getField(fieldName);
      
      if (field == null)
         return null;

      if (expected.isInstance(field))
         return expected.cast(field);
      
      if (field instanceof SimpleValue)
      {
         SimpleValue value = (SimpleValue) field;
         Object result = value.getValue();
         if (result == null)
            return null;
         return expected.cast(result);
      }
      
      throw new IllegalStateException("Field " + fieldName + " with value " + field + " is not of the expected type: " + expected.getName());
   }
   
   // TODO metaType stuff
   public void setField(String fieldName, Serializable value)
   {
      if (fieldName == null)
         throw new IllegalArgumentException("Null field name");
      
      getFields().setField(fieldName, value);
   }
   
   public String getName()
   {
      return name;
   }

   public String getMappedName()
   {
      return getField(Fields.MAPPED_NAME, String.class);
   }

   public String getDescription()
   {
      return getField(Fields.DESCRIPTION, String.class);
   }
   
   /**
    * Set the description
    * 
    * @param description the description
    */
   public void setDescription(String description)
   {
      setField(Fields.DESCRIPTION, description);
   }


   /**
    * Get the annotations associated with the property
    * @return the annotations associated with the property
    */
   @SuppressWarnings("unchecked")
   public Map<String, Annotation> getAnnotations()
   {
      Object set = getField(Fields.ANNOTATIONS, Object.class);
      return (Map) set;
   }

   public void setAnnotations(Map<String, Annotation> annotations)
   {
      setField(Fields.ANNOTATIONS, (Serializable) annotations);      
   }

   
   public boolean hasAnnotation(String key)
   {
      boolean hasAnnotation = false;
      // Look to the ManagementProperty annotation
      Map<String, Annotation> annotations = getAnnotations();
      if(annotations != null)
      {
         hasAnnotation = annotations.containsKey(key);
      }
      return hasAnnotation;
   }

   /**
    * See if the property has the indicated ViewUse among its
    * @ManagementProperty annotation or VIEW_USE field uses.
    * 
    * @param use - the ViewUse to check for
    * @return true if the ViewUse exists in the property uses, false otherwise
    */
   public boolean hasViewUse(ViewUse use)
   {
      boolean hasViewUse = false;
      ViewUse[] uses = getViewUse();
      if(uses != null)
      {
         for(ViewUse vu : uses)
         {
            hasViewUse |= vu == use;
         }
      }
      return hasViewUse;
   }

   public MetaType getMetaType()
   {
      return getField(Fields.META_TYPE, MetaType.class);
   }
   
   /**
    * Set the meta type
    * 
    * @param type the meta type
    */
   public void setMetaType(MetaType type)
   {
      setField(Fields.META_TYPE, type);
   }

   public MetaValue getValue()
   {
      return getField(Fields.VALUE, MetaValue.class);
   }

   public void setValue(MetaValue value)
   {
      // Check for a change 
      MetaValue oldValue = getValue();
      if(oldValue != value)
      {
         boolean isModified = true;
         if(value != null)
            isModified = ! value.equals(oldValue);
         setModified(isModified);
      }
      setField(Fields.VALUE, value);
   }

   public ViewUse[] getViewUse()
   {
      ViewUse[] use = {};
      ViewUse[] useField = getField(Fields.VIEW_USE, ViewUse[].class);
      // Also look to the ManagementProperty annotation
      Map<String, Annotation> annotations = getAnnotations();
      if(annotations != null)
      {
         ManagementProperty mp = (ManagementProperty) annotations.get(ManagementProperty.class.getName());
         if(mp != null)
         {
            use = mp.use();
            if(useField != null && useField.length > 0)
            {
               HashSet<ViewUse> uses = new HashSet<ViewUse>();
               for(ViewUse vu : use)
               {
                  uses.add(vu);
               }
               for(ViewUse vu : useField)
               {
                  uses.add(vu);
               }
               use = new ViewUse[uses.size()];
               uses.toArray(use);
            }
         }
      }
      else if(useField != null)
      {
         use = useField;
      }
      return use;
   }
   public void setViewUse(ViewUse[] use)
   {
      setField(Fields.VIEW_USE, use);
   }

   public Collection<String> getAdminViewUses()
   {
      Collection<String> adminViews = getField(Fields.ADMIN_VIEWS, Collection.class);
      // Also look to the ManagementProperty annotation
      Map<String, Annotation> annotations = getAnnotations();
      if(annotations != null)
      {
         ManagementProperty mp = (ManagementProperty) annotations.get(ManagementProperty.class.getName());
         if(mp != null)
         {
            HashSet<String> views = new HashSet<String>();
            if(adminViews != null)
            {
               views.addAll(adminViews);
            }
            String[] mpViews = mp.adminViews();
            for(String view : mpViews)
               views.add(view);
            adminViews = views;
         }
      }
      if(adminViews == null)
         adminViews = Collections.emptySet();
      return adminViews; 
   }
   public void setAdminViewUses(Collection<String> viewUses)
   {
      setField(Fields.ADMIN_VIEWS, (Serializable) viewUses);
   }

   public ActivationPolicy getActivationPolicy()
   {
      ActivationPolicy activationPolicy = getField(Fields.ACTIVATION_POLICY, ActivationPolicy.class);
      return activationPolicy;
   }
   public void setActivationPolicy(ActivationPolicy policy)
   {
      setField(Fields.ACTIVATION_POLICY, policy);
   }

   @SuppressWarnings("unchecked")
   public Set<MetaValue> getLegalValues()
   {
      return getField(Fields.LEGAL_VALUES, Set.class);
   }
   
   /**
    * Set the legal values
    * 
    * @param values the values
    */
   public void setLegalValues(Set<MetaValue> values)
   {
      setField(Fields.LEGAL_VALUES, (Serializable)values);
   }

   public MetaValue getDefaultValue()
   {
      MetaValue field = getField(Fields.DEFAULT_VALUE, MetaValue.class);
      return field;
   }

   public Comparable<MetaValue> getMinimumValue()
   {
      return getField(Fields.MINIMUM_VALUE, Comparable.class);
   }
   
   /**
    * Set the minimum value
    * 
    * @param value the value
    */
   public void setMinimumValue(Comparable<MetaValue> value)
   {
      setField(Fields.MINIMUM_VALUE, (Serializable)value);
   }

   public Comparable<MetaValue> getMaximumValue()
   {
      Comparable<MetaValue> field = getField(Fields.MAXIMUM_VALUE, Comparable.class);
      return field;
   }

   /**
    * Set the maximum value
    * 
    * @param value the value
    */
   public void setMaximumValue(Comparable<MetaValue> value)
   {
      setField(Fields.MAXIMUM_VALUE, (Serializable)value);
   }

   public String checkValidValue(MetaValue value)
   {
      Comparable<MetaValue> min = getMinimumValue();
      if(min != null)
      {
         if(min.compareTo(value) > 0)
            return "min("+min+") > "+value;
      }
      Comparable<MetaValue> max = getMaximumValue();
      if(max != null)
      {
         if(max.compareTo(value) < 0)
            return "max("+max+") < "+value;
      }
      Set<MetaValue> legalValues = getLegalValues();
      if(legalValues != null && legalValues.size() > 0)
      {
         if(legalValues.contains(value) == false)
            return legalValues+" does not contain: "+value;
      }
      return null;
   }
   
   public boolean isMandatory()
   {
      Boolean result = getField(Fields.MANDATORY, Boolean.class);
      if (result == null)
         return false;
      return result;
   }

   public boolean isReadOnly()
   {
      Boolean result = getField(Fields.READ_ONLY, Boolean.class);
      if (result == null)
         return false;
      return result;
   }
   public void setReadOnly(boolean flag)
   {
      if (flag)
         setField(Fields.READ_ONLY, flag);
      else
         setField(Fields.READ_ONLY, null);
   }

   public boolean isModified()
   {
      Boolean result = getField(Fields.MODIFIED, Boolean.class);
      if (result == null)
         return false;
      return result;
   }
   public void setModified(boolean flag)
   {
      if (flag)
         setField(Fields.MODIFIED, flag);
      else
         setField(Fields.MODIFIED, null);
   }

   /**
    * Set whether the field is mandatory
    * 
    * @param flag true for mandatory
    */
   public void setMandatory(boolean flag)
   {
      if (flag)
         setField(Fields.MANDATORY, flag);
      else
         setField(Fields.MANDATORY, null);
   }

   public boolean isRemoved()
   {
      Boolean result = getField(Fields.REMOVED, Boolean.class);
      if (result == null)
         return false;
      return result;
   }
   
   /**
    * Set whether the property is removed
    * 
    * @param flag true for removed
    */
   public void setRemoved(boolean flag)
   {
      if (flag)
         setField(Fields.REMOVED, flag);
      else
         setField(Fields.REMOVED, null);
   }

   public <T> T getTransientAttachment(Class<T> expectedType)
   {
      T tvalue = null;
      Object value = getTransientAttachment(expectedType.getName());
      if(value != null)
         tvalue = expectedType.cast(value);
      return tvalue;
   }

   public Object getTransientAttachment(String name)
   {
      Object value = null;
      if(transientAttachments != null)
         value = transientAttachments.get(name);
      return value;
   }

   public synchronized void setTransientAttachment(String name, Object attachment)
   {
      if(transientAttachments == null)
         transientAttachments = new HashMap<String, Object>();
      transientAttachments.put(name, attachment);
   }

   @Override
   public String toString()
   {
      StringBuilder tmp = new StringBuilder("ManagedProperty");
      tmp.append('{');
      tmp.append(name);
      if( getMappedName() != null )
      {
         tmp.append(',');
         tmp.append(getMappedName());
      }
      tmp.append(",metaType=");
      tmp.append(this.getMetaType());
      tmp.append('}');
      return tmp.toString(); 
   }

   @Override
   public int hashCode()
   {
      return name.hashCode(); 
   }

   @Override
   public boolean equals(Object obj)
   {
      if (obj == this)
         return true;
      if (obj == null || obj instanceof ManagedProperty == false)
         return false;
      
      ManagedProperty other = (ManagedProperty) obj;
      return getName().equals(other.getName());
   }
   
   public ManagedProperty copy()
   {
      Fields fieldsCopy = fields.copy();
      ManagedProperty mp = new ManagedPropertyImpl(fieldsCopy);
      return mp;
   }

   /**
    * Initialise a ManagedPropertyImpl.
    * 
    * @param managedObject the managed object, may be null
    * @param fields the fields
    * @throws IllegalArgumentException for null fields or
    *    missing Fields.NAME
    */
   private void init(ManagedObject managedObject, Fields fields)
   {
      if (fields == null)
         throw new IllegalArgumentException("Null fields");
      
      this.managedObject = managedObject;
      this.fields = fields;
      
      name = getField(Fields.NAME, String.class);
      if (name == null)
         throw new IllegalArgumentException("No " + Fields.NAME + " in fields");
   }

   /**
    * Read from a stream
    * 
    * @param in the stream
    * @throws IOException for IO problem
    * @throws ClassNotFoundException for a classloading problem
    */
   private void readObject(ObjectInputStream in)
      throws IOException, ClassNotFoundException
   {
      int version = in.readInt();
      if( version == VERSION1 )
         readVersion1(in);
      else
         throw new InvalidObjectException("Unknown version="+version);
   }
   /**
    * Write out the property fields
    * @param out
    * @throws IOException
    */
   private void writeObject(ObjectOutputStream out)
      throws IOException
   {
      out.writeInt(STREAM_VERSION);
      out.writeObject(fields);
      out.writeObject(managedObject);
      out.writeObject(targetManagedObject);
   }

   /**
    * The VERSION1 expected format: 
    * - Fields fields
    * - ManagedObject managedObject
    */
   private void readVersion1(ObjectInputStream in)
      throws IOException, ClassNotFoundException
   {
      fields = (Fields) in.readObject();
      name = getField(Fields.NAME, String.class);
      if (name == null)
         throw new IOException("No " + Fields.NAME + " in fields");
      managedObject = (ManagedObject) in.readObject();      
      targetManagedObject = (ManagedObject) in.readObject();      
   }
}
