/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.bean.switchboard.resource.provider;

import java.util.HashSet;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.managed.bean.metadata.ManagedBeanMetaData;
import org.jboss.switchboard.impl.resource.LinkRefResource;
import org.jboss.switchboard.javaee.environment.ResourceEnvRefType;
import org.jboss.switchboard.javaee.jboss.environment.JBossResourceEnvRefType;
import org.jboss.switchboard.javaee.util.InjectionTargetUtil;
import org.jboss.switchboard.mc.spi.MCBasedResourceProvider;
import org.jboss.switchboard.spi.Resource;

/**
 * {@link MCBasedResourceProvider} for resolving Managed Bean resource-env-ref
 *
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class ManagedBeanResourceEnvRefResourceProvider extends AbstractManagedBeanRefResourceProvider implements MCBasedResourceProvider<JBossResourceEnvRefType>
{

   @Override
   public Class<JBossResourceEnvRefType> getEnvironmentEntryType()
   {
      return JBossResourceEnvRefType.class;
   }

   /**
    * First checks for the presence of any lookup name on the passed {@link JBossResourceEnvRefType}.
    * If a lookup name is set, then this method returns a {@link LinkRefResource} to that lookup name.
    * If no lookup name is set, then this method scans the passed deployment unit and any of its nested
    * deployment units to find a managed bean whose class name equals the "type" returned of the 
    * {@link JBossResourceEnvRefType resource-evn-ref}. If such a managed bean is found, then this
    * method returns a {@link LinkRefResource} to java:module/<managed bean name>
    * 
    * If the passed {@link JBossResourceEnvRefType resource-env-ref} cannot be resolved, then this method
    * returns null.
    */
   @Override
   public Resource provide(DeploymentUnit unit, JBossResourceEnvRefType resEnvRef)
   {
      // first check lookup name
      String lookupName = resEnvRef.getLookupName();
      if (lookupName != null && !lookupName.trim().isEmpty())
      {
         return new LinkRefResource(lookupName);
      }
      String type = this.getResourceEnvRefType(unit.getClassLoader(), resEnvRef);
      if (type == null)
      {
         return null;
      }
      ManagedBeanMetaData managedBean = this.resolveWithinDeploymentUnit(unit, new HashSet<DeploymentUnit>(), type);
      if (managedBean != null)
      {
         String jndiName = "java:module/" + managedBean.getName();
         return new LinkRefResource(jndiName);
      }
      return null;
   }
   
   /**
    * Returns the resource-env-ref-type for the passed {@link ResourceEnvRefType}.
    * <p>
    *   If the passed resource-env-ref has the resource-env-ref-type explicitly specified, then
    *   that value is returned. Else, this method checks for the presence of any
    *   injection targets for this resource-env-ref. If there's a injection target, then
    *   the resource-env-ref-type is deduced based on the field/method of the injection target. 
    * </p>
    * <p>
    *   This method returns null if the res-type isn't explicitly specified and 
    *   if the resource-env-ref-type could not be deduced from the injection targets of this
    *   resource-env-ref.
    * </p>
    * 
    * @param cl The {@link ClassLoader} to be used during processing the metadata
    * @param resourceEnvRef The Java EE resource-env-ref
    * @return
    */
   private String getResourceEnvRefType(ClassLoader cl, ResourceEnvRefType resourceEnvRef)
   {
      // first check whether the type is explicitly specified
      String explicitType = resourceEnvRef.getResourceType();
      if (explicitType != null && !explicitType.isEmpty())
      {
         return explicitType;
      }
      Class<?> type = InjectionTargetUtil.getInjectionTargetPropertyType(cl, resourceEnvRef);
      return type == null ? null : type.getName();
   }

}
