package org.jboss.maven.plugins.injection;

import java.util.List;
import java.util.ArrayList;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.URLClassLoader;
import java.io.File;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.jboss.maven.plugins.injection.process.InjectionDescriptor;
import org.jboss.maven.plugins.injection.process.InjectionTarget;

/**
 * Mojo responsible for injecting values into various targets as part of a build process.  One example of this
 * is injecting the project version into a class file (bytcode manip) or a text file or etc so that these various
 * targets do not need to be synch'ed manually.
 *
 * @author Steve Ebersole
 */
public abstract class AbstractInjectionMojo extends AbstractMojo {
	/**
     * The Maven Project Object
     *
     * @parameter expression="${project}"
     * @required
     */
    protected MavenProject project;

	protected void prepare() throws MojoExecutionException {
	}

	protected abstract List<InjectionDescriptor> getInjectionDescriptors() throws MojoExecutionException;

	protected void finishUp() {
	}

	public void execute() throws MojoExecutionException, MojoFailureException {
		prepare();

		for ( InjectionDescriptor descriptor : getInjectionDescriptors() ) {
			final String injectionValue = descriptor.getValue();
			for ( InjectionTarget target : descriptor.getTargets() ) {
				target.inject( injectionValue );
			}
		}

		finishUp();
	}

	protected final String resolveExpression(String expression) {
		return expression;
	}

	protected final ClassLoader buildProjectCompileClassLoader() throws MojoExecutionException {
		ArrayList<URL> classPathUrls = new ArrayList<URL>();

		// we should also add the compile classpath elements...
		for ( String path : projectCompileClasspathElements() ) {
			try {
				getLog().info( "Adding project compile classpath element : " + path );
				classPathUrls.add( new File( path ).toURI().toURL() );
			}
			catch ( MalformedURLException e ) {
				throw new MojoExecutionException( "Unable to build path URL [" + path + "]" );
			}
		}
		return new URLClassLoader( classPathUrls.toArray( new URL[classPathUrls.size()] ), getClass().getClassLoader() );
	}

	protected final List<String> projectCompileClasspathElements() throws MojoExecutionException {
		try {
			return ( List<String> ) project.getCompileClasspathElements();
		}
		catch ( DependencyResolutionRequiredException e ) {
			throw new MojoExecutionException( "Call to Project#getCompileClasspathElements required dependency resolution" );
		}
	}
}
