/*
 * Copyright (c) 2009, Red Hat Middleware LLC or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Middleware LLC.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 */
package org.jboss.maven.plugins.injection;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javassist.ClassPool;
import javassist.CtClass;
import javassist.LoaderClassPath;
import javassist.CtMethod;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstantAttribute;
import javassist.bytecode.FieldInfo;
import org.apache.maven.plugin.MojoExecutionException;
import org.jboss.maven.plugins.injection.process.InjectionDescriptor;
import org.jboss.maven.plugins.injection.process.InjectionTarget;

/**
 * Used to inject resolved expression values into compiled bytecode.
 * <p/>
 * TODO : add checks as to whether the injection is needed to avoid file timestamp changes.
 * Basically we should skip the injection if the class file field is already the injection value...
 * 
 * @goal bytecode
 * @phase compile
 * @
 *
 * @author Steve Ebersole
 */
public class BytecodeInjectionMojo extends AbstractInjectionMojo {
	/**
	 * The injections to be performed.
	 *
	 * @parameter
	 * @required
	 */
	protected BytecodeInjection[] bytecodeInjections;

	private LoaderClassPath loaderClassPath;
	private ClassPool classPool;

	@Override
	protected void prepare() throws MojoExecutionException {
		super.prepare();
		loaderClassPath = new LoaderClassPath( buildProjectCompileClassLoader() );
		classPool = new ClassPool();
		classPool.appendClassPath( loaderClassPath );
	}

	protected List<InjectionDescriptor> getInjectionDescriptors() throws MojoExecutionException {
		ArrayList<InjectionDescriptor> descriptors = new ArrayList<InjectionDescriptor>();
		for ( BytecodeInjection injection : bytecodeInjections ) {
			descriptors.add( generateDescriptor( injection ) );
		}
		return descriptors;
	}

	private InjectionDescriptor generateDescriptor(BytecodeInjection injection) throws MojoExecutionException {
		InjectionDescriptor descriptor = new InjectionDescriptor( resolveExpression( injection.getExpression() ) );
		for ( TargetMember member : injection.getTargetMembers() ) {
			descriptor.getTargets().add( generateBytecodeInjectionTarget( member ) );
		}
		return descriptor;
	}

	private InjectionTarget generateBytecodeInjectionTarget(TargetMember member) throws MojoExecutionException {
		if ( member instanceof Constant ) {
			return new ConstantInjectionTarget( ( Constant ) member );
		}
		else if ( member instanceof MethodBodyReturn ) {
			return new MethodBodyReturnReplacementTarget( ( MethodBodyReturn ) member );
		}
		throw new MojoExecutionException( "Unexpected injection member type : " + member );
	}

	private abstract class BaseInjectionTarget implements InjectionTarget {
		private final TargetMember targetMember;

		public TargetMember getTargetMember() {
			return targetMember;
		}

		private final File classFileLocation;

		public File getClassFileLocation() {
			return classFileLocation;
		}

		private final CtClass ctClass;

		public CtClass getCtClass() {
			return ctClass;
		}

		protected BaseInjectionTarget(TargetMember targetMember) throws MojoExecutionException {
			this.targetMember = targetMember;
			try {
				classFileLocation = new File( loaderClassPath.find( targetMember.getClassName() ).toURI() );
				ctClass = classPool.get( targetMember.getClassName() );
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve class file path", e );
			}
		}

		protected void writeOutChanges() throws MojoExecutionException {
			getLog().info( "writing injection changes back [" + classFileLocation.getAbsolutePath() + "]" );
			ClassFile classFile = ctClass.getClassFile();
			classFile.compact();
			try {
				DataOutputStream out = new DataOutputStream( new BufferedOutputStream( new FileOutputStream( classFileLocation ) ) );
				try {

					classFile.write( out );
					out.flush();
					if ( ! classFileLocation.setLastModified( System.currentTimeMillis() ) ) {
						getLog().info( "Unable to manually update class file timestamp" );
					}
				}
				finally {
					out.close();
				}
			}
			catch ( IOException e ) {
				throw new MojoExecutionException( "Unable to write out modified class file", e );
			}
		}
	}

	private class ConstantInjectionTarget extends BaseInjectionTarget {
		private final FieldInfo ctFieldInfo;

		private ConstantInjectionTarget(Constant constant) throws MojoExecutionException {
			super( constant );

			try {
				ctFieldInfo = getCtClass().getField( constant.getFieldName() ).getFieldInfo();
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve class field [" + constant.getQualifiedName() + "]", e );
			}
		}

		public void inject(String value) throws MojoExecutionException {
			ctFieldInfo.addAttribute(
					new ConstantAttribute(
							ctFieldInfo.getConstPool(),
							ctFieldInfo.getConstPool().addStringInfo( value )
					)
			);

			writeOutChanges();
		}
	}

	private class MethodBodyReturnReplacementTarget extends BaseInjectionTarget {
		private final CtMethod ctMethod;

		private MethodBodyReturnReplacementTarget(MethodBodyReturn method) throws MojoExecutionException {
			super( method );

			try {
				for ( CtMethod ctMethod : getCtClass().getMethods() ) {
					if ( method.getMethodName().equals( ctMethod.getName() ) ) {
						this.ctMethod = ctMethod;
						return;
					}
				}
				throw new MojoExecutionException( "Could not locate method [" + method.getQualifiedName() + "]" );
			}
			catch ( Throwable e ) {
				throw new MojoExecutionException( "Unable to resolve method [" + method.getQualifiedName() + "]", e );
			}
		}

		public void inject(String value) throws MojoExecutionException {
			try {
				ctMethod.setBody( "{return \"" + value + "\";}" );
			}
			catch ( Throwable t ) {
				throw new MojoExecutionException( "Unable to replace method body", t );
			}
			writeOutChanges();
		}
	}
}
