/*
 * JBoss, Home of Professional Open Source
 * Copyright 2013, Red Hat, Inc. and/or its affiliates, and individual
 * contributors by the @authors tag. See the copyright.txt in the
 * distribution for a full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.maven.plugins.qstools.checkers;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.component.annotations.Component;
import org.codehaus.plexus.component.annotations.Requirement;
import org.jboss.maven.plugins.qstools.QSToolsException;
import org.jboss.maven.plugins.qstools.common.UnusedPropertiesUtil;
import org.jboss.maven.plugins.qstools.config.ConfigurationProvider;
import org.jboss.maven.plugins.qstools.config.Rules;

/**
 * @author Rafael Benevides
 *
 */
@Component(role = QSChecker.class, hint = "unusedPropertiesChecker")
public class UnusedPropertiesChecker implements QSChecker {

    private int violationsQtd;

    @Requirement
    private ConfigurationProvider configurationProvider;

    @Requirement
    private UnusedPropertiesUtil unusedPropertiesUtil;

    private String checkerMessage;

    /*
     * (non-Javadoc)
     * 
     * @see org.jboss.maven.plugins.qstools.QSChecker#getViolatonsQtd()
     */
    @Override
    public int getViolatonsQtd() {
        return violationsQtd;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.jboss.maven.plugins.qstools.QSChecker#resetViolationsQtd()
     */
    @Override
    public void resetViolationsQtd() {
        violationsQtd = 0;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.jboss.maven.plugins.qstools.QSChecker#getCheckerDescription()
     */
    @Override
    public String getCheckerDescription() {
        return "Checks if a project and its modules has a declared and unused propery";
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.jboss.maven.plugins.qstools.QSChecker#check(org.apache.maven.project.MavenProject,
     * org.apache.maven.execution.MavenSession, java.util.List, org.apache.maven.plugin.logging.Log)
     */
    @Override
    public Map<String, List<Violation>> check(MavenProject project, MavenSession mavenSession, List<MavenProject> reactorProjects, Log log) throws QSToolsException {
        Map<String, List<Violation>> results = new TreeMap<String, List<Violation>>();
        Rules rules = configurationProvider.getQuickstartsRules(project.getGroupId());
        if (rules.isCheckerIgnored(this.getClass())) {
            checkerMessage = "This checker is ignored for this groupId in config file.";
        } else {
            try {

                List<UnusedPropertiesUtil.PomInformation> unusedPropertyInfo = unusedPropertiesUtil.findUnusedProperties(reactorProjects, rules);

                // Construct a violation for each unused property
                for (UnusedPropertiesUtil.PomInformation pi : unusedPropertyInfo) {
                    // Get relative path based on maven work dir
                    String rootDirectory = (mavenSession.getExecutionRootDirectory() + File.separator).replace("\\", "\\\\");
                    String fileAsString = pi.getProject().getFile().getAbsolutePath().replace(rootDirectory, "");
                    if (results.get(fileAsString) == null) {
                        results.put(fileAsString, new ArrayList<Violation>());
                    }
                    String msg = "Property [%s] was declared but was never used";
                    results.get(fileAsString).add(new Violation(getClass(), pi.getLine(), String.format(msg, pi.getProperty())));
                    violationsQtd++;
                }
                if (getCheckerMessage() != null) {
                    log.info("--> Checker Message: " + getCheckerMessage());
                }
                if (violationsQtd > 0) {
                    log.info("There are " + violationsQtd + " checkers violations");
                }
            } catch (Exception e) {
                throw new QSToolsException(e);
            }
        }
        return results;
    }

    @Override
    public String getCheckerMessage() {
        return checkerMessage;
    }

}
