/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.mc.common;

import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.kernel.Kernel;
import org.jboss.kernel.spi.dependency.KernelController;

/**
 * Simple MC thread local helper.
 * 
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class ThreadLocalUtils
{
   private static ThreadLocal<Kernel> tlKernel = new ThreadLocal<Kernel>();
   private static ThreadLocal<DeploymentUnit> tlDU = new ThreadLocal<DeploymentUnit>();

   /**
    * Put Kernel.
    *
    * @param kernel the kernel instance
    */
   public static void putKernel(Kernel kernel)
   {
      tlKernel.set(kernel);
   }

   /**
    * Get Kernel.
    *
    * @return the kernel
    */
   public static Kernel getKernel()
   {
      return tlKernel.get();   
   }

   /**
    * Remove the Kernel.
    */
   public static void removeKernel()
   {
      tlKernel.remove();
   }

   /**
    * Get controller.
    *
    * @return the controller
    */
   public static KernelController getController()
   {
      Kernel kernel = getKernel();
      return (kernel != null) ? kernel.getController() : null;
   }

   /**
    * Put DeploymentUnit.
    *
    * @param unit the deployment unit
    */
   public static void putUnit(DeploymentUnit unit)
   {
      tlDU.set(unit);
   }

   /**
    * Get DeploymentUnit.
    *
    * @return the deployment unit
    */
   public static DeploymentUnit getUnit()
   {
      return tlDU.get();
   }

   /**
    * Remove DeploymentUnit.
    */
   public static void removeUnit()
   {
      tlDU.remove();
   }

   /**
    * Get attachment.
    *
    * @param attachmentClass the attachment class
    * @param <T> exact attachment type
    * @return the attachment
    */
   public static <T> T getAttachment(Class<T> attachmentClass)
   {
      DeploymentUnit unit = getUnit();
      return (unit != null) ? unit.getAttachment(attachmentClass) : null;
   }

   /**
    * Get attachment.
    *
    * @param attachmentName the attachment name
    * @return the attachment
    */
   public static Object getAttachment(String attachmentName)
   {
      DeploymentUnit unit = getUnit();
      return (unit != null) ? unit.getAttachment(attachmentName) : null;
   }

   /**
    * Get attachment.
    *
    * @param attachmentName the attachment name
    * @param attachmentClass the attachment class
    * @param <T> exact attachment type
    * @return the attachment
    */
   public static <T> T getAttachment(String attachmentName, Class<T> attachmentClass)
   {
      DeploymentUnit unit = getUnit();
      return (unit != null) ? unit.getAttachment(attachmentName, attachmentClass) : null;
   }
}
