/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.mcann.indexer;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A HTML report generator
 * @author <a href="mailto:jesper.pedersen@jboss.org">Jesper Pedersen</a>
 */
public class HTMLWriter
{
   private static Logger log = Logger.getLogger(HTMLWriter.class.getName());

   /** New line character */
   protected static final String NEW_LINE = System.getProperty("line.separator");

   /**
    * Constructor
    */
   private HTMLWriter()
   {
   }

   /**
    * Generate CSS files
    * @param directory Where the reports go
    */
   public static void generateCSS(File directory)
   {
      byte buffer[] = new byte[8192];
      int bytesRead;

      InputStream is = null;
      OutputStream os = null;
      try
      {
         is = HTMLWriter.class.getClassLoader().getResourceAsStream("style.css");

         if (is != null)
         {
            os = new FileOutputStream(directory.getAbsolutePath() + File.separator + "style.css");
            while ((bytesRead = is.read(buffer)) != -1)
            {
               os.write(buffer, 0, bytesRead);
            }
            
            os.flush();
         }
      }
      catch (Throwable t)
      {
         log.log(Level.SEVERE, t.getMessage(), t);
      }
      finally
      {
         try
         {
            if (is != null)
               is.close();
         }
         catch (IOException ioe)
         {
            // Ignore
         }

         try
         {
            if (os != null)
               os.close();
         }
         catch (IOException ioe)
         {
            // Ignore
         }
      }
   }

   /**
    * Generate report
    * @param directory Where the reports go
    * @param name The name of the report
    * @param data The raw data
    */
   public static void generateReport(File directory, String name, Map<String, Collection<String>> data)
   {
      BufferedWriter bw = null;
      try
      {
         FileWriter fw = new FileWriter(directory.getAbsolutePath() + File.separator + name + ".html");
         bw = new BufferedWriter(fw, 8192);

         bw.write("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"" +
                  "\"http://www.w3.org/TR/html4/loose.dtd\">" + NEW_LINE);
         bw.write("<html>" + NEW_LINE);
         bw.write("<head>" + NEW_LINE);
         bw.write("  <title>Papaki: " + name + "</title>" + NEW_LINE);
         bw.write("  <meta http-equiv=\"Content-Type\" content=\"text/html;charset=utf-8\">" + NEW_LINE);
         bw.write("  <link rel=\"stylesheet\" type=\"text/css\" href=\"style.css\">" + NEW_LINE);
         bw.write("</head>" + NEW_LINE);
         bw.write("<body>" + NEW_LINE);
         bw.write(NEW_LINE);

         bw.write("<h1>Papaki: " + name + "</h1>" + NEW_LINE);

         bw.write("<table>" + NEW_LINE);

         bw.write("  <tr>" + NEW_LINE);
         bw.write("     <th>Annotation</th>" + NEW_LINE);
         bw.write("     <th>Location</th>" + NEW_LINE);
         bw.write("  </tr>" + NEW_LINE);

         boolean odd = true;

         Iterator<Map.Entry<String, Collection<String>>> it = data.entrySet().iterator();
         while (it.hasNext())
         {
            Map.Entry<String, Collection<String>> entry = it.next();

            String annotation = entry.getKey();
            Collection<String> classes = entry.getValue();

            if (odd)
            {
               bw.write("  <tr class=\"rowodd\">" + NEW_LINE);
            }
            else
            {
               bw.write("  <tr class=\"roweven\">" + NEW_LINE);
            }
            bw.write("     <td>" + annotation + "</td>" + NEW_LINE);
            bw.write("     <td>");

            Iterator<String> cit = classes.iterator();
            while (cit.hasNext())
            {
               bw.write(cit.next());

               if (cit.hasNext())
               {
                  bw.write(", ");
               }
            }

            bw.write("</td>" + NEW_LINE);
            bw.write("  </tr>" + NEW_LINE);
         
            odd = !odd;
         }

         bw.write("</table>" + NEW_LINE);


         bw.write(NEW_LINE);
         bw.write("<p>" + NEW_LINE);
         bw.write("<hr>" + NEW_LINE);
         bw.write("Generated by: <a href=\"http://www.jboss.org/projects/papaki\">Papaki</a>" + NEW_LINE);
         bw.write(NEW_LINE);
         bw.write("</body>" + NEW_LINE);
         bw.write("</html>" + NEW_LINE);

         bw.flush();
      }
      catch (Throwable t)
      {
         log.log(Level.SEVERE, t.getMessage(), t);
      }
      finally
      {
         try
         {
            if (bw != null)
               bw.close();
         }
         catch (IOException ioe)
         {
            // Ignore
         }
      }
   }
}
