/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.mcann.indexer;

import org.jboss.mcann.AnnotationRepository;
import org.jboss.mcann.AnnotationScanner;
import org.jboss.mcann.AnnotationScannerFactory;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.GZIPOutputStream;

/**
 * The Papaki Indexer
 * @author <a href="mailto:jesper.pedersen@jboss.org">Jesper Pedersen</a>
 */
public class Main
{
   private static Logger log = Logger.getLogger(Main.class.getName());

   /**
    * Constructor
    */
   public Main()
   {
   }

   /**
    * Store
    * @param ari The annotation repository implementation
    * @param directory The destination directory
    * @exception Exception Thrown if an error occurs
    */
   public void store(AnnotationRepository ari, File directory) throws Exception
   {
      if (ari == null)
         throw new IllegalArgumentException("ARI is null");

      if (directory == null || !directory.exists())
         throw new IllegalArgumentException("Directory is null");

      File file = new File(directory, AnnotationRepository.MCANN_METADATA_BINARY);
      FileOutputStream fos = new FileOutputStream(file);
      BufferedOutputStream bos = new BufferedOutputStream(fos);
      GZIPOutputStream gos = new GZIPOutputStream(bos);
      ObjectOutputStream oos = new ObjectOutputStream(gos);

      oos.writeObject(ari);

      oos.flush();
      oos.close();
   }

   /**
    * Usage
    */
   private static void usage()
   {
      System.out.println("Usage: Indexer <input-jar> <output-jar>");
   }

   /**
    * Main
    * @param args The program arguments
    */
   public static void main(String[] args)
   {
      try
      {
         if (args.length == 2)
         {
            Main m = new Main();
            
            File input = new File(args[0]);
            File output = new File(args[1]);

            File tmp = new File(System.getProperty("java.io.tmpdir"));

            File root = FileUtil.extract(input, tmp);
            File destination = new File(root, "META-INF");

            AnnotationScanner as = AnnotationScannerFactory.getStrategy(AnnotationScannerFactory.JAVA_LANG_REFLECT);
            URL[] urls = {input.toURI().toURL()};
            URLClassLoader ucl = new URLClassLoader(urls);
            AnnotationRepository ari = as.scan(ucl, urls);

            m.store(ari, destination);

            FileUtil.compress(root, output);
               
            FileUtil.recursiveDelete(root);
         }
         else
         {
            usage();
         }
      }
      catch (Throwable t)
      {
         log.log(Level.SEVERE, t.getMessage(), t);
      }
   }
}
