/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.metadata.annotation.creator.jboss;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.metadata.annotation.creator.AbstractFinderUser;
import org.jboss.metadata.annotation.creator.EJBClassProcessor;
import org.jboss.metadata.annotation.creator.EJBFieldProcessor;
import org.jboss.metadata.annotation.creator.EJBMethodProcessor;
import org.jboss.metadata.annotation.creator.EJBsClassProcessor;
import org.jboss.metadata.annotation.creator.PersistenceContextClassProcessor;
import org.jboss.metadata.annotation.creator.PersistenceContextFieldProcessor;
import org.jboss.metadata.annotation.creator.PersistenceContextMethodProcessor;
import org.jboss.metadata.annotation.creator.Processor;
import org.jboss.metadata.annotation.creator.ResourceClassProcessor;
import org.jboss.metadata.annotation.creator.ResourceFieldProcessor;
import org.jboss.metadata.annotation.creator.ResourceMethodProcessor;
import org.jboss.metadata.annotation.creator.ResourcesClassProcessor;
import org.jboss.metadata.annotation.creator.ws.WebServiceRefClassProcessor;
import org.jboss.metadata.annotation.creator.ws.WebServiceRefFieldProcessor;
import org.jboss.metadata.annotation.creator.ws.WebServiceRefMethodProcessor;
import org.jboss.metadata.annotation.creator.ws.WebServiceRefsClassProcessor;
import org.jboss.metadata.annotation.finder.AnnotationFinder;
import org.jboss.metadata.ejb.jboss.JBossEnvironmentRefsGroupMetaData;
import org.jboss.metadata.javaee.jboss.JBossServiceReferencesMetaData;
import org.jboss.metadata.javaee.spec.AnnotatedEJBReferencesMetaData;
import org.jboss.metadata.javaee.spec.EnvironmentRefsGroupMetaData;
import org.jboss.metadata.javaee.spec.PersistenceContextReferencesMetaData;
import org.jboss.metadata.javaee.spec.PersistenceUnitReferencesMetaData;
import org.jboss.metadata.javaee.spec.RemoteEnvironmentRefsGroupMetaData;
import org.jboss.metadata.javaee.spec.ServiceReferencesMetaData;

/**
 * A base ejb component processor.
 * @param MD - the component metadata type
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 72364 $
 */
public abstract class AbstractComponentProcessor<MD>
   extends AbstractFinderUser
{
   private static Logger log = Logger.getLogger(AbstractComponentProcessor.class);

   /** Map<Processor metdata type class, List<Processor for classes>> */
   protected Map<Class<?>, List<Processor<Object, Class<?>>>> typeProcessors;
   /** Map<Processor metdata type class, List<Processor for fields>> */
   protected Map<Class<?>, List<Processor<Object, Field>>> fieldProcessors;
   /** Map<Processor metdata type class, List<Processor for methods>> */
   protected Map<Class<?>, List<Processor<Object, Method>>> methodProcessors;
   /** */
   protected Set<Class> boundedTypes = new HashSet<Class>();

   /**
    * @param finder
    */
   public AbstractComponentProcessor(AnnotationFinder<AnnotatedElement> finder)
   {
      super(finder);
      typeProcessors = new HashMap<Class<?>, List<Processor<Object,Class<?>>>>();
      fieldProcessors = new HashMap<Class<?>, List<Processor<Object,Field>>>();
      methodProcessors = new HashMap<Class<?>, List<Processor<Object,Method>>>();

      // Add component environment processors
      // @Resources/@Resource
      addMethodProcessor(new ResourceMethodProcessor(finder));
      addFieldProcessor(new ResourceFieldProcessor(finder));
      addTypeProcessor(new ResourceClassProcessor(finder));
      addTypeProcessor(new ResourcesClassProcessor(finder));
      // @EJBs/@EJB
      addMethodProcessor(new EJBMethodProcessor(finder));
      addFieldProcessor(new EJBFieldProcessor(finder));
      addTypeProcessor(new EJBClassProcessor(finder));
      addTypeProcessor(new EJBsClassProcessor(finder));
      // @PersistenceContext
      addFieldProcessor(new PersistenceContextFieldProcessor(finder));
      addMethodProcessor(new PersistenceContextMethodProcessor(finder));
      addTypeProcessor(new PersistenceContextClassProcessor(finder));
      // @PostConstruct/@PreDestroy
      addMethodProcessor(new PostConstructMethodProcessor(finder));
      addMethodProcessor(new PreDestroyMethodProcessor(finder));
      // @WebServiceRef (includes @HandlerChain)
      addMethodProcessor(new WebServiceRefMethodProcessor(finder));
      addFieldProcessor(new WebServiceRefFieldProcessor(finder));  
      addTypeProcessor(new WebServiceRefClassProcessor(finder));
      addTypeProcessor(new WebServiceRefsClassProcessor(finder));
   }

   public void addFieldProcessor(Processor processor)
   {
      Class processorType = getProcessorMetaDataType(processor);
      log.debug("addFieldProcessor: "+processor+", for type: "+processorType);
      List<Processor<Object, Field>> processors = fieldProcessors.get(processorType);
      if(processors == null)
      {
         processors = new ArrayList<Processor<Object, Field>>();
         fieldProcessors.put(processorType, processors);
      }
      processors.add(processor);
   }

   public void addMethodProcessor(Processor processor)
   {
      Class processorType = getProcessorMetaDataType(processor);
      log.debug("addMethodProcessor: "+processor+", for type: "+processorType);
      List<Processor<Object, Method>> processors = methodProcessors.get(processorType);
      if(processors == null)
      {
         processors = new ArrayList<Processor<Object, Method>>();
         methodProcessors.put(processorType, processors);
      }
      processors.add(processor);
   }
   public void addTypeProcessor(Processor processor)
   {
      Class processorType = getProcessorMetaDataType(processor);
      log.debug("addTypeProcessor: "+processor+", for type: "+processorType);
      List<Processor<Object, Class<?>>> processors = typeProcessors.get(processorType);
      if(processors == null)
      {
         processors = new ArrayList<Processor<Object, Class<?>>>();
         typeProcessors.put(processorType, processors);
      }
      processors.add(processor);
   }

   /**
    * Process type for remote environment metadata related annotations
    * @param metaData
    * @param type
    */
   public void process(RemoteEnvironmentRefsGroupMetaData metaData, Class<?> type)
   {
      // @Resources/@Resource
      processClass(metaData, type, JBossEnvironmentRefsGroupMetaData.class);

      // @EJBs/@EJB
      AnnotatedEJBReferencesMetaData aejbRefs = metaData.getAnnotatedEjbReferences();
      if(aejbRefs == null)
      {
         aejbRefs = new AnnotatedEJBReferencesMetaData();
         metaData.setAnnotatedEjbReferences(aejbRefs);
      }
      processClass(aejbRefs, type);

      // @PersistenceUnit
      PersistenceUnitReferencesMetaData puRefs = metaData.getPersistenceUnitRefs();
      if(puRefs == null)
      {
         puRefs = new PersistenceUnitReferencesMetaData();
         metaData.setPersistenceUnitRefs(puRefs);
      }
      processClass(puRefs, type);
      // @WebServiceRefs/@WebServiceRef
      ServiceReferencesMetaData wsRefs = metaData.getServiceReferences();
      if(wsRefs == null)
      {
         wsRefs = new JBossServiceReferencesMetaData();
         metaData.setServiceReferences(wsRefs);
      }
      processClass(wsRefs, type);
   }
   /**
    * Process type for environment metadata related annotations
    * @param metaData
    * @param type
    */
   public void process(EnvironmentRefsGroupMetaData metaData, Class<?> type)
   {
      this.process((RemoteEnvironmentRefsGroupMetaData)metaData, type);

      // @PersistenceContext
      PersistenceContextReferencesMetaData pcRefs = metaData.getPersistenceContextRefs();
      if(pcRefs == null)
      {
         pcRefs = new PersistenceContextReferencesMetaData();
         metaData.setPersistenceContextRefs(pcRefs);
      }
      processClass(pcRefs, type);
   }
   /**
    * Process type for component metadata related annotations
    * @param metaData
    * @param type
    */
   public void process(MD metaData, Class<?> type)
   {
      processClass(metaData, type);
   }

   /**
    * Process a the cls annotations at the type, method and field levels
    * into the argument metaData. Only processors registered for the metaData
    * type will be run.
    * 
    * @param <T>
    * @param metaData
    * @param cls
    */
   protected <T> void processClass(T metaData, Class<?> cls)
   {
      Class<?> type = metaData.getClass();
      boolean trace = log.isTraceEnabled();
      int processorCount = processClass(metaData, cls, type);
      if(trace)
         log.trace("Found "+processorCount+" processors for type: "+type);
      // See if metaData has a bounded type
      type = type.getSuperclass();
      while(boundedTypes.contains(type) == false && type != Object.class)
      {
         type = type.getSuperclass();
      }
      // 
      if(type != Object.class)
      {
         if(trace)
            log.trace("Reprocessing class using bounded processor type: "+type);
         processClass(metaData, cls, type);
      }
      // Also process the interfaces of the metadata class
      for(Class<?> iface : metaData.getClass().getInterfaces())
      {
         if(boundedTypes.contains(iface));
         {
            processClass(metaData, cls, iface);
         }
      }
   }
   private <T> int processClass(T metaData, Class<?> cls, Class processorType)
   {
      boolean trace = log.isTraceEnabled();
      int processorCount = 0;
      if(trace)
         log.trace("processClass for metaData: "+processorType+", class: "+cls);

      List<Processor<Object, Class<?>>> tps = typeProcessors.get(processorType);
      if(tps != null)
      {
         processorCount += tps.size();
         if(trace)
            log.trace("typeProcessors("+tps.size()+") for metaData: "+tps);
         for(Processor<Object, Class<?>> processor : tps)
         {
            processor.process(metaData, cls);
         }
         
         for(Class<?> intf : cls.getInterfaces())
         {
            for(Processor<Object, Class<?>> processor : tps)
            {
               processor.process(metaData, intf);
            }
         }
      }
      
      List<Processor<Object, Field>> fps = fieldProcessors.get(processorType);
      if(fps != null)
      {
         processorCount += fps.size();
         if(trace)
            log.trace("fieldProcessors("+fps.size()+") for metaData: "+fps);
         Field[] fields = {};
         try
         {
            fields = cls.getDeclaredFields();
         }
         catch(Throwable e)
         {
            log.debug("Failed to get DeclaredFields for: "+cls, e);
         }
         for(Field field : fields)
         {
            for(Processor<Object, Field> processor : fps)
            {
               processor.process(metaData, field);
            }
         }
      }

      List<Processor<Object, Method>> mps = methodProcessors.get(processorType);
      if(mps != null)
      {
         processorCount += mps.size();
         if(trace)
            log.trace("methodProcessors("+mps.size()+") for metaData: "+mps);
         Method[] methods = {};
         try
         {
            methods = cls.getDeclaredMethods();
         }
         catch(Throwable e)
         {
            log.debug("Failed to get DeclaredMethods for: "+cls, e);            
         }
         for(Method method : methods)
         {
            if(trace)
               log.trace("process method " + method);
            for(Processor<Object, Method> processor : mps)
            {
               processor.process(metaData, method);
            }
         }
      }
      
      if(cls.getSuperclass() != null && cls.getSuperclass() != Object.class)
         processorCount += processClass(metaData, cls.getSuperclass(), processorType);
      return processorCount;
   }

   /**
    * Determine the Processor<T, ?> T generic processorType class.
    * 
    * @param processor
    * @return The Class for the T parameter type. If this was an upper bound,
    * this will have been added to the boundedTypes set.
    */
   protected Class getProcessorMetaDataType(Processor processor)
   {
      // Find the Proccessor<T, ?> interface
      Type[] interfaces = processor.getClass().getGenericInterfaces();
      Type processorType = null;
      for(Type t : interfaces)
      {
         ParameterizedType pt = (ParameterizedType) t;
         Type rawType = pt.getRawType();
         if((rawType instanceof Class) && ((Class)rawType).getName().equals("org.jboss.metadata.annotation.creator.Processor"))
         {
            processorType = t;
            break;
         }
      }
      if(processorType == null)
         throw new IllegalStateException("No generic Processor interface found on: "+processor);

      // Get the type of the T parameter
      ParameterizedType pt = (ParameterizedType) processorType;
      Type t0 = pt.getActualTypeArguments()[0];
      Class t = null;
      if(t0 instanceof Class)
         t = (Class) t0;
      else if(t0 instanceof TypeVariable)
      {
         TypeVariable tv = (TypeVariable) t0;
         t = (Class)tv.getBounds()[0];
         boundedTypes.add(t);
      }
      return t;
   }

}
