/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.metadata.ejb.jboss.jndipolicy.plugins;

import org.jboss.metadata.ejb.jboss.jndipolicy.spi.DefaultJndiBindingPolicy;
import org.jboss.metadata.ejb.jboss.jndipolicy.spi.EjbDeploymentSummary;
import org.jboss.metadata.ejb.jboss.jndipolicy.spi.KnownInterfaces;
import org.jboss.metadata.ejb.jboss.jndipolicy.spi.KnownInterfaces.KnownInterfaceType;

/**
 * A basic implementation of DefaultJndiBindingPolicy that uses a base jndi
 * name and generates the following from that:
 * getDefaultLocalHomeJndiName : baseJndiName + "/" + KnownInterfaces.LOCAL_HOME
 * getDefaultLocalJndiName : baseJndiName + "/" + KnownInterfaces.LOCAL
 * getDefaultRemoteHomeJndiName : baseJndiName + "/" + KnownInterfaces.HOME
 * getDefaultRemoteJndiName : baseJndiName + "/" + KnownInterfaces.REMOTE
 * getJndiName : baseJndiName
 * getJndiName(String iface) : baseJndiName + "/" + iface (if iface is not in KnownInterfaces)
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision:$
 */
public class BasicJndiBindingPolicy implements DefaultJndiBindingPolicy
{
   private String baseJndiName;

   public BasicJndiBindingPolicy()
   {
      
   }
   public BasicJndiBindingPolicy(String baseJndiName)
   {
      this.baseJndiName = baseJndiName;
   }
   
   public String getBaseJndiName()
   {
      return baseJndiName;
   }
   public void setBaseJndiName(String baseJndiName)
   {
      this.baseJndiName = baseJndiName;
   }

   public String getDefaultLocalHomeJndiName(EjbDeploymentSummary summary)
   {
      return baseJndiName + "/" + KnownInterfaces.LOCAL_HOME;
   }

   public String getDefaultLocalJndiName(EjbDeploymentSummary summary)
   {
      return baseJndiName + "/" + KnownInterfaces.LOCAL;
   }

   public String getDefaultRemoteHomeJndiName(EjbDeploymentSummary summary)
   {
      return baseJndiName + "/" + KnownInterfaces.HOME;
   }

   public String getDefaultRemoteJndiName(EjbDeploymentSummary summary)
   {
      return baseJndiName + "/" + KnownInterfaces.REMOTE;
   }

   public String getJndiName(EjbDeploymentSummary summary)
   {
      return baseJndiName;
   }

   public String getJndiName(EjbDeploymentSummary summary, String iface,
         KnownInterfaceType ifaceType)
   {
      String jndiName = null;
      if(KnownInterfaces.isKnownInterface(iface))
         iface = null;

      if(ifaceType == KnownInterfaceType.UNKNOWN)
      {
         if(iface == null)
            jndiName = baseJndiName;
         else
            jndiName = baseJndiName + "/" + iface;
      }
      else
      {
         boolean is3x = summary.getBeanMD().getJBossMetaData().isEJB3x();
         
         if(is3x)
         {
            // EJB 3
            switch(ifaceType)
            {
               // base-name / (local|remote) - iface
               case BUSINESS_LOCAL:
               case BUSINESS_REMOTE:
               case UNKNOWN:
                  if(iface == null)
                     jndiName = baseJndiName + "/" + ifaceType.toSuffix();
                  else
                     jndiName = baseJndiName + "/" + ifaceType.toSuffix() + "-" + iface;
               break;
               case LOCAL_HOME:
                     // base-name / (local|remote)Home
                     jndiName = baseJndiName + "/" + ifaceType.toSuffix();
                  break;
               case REMOTE_HOME:
                     // base-name / (local|remote)Home
                     jndiName = baseJndiName + "/" + ifaceType.toSuffix();
                  break;
            }
         }
         else
         {
            // EJB 2.x
            switch(ifaceType)
            {               
               // The local home jndi name (same for getLocalJndiName and getLocalHomeJndiName)
               case BUSINESS_LOCAL:
               case LOCAL_HOME:
                     // Use the bean local jndi name or generated name for ejb2.x local homes
                     jndiName = summary.getBeanMD().determineLocalJndiName();
                  break;
                  
               // The home jndi name (same for getJndiName and getHomeJndiName) 
               case BUSINESS_REMOTE:
               case REMOTE_HOME:
                     // Don't append the iface suffix for ejb2.x homes
                     jndiName = baseJndiName;
                  break;
            }
         }

      }
      return jndiName;
   }

}
