/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.domain.management.security;

import java.io.ByteArrayInputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import javax.security.auth.x500.X500Principal;
import org.jboss.as.domain.management.logging.DomainManagementLogger;
import org.jboss.logging.Logger;
import org.wildfly.common.Assert;
import org.wildfly.security.util.ByteStringBuilder;

final class X509CertificateBuilder {
    private static final ZonedDateTime LATEST_VALID = ZonedDateTime.of(9999, 12, 31, 23, 59, 59, 0, ZoneOffset.UTC);
    private static DomainManagementLogger log = (DomainManagementLogger)Logger.getMessageLogger(DomainManagementLogger.class, (String)"org.jboss.as.domain.management.certificate-generation");
    private int version = 3;
    private BigInteger serialNumber = BigInteger.ONE;
    private X500Principal subjectDn;
    private byte[] subjectUniqueId;
    private X500Principal issuerDn;
    private byte[] issuerUniqueId;
    private ZonedDateTime notValidBefore = ZonedDateTime.now();
    private ZonedDateTime notValidAfter = LATEST_VALID;
    private PublicKey publicKey;
    private PrivateKey signingKey;
    private String signatureAlgorithmName;

    public int getVersion() {
        return this.version;
    }

    public X509CertificateBuilder setVersion(int version) {
        Assert.checkMinimumParameter((String)"version", (int)1, (int)version);
        Assert.checkMaximumParameter((String)"version", (int)3, (int)version);
        this.version = version;
        return this;
    }

    public BigInteger getSerialNumber() {
        return this.serialNumber;
    }

    public X509CertificateBuilder setSerialNumber(BigInteger serialNumber) {
        Assert.checkNotNullParam((String)"serialNumber", (Object)serialNumber);
        if (BigInteger.ONE.compareTo(serialNumber) > 0) {
            throw log.serialNumberTooSmall();
        }
        if (serialNumber.bitLength() > 160) {
            throw log.serialNumberTooLarge();
        }
        this.serialNumber = serialNumber;
        return this;
    }

    public X500Principal getSubjectDn() {
        return this.subjectDn;
    }

    public X509CertificateBuilder setSubjectDn(X500Principal subjectDn) {
        Assert.checkNotNullParam((String)"subjectDn", (Object)subjectDn);
        this.subjectDn = subjectDn;
        return this;
    }

    public byte[] getSubjectUniqueId() {
        return this.subjectUniqueId;
    }

    public X509CertificateBuilder setSubjectUniqueId(byte[] subjectUniqueId) {
        Assert.checkNotNullParam((String)"subjectUniqueId", (Object)subjectUniqueId);
        this.subjectUniqueId = subjectUniqueId;
        return this;
    }

    public X500Principal getIssuerDn() {
        return this.issuerDn;
    }

    public X509CertificateBuilder setIssuerDn(X500Principal issuerDn) {
        Assert.checkNotNullParam((String)"issuerDn", (Object)issuerDn);
        this.issuerDn = issuerDn;
        return this;
    }

    public byte[] getIssuerUniqueId() {
        return this.issuerUniqueId;
    }

    public X509CertificateBuilder setIssuerUniqueId(byte[] issuerUniqueId) {
        Assert.checkNotNullParam((String)"issuerUniqueId", (Object)issuerUniqueId);
        this.issuerUniqueId = issuerUniqueId;
        return this;
    }

    public ZonedDateTime getNotValidBefore() {
        return this.notValidBefore;
    }

    public X509CertificateBuilder setNotValidBefore(ZonedDateTime notValidBefore) {
        Assert.checkNotNullParam((String)"notValidBefore", (Object)notValidBefore);
        this.notValidBefore = notValidBefore;
        return this;
    }

    public ZonedDateTime getNotValidAfter() {
        return this.notValidAfter;
    }

    public X509CertificateBuilder setNotValidAfter(ZonedDateTime notValidAfter) {
        Assert.checkNotNullParam((String)"notValidAfter", (Object)notValidAfter);
        this.notValidAfter = notValidAfter;
        return this;
    }

    public PublicKey getPublicKey() {
        return this.publicKey;
    }

    public X509CertificateBuilder setPublicKey(PublicKey publicKey) {
        Assert.checkNotNullParam((String)"publicKey", (Object)publicKey);
        this.publicKey = publicKey;
        return this;
    }

    public PrivateKey getSigningKey() {
        return this.signingKey;
    }

    public X509CertificateBuilder setSigningKey(PrivateKey signingKey) {
        Assert.checkNotNullParam((String)"signingKey", (Object)signingKey);
        this.signingKey = signingKey;
        return this;
    }

    public String getSignatureAlgorithmName() {
        return this.signatureAlgorithmName;
    }

    public X509CertificateBuilder setSignatureAlgorithmName(String signatureAlgorithmName) {
        Assert.checkNotNullParam((String)"signatureAlgorithmName", (Object)signatureAlgorithmName);
        this.signatureAlgorithmName = signatureAlgorithmName;
        return this;
    }

    public X509Certificate build() throws CertificateException {
        byte[] tbsCertificate = this.getTBSBytes();
        ByteStringBuilder b = new ByteStringBuilder();
        DEREncoder derEncoder = new DEREncoder(b);
        derEncoder.startSequence();
        derEncoder.writeEncoded(tbsCertificate);
        String signatureAlgorithmName = this.signatureAlgorithmName;
        String signatureAlgorithmOid = ASN1.oidFromSignatureAlgorithm(signatureAlgorithmName);
        derEncoder.startSequence();
        derEncoder.encodeObjectIdentifier(signatureAlgorithmOid);
        derEncoder.endSequence();
        try {
            Signature signature = Signature.getInstance(signatureAlgorithmName);
            signature.initSign(this.signingKey);
            signature.update(tbsCertificate);
            derEncoder.encodeBitString(signature.sign());
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | SignatureException e) {
            throw log.certSigningFailed(e);
        }
        derEncoder.endSequence();
        byte[] bytes = b.toArray();
        CertificateFactory factory = CertificateFactory.getInstance("X.509");
        return (X509Certificate)factory.generateCertificate(new ByteArrayInputStream(bytes));
    }

    byte[] getTBSBytes() {
        X509EncodedKeySpec keySpec;
        BigInteger serialNumber = this.serialNumber;
        int version = this.version;
        String signatureAlgorithmName = this.signatureAlgorithmName;
        if (signatureAlgorithmName == null) {
            throw log.noSignatureAlgorithmNameGiven();
        }
        String signatureAlgorithmOid = ASN1.oidFromSignatureAlgorithm(signatureAlgorithmName);
        if (signatureAlgorithmOid == null) {
            throw log.unknownSignatureAlgorithmName(signatureAlgorithmName);
        }
        PrivateKey signingKey = this.signingKey;
        if (signingKey == null) {
            throw log.noSigningKeyGiven();
        }
        String signingKeyAlgorithm = signingKey.getAlgorithm();
        if (!signatureAlgorithmName.endsWith("with" + signingKeyAlgorithm) || signatureAlgorithmName.contains("with" + signingKeyAlgorithm + "and")) {
            throw log.signingKeyNotCompatWithSig(signingKeyAlgorithm, signatureAlgorithmName);
        }
        ZonedDateTime notValidBefore = this.notValidBefore;
        ZonedDateTime notValidAfter = this.notValidAfter;
        if (notValidBefore.compareTo(notValidAfter) > 0) {
            throw log.validAfterBeforeValidBefore(notValidBefore, notValidAfter);
        }
        X500Principal issuerDn = this.issuerDn;
        if (issuerDn == null) {
            throw log.noIssuerDnGiven();
        }
        X500Principal subjectDn = this.subjectDn;
        PublicKey publicKey = this.publicKey;
        if (publicKey == null) {
            throw log.noPublicKeyGiven();
        }
        byte[] issuerUniqueId = this.issuerUniqueId;
        byte[] subjectUniqueId = this.subjectUniqueId;
        if (version < 2 && (issuerUniqueId != null || subjectUniqueId != null)) {
            throw log.uniqueIdNotAllowed();
        }
        ByteStringBuilder b = new ByteStringBuilder();
        DEREncoder derEncoder = new DEREncoder(b);
        derEncoder.startSequence();
        derEncoder.startExplicit(0);
        derEncoder.encodeInteger(version - 1);
        derEncoder.endExplicit();
        derEncoder.encodeInteger(serialNumber);
        derEncoder.startSequence();
        derEncoder.encodeObjectIdentifier(signatureAlgorithmOid);
        derEncoder.endSequence();
        derEncoder.writeEncoded(issuerDn.getEncoded());
        derEncoder.startSequence();
        derEncoder.encodeGeneralizedTime(notValidBefore.withZoneSameInstant(ZoneOffset.UTC));
        derEncoder.encodeGeneralizedTime(notValidAfter.withZoneSameInstant(ZoneOffset.UTC));
        derEncoder.endSequence();
        if (subjectDn != null) {
            derEncoder.writeEncoded(subjectDn.getEncoded());
        }
        String publicKeyAlgorithm = publicKey.getAlgorithm();
        try {
            KeyFactory keyFactory = KeyFactory.getInstance(publicKeyAlgorithm);
            Key translatedKey = keyFactory.translateKey(publicKey);
            keySpec = keyFactory.getKeySpec(translatedKey, X509EncodedKeySpec.class);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw log.invalidKeyForCert(publicKeyAlgorithm, e);
        }
        derEncoder.writeEncoded(keySpec.getEncoded());
        if (issuerUniqueId != null) {
            derEncoder.encodeImplicit(1);
            derEncoder.encodeBitString(issuerUniqueId);
        }
        if (subjectUniqueId != null) {
            derEncoder.encodeImplicit(2);
            derEncoder.encodeBitString(subjectUniqueId);
        }
        derEncoder.endSequence();
        return b.toArray();
    }

    static class ASN1 {
        public static final int BOOLEAN_TYPE = 1;
        public static final int INTEGER_TYPE = 2;
        public static final int BIT_STRING_TYPE = 3;
        public static final int OCTET_STRING_TYPE = 4;
        public static final int NULL_TYPE = 5;
        public static final int OBJECT_IDENTIFIER_TYPE = 6;
        public static final int UTF8_STRING_TYPE = 12;
        public static final int PRINTABLE_STRING_TYPE = 19;
        public static final int IA5_STRING_TYPE = 22;
        public static final int GENERALIZED_TIME_TYPE = 24;
        public static final int UNIVERSAL_STRING_TYPE = 28;
        public static final int BMP_STRING_TYPE = 30;
        public static final int SEQUENCE_TYPE = 48;
        public static final int SET_TYPE = 49;
        public static final int CONSTRUCTED_MASK = 32;
        public static final int APPLICATION_SPECIFIC_MASK = 64;
        public static final int CONTEXT_SPECIFIC_MASK = 128;
        public static final int CLASS_MASK = 192;
        public static final int TAG_NUMBER_MASK = 31;
        public static final String OID_SHA1_WITH_DSA = "1.2.840.10040.4.3";
        public static final String OID_SHA1_WITH_ECDSA = "1.2.840.10045.4.1";
        public static final String OID_SHA224_WITH_ECDSA = "1.2.840.10045.4.3.1";
        public static final String OID_SHA256_WITH_ECDSA = "1.2.840.10045.4.3.2";
        public static final String OID_SHA384_WITH_ECDSA = "1.2.840.10045.4.3.3";
        public static final String OID_SHA512_WITH_ECDSA = "1.2.840.10045.4.3.4";
        public static final String OID_MD2_WITH_RSA = "1.2.840.113549.1.1.2";
        public static final String OID_MD4_WITH_RSA = "1.2.840.113549.1.1.3";
        public static final String OID_MD5_WITH_RSA = "1.2.840.113549.1.1.4";
        public static final String OID_SHA1_WITH_RSA = "1.2.840.113549.1.1.5";
        public static final String OID_SHA256_WITH_RSA = "1.2.840.113549.1.1.11";
        public static final String OID_SHA384_WITH_RSA = "1.2.840.113549.1.1.12";
        public static final String OID_SHA512_WITH_RSA = "1.2.840.113549.1.1.13";

        ASN1() {
        }

        public static String oidFromSignatureAlgorithm(String algorithmName) {
            switch (algorithmName) {
                case "NONEwithRSA": {
                    return null;
                }
                case "MD2withRSA": {
                    return OID_MD2_WITH_RSA;
                }
                case "MD5withRSA": {
                    return OID_MD5_WITH_RSA;
                }
                case "SHA1withRSA": {
                    return OID_SHA1_WITH_RSA;
                }
                case "SHA256withRSA": {
                    return OID_SHA256_WITH_RSA;
                }
                case "SHA384withRSA": {
                    return OID_SHA384_WITH_RSA;
                }
                case "SHA512withRSA": {
                    return OID_SHA512_WITH_RSA;
                }
                case "NONEwithDSA": {
                    return null;
                }
                case "SHA1withDSA": {
                    return OID_SHA1_WITH_DSA;
                }
                case "NONEwithECDSA": {
                    return null;
                }
                case "ECDSA": 
                case "SHA1withECDSA": {
                    return OID_SHA1_WITH_ECDSA;
                }
                case "SHA256withECDSA": {
                    return OID_SHA256_WITH_ECDSA;
                }
                case "SHA384withECDSA": {
                    return OID_SHA384_WITH_ECDSA;
                }
                case "SHA512withECDSA": {
                    return OID_SHA512_WITH_ECDSA;
                }
            }
            return null;
        }
    }

    static class DEREncoder {
        private static final long LARGEST_UNSHIFTED_LONG = 0xCCCCCCCCCCCCCCCL;
        private static final TagComparator TAG_COMPARATOR = new TagComparator();
        private final ArrayDeque<EncoderState> states = new ArrayDeque();
        private final ArrayList<ByteStringBuilder> buffers = new ArrayList();
        private ByteStringBuilder currentBuffer;
        private int currentBufferPos = -1;
        private final ByteStringBuilder target;
        private int implicitTag = -1;
        private static final DateTimeFormatter GENERALIZED_TIME_FORMAT = DateTimeFormatter.ofPattern("yyyyMMddHHmmssX");
        private static final BigInteger[] digits = new BigInteger[]{BigInteger.ZERO, BigInteger.ONE, BigInteger.valueOf(2L), BigInteger.valueOf(3L), BigInteger.valueOf(4L), BigInteger.valueOf(5L), BigInteger.valueOf(6L), BigInteger.valueOf(7L), BigInteger.valueOf(8L), BigInteger.valueOf(9L)};

        public DEREncoder(ByteStringBuilder target) {
            this.target = target;
            this.currentBuffer = target;
        }

        public void startSequence() {
            this.startConstructedElement(48);
        }

        public void startSet() {
            this.startConstructedElement(49);
        }

        public void startSetOf() {
            this.startSet();
        }

        public void startExplicit(int number) {
            this.startExplicit(128, number);
        }

        public void startExplicit(int clazz, int number) {
            int explicitTag = clazz | 0x20 | number;
            this.startConstructedElement(explicitTag);
        }

        private void startConstructedElement(int tag) {
            EncoderState lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                this.updateCurrentBuffer();
                lastState.addChildElement(tag, this.currentBufferPos);
            }
            this.writeTag(tag, this.currentBuffer);
            if (tag != 49) {
                this.updateCurrentBuffer();
            }
            this.states.add(new EncoderState(tag, this.currentBufferPos));
        }

        public void endSequence() throws IllegalStateException {
            EncoderState lastState = this.states.peekLast();
            if (lastState == null || lastState.getTag() != 48) {
                throw log.noSequenceToEnd();
            }
            this.endConstructedElement();
        }

        public void endExplicit() throws IllegalStateException {
            EncoderState lastState = this.states.peekLast();
            if (lastState == null || lastState.getTag() == 48 || lastState.getTag() == 49 || (lastState.getTag() & 0x20) == 0) {
                throw log.noExplicitlyTaggedElementToEnd();
            }
            this.endConstructedElement();
        }

        private void endConstructedElement() {
            ByteStringBuilder dest = this.currentBufferPos > 0 ? this.buffers.get(this.currentBufferPos - 1) : this.target;
            int length = this.currentBuffer.length();
            int numLengthOctets = this.writeLength(length, dest);
            dest.append(this.currentBuffer);
            this.currentBuffer.setLength(0);
            this.currentBuffer = dest;
            --this.currentBufferPos;
            this.states.removeLast();
            EncoderState lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                lastState.addChildLength(1 + numLengthOctets + length);
            }
        }

        public void endSet() throws IllegalStateException {
            this.endSet(TAG_COMPARATOR);
        }

        private void endSet(Comparator<EncoderState> comparator) {
            EncoderState lastState = this.states.peekLast();
            if (lastState == null || lastState.getTag() != 49) {
                throw log.noSetToEnd();
            }
            LinkedList<EncoderState> childElements = lastState.getSortedChildElements(comparator);
            int setBufferPos = lastState.getBufferPos();
            ByteStringBuilder dest = setBufferPos >= 0 ? this.buffers.get(setBufferPos) : this.target;
            int childLength = lastState.getChildLength();
            int numLengthOctets = this.writeLength(lastState.getChildLength(), dest);
            for (EncoderState element : childElements) {
                ByteStringBuilder contents = this.buffers.get(element.getBufferPos());
                dest.append(contents);
                contents.setLength(0);
            }
            this.currentBuffer = dest;
            this.currentBufferPos = setBufferPos;
            this.states.removeLast();
            lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                lastState.addChildLength(1 + numLengthOctets + childLength);
            }
        }

        public void encodeBitString(byte[] str) {
            this.encodeBitString(str, 0);
        }

        public void encodeBitString(byte[] str, int numUnusedBits) {
            byte[] contents = new byte[str.length + 1];
            contents[0] = (byte)numUnusedBits;
            System.arraycopy(str, 0, contents, 1, str.length);
            this.writeElement(3, contents);
        }

        public void encodeGeneralizedTime(ZonedDateTime time) {
            this.writeElement(24, GENERALIZED_TIME_FORMAT.format(time).getBytes(StandardCharsets.UTF_8));
        }

        public void encodeObjectIdentifier(String objectIdentifier) {
            char c;
            int len = objectIdentifier.length();
            if (len == 0) {
                throw log.asnOidMustHaveAtLeast2Components();
            }
            int offs = 0;
            int idx = 0;
            long t = 0L;
            int numComponents = 0;
            int first = -1;
            ByteStringBuilder contents = new ByteStringBuilder();
            block0: while (true) {
                if (Character.isDigit(c = objectIdentifier.charAt(offs + idx++))) {
                    int digit = Character.digit(c, 10);
                    if (t > 0xCCCCCCCCCCCCCCCL) {
                        BigInteger bi = BigInteger.valueOf(t).multiply(BigInteger.TEN).add(digits[digit]);
                        t = 0L;
                        do {
                            if (!Character.isDigit(c = objectIdentifier.charAt(offs + idx++))) {
                                if (c == '.') {
                                    if (numComponents == 0) {
                                        first = this.validateFirstOIDComponent(bi);
                                    } else {
                                        this.encodeOIDComponent(bi, contents, numComponents, first);
                                    }
                                    ++numComponents;
                                    continue block0;
                                }
                                throw log.asnInvalidOidCharacter();
                            }
                            digit = Character.digit(c, 10);
                            bi = bi.multiply(BigInteger.TEN).add(digits[digit]);
                        } while (idx != len);
                        if (numComponents == 0) {
                            throw log.asnOidMustHaveAtLeast2Components();
                        }
                        this.encodeOIDComponent(bi, contents, numComponents, first);
                        this.writeElement(6, contents);
                        return;
                    }
                    t = 10L * t + (long)digit;
                } else if (c == '.') {
                    if (numComponents == 0) {
                        first = this.validateFirstOIDComponent(t);
                    } else {
                        this.encodeOIDComponent(t, contents, numComponents, first);
                    }
                    ++numComponents;
                    t = 0L;
                } else {
                    throw log.asnInvalidOidCharacter();
                }
                if (idx == len) break;
            }
            if (c == '.') {
                throw log.asnInvalidOidCharacter();
            }
            if (numComponents == 0) {
                throw log.asnOidMustHaveAtLeast2Components();
            }
            this.encodeOIDComponent(t, contents, numComponents, first);
            this.writeElement(6, contents);
        }

        public void encodeImplicit(int number) {
            this.encodeImplicit(128, number);
        }

        public void encodeImplicit(int clazz, int number) {
            if (this.implicitTag == -1) {
                this.implicitTag = clazz | number;
            }
        }

        void encodeInteger(int integer) {
            this.encodeInteger(BigInteger.valueOf(integer));
        }

        public void encodeInteger(BigInteger integer) {
            this.writeElement(2, integer.toByteArray());
        }

        public void writeEncoded(byte[] encoded) {
            EncoderState lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                this.updateCurrentBuffer();
                lastState.addChildElement(encoded[0], this.currentBufferPos);
            }
            if (this.implicitTag != -1) {
                this.writeTag(encoded[0], this.currentBuffer);
                this.currentBuffer.append(encoded, 1, encoded.length - 1);
            } else {
                this.currentBuffer.append(encoded);
            }
            if (lastState != null && lastState.getTag() == 49) {
                lastState.addChildLength(this.currentBuffer.length());
            }
        }

        public void flush() {
            while (this.states.size() != 0) {
                EncoderState lastState = this.states.peekLast();
                if (lastState.getTag() == 48) {
                    this.endSequence();
                    continue;
                }
                if (lastState.getTag() != 49) continue;
                this.endSet();
            }
        }

        private int validateFirstOIDComponent(long value) {
            if (value < 0L || value > 2L) {
                throw log.asnInvalidValueForFirstOidComponent();
            }
            return (int)value;
        }

        private int validateFirstOIDComponent(BigInteger value) {
            if (value.compareTo(BigInteger.valueOf(0L)) == -1 || value.compareTo(BigInteger.valueOf(2L)) == 1) {
                throw log.asnInvalidValueForFirstOidComponent();
            }
            return value.intValue();
        }

        private void validateSecondOIDComponent(long second, int first) {
            if (first < 2 && second > 39L) {
                throw log.asnInvalidValueForSecondOidComponent();
            }
        }

        private void validateSecondOIDComponent(BigInteger second, int first) {
            if (first < 2 && second.compareTo(BigInteger.valueOf(39L)) == 1) {
                throw log.asnInvalidValueForSecondOidComponent();
            }
        }

        private void encodeOIDComponent(long value, ByteStringBuilder contents, int numComponents, int first) {
            if (numComponents == 1) {
                this.validateSecondOIDComponent(value, first);
                this.encodeOIDComponent(value + (long)(40 * first), contents);
            } else {
                this.encodeOIDComponent(value, contents);
            }
        }

        private void encodeOIDComponent(BigInteger value, ByteStringBuilder contents, int numComponents, int first) {
            if (numComponents == 1) {
                this.validateSecondOIDComponent(value, first);
                this.encodeOIDComponent(value.add(BigInteger.valueOf(40 * first)), contents);
            } else {
                this.encodeOIDComponent(value, contents);
            }
        }

        private void encodeOIDComponent(long value, ByteStringBuilder contents) {
            int octet;
            for (int shift = 56; shift > 0; shift -= 7) {
                if (value < 1L << shift) continue;
                octet = (int)(value >>> shift | 0x80L);
                contents.append((byte)octet);
            }
            octet = (int)(value & 0x7FL);
            contents.append((byte)octet);
        }

        private void encodeOIDComponent(BigInteger value, ByteStringBuilder contents) {
            int numBytes = (value.bitLength() + 6) / 7;
            if (numBytes == 0) {
                contents.append((byte)0);
            } else {
                byte[] result = new byte[numBytes];
                BigInteger currValue = value;
                for (int i = numBytes - 1; i >= 0; --i) {
                    result[i] = (byte)(currValue.intValue() & 0x7F | 0x80);
                    currValue = currValue.shiftRight(7);
                }
                int n = numBytes - 1;
                result[n] = (byte)(result[n] & 0x7F);
                contents.append(result);
            }
        }

        private void writeTag(int tag, ByteStringBuilder dest) {
            int constructed = tag & 0x20;
            if (this.implicitTag != -1) {
                tag = this.implicitTag | constructed;
                this.implicitTag = -1;
            }
            int tagClass = tag & 0xC0;
            int tagNumber = tag & 0x1F;
            if (tagNumber < 31) {
                dest.append((byte)(tagClass | constructed | tagNumber));
            } else {
                dest.append((byte)(tagClass | constructed | 0x1F));
                if (tagNumber < 128) {
                    dest.append((byte)tagNumber);
                } else {
                    int octet;
                    for (int shift = 28; shift > 0; shift -= 7) {
                        if (tagNumber < 1 << shift) continue;
                        octet = tagNumber >>> shift | 0x80;
                        dest.append((byte)octet);
                    }
                    octet = tagNumber & 0x7F;
                    dest.append((byte)octet);
                }
            }
        }

        private int writeLength(int length, ByteStringBuilder dest) {
            int numLengthOctets;
            if (length < 0) {
                throw log.asnInvalidLength();
            }
            if (length <= 127) {
                numLengthOctets = 1;
            } else {
                numLengthOctets = 1;
                int value = length;
                while ((value >>>= 8) != 0) {
                    ++numLengthOctets;
                }
            }
            if (length > 127) {
                dest.append((byte)(numLengthOctets | 0x80));
            }
            for (int i = (numLengthOctets - 1) * 8; i >= 0; i -= 8) {
                dest.append((byte)(length >> i));
            }
            return numLengthOctets;
        }

        private void updateCurrentBuffer() {
            ++this.currentBufferPos;
            if (this.currentBufferPos < this.buffers.size()) {
                this.currentBuffer = this.buffers.get(this.currentBufferPos);
            } else {
                ByteStringBuilder buffer = new ByteStringBuilder();
                this.buffers.add(buffer);
                this.currentBuffer = buffer;
            }
        }

        private void writeElement(int tag, byte[] contents) {
            EncoderState lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                this.updateCurrentBuffer();
                lastState.addChildElement(tag, this.currentBufferPos);
            }
            this.writeTag(tag, this.currentBuffer);
            this.writeLength(contents.length, this.currentBuffer);
            this.currentBuffer.append(contents);
            if (lastState != null && lastState.getTag() == 49) {
                lastState.addChildLength(this.currentBuffer.length());
            }
        }

        private void writeElement(int tag, ByteStringBuilder contents) {
            EncoderState lastState = this.states.peekLast();
            if (lastState != null && lastState.getTag() == 49) {
                this.updateCurrentBuffer();
                lastState.addChildElement(tag, this.currentBufferPos);
            }
            this.writeTag(tag, this.currentBuffer);
            this.writeLength(contents.length(), this.currentBuffer);
            this.currentBuffer.append(contents);
            if (lastState != null && lastState.getTag() == 49) {
                lastState.addChildLength(this.currentBuffer.length());
            }
        }

        private static class TagComparator
        implements Comparator<EncoderState> {
            private TagComparator() {
            }

            @Override
            public int compare(EncoderState state1, EncoderState state2) {
                return (state1.getTag() | 0x20) - (state2.getTag() | 0x20);
            }
        }

        private class EncoderState {
            private final int tag;
            private final int bufferPos;
            private LinkedList<EncoderState> childElements = new LinkedList();
            private int childLength = 0;

            public EncoderState(int tag, int bufferPos) {
                this.tag = tag;
                this.bufferPos = bufferPos;
            }

            public int getTag() {
                return this.tag;
            }

            public int getBufferPos() {
                return this.bufferPos;
            }

            public ByteStringBuilder getBuffer() {
                return (ByteStringBuilder)DEREncoder.this.buffers.get(this.getBufferPos());
            }

            public int getChildLength() {
                return this.childLength;
            }

            public LinkedList<EncoderState> getSortedChildElements(Comparator<EncoderState> comparator) {
                Collections.sort(this.childElements, comparator);
                return this.childElements;
            }

            public void addChildElement(int tag, int bufferPos) {
                this.childElements.add(new EncoderState(tag, bufferPos));
            }

            public void addChildLength(int length) {
                this.childLength += length;
            }
        }
    }
}

