/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.as.repository;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.AclEntry;
import java.nio.file.attribute.AclFileAttributeView;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import org.jboss.as.repository.ContentFilter;
import org.jboss.as.repository.ContentRepositoryElement;
import org.jboss.as.repository.logging.DeploymentRepositoryLogger;

public class PathUtil {
    public static void copyRecursively(final Path source, final Path target, boolean overwrite) throws IOException {
        final CopyOption[] options = overwrite ? new CopyOption[]{StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING} : new CopyOption[]{StandardCopyOption.COPY_ATTRIBUTES};
        Files.walkFileTree(source, (FileVisitor<? super Path>)new FileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Files.copy(dir, target.resolve(source.relativize(dir)), options);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.copy(file, target.resolve(source.relativize(file)), options);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                DeploymentRepositoryLogger.ROOT_LOGGER.cannotCopyFile(exc, file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void deleteSilentlyRecursively(Path path) {
        if (path != null) {
            try {
                PathUtil.deleteRecursively(path);
            }
            catch (IOException ioex) {
                DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteFile(ioex, path);
            }
        }
    }

    public static void deleteRecursively(final Path path) throws IOException {
        DeploymentRepositoryLogger.ROOT_LOGGER.debugf("Deleting %s recursively", path);
        if (Files.exists(path, new LinkOption[0])) {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    DeploymentRepositoryLogger.ROOT_LOGGER.cannotDeleteFile(exc, path);
                    throw exc;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
    }

    public static final Path resolveSecurely(Path rootPath, String path) {
        Path resolvedPath;
        if (path == null || path.isEmpty()) {
            resolvedPath = rootPath.normalize();
        } else {
            String relativePath = PathUtil.removeSuperflousSlashes(path);
            resolvedPath = rootPath.resolve(relativePath).normalize();
        }
        if (!resolvedPath.startsWith(rootPath)) {
            throw DeploymentRepositoryLogger.ROOT_LOGGER.forbiddenPath(path);
        }
        return resolvedPath;
    }

    private static String removeSuperflousSlashes(String path) {
        if (path.startsWith("/")) {
            return PathUtil.removeSuperflousSlashes(path.substring(1));
        }
        return path;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static final boolean isArchive(Path path) throws IOException {
        if (!Files.exists(path, new LinkOption[0])) return false;
        if (!Files.isRegularFile(path, new LinkOption[0])) return false;
        try (ZipFile zip = new ZipFile(path.toFile());){
            boolean bl = true;
            return bl;
        }
        catch (ZipException e) {
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static final boolean isArchive(InputStream in) throws IOException {
        if (in == null) return false;
        try (ZipInputStream zip = new ZipInputStream(in);){
            boolean bl = zip.getNextEntry() != null;
            return bl;
        }
        catch (ZipException e) {
            return false;
        }
    }

    public static List<ContentRepositoryElement> listFiles(final Path rootPath, Path tempDir, final ContentFilter filter) throws IOException {
        final ArrayList<ContentRepositoryElement> result = new ArrayList<ContentRepositoryElement>();
        if (Files.exists(rootPath, new LinkOption[0])) {
            if (PathUtil.isArchive(rootPath)) {
                return PathUtil.listZipContent(rootPath, filter);
            }
        } else {
            Path file = PathUtil.getFile(rootPath);
            if (PathUtil.isArchive(file)) {
                Path relativePath = file.relativize(rootPath);
                Path target = PathUtil.createTempDirectory(tempDir, "unarchive");
                PathUtil.unzip(file, target);
                return PathUtil.listFiles(target.resolve(relativePath), tempDir, filter);
            }
            throw new FileNotFoundException(rootPath.toString());
        }
        Files.walkFileTree(rootPath, (FileVisitor<? super Path>)new FileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (filter.acceptFile(rootPath, file)) {
                    result.add(ContentRepositoryElement.createFile(this.formatPath(rootPath.relativize(file)), Files.size(file)));
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                String directoryPath;
                if (filter.acceptDirectory(rootPath, dir) && !"/".equals(directoryPath = this.formatDirectoryPath(rootPath.relativize(dir)))) {
                    result.add(ContentRepositoryElement.createFolder(directoryPath));
                }
                return FileVisitResult.CONTINUE;
            }

            private String formatDirectoryPath(Path path) {
                return this.formatPath(path) + '/';
            }

            private String formatPath(Path path) {
                return path.toString().replace(File.separatorChar, '/');
            }
        });
        return result;
    }

    private static List<ContentRepositoryElement> listZipContent(Path zipFilePath, ContentFilter filter) throws IOException {
        ArrayList<ContentRepositoryElement> result = new ArrayList<ContentRepositoryElement>();
        try (ZipFile zip = new ZipFile(zipFilePath.toFile());){
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                String name = entry.getName();
                Path entryPath = zipFilePath.resolve(name);
                if (entry.isDirectory()) {
                    if (!filter.acceptDirectory(zipFilePath, entryPath)) continue;
                    result.add(ContentRepositoryElement.createFolder(name));
                    continue;
                }
                InputStream in = zip.getInputStream(entry);
                Throwable throwable = null;
                try {
                    if (!filter.acceptFile(zipFilePath, entryPath, in)) continue;
                    result.add(ContentRepositoryElement.createFile(name, entry.getSize()));
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (in == null) continue;
                    if (throwable != null) {
                        try {
                            in.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    in.close();
                }
            }
        }
        return result;
    }

    private static List<FileAttribute<Set<PosixFilePermission>>> getPosixAttributes(Path file) throws IOException {
        PosixFileAttributeView posixView;
        if (Files.exists(file, new LinkOption[0]) && Files.getFileStore(file).supportsFileAttributeView(PosixFileAttributeView.class) && (posixView = Files.getFileAttributeView(file, PosixFileAttributeView.class, new LinkOption[0])) != null) {
            return Collections.singletonList(PosixFilePermissions.asFileAttribute(posixView.readAttributes().permissions()));
        }
        return Collections.emptyList();
    }

    private static List<FileAttribute<List<AclEntry>>> getAclAttributes(Path file) throws IOException {
        AclFileAttributeView aclView;
        if (Files.exists(file, new LinkOption[0]) && Files.getFileStore(file).supportsFileAttributeView(AclFileAttributeView.class) && (aclView = Files.getFileAttributeView(file, AclFileAttributeView.class, new LinkOption[0])) != null) {
            final List<AclEntry> entries = aclView.getAcl();
            return Collections.singletonList(new FileAttribute<List<AclEntry>>(){

                @Override
                public List<AclEntry> value() {
                    return entries;
                }

                @Override
                public String name() {
                    return "acl:acl";
                }
            });
        }
        return Collections.emptyList();
    }

    public static Path createTempDirectory(Path dir, String prefix) throws IOException {
        try {
            return Files.createTempDirectory(dir, prefix, new FileAttribute[0]);
        }
        catch (UnsupportedOperationException unsupportedOperationException) {
            return Files.createTempDirectory(dir, prefix, new FileAttribute[0]);
        }
    }

    public static void unzip(Path zip, Path target) throws IOException {
        try (ZipFile zipFile = new ZipFile(zip.toFile());){
            PathUtil.unzip(zipFile, target);
        }
    }

    private static void unzip(ZipFile zip, Path targetDir) throws IOException {
        Enumeration<? extends ZipEntry> entries = zip.entries();
        while (entries.hasMoreElements()) {
            ZipEntry entry = entries.nextElement();
            String name = entry.getName();
            Path current = targetDir.resolve(name);
            if (entry.isDirectory()) {
                if (!Files.exists(current, new LinkOption[0])) {
                    Files.createDirectories(current, new FileAttribute[0]);
                }
            } else {
                if (Files.notExists(current.getParent(), new LinkOption[0])) {
                    Files.createDirectories(current.getParent(), new FileAttribute[0]);
                }
                try (InputStream eis = zip.getInputStream(entry);){
                    Files.copy(eis, current, new CopyOption[0]);
                }
            }
            try {
                Files.getFileAttributeView(current, BasicFileAttributeView.class, new LinkOption[0]).setTimes(entry.getLastModifiedTime(), entry.getLastAccessTime(), entry.getCreationTime());
            }
            catch (IOException iOException) {}
        }
    }

    public static String getFileExtension(Path path) {
        String fileName = path.getFileName().toString();
        int separator = fileName.lastIndexOf(46);
        if (separator > 0) {
            return fileName.substring(separator);
        }
        return "";
    }

    public static Path readFile(Path src, Path tempDir) throws IOException {
        if (PathUtil.isFile(src)) {
            return src;
        }
        Path file = PathUtil.getFile(src);
        if (PathUtil.isArchive(file)) {
            Path relativePath = file.relativize(src);
            Path target = PathUtil.createTempDirectory(tempDir, "unarchive");
            PathUtil.unzip(file, target);
            return PathUtil.readFile(target.resolve(relativePath), tempDir);
        }
        throw new FileNotFoundException(src.toString());
    }

    private static Path getFile(Path src) throws FileNotFoundException {
        if (src.getNameCount() > 1) {
            Path parent = src.getParent();
            if (PathUtil.isFile(parent)) {
                return parent;
            }
            return PathUtil.getFile(parent);
        }
        throw new FileNotFoundException(src.toString());
    }

    private static boolean isFile(Path src) {
        return Files.exists(src, new LinkOption[0]) && Files.isRegularFile(src, new LinkOption[0]);
    }
}

