/*
 * Decompiled with CFR 0.152.
 */
package org.jboss.migration.core;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.HashMap;
import java.util.Map;
import org.jboss.migration.core.ServerMigrationFailureException;
import org.jboss.migration.core.logger.ServerMigrationLogger;

public class MigrationFiles {
    private static final CopyOption[] COPY_OPTIONS = new CopyOption[]{StandardCopyOption.REPLACE_EXISTING};
    private final Map<Path, Path> copiedFiles = new HashMap<Path, Path>();

    MigrationFiles() {
    }

    public synchronized void copy(Path source, Path target) throws IllegalArgumentException, ServerMigrationFailureException {
        if (!Files.exists(source, new LinkOption[0])) {
            throw ServerMigrationLogger.ROOT_LOGGER.sourceFileDoesNotExists(source);
        }
        Path existentCopySource = this.copiedFiles.get(target);
        if (existentCopySource != null) {
            ServerMigrationLogger.ROOT_LOGGER.debugf("Skipping previously copied file %s", source);
        } else {
            try {
                Files.createDirectories(target.getParent(), new FileAttribute[0]);
                Files.walkFileTree(source, new CopyVisitor(source, target, this.copiedFiles));
            }
            catch (IOException e) {
                throw new ServerMigrationFailureException("File copy failed", e);
            }
        }
    }

    static class CopyVisitor
    extends SimpleFileVisitor<Path> {
        private final Map<Path, Path> copiedFiles;
        private final Path source;
        private final Path target;

        CopyVisitor(Path source, Path target, Map<Path, Path> copiedFiles) {
            this.source = source;
            this.target = target;
            this.copiedFiles = copiedFiles;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            return this.copy(dir);
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            return this.copy(file);
        }

        private Path getTargetPath(Path sourceFile) {
            return this.target.resolve(this.source.relativize(sourceFile));
        }

        private synchronized FileVisitResult copy(Path sourcePath) throws ServerMigrationFailureException {
            Path targetPath = this.getTargetPath(sourcePath);
            Path previousSourcePath = this.copiedFiles.put(targetPath, sourcePath);
            if (previousSourcePath != null) {
                if (previousSourcePath.equals(sourcePath)) {
                    return FileVisitResult.CONTINUE;
                }
                throw new ServerMigrationFailureException("Target " + targetPath + " previously copied and sources mismatch: new = " + sourcePath + ", previous = " + previousSourcePath);
            }
            try {
                if (Files.exists(targetPath, new LinkOption[0])) {
                    if (!Files.isDirectory(targetPath, new LinkOption[0])) {
                        Path backupPath = targetPath.resolveSibling(targetPath.getFileName().toString() + ".beforeMigration");
                        Files.move(targetPath, backupPath, COPY_OPTIONS);
                    } else {
                        return FileVisitResult.CONTINUE;
                    }
                }
                ServerMigrationLogger.ROOT_LOGGER.tracef("Copying file %s", targetPath);
                Files.copy(sourcePath, targetPath, COPY_OPTIONS);
            }
            catch (IOException e) {
                throw new ServerMigrationFailureException("File copy failed.", e);
            }
            ServerMigrationLogger.ROOT_LOGGER.debugf("File %s copied to %s.", sourcePath, targetPath);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            Path targetDir = this.getTargetPath(dir);
            FileTime time = Files.getLastModifiedTime(dir, new LinkOption[0]);
            Files.setLastModifiedTime(targetDir, time);
            return FileVisitResult.CONTINUE;
        }
    }

    static class BackupVisitor
    extends SimpleFileVisitor<Path> {
        private final Path source;
        private final Path backup;
        private final Map<Path, Path> copiedFiles;

        BackupVisitor(Path source, Map<Path, Path> copiedFiles) {
            this.source = source;
            this.copiedFiles = copiedFiles;
            this.backup = source.resolveSibling(source.getFileName().toString() + ".beforeMigration");
        }

        @Override
        public FileVisitResult preVisitDirectory(Path sourcePath, BasicFileAttributes attrs) throws IOException {
            boolean backup = true;
            for (Path copiedTarget : this.copiedFiles.keySet()) {
                if (!copiedTarget.equals(sourcePath) && !copiedTarget.startsWith(sourcePath)) continue;
                backup = false;
                break;
            }
            if (backup) {
                Path backupPath = this.getBackupPath(sourcePath);
                Files.move(sourcePath, backupPath, COPY_OPTIONS);
                return Files.newDirectoryStream(backupPath).iterator().hasNext() ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
            }
            return Files.newDirectoryStream(sourcePath).iterator().hasNext() ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path sourcePath, BasicFileAttributes attrs) throws IOException {
            Path backupPath = this.getBackupPath(sourcePath);
            Files.move(sourcePath, backupPath, COPY_OPTIONS);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path sourcePath, IOException exc) throws IOException {
            Path backupPath = this.getBackupPath(sourcePath);
            if (Files.exists(backupPath, new LinkOption[0])) {
                FileTime time = Files.getLastModifiedTime(sourcePath, new LinkOption[0]);
                Files.setLastModifiedTime(backupPath, time);
            }
            return FileVisitResult.CONTINUE;
        }

        private Path getBackupPath(Path sourcePath) {
            return this.backup.resolve(this.source.relativize(sourcePath));
        }
    }
}

