/*
 * Copyright 2015 Red Hat Inc. and/or its affiliates and other contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.qa.jcontainer;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

public class Configuration {

	protected final File directory;
	protected final String host;
	protected final Integer port;
	protected final String username;
	protected final String password;
	protected final String xms;
	protected final String xmx;
	protected final String maxPermSize;
	protected final List<String> params;

	protected Configuration(Builder<?> builder) {
		// Mandatory properties
		checkMandatoryProperty("directory", directory = builder.directory);
		checkMandatoryProperty("host", host = builder.host);
		checkMandatoryProperty("port", port = builder.port);
		checkMandatoryProperty("username", username = builder.username);
		checkMandatoryProperty("password", password = builder.password);

		// Optional properties
		xms = builder.xms;
		xmx = builder.xmx;
		maxPermSize = builder.maxPermSize;
		params = builder.params;
	}

	protected void checkMandatoryProperty(String name, Object value) {
		if (value == null) {
			throw new IllegalArgumentException(String.format("Property '%s' is mandatory", name));
		}
	}

	public File getDirectory() {
		return directory;
	}

	public String getHost() {
		return host;
	}

	public Integer getPort() {
		return port;
	}

	public String getUsername() {
		return username;
	}

	public String getPassword() {
		return password;
	}

	public String getXms() {
		return xms;
	}

	public String getXmx() {
		return xmx;
	}

	public String getMaxPermSize() {
		return maxPermSize;
	}

	public List<String> getParams() {
		return params;
	}

	public List<String> generateCommand() {
		final List<String> cmd = new ArrayList<>();

		// Add binary java file
		if (System.getProperty("java.home") == null) {
			throw new IllegalStateException("System property 'java.home' is not set");
		}
		final File javaHome = new File(System.getProperty("java.home"));
		final String javaExec = javaHome + File.separator + "bin" + File.separator + "java";
		cmd.add(javaExec);

		// Add optional properties
		if (xms != null) {
			cmd.add("-Xms" + xms);
		}
		if (xmx != null) {
			cmd.add("-Xmx" + xmx);
		}
		if (maxPermSize != null) {
			cmd.add("-XX:MaxPermSize=" + maxPermSize);
		}
		if (params != null) {
			cmd.addAll(params);
		}
		return cmd;
	}

	public abstract static class Builder<T extends Builder<T>> {

		protected File directory;
		protected String host;
		protected Integer port;
		protected String username;
		protected String password;
		protected String xms;
		protected String xmx;
		protected String maxPermSize;
		protected List<String> params;

		public Builder() {
			host = "localhost";
			port = 8080;
			username = "admin";
			password = "admin";
			xms = "64m";
			xmx = "256m";
			maxPermSize = "512m";
			params = new ArrayList<>();
		}

		protected abstract T self();

		public T directory(String directory) {
			this.directory = new File(directory);
			return self();
		}

		public T host(String host) {
			this.host = host;
			return self();
		}

		public T port(Integer port) {
			this.port = port;
			return self();
		}

		public T username(String username) {
			this.username = username;
			return self();
		}

		public T password(String password) {
			this.password = password;
			return self();
		}

		public T xms(String xms) {
			this.xms = xms;
			return self();
		}

		public T xmx(String xmx) {
			this.xmx = xmx;
			return self();
		}

		public T maxPermSize(String maxPermSize) {
			this.maxPermSize = maxPermSize;
			return self();
		}

		public T params(List<String> params) {
			this.params.addAll(params);
			return self();
		}

		public T param(String param) {
			this.params.add(param);
			return self();
		}
	}
}
